import { unitTest } from 'tests/utils/module';
import sinon from 'sinon';
import { waitFor } from 'tests/utils/waitFor';
import { setLoggedIn, expectStreamInfo, expectVideoInfo,
         expectChannelInfo } from 'tests/utils/api';
import { ACTION_SET_STREAMMETADATA, setVODStreamMetadata, FETCH_METADATA_TIMEOUT,
         setClipStreamMetadata, setLiveStreamMetadata, fetchLiveStreamMetadata,
         fetchAndSetStreamMetadata } from 'actions/stream-metadata';
import { ACTION_UPDATE_VIEWERCOUNT } from 'actions/viewercount';
import { LiveTwitchContentStream } from 'stream/twitch-live';
import { VODTwitchContentStream } from 'stream/twitch-vod';
import { DEFAULT_STREAM_METADATA } from 'tests/fixtures/stream-metadata';
import { CLIP_RESPONSE_FULL, CLIP_RESPONSE_MINIMUM, createClipMetadataFromResponse } from 'tests/fixtures/clip';

const channelInfo = Object.freeze({
    // eslint-disable-next-line camelcase
    display_name: 'Monstercat',
    name: 'monstercat',
    _id: 123456,
    partner: true,
    status: 'playing music or something',
    // eslint-disable-next-line camelcase
    video_banner: 'https://twitch.tv/monstercat/banner',
    logo: 'https://twitch.tv/monstercat/logo',
    url: 'https://twitch.tv/monstercat/',
    game: 'Music',
    mature: false,
});

const normalizedChannelInfo = Object.freeze({
    channel: {
        displayName: channelInfo.display_name,
        name: channelInfo.name,
        id: channelInfo._id,
        partner: channelInfo.partner,
        status: channelInfo.status,
        videoBanner: channelInfo.video_banner,
        logo: channelInfo.logo,
        mature: channelInfo.mature,
    },
    game: channelInfo.game,
    url: channelInfo.url,
});

unitTest('actions | stream-metadata', function() {
    QUnit.test('setVODStreamMetadata should update streamMetadata with normalized videoInfo data', function(assert) {
        const videoInfo = {
            /* eslint-disable camelcase */
            broadcast_id: 0,
            broadcast_type: 'archive',
            title: 'This is a VOD',
            channel: {
                name: 'channelName',
                _id: 123456,
                display_name: 'displayName',
                partner: false,
                status: 'status',
                video_banner: 'banner',
                logo: 'logo',
                mature: false,
            },
            game: 'LoL',
            created_at: 'some time',
            url: 'https://twitch.tv/channelName/v/12345',
            preview: 'some preview',
            /* eslint-enable camelcase */
        };
        const expectedStreamMetadata = {
            broadcastID: videoInfo.broadcast_id,
            type: videoInfo.broadcast_type,
            name: videoInfo.title,
            channel: {
                displayName: videoInfo.channel.display_name,
                name: videoInfo.channel.name,
                id: videoInfo.channel._id,
                partner: false,
                status: videoInfo.status,
                videoBanner: videoInfo.channel.video_banner,
                logo: videoInfo.channel.logo,
                mature: videoInfo.channel.mature,
            },
            createdAt: videoInfo.created_at,
            game: videoInfo.game,
            preview: {
                template: 'some preview',
            },
            url: 'https://twitch.tv/channelName/v/12345',
            viewers: 0,
        };
        const action = setVODStreamMetadata(videoInfo);

        assert.deepEqual(action, {
            type: ACTION_SET_STREAMMETADATA,
            streamMetadata: expectedStreamMetadata,
        });
    });

    QUnit.test('setLiveStreamMetadata should update streamMetadata with pertinent streamInfo data', function(assert) {
        /* eslint-disable camelcase */
        const streamInfo = {
            _id: 0,
            created_at: '2016-08-12T17:22:02Z',
            game: 'Music',
            preview: {
                template: 'template',
            },
            channel: channelInfo,
            viewers: 123,
            stream_type: 'live',
            mature: false,
        };
        /* eslint-enable camelcase */
        const expectedStreamMetadata = {
            broadcastID: streamInfo._id,
            type: 'live',
            name: '',
            channel: {
                displayName: streamInfo.channel.display_name,
                name: streamInfo.channel.name,
                id: streamInfo.channel._id,
                partner: streamInfo.channel.partner,
                status: streamInfo.channel.status,
                videoBanner: streamInfo.channel.video_banner,
                logo: streamInfo.channel.logo,
                mature: streamInfo.channel.mature,
            },
            createdAt: streamInfo.created_at,
            game: streamInfo.game,
            preview: {
                template: streamInfo.preview.template,
            },
            streamType: streamInfo.stream_type,
            viewers: 123,
            url: streamInfo.channel.url,
        };
        const action = setLiveStreamMetadata(streamInfo);

        assert.deepEqual(action, {
            type: ACTION_SET_STREAMMETADATA,
            streamMetadata: expectedStreamMetadata,
        });
    });

    QUnit.module('fetchLiveStreamMetadata', function(hooks) {
        hooks.beforeEach(function() {
            this.api.setLoggedIn(false);
            /* eslint-disable camelcase */
            this.responseInfo = {
                _id: 0,
                created_at: '2016-08-12T17:22:02Z',
                game: 'Music',
                preview: {
                    template: 'template',
                },
            };
            /* eslint-enable camelcase */
        });

        QUnit.test('fetchLiveStreamMetadata should return a function', function(assert) {
            const action = fetchLiveStreamMetadata();
            assert.equal(typeof(action), 'function');
        });

        QUnit.test('fetchLiveStreamMetadata should update streamMetadata property upon success', function(assert) {
            const broadcastID = parseInt(QUnit.config.current.testId, 36);
            this.responseInfo._id = broadcastID;

            const streamData = expectStreamInfo(
                channelInfo.name,
                true,
                channelInfo,
                this.responseInfo
            );

            /* eslint-disable camelcase */
            const expectedStreamMetadata = {
                broadcastID: this.responseInfo._id,
                channel: normalizedChannelInfo.channel,
                createdAt: this.responseInfo.created_at,
                game: this.responseInfo.game,
                preview: {
                    template: this.responseInfo.preview.template,
                },
                url: normalizedChannelInfo.url,
                viewers: streamData.stream.viewers,
                name: '',
                type: 'live',
                streamType: 'live',
            };
            /* eslint-enable camelcase */
            const state = {
                window: {},
                streamMetadata: DEFAULT_STREAM_METADATA,
                stream: new LiveTwitchContentStream(channelInfo.name, Promise.resolve({}), {}),
            };
            const getState = () => {
                return state;
            };
            const dispatch = sinon.spy();
            const setStreamMetadataSpy = dispatch.withArgs(sinon.match({
                type: ACTION_SET_STREAMMETADATA,
            }));
            const updateViewerCountSpy = dispatch.withArgs(sinon.match({
                type: ACTION_UPDATE_VIEWERCOUNT,
            }));

            const action = fetchLiveStreamMetadata(broadcastID);
            action(dispatch, getState);
            return waitFor(() => setStreamMetadataSpy.called && updateViewerCountSpy.called).then(() => {
                assert.deepEqual(
                    setStreamMetadataSpy.firstCall.args[0],
                    {
                        type: ACTION_SET_STREAMMETADATA,
                        streamMetadata: expectedStreamMetadata,
                    },
                    'should set stream metaata with new info'
                );
                assert.deepEqual(
                    updateViewerCountSpy.firstCall.args[0],
                    {
                        type: ACTION_UPDATE_VIEWERCOUNT,
                        count: streamData.stream.viewers,
                    },
                    'should update viewer count with new info'
                );
            });
        });

        QUnit.test('fetchLiveStreamMetadata should re-run upon null stream value', function(assert) {
            const broadcastID = parseInt(QUnit.config.current.testId, 36);
            const setTimeout = sinon.spy();
            const state = {
                window: {
                    setTimeout,
                },
                streamMetadata: DEFAULT_STREAM_METADATA,
                stream: new LiveTwitchContentStream(channelInfo.name, Promise.resolve({}), {}),
            };
            expectStreamInfo(channelInfo.name, false, channelInfo, this.responseInfo);
            const getState = () => {
                return state;
            };
            const dispatch = sinon.spy();

            const action = fetchLiveStreamMetadata(broadcastID);
            action(dispatch, getState);

            return waitFor(() => setTimeout.called).then(() => {
                assert.equal(setTimeout.callCount, 1);
                assert.ok(setTimeout.calledWith(sinon.match.func, FETCH_METADATA_TIMEOUT[0])); // first timeout = 15s
            });
        });

        QUnit.test('fetchLiveStreamMetadata should re-run upon api response with wrong id', function(assert) {
            const broadcastID = parseInt(QUnit.config.current.testId, 36);
            this.responseInfo._id = broadcastID + 5;
            const setTimeout = sinon.spy();
            const state = {
                window: {
                    setTimeout,
                },
                streamMetadata: DEFAULT_STREAM_METADATA,
                stream: new LiveTwitchContentStream(channelInfo.name, Promise.resolve({}), {}),
            };
            expectStreamInfo(channelInfo.name, true, channelInfo, this.responseInfo);
            const getState = () => {
                return state;
            };
            const dispatch = sinon.spy();

            const action = fetchLiveStreamMetadata(broadcastID);
            action(dispatch, getState);

            return waitFor(() => setTimeout.called).then(() => {
                assert.equal(setTimeout.callCount, 1);
                assert.ok(setTimeout.calledWith(sinon.match.func, FETCH_METADATA_TIMEOUT[0]));
            });
        });
    });

    QUnit.module('fetchAndSetStreamMetadata', function(hooks) {
        hooks.beforeEach(function() {
            setLoggedIn(true);

            this.getState = () => {
                return {};
            };
        });

        QUnit.module('for given live stream', function(hooks) {
            hooks.beforeEach(function() {
                this.streamInfo = {
                    /* eslint-disable camelcase */
                    _id: 123456,
                    created_at: 'some time',
                    game: 'Music',
                    preview: {
                        template: 'https://www.twitch.tv/some_template',
                    },
                    url: 'https://www.twitch.tv/monstercat',
                    viewers: 12345,
                    /* eslint-enable camelcase */
                };
                this.streamResponse = expectStreamInfo(
                    channelInfo.name,
                    true,
                    channelInfo,
                    this.streamInfo
                );
                this.stream = new LiveTwitchContentStream(channelInfo.name, Promise.resolve({}), {}, {}, {});
            });

            QUnit.test('should dispatch setLiveStreamMetadata with channel info', function(assert) {
                const dispatch = sinon.spy();
                const fetchAndSetAction = fetchAndSetStreamMetadata(this.stream);
                fetchAndSetAction(dispatch, this.getState);
                return waitFor(() => dispatch.called).then(() => {
                    assert.ok(dispatch.calledWith(setLiveStreamMetadata(this.streamResponse.stream)));
                });
            });
        });

        QUnit.module('for given offline stream', function(hooks) {
            hooks.beforeEach(function() {
                this.streamResponse = expectStreamInfo(
                    channelInfo.name,
                    false,
                    channelInfo
                );
                expectChannelInfo(channelInfo.name, channelInfo);

                this.stream = new LiveTwitchContentStream(channelInfo.name, Promise.resolve({}), {}, {}, {});
            });

            QUnit.test('should dispatch ACTION_SET_STREAMMETADATA with normalized channel info', function(assert) {
                const dispatch = sinon.spy();
                const fetchAndSetAction = fetchAndSetStreamMetadata(this.stream);
                fetchAndSetAction(dispatch, this.getState);
                return waitFor(() => dispatch.called).then(() => {
                    assert.deepEqual(
                        dispatch.firstCall.args[0],
                        {
                            type: ACTION_SET_STREAMMETADATA,
                            streamMetadata: normalizedChannelInfo,
                        },
                        'should dispatch ACTION_SET_STREAMMETADATA with normalized channel info');
                });
            });
        });

        QUnit.module('for given vod stream', function(hooks) {
            hooks.beforeEach(function() {
                this.vodId = 'v12345';
                this.vodInfo = {
                    /* eslint-disable camelcase */
                    broadcast_id: 0,
                    broadcast_type: 'archive',
                    title: 'Some music VOD',
                    channel: {
                        name: 'monstercat',
                        _id: 123456,
                        display_name: 'Monstercat',
                        partner: true,
                        status: 'I played music!',
                        video_banner: 'https://twitch.tv/monstercat/banner',
                        logo: 'https://twitch.tv/monstercat/logo',
                        mature: false,
                    },
                    game: 'Music',
                    created_at: 'some time',
                    url: 'https://twitch.tv/monstercat/v/12345',
                    preview: 'https://www.twitch.tv/some_template',
                    /* eslint-enable camelcase */
                };

                // channel api is called to fetch more metadata for video
                expectChannelInfo(this.vodInfo.channel.name, channelInfo);
                this.vodResponse = expectVideoInfo(this.vodId, this.vodInfo);

                this.stream = new VODTwitchContentStream(this.vodId, Promise.resolve({}), {}, {}, {});
            });

            QUnit.test('should dispatch setVODStreamMetadata with video info', function(assert) {
                const dispatch = sinon.spy();
                const fetchAndSetAction = fetchAndSetStreamMetadata(this.stream);
                fetchAndSetAction(dispatch, this.getState);
                return waitFor(() => dispatch.called).then(() => {
                    assert.deepEqual(
                        dispatch.firstCall.args[0],
                        setVODStreamMetadata(this.vodResponse),
                        'should dispatch ACTION_SET_STREAMMETADATA with normalized videoInfo');
                });
            });
        });
    });

    QUnit.test('setClipStreamMetadata should update streamMetadata with clip metadata', function(assert) {
        const action = setClipStreamMetadata(CLIP_RESPONSE_FULL);

        assert.deepEqual(action, {
            type: ACTION_SET_STREAMMETADATA,
            streamMetadata: createClipMetadataFromResponse(CLIP_RESPONSE_FULL),
        });
    });

    QUnit.test('setClipStreamMetadata should not error with minimum clip data', function(assert) {
        const action = setClipStreamMetadata(CLIP_RESPONSE_MINIMUM);

        assert.deepEqual(action, {
            type: ACTION_SET_STREAMMETADATA,
            streamMetadata: createClipMetadataFromResponse(CLIP_RESPONSE_MINIMUM),
        });
    });
});
