import sinon from 'sinon';
import { unitTest } from 'tests/utils/module';
import isFunction from 'lodash/isFunction';
import { ACTION_SET_QUALITIES } from 'actions/quality';
import * as StreamActions from 'actions/stream';
import * as StreamMetadataActions from 'actions/stream-metadata';
import * as ExtensionsActions from 'actions/extensions';
import * as AnalyticsActions from 'actions/analytics';
import { trackEvent } from 'actions/analytics-tracker';
import { VIDEO_PLAY_LOAD_START, VIDEO_PLAY_OAUTH, VIDEO_PLAY_NAUTH } from 'analytics/analytics';
import { LiveTwitchContentStream, CONTENT_MODE_LIVE } from 'stream/twitch-live';
import { VODTwitchContentStream, CONTENT_MODE_VOD } from 'stream/twitch-vod';
import { VideoSourceContentStream } from 'stream/video-source';
import { ProvidedContentStream } from 'stream/provided';
import { ClipContentStream } from 'stream/clip';
import * as PlayerType from 'util/player-type';
import { waitFor } from 'tests/utils/waitFor';
import { CLIP_RESPONSE_MINIMUM } from 'tests/fixtures/clip';

unitTest('actions | stream', function(hooks) {
    hooks.beforeEach(function() {
        this.usherParams = {
            godlike: true,
        };
        this.usherHostOverride = '';
        this.accessToken = {
            params: {
                // eslint-disable-next-line camelcase
                need_https: false,
            },
        };
        this.getState = () => {
            return {
                ads: {
                    adblock: false,
                },
                analytics: {
                    playSessionStartTime: 0,
                },
                env: {
                    playerType: PlayerType.PLAYER_SITE,
                    platform: 'web-test',
                },
                experiments: {
                    get() {
                        return Promise.resolve();
                    },
                },
                playback: {
                    autoplay: true,
                },
                usher: {
                    hostOverride: this.usherHostOverride,
                    params: this.usherParams,
                },
                accessToken: this.accessToken,
            };
        };

        this.dateNow = parseInt(QUnit.config.current.testId, 36);
        sinon.stub(Date, 'now').returns(this.dateNow);

        sinon.spy(StreamMetadataActions, 'fetchAndSetStreamMetadata');
        sinon.spy(ExtensionsActions, 'fetchExtensions');
        sinon.spy(ExtensionsActions, 'subscribeToExtensionControl');

        // Helper function to wait until end of `setStream`
        // 2 calls for metadata: oauth token and access token
        this.waitForSetStream = testsFn => {
            return waitFor(() => this.api.calls().matched.length === 2).then(testsFn);
        };
    });

    hooks.afterEach(function() {
        Date.now.restore();
        StreamMetadataActions.fetchAndSetStreamMetadata.restore();
        ExtensionsActions.fetchExtensions.restore();
        ExtensionsActions.subscribeToExtensionControl.restore();
    });

    QUnit.module('setStream (channel)', function(hooks) {
        hooks.beforeEach(function() {
            this.channelName = `channel_${QUnit.config.current.testId}`;

            this.api.setLoggedIn(true);
            this.api.expectUserInfo({
                geo: 'US',
            });
            this.api.expectChannelAccessToken(this.channelName, {
                canAccess: true,
                accessTokenParams: {},
            });
        });

        QUnit.test('should asynchronously dispatch an action to set the stream', function(assert) {
            const dispatch = sinon.spy();
            const action = StreamActions.setStream({
                contentType: StreamActions.TYPE_CHANNEL,
                contentId: this.channelName,
            });

            action(dispatch, this.getState);

            return this.waitForSetStream(() => {
                // dispatch calls: reset play session, set stream, fetch and set metadata
                assert.equal(dispatch.callCount, 8, 'should call dispatch 8 times');
                assert.ok(dispatch.calledWith(sinon.match({
                    type: StreamActions.ACTION_SET_STREAM,
                    stream: sinon.match.instanceOf(LiveTwitchContentStream).and(sinon.match({
                        channel: this.channelName,
                    })),
                })));
            });
        });

        QUnit.test('should dispatch an action to reset play session', function(assert) {
            const dispatch = sinon.spy();
            const action = StreamActions.setStream({
                contentType: StreamActions.TYPE_CHANNEL,
                contentId: this.channelName,
            });

            action(dispatch, this.getState);

            return this.waitForSetStream(() => {
                assert.ok(dispatch.calledWithMatch({
                    type: AnalyticsActions.ACTION_RESET_PLAY_SESSION,
                }));
            });
        });

        QUnit.test('should send tracking event video_play_load_start', function(assert) {
            const dispatch = sinon.spy();
            const action = StreamActions.setStream({
                contentType: StreamActions.TYPE_CHANNEL,
                contentId: this.channelName,
            });

            action(dispatch, this.getState);

            return this.waitForSetStream(() => {
                assert.ok(dispatch.calledWith(trackEvent(VIDEO_PLAY_LOAD_START, sinon.match({
                    channel: this.channelName,
                    autoplay: sinon.match.bool,
                }))));

                assert.ok(!dispatch.calledWith(trackEvent(VIDEO_PLAY_LOAD_START, sinon.match({
                    // eslint-disable-next-line camelcase
                    vod_id: sinon.match.defined,
                }))));
            });
        });

        QUnit.test('should send tracking event video_play_oauth', function(assert) {
            const dispatch = sinon.spy();
            const action = StreamActions.setStream({
                contentType: StreamActions.TYPE_CHANNEL,
                contentId: this.channelName,
            });

            action(dispatch, this.getState);

            return this.waitForSetStream(() => {
                assert.ok(dispatch.calledWith(trackEvent(VIDEO_PLAY_OAUTH, sinon.match({
                    // eslint-disable-next-line camelcase
                    time_since_load_start: this.dateNow,
                }))));
            });
        });

        QUnit.test('should send tracking event video_play_nauth when `streamUrl` is accessed', function(assert) {
            const dispatch = sinon.spy();
            const action = StreamActions.setStream({
                contentType: StreamActions.TYPE_CHANNEL,
                contentId: this.channelName,
            });

            action(dispatch, this.getState);

            return this.waitForSetStream(() => {
                const stream = dispatch.thirdCall.args[0].stream;
                stream.streamUrl.then(() => {
                    assert.ok(dispatch.calledWith(trackEvent(VIDEO_PLAY_NAUTH, sinon.match({
                        // eslint-disable-next-line camelcase
                        time_since_load_start: this.dateNow,
                    }))));
                });
            });
        });

        QUnit.test('should sanitize the input channel name', function(assert) {
            const invalidChannel = 'lol@th!$name';
            const sanitizedChannel = 'lolthname';

            this.api.expectChannelAccessToken(sanitizedChannel, {
                canAccess: true,
                accessTokenParams: {},
            });

            const dispatch = sinon.spy();
            const action = StreamActions.setStream({
                contentType: StreamActions.TYPE_CHANNEL,
                contentId: invalidChannel,
            });

            action(dispatch, this.getState);

            return this.waitForSetStream(() => {
                assert.ok(dispatch.calledWithMatch({
                    stream: {
                        channel: sanitizedChannel,
                    },
                }));
            });
        });

        QUnit.test('passes along usher configuration information', function(assert) {
            const dispatch = sinon.spy();
            const action = StreamActions.setStream({
                contentType: StreamActions.TYPE_CHANNEL,
                contentId: this.channelName,
            });

            action(dispatch, this.getState);

            return this.waitForSetStream(() => {
                const stream = dispatch.thirdCall.args[0].stream;
                assert.deepEqual(stream._usherParams, this.usherParams);
            });
        });

        QUnit.test('passes along access token parameters', function(assert) {
            const dispatch = sinon.spy();
            const action = StreamActions.setStream({
                contentType: StreamActions.TYPE_CHANNEL,
                contentId: this.channelName,
            });

            action(dispatch, this.getState);

            return this.waitForSetStream(() => {
                const stream = dispatch.thirdCall.args[0].stream;
                assert.deepEqual(stream._accessTokenParams, this.accessToken.params);
            });
        });

        QUnit.test('should dispatch an action to fetch and set stream metadata', function(assert) {
            const dispatch = sinon.spy();
            const action = StreamActions.setStream({
                contentType: StreamActions.TYPE_CHANNEL,
                contentId: this.channelName,
            });

            action(dispatch, this.getState);

            return this.waitForSetStream(() => {
                // eslint-disable-next-line max-len
                assert.equal(StreamMetadataActions.fetchAndSetStreamMetadata.callCount, 1, 'fetch and set action called once');
                const [[stream]] = StreamMetadataActions.fetchAndSetStreamMetadata.args;
                assert.equal(stream.contentType, CONTENT_MODE_LIVE, 'stream passed is of live type');
                assert.equal(stream.channel, this.channelName, 'stream passed includes correct channel name');
            });
        });

        QUnit.test('should dispatch an action to fetch extensions', function(assert) {
            const dispatch = sinon.spy();
            const action = StreamActions.setStream({
                contentType: StreamActions.TYPE_CHANNEL,
                contentId: this.channelName,
            });

            action(dispatch, this.getState);

            return this.waitForSetStream(() => {
                // eslint-disable-next-line max-len
                assert.equal(ExtensionsActions.fetchExtensions.callCount, 1, 'fetch extensions action is called once');
                const [[channel]] = ExtensionsActions.fetchExtensions.args;
                assert.equal(channel, this.channelName, 'the current channel name is passed');
            });
        });

        QUnit.test('should dispatch an action to subscibe to extension control', function(assert) {
            const dispatch = sinon.spy();
            const action = StreamActions.setStream({
                contentType: StreamActions.TYPE_CHANNEL,
                contentId: this.channelName,
            });

            action(dispatch, this.getState);

            return this.waitForSetStream(() => {
                // eslint-disable-next-line max-len
                assert.equal(ExtensionsActions.subscribeToExtensionControl.callCount, 1, 'subscribe to extension control action once');
                const [[channel]] = ExtensionsActions.subscribeToExtensionControl.args;
                assert.equal(channel, this.channelName, 'the current channel name is passed');
            });
        });

        QUnit.module('when logged in', function(hooks) {
            hooks.beforeEach(function() {
                this.oauthResponse = this.api.setLoggedIn(true);
            });

            QUnit.test('should pass the oauth token to the live stream', function(assert) {
                const dispatch = sinon.spy();
                const action = StreamActions.setStream({
                    contentType: StreamActions.TYPE_CHANNEL,
                    contentId: this.channelName,
                });

                action(dispatch, this.getState);

                return this.waitForSetStream(() => {
                    // TODO: make order agnostic if possible
                    return dispatch.thirdCall.args[0].stream._oAuthToken.then(token => {
                        assert.equal(token, this.oauthResponse.token);
                    });
                });
            });
        });

        QUnit.module('when not logged in', function(hooks) {
            hooks.beforeEach(function() {
                this.api.clearFakeResponses();
                this.api.expectChannelAccessToken(this.channelName, {
                    canAccess: true,
                    accessTokenParams: {},
                });
                this.oauthResponse = this.api.setLoggedIn(false);
            });

            QUnit.test('should pass an empty oauth token to the live stream', function(assert) {
                const dispatch = sinon.spy();
                const action = StreamActions.setStream({
                    contentType: StreamActions.TYPE_CHANNEL,
                    contentId: this.channelName,
                });

                action(dispatch, this.getState);

                return this.waitForSetStream(() => {
                    // TODO: make order agnostic if possible
                    return dispatch.thirdCall.args[0].stream._oAuthToken.then(token => {
                        assert.equal(token, null);
                    });
                });
            });
        });
    });

    QUnit.module('setStream (video)', function(hooks) {
        hooks.beforeEach(function() {
            this.vodId = `v${QUnit.config.current.testId}`;

            this.api.setLoggedIn(true);
            this.api.expectUserInfo({
                geo: 'US',
            });
            this.api.expectVodAccessToken(this.vodId, {
                canAccess: true,
                accessTokenParams: {},
            });
        });

        QUnit.test('should asynchronously dispatch an action to set the stream', function(assert) {
            const dispatch = sinon.spy();
            const action = StreamActions.setStream({
                contentType: StreamActions.TYPE_VIDEO,
                contentId: this.vodId,
            });

            assert.ok(isFunction(action));
            action(dispatch, this.getState);

            return this.waitForSetStream(() => {
                assert.equal(dispatch.callCount, 8);
                dispatch.calledWith(sinon.match({
                    type: StreamActions.ACTION_SET_STREAM,
                    stream: sinon.match.instanceOf(VODTwitchContentStream).and(sinon.match({
                        videoId: this.vodId,
                    })),
                }));
            });
        });

        QUnit.test('should sanitize the input VOD id', function(assert) {
            const invalidVodId = 'v*d$r$w33t';
            const sanitizedVodId = 'vdrw33t';

            this.api.expectVodAccessToken(sanitizedVodId, {
                canAccess: true,
                accessTokenParams: {},
            });

            const dispatch = sinon.spy();
            const action = StreamActions.setStream({
                contentType: StreamActions.TYPE_VIDEO,
                contentId: invalidVodId,
            });

            action(dispatch, this.getState);

            return this.waitForSetStream(() => {
                assert.ok(dispatch.calledWithMatch({
                    stream: {
                        videoId: sanitizedVodId,
                    },
                }));
            });
        });

        QUnit.test('should dispatch an action to reset play session', function(assert) {
            const dispatch = sinon.spy();
            const action = StreamActions.setStream({
                contentType: StreamActions.TYPE_VIDEO,
                contentId: this.vodId,
            });

            action(dispatch, this.getState);

            return this.waitForSetStream(() => {
                assert.ok(dispatch.calledWithMatch({
                    type: AnalyticsActions.ACTION_RESET_PLAY_SESSION,
                }));
            });
        });

        QUnit.test('should send tracking event video_play_load_start', function(assert) {
            const dispatch = sinon.spy();
            const action = StreamActions.setStream({
                contentType: StreamActions.TYPE_VIDEO,
                contentId: this.vodId,
            });

            action(dispatch, this.getState);

            return this.waitForSetStream(() => {
                assert.ok(dispatch.calledWith(trackEvent(VIDEO_PLAY_LOAD_START, sinon.match({
                    // eslint-disable-next-line camelcase
                    vod_id: this.vodId,
                    autoplay: sinon.match.bool,
                }))));

                assert.ok(!dispatch.calledWith(trackEvent(VIDEO_PLAY_LOAD_START, sinon.match({
                    channel: sinon.match.defined,
                }))));
            });
        });

        QUnit.test('should send tracking event video_play_oauth', function(assert) {
            const dispatch = sinon.spy();
            const action = StreamActions.setStream({
                contentType: StreamActions.TYPE_VIDEO,
                contentId: this.vodId,
            });

            action(dispatch, this.getState);

            return this.waitForSetStream(() => {
                assert.ok(dispatch.calledWith(trackEvent(VIDEO_PLAY_OAUTH, sinon.match({
                    // eslint-disable-next-line camelcase
                    time_since_load_start: this.dateNow,
                }))));
            });
        });

        QUnit.test('should send tracking event video_play_nauth when `streamUrl` is accessed', function(assert) {
            const dispatch = sinon.spy();
            const action = StreamActions.setStream({
                contentType: StreamActions.TYPE_VIDEO,
                contentId: this.vodId,
            });

            action(dispatch, this.getState);

            return this.waitForSetStream(() => {
                const stream = dispatch.thirdCall.args[0].stream;
                stream.streamUrl.then(() => {
                    assert.ok(dispatch.calledWith(trackEvent(VIDEO_PLAY_NAUTH, sinon.match({
                        // eslint-disable-next-line camelcase
                        time_since_load_start: this.dateNow,
                    }))));
                });
            });
        });

        QUnit.test('passes along usher configuration information', function(assert) {
            const dispatch = sinon.spy();
            const action = StreamActions.setStream({
                contentType: StreamActions.TYPE_VIDEO,
                contentId: this.vodId,
            });

            action(dispatch, this.getState);

            return this.waitForSetStream(() => {
                const stream = dispatch.thirdCall.args[0].stream;
                assert.deepEqual(stream._usherParams, this.usherParams);
            });
        });

        QUnit.test('passes along access token params', function(assert) {
            const dispatch = sinon.spy();
            const action = StreamActions.setStream({
                contentType: StreamActions.TYPE_VIDEO,
                contentId: this.vodId,
            });

            action(dispatch, this.getState);

            return this.waitForSetStream(() => {
                const stream = dispatch.thirdCall.args[0].stream;
                assert.deepEqual(stream._accessTokenParams, this.accessToken.params);
            });
        });

        QUnit.test('should dispatch an action to fetch and set stream metadata', function(assert) {
            const dispatch = sinon.spy();
            const action = StreamActions.setStream({
                contentType: StreamActions.TYPE_VIDEO,
                contentId: this.vodId,
            });

            action(dispatch, this.getState);

            return this.waitForSetStream(() => {
                // eslint-disable-next-line max-len
                assert.equal(StreamMetadataActions.fetchAndSetStreamMetadata.callCount, 1, 'fetch and set action called once');
                const [[stream]] = StreamMetadataActions.fetchAndSetStreamMetadata.args;
                assert.equal(stream.contentType, CONTENT_MODE_VOD, 'stream passed is of vod type');
                assert.equal(stream.videoId, this.vodId, 'stream passed includes correct vod id');
            });
        });

        QUnit.module('when logged in', function(hooks) {
            hooks.beforeEach(function() {
                this.oauthResponse = this.api.setLoggedIn(true);
            });

            QUnit.test('should pass the oauth token to the VOD stream', function(assert) {
                const dispatch = sinon.spy();
                const action = StreamActions.setStream({
                    contentType: StreamActions.TYPE_VIDEO,
                    contentId: this.vodId,
                });

                action(dispatch, this.getState);

                return this.waitForSetStream(() => {
                    // TODO: make order agnostic if possible
                    return dispatch.thirdCall.args[0].stream._oAuthToken.then(token => {
                        assert.equal(token, this.oauthResponse.token);
                    });
                });
            });
        });

        QUnit.module('when not logged in', function(hooks) {
            hooks.beforeEach(function() {
                this.api.clearFakeResponses();
                this.oauthResponse = this.api.setLoggedIn(false);
                this.api.expectVodAccessToken(this.vodId, {
                    canAccess: true,
                    accessTokenParams: {},
                });
            });

            QUnit.test('should pass an empty oauth token to the VOD stream', function(assert) {
                const dispatch = sinon.spy();
                const action = StreamActions.setStream({
                    contentType: StreamActions.TYPE_VIDEO,
                    contentId: this.vodId,
                });

                action(dispatch, this.getState);

                return this.waitForSetStream(() => {
                    // TODO: make order agnostic if possible
                    return dispatch.thirdCall.args[0].stream._oAuthToken.then(token => {
                        assert.equal(token, null);
                    });
                });
            });
        });
    });

    QUnit.module('setStream (provided)', function(hooks) {
        hooks.beforeEach(function() {
            this.contentUrl = 'fakecontenturl';
            this.contentId = 'acontentid';
            this.customerId = 'acustomerid';
            this.usherHostOverride = 'newhost';
            this.dispatchSpy = sinon.spy();
            this.waitForSetStream = testsFn => {
                return waitFor(() => this.dispatchSpy.callCount === 3).then(testsFn);
            };
        });

        QUnit.test('should asynchronously dispatch an action to set the stream', function(assert) {
            const action = StreamActions.setStream({
                contentType: StreamActions.TYPE_PROVIDED,
                contentId: this.contentId,
                customerId: this.customerId,
            });

            assert.ok(isFunction(action), 'is a thunk action');
            action(this.dispatchSpy, this.getState);

            return this.waitForSetStream(() => {
                this.dispatchSpy.calledWith(sinon.match({
                    type: StreamActions.ACTION_SET_STREAM,
                    stream: sinon.match.instanceOf(ProvidedContentStream).and(sinon.match({
                        contentId: this.contentId,
                        customerId: this.customerId,
                        usherHostOverride: this.usherHostOverride,
                    })),
                }));
            });
        });

        QUnit.test('should dispatch an action to reset play session', function(assert) {
            const action = StreamActions.setStream({
                contentType: StreamActions.TYPE_PROVIDED,
                contentId: this.contentId,
                customerId: this.customerId,
            });
            action(this.dispatchSpy, this.getState);

            return this.waitForSetStream(() => {
                assert.ok(this.dispatchSpy.calledWithMatch({
                    type: AnalyticsActions.ACTION_RESET_PLAY_SESSION,
                }));
            });
        });

        QUnit.test('should send tracking event video_play_load_start', function(assert) {
            const action = StreamActions.setStream({
                contentType: StreamActions.TYPE_PROVIDED,
                contentId: this.contentId,
                customerId: this.customerId,
            });
            action(this.dispatchSpy, this.getState);

            return this.waitForSetStream(() => {
                assert.ok(this.dispatchSpy.calledWith(trackEvent(VIDEO_PLAY_LOAD_START, sinon.match({
                    autoplay: sinon.match.bool,
                    content_id: this.contentId, // eslint-disable-line camelcase
                    customer_id: this.customerId, // eslint-disable-line camelcase
                }))));
            });
        });

        QUnit.test('should not send tracking event video_play_oauth', function(assert) {
            const action = StreamActions.setStream({
                contentType: StreamActions.TYPE_PROVIDED,
                contentId: this.contentId,
                customerId: this.customerId,
            });
            action(this.dispatchSpy, this.getState);

            return this.waitForSetStream(() => {
                assert.ok(!this.dispatchSpy.calledWith(trackEvent(VIDEO_PLAY_OAUTH, sinon.match({
                    // eslint-disable-next-line camelcase
                    time_since_load_start: this.dateNow,
                }))));
            });
        });

        QUnit.test('should not send tracking event video_play_nauth when `streamUrl` is accessed', function(assert) {
            const action = StreamActions.setStream({
                contentType: StreamActions.TYPE_PROVIDED,
                contentId: this.contentId,
                customerId: this.customerId,
            });
            action(this.dispatchSpy, this.getState);

            return this.waitForSetStream(() => {
                const stream = this.dispatchSpy.thirdCall.args[0].stream;
                stream.streamUrl.then(() => {
                    assert.ok(!this.dispatchSpy.calledWith(trackEvent(VIDEO_PLAY_NAUTH, sinon.match({
                        // eslint-disable-next-line camelcase
                        time_since_load_start: this.dateNow,
                    }))));
                });
            });
        });

        // eslint-disable-next-line max-len
        QUnit.test('should not dispatch actions for stream metadata, extensions, or extensions subscribe ', function(assert) {
            const action = StreamActions.setStream({
                contentType: StreamActions.TYPE_PROVIDED,
                contentId: this.contentId,
                customerId: this.customerId,
            });
            action(this.dispatchSpy, this.getState);

            return this.waitForSetStream(() => {
                // eslint-disable-next-line max-len
                assert.equal(StreamMetadataActions.fetchAndSetStreamMetadata.callCount, 0, 'fetch and set action not called');
                assert.equal(ExtensionsActions.fetchExtensions.callCount, 0, 'fetch extensions action not called');
                // eslint-disable-next-line max-len
                assert.equal(ExtensionsActions.subscribeToExtensionControl.callCount, 0, 'subscribe to extension control action not called');
            });
        });
    });

    QUnit.module('setStream (clip)', function(hooks) {
        hooks.beforeEach(function() {
            this.slug = 'SomeClipSlug';
            this.api.setLoggedIn(true);
            this.api.expectClipStatus(this.slug, {
                // eslint-disable-next-line camelcase
                quality_options: [],
            });
            this.api.expectClipInfo(CLIP_RESPONSE_MINIMUM);
            this.api.expectClipView(this.slug, {});
            this.dispatchSpy = sinon.spy();
        });

        QUnit.test('should asynchronously dispatch an action to set the stream', function(assert) {
            const action = StreamActions.setStream({
                contentType: StreamActions.TYPE_CLIP,
                contentId: this.slug,
            });

            assert.ok(isFunction(action), 'is a thunk action');
            return action(this.dispatchSpy, this.getState).then(() => {
                this.dispatchSpy.calledWith(sinon.match({
                    type: StreamActions.ACTION_SET_STREAM,
                    stream: sinon.match.instanceOf(ClipContentStream).and(sinon.match({
                        contentId: this.slug,
                    })),
                }));
            });
        });

        QUnit.test('should dispatch an action to reset play session', function(assert) {
            const action = StreamActions.setStream({
                contentType: StreamActions.TYPE_CLIP,
                contentId: this.slug,
            });

            return action(this.dispatchSpy, this.getState).then(() => {
                assert.ok(this.dispatchSpy.calledWithMatch({
                    type: AnalyticsActions.ACTION_RESET_PLAY_SESSION,
                }));
            });
        });

        QUnit.test('should send tracking event video_play_load_start', function(assert) {
            const action = StreamActions.setStream({
                contentType: StreamActions.TYPE_CLIP,
                contentId: this.slug,
            });

            return action(this.dispatchSpy, this.getState).then(() => {
                assert.ok(this.dispatchSpy.calledWith(trackEvent(VIDEO_PLAY_LOAD_START, sinon.match({
                    autoplay: sinon.match.bool,
                }))));
            });
        });

        QUnit.test('should not send tracking event video_play_oauth', function(assert) {
            const action = StreamActions.setStream({
                contentType: StreamActions.TYPE_CLIP,
                contentId: this.slug,
            });

            return action(this.dispatchSpy, this.getState).then(() => {
                assert.ok(!this.dispatchSpy.calledWith(trackEvent(VIDEO_PLAY_OAUTH, sinon.match({
                    // eslint-disable-next-line camelcase
                    time_since_load_start: this.dateNow,
                }))));
            });
        });

        QUnit.test('should not send tracking event video_play_nauth when `streamUrl` is accessed', function(assert) {
            const action = StreamActions.setStream({
                contentType: StreamActions.TYPE_CLIP,
                contentId: this.slug,
            });

            return action(this.dispatchSpy, this.getState).then(() => {
                const stream = this.dispatchSpy.getCall(3).args[0].stream;
                stream.streamUrl.then(() => {
                    assert.ok(!this.dispatchSpy.calledWith(trackEvent(VIDEO_PLAY_NAUTH, sinon.match({
                        // eslint-disable-next-line camelcase
                        time_since_load_start: this.dateNow,
                    }))));
                });
            });
        });

        QUnit.test('should dispatch action for set quality', function(assert) {
            const action = StreamActions.setStream({
                contentType: StreamActions.TYPE_CLIP,
                contentId: this.slug,
            });

            return action(this.dispatchSpy, this.getState).then(() => {
                const qualities = this.dispatchSpy.getCall(2).args[0];
                assert.equal(qualities.type, ACTION_SET_QUALITIES);
            });
        });

        QUnit.test('should dispatch action for stream metadata', function(assert) {
            const action = StreamActions.setStream({
                contentType: StreamActions.TYPE_CLIP,
                contentId: this.slug,
            });

            return action(this.dispatchSpy, this.getState).then(() => {
                assert.ok(this.dispatchSpy.calledWithMatch({
                    type: StreamMetadataActions.ACTION_SET_STREAMMETADATA,
                }));
            });
        });
    });

    QUnit.module('setStream (video-source)', function(hooks) {
        hooks.beforeEach(function() {
            this.url = 'https://twitch.tv/some.mp4';
            this.dispatchSpy = sinon.spy();
        });

        QUnit.test('should asynchronously dispatch an action to set the stream', function(assert) {
            const action = StreamActions.setStream({
                contentType: StreamActions.TYPE_VIDEO_SOURCE,
                contentId: this.url,
            });

            assert.ok(isFunction(action), 'is a thunk action');
            action(this.dispatchSpy, this.getState);
            this.dispatchSpy.calledWith(sinon.match({
                type: StreamActions.ACTION_SET_STREAM,
                stream: sinon.match.instanceOf(VideoSourceContentStream).and(sinon.match({
                    contentId: this.url,
                })),
            }));
        });

        QUnit.test('should dispatch an action to reset play session', function(assert) {
            const action = StreamActions.setStream({
                contentType: StreamActions.TYPE_VIDEO_SOURCE,
                contentId: this.url,
            });

            action(this.dispatchSpy, this.getState);
            assert.ok(this.dispatchSpy.calledWithMatch({
                type: AnalyticsActions.ACTION_RESET_PLAY_SESSION,
            }));
        });

        QUnit.test('should send tracking event video_play_load_start', function(assert) {
            const action = StreamActions.setStream({
                contentType: StreamActions.TYPE_VIDEO_SOURCE,
                contentId: this.url,
            });

            action(this.dispatchSpy, this.getState);
            assert.ok(this.dispatchSpy.calledWith(trackEvent(VIDEO_PLAY_LOAD_START, sinon.match({
                autoplay: sinon.match.bool,
            }))));
        });

        QUnit.test('should not send tracking event video_play_oauth', function(assert) {
            const action = StreamActions.setStream({
                contentType: StreamActions.TYPE_VIDEO_SOURCE,
                contentId: this.url,
            });

            action(this.dispatchSpy, this.getState);
            assert.ok(!this.dispatchSpy.calledWith(trackEvent(VIDEO_PLAY_OAUTH, sinon.match({
                // eslint-disable-next-line camelcase
                time_since_load_start: this.dateNow,
            }))));
        });

        QUnit.test('should not send tracking event video_play_nauth when `streamUrl` is accessed', function(assert) {
            const action = StreamActions.setStream({
                contentType: StreamActions.TYPE_VIDEO_SOURCE,
                contentId: this.url,
            });

            action(this.dispatchSpy, this.getState);
            const stream = this.dispatchSpy.getCall(2).args[0].stream;
            return stream.streamUrl.then(() => {
                assert.ok(!this.dispatchSpy.calledWith(trackEvent(VIDEO_PLAY_NAUTH, sinon.match({
                    // eslint-disable-next-line camelcase
                    time_since_load_start: this.dateNow,
                }))));
            });
        });
    });
});
