import * as API from 'api';
import sinon from 'sinon';
import { fetchChannelSubscriptions, fetchSubscriptionProducts, ACTION_CHANNEL_SUBSCRIPTIONS_FETCHED,
         ACTION_SUBSCRIPTION_PRODUCTS_FETCHED } from 'actions/subscriptions';

QUnit.module('actions | subscriptions', function(hooks) {
    hooks.beforeEach(function() {
        this.sandbox = sinon.sandbox.create();
        const userId = 123;
        const channelId = 321;
        const channelName = 'foo';

        this.state = {
            streamMetadata: {
                channel: {
                    id: channelId,
                    name: channelName,
                },
            },
        };

        this.getState = () => this.state;

        this.dispatch = sinon.spy();

        this.dispatchFetchChannelSubcriptions = () => {
            return fetchChannelSubscriptions(userId, channelId)(this.dispatch, this.getState);
        };

        this.dispatchFetchSubscriptionProducts = () => {
            return fetchSubscriptionProducts(channelName)(this.dispatch, this.getState);
        };
    });

    hooks.afterEach(function() {
        this.sandbox.restore();
    });

    QUnit.module('fetchChannelSubscriptions', function() {
        QUnit.test('dispatches subscribed state values when successful', function(assert) {
            this.sandbox.stub(API, 'getSubscriptionInfo', () => Promise.resolve());

            return this.dispatchFetchChannelSubcriptions().then(() => {
                assert.ok(this.dispatch.calledWith({
                    type: ACTION_CHANNEL_SUBSCRIPTIONS_FETCHED,
                    subscriptions: {
                        isSubscribed: true,
                    },
                }), 'dispatched channelSubscriptionsFetched with isSubscribed and isSubscribable to be true');
            });
        });

        QUnit.test('dispatches not subscribed state values when request returns an error', function(assert) {
            this.sandbox.stub(API, 'getSubscriptionInfo', () => Promise.reject());

            return this.dispatchFetchChannelSubcriptions().then(() => {
                assert.ok(this.dispatch.calledWith({
                    type: ACTION_CHANNEL_SUBSCRIPTIONS_FETCHED,
                    subscriptions: {
                        isSubscribed: false,
                    },
                }), 'dispatched channelSubscriptionsFetched with isSubscribed: false');
            });
        });

        QUnit.test('dispatches nothing on success when channel has changed', function(assert) {
            this.sandbox.stub(API, 'getSubscriptionInfo', () => Promise.resolve());

            this.state.streamMetadata.channel.id = 42;

            return this.dispatchFetchChannelSubcriptions().then(() => {
                assert.equal(this.dispatch.callCount, 0, 'dispatch not called');
            });
        });

        QUnit.test('dispatches nothing on failure when channel has changed', function(assert) {
            this.sandbox.stub(API, 'getSubscriptionInfo', () => Promise.reject());

            this.state.streamMetadata.channel.id = 42;

            return this.dispatchFetchChannelSubcriptions().then(() => {
                assert.equal(this.dispatch.callCount, 0, 'dispatch not called');
            });
        });
    });

    QUnit.module('fetchSubscriptionProducts', function() {
        QUnit.test('dispatches subscribability of channel when successful', function(assert) {
            this.sandbox.stub(API, 'getSubscriptionProducts', () => Promise.resolve());

            return this.dispatchFetchSubscriptionProducts().then(() => {
                assert.ok(this.dispatch.calledWith({
                    type: ACTION_SUBSCRIPTION_PRODUCTS_FETCHED,
                    subscriptions: {
                        isSubscribable: true,
                    },
                }), 'dispatched subscriptionProductsFetched with isSubscribable to be true');
            });
        });

        QUnit.test('dispatches not subscribable state of channel on error', function(assert) {
            this.sandbox.stub(API, 'getSubscriptionProducts', () => Promise.reject());

            return this.dispatchFetchSubscriptionProducts().then(() => {
                assert.ok(this.dispatch.calledWith({
                    type: ACTION_SUBSCRIPTION_PRODUCTS_FETCHED,
                    subscriptions: {
                        isSubscribable: false,
                    },
                }), 'dispatched subscriptionProductsFetched isSubscribable to be false');
            });
        });

        QUnit.test('dispatches nothing on success when channel has changed', function(assert) {
            this.sandbox.stub(API, 'getSubscriptionProducts', () => Promise.reject());

            this.state.streamMetadata.channel.name = 'bar';

            return this.dispatchFetchSubscriptionProducts().then(() => {
                assert.equal(this.dispatch.callCount, 0, 'dispatch not called');
            });
        });

        QUnit.test('dispatches nothing on failure when channel has changed', function(assert) {
            this.sandbox.stub(API, 'getSubscriptionProducts', () => Promise.reject());

            this.state.streamMetadata.channel.name = 'bar';

            return this.dispatchFetchSubscriptionProducts().then(() => {
                assert.equal(this.dispatch.callCount, 0, 'dispatch not called');
            });
        });
    });
});
