import sinon from 'sinon';
import { unitTest } from 'tests/utils/module';
import { AAXManager, AAX_TWITCH_PUBLISHER_ID, AAX_VIDEO_AD_SERVER } from 'ads/aax-manager';
import { AdRollTypes } from 'actions/ads';
import { init as initStore } from 'state';
import { setAnalyticsTracker } from 'actions/analytics-tracker';
import { buildFakeWindow } from 'tests/fakes/window.fake';
import { setWindow } from 'actions/window';
import * as AdSpadeEvents from 'analytics/spade-events';
import { LiveTwitchContentStream } from 'stream/twitch-live';
import { VODTwitchContentStream } from 'stream/twitch-vod';
import { ProvidedContentStream } from 'stream/provided';
import { nullContentStream } from 'stream/null';
import { ACTION_SET_STREAM } from 'actions/stream';

unitTest('ads | aax-manager', function(hooks) {
    hooks.beforeEach(function() {
        this.store = initStore();
        this.options = {};

        const nowStub = sinon.stub();
        this._dateNowDiff = 100;
        nowStub.onCall(0).returns(this._dateNowDiff);
        nowStub.onCall(1).returns(this._dateNowDiff + this._dateNowDiff);
        this.window = buildFakeWindow({
            Date: {
                now: nowStub,
            },
        });

        this.aaxManager = new AAXManager(AAX_TWITCH_PUBLISHER_ID, AAX_VIDEO_AD_SERVER, this.store, this.options);
        this.sampleBaseBids = {
            amznbid: 'v_bzcv7k',
            amzniid: 'Iqu4Nb1K7kvKgs4C0-C5pqUAAAFbwjT_eQYAAAvcBFiI8xw',
            mediaType: 'video',
            encodedQsParams: '%26amzniid%3DIqu4Nb1K7kvKgs4C0-C5pqUAAAFbwjT_eQYAAAvcBFiI8xw%26amznbid%3Dv_bzcv7k',
            qsParams: '&amzniid=Iqu4Nb1K7kvKgs4C0-C5pqUAAAFbwjT_eQYAAAvcBFiI8xw&amznbid=v_bzcv7k',
        };
        this.samplePrerollBids = [Object.assign({ slotID: 'twitch-preroll' }, this.sampleBaseBids)];
        this.sampleMidrollBids = [Object.assign({ slotID: 'twitch-midroll-1' }, this.sampleBaseBids)];
        this.samplePostrollBids = [Object.assign({ slotID: 'twitch-postroll' }, this.sampleBaseBids)];

        this.adRequestContext = {
            duration: 30,
        };

        this.analytics = {
            trackEvent: sinon.spy(),
        };
        this.store.dispatch(setAnalyticsTracker(this.analytics));
        this.store.dispatch(setWindow(this.window));

        this.fetchBidsStub = sinon.stub(this.aaxManager._apstag, 'fetchBids');
        this.adsRequestStub = sinon.stub(this.aaxManager, '_getAdsRequestContextForPreroll');
        this.adsRequestStub.returns(Promise.resolve({}));
        this.willDeclineStub = sinon.stub(this.aaxManager, '_willDeclineAds').returns(false);
    });

    hooks.afterEach(function() {
        this.fetchBidsStub.restore();
        this.adsRequestStub.restore();
        this.willDeclineStub.restore();
    });

    QUnit.module('AAXManager', function() {
        QUnit.test('should initialize apstag', function(assert) {
            assert.ok(this.aaxManager._apstag);
        });

        QUnit.test('should preload preroll bids on first live stream change', function(assert) {
            this.aaxManager = new AAXManager(AAX_TWITCH_PUBLISHER_ID, AAX_VIDEO_AD_SERVER, this.store, this.options);
            sinon.spy(this.aaxManager, 'preloadPrerollBids');

            assert.equal(this.aaxManager.preloadPrerollBids.callCount, 0, 'preroll bids have not been preloaded');
            this.store.dispatch({
                type: ACTION_SET_STREAM,
                stream: new LiveTwitchContentStream('a live channel'),
            });

            assert.equal(this.aaxManager.preloadPrerollBids.callCount, 1, 'preroll bids have been preloaded');
        });

        QUnit.test('should preload preroll bids on first vod stream change', function(assert) {
            this.aaxManager = new AAXManager(AAX_TWITCH_PUBLISHER_ID, AAX_VIDEO_AD_SERVER, this.store, this.options);
            sinon.spy(this.aaxManager, 'preloadPrerollBids');

            assert.equal(this.aaxManager.preloadPrerollBids.callCount, 0, 'preroll bids have not been preloaded');
            this.store.dispatch({
                type: ACTION_SET_STREAM,
                stream: new VODTwitchContentStream('a live vod'),
            });

            assert.equal(this.aaxManager.preloadPrerollBids.callCount, 1, 'preroll bids have been preloaded');
        });

        // eslint-disable-next-line max-len
        QUnit.test('should not preload preroll bids on subsequent stream changes after the first one', function(assert) {
            this.aaxManager = new AAXManager(AAX_TWITCH_PUBLISHER_ID, AAX_VIDEO_AD_SERVER, this.store, this.options);
            sinon.spy(this.aaxManager, 'preloadPrerollBids');

            assert.equal(this.aaxManager.preloadPrerollBids.callCount, 0, 'preroll bids have not been preloaded');
            this.store.dispatch({
                type: ACTION_SET_STREAM,
                stream: new VODTwitchContentStream('a live channel'),
            });

            assert.equal(this.aaxManager.preloadPrerollBids.callCount, 1, 'preroll bids have been preloaded');
            this.aaxManager.preloadPrerollBids.reset();

            this.store.dispatch({
                type: ACTION_SET_STREAM,
                stream: new VODTwitchContentStream('a live channel'),
            });

            assert.equal(this.aaxManager.preloadPrerollBids.callCount, 0, 'preroll bids have not been preloaded');
        });

        [
            new ProvidedContentStream({
                contentId: 'acontentid',
                customerId: 'acustomerid',
            }),
            nullContentStream,
        ].forEach(stream => {
            // eslint-disable-next-line max-len
            QUnit.test(`should not preload preroll bids on first ${stream.contentType} stream change`, function(assert) {
                this.aaxManager = new AAXManager(
                    AAX_TWITCH_PUBLISHER_ID,
                    AAX_VIDEO_AD_SERVER,
                    this.store,
                    this.options
                );
                sinon.spy(this.aaxManager, 'preloadPrerollBids');

                assert.equal(this.aaxManager.preloadPrerollBids.callCount, 0, 'preroll bids have not been preloaded');
                this.store.dispatch({
                    type: ACTION_SET_STREAM,
                    stream,
                });

                assert.equal(this.aaxManager.preloadPrerollBids.callCount, 0, 'preroll bids have not been preloaded');
            });
        });

        QUnit.test('bad apstag object exception is caught and sends event', function(assert) {
            this.window.apstag = {};
            this.aaxManager = new AAXManager(AAX_TWITCH_PUBLISHER_ID, AAX_VIDEO_AD_SERVER, this.store, this.options);
            assert.ok(this.aaxManager._apstag);
            assert.equal(this.analytics.trackEvent.callCount, 1, 'sent one spade events');
            const eventName = this.analytics.trackEvent.firstCall.args[0];
            assert.equal(eventName, AdSpadeEvents.AAX_AD_AUCTION_INIT_ERROR, 'first event is error');
        });

        QUnit.test('can fetch preroll bid after preroll is preloaded', function(assert) {
            this.fetchBidsStub.callsArgWith(1, this.samplePrerollBids);
            this.adRequestContext.adType = AdRollTypes.PREROLL;
            return this.aaxManager.preloadPrerollBids().then(() => {
                return this.aaxManager.fetchBids(this.adRequestContext).then(bids => {
                    assert.deepEqual(bids, this.samplePrerollBids);
                });
            });
        });

        QUnit.test('can fetch preroll bid before preroll is preloaded', function(assert) {
            this.fetchBidsStub.callsArgWith(1, this.samplePrerollBids);
            this.adRequestContext.adType = AdRollTypes.PREROLL;

            this.aaxManager.preloadPrerollBids();
            return this.aaxManager.fetchBids(this.adRequestContext).then(bids => {
                assert.deepEqual(bids, this.samplePrerollBids);
            });
        });

        QUnit.test('can fetch preroll bid without preloading', function(assert) {
            this.fetchBidsStub.callsArgWith(1, this.samplePrerollBids);
            this.adRequestContext.adType = AdRollTypes.PREROLL;

            return this.aaxManager.fetchBids(this.adRequestContext).then(bids => {
                assert.deepEqual(bids, this.samplePrerollBids);
            });
        });

        QUnit.test('can fetch midroll bid', function(assert) {
            this.fetchBidsStub.callsArgWith(1, this.sampleMidrollBids);
            this.adRequestContext.adType = AdRollTypes.MIDROLL;
            return this.aaxManager.fetchBids(this.adRequestContext).then(bids => {
                assert.deepEqual(bids, this.sampleMidrollBids);
            });
        });

        QUnit.test('can fetch postroll bid', function(assert) {
            this.fetchBidsStub.callsArgWith(1, this.samplePostrollBids);
            this.adRequestContext.adType = AdRollTypes.POSTROLL;
            return this.aaxManager.fetchBids(this.adRequestContext).then(bids => {
                assert.deepEqual(bids, this.samplePostrollBids);
            });
        });

        QUnit.test('can handle no preroll bids returned', function(assert) {
            this.fetchBidsStub.callsArgWith(1, []);
            this.adRequestContext.adType = AdRollTypes.PREROLL;
            return this.aaxManager.fetchBids(this.adRequestContext).then(bids => {
                assert.deepEqual(bids, []);
            });
        });

        QUnit.test('can handle no midroll bids returned', function(assert) {
            this.fetchBidsStub.callsArgWith(1, []);
            this.adRequestContext.adType = AdRollTypes.MIDROLL;
            return this.aaxManager.fetchBids(this.adRequestContext).then(bids => {
                assert.deepEqual(bids, []);
            });
        });

        QUnit.test('can handle no postroll bids returned', function(assert) {
            this.fetchBidsStub.callsArgWith(1, []);
            this.adRequestContext.adType = AdRollTypes.POSTROLL;
            return this.aaxManager.fetchBids(this.adRequestContext).then(bids => {
                assert.deepEqual(bids, []);
            });
        });

        QUnit.test('can add amazon bid to AdRequestContext', function(assert) {
            const adRequestContext = this.aaxManager.getAdRequestContextWithAmazonBids(this.samplePrerollBids, {});
            // eslint-disable-next-line max-len
            assert.equal(adRequestContext.amzniid, this.samplePrerollBids[0].amzniid, 'adRequestContext amzniid correct');
            // eslint-disable-next-line max-len
            assert.equal(adRequestContext.amznbid, this.samplePrerollBids[0].amznbid, 'adRequestContext amznbid correct');
            // eslint-disable-next-line max-len
            assert.equal(adRequestContext.qsParams, this.samplePrerollBids[0].qsParams, 'adRequestContext qsParams correct');
        });

        QUnit.test('will not preload if decline ads', function(assert) {
            this.willDeclineStub.returns(true);
            const spy = sinon.spy(this.aaxManager, '_fetchVideoBids');
            return this.aaxManager.preloadPrerollBids().then(() => {
                assert.ok(!spy.called);
            });
        });

        QUnit.test('should send spade events on successful fetchBids', function(assert) {
            this.fetchBidsStub.callsArgWith(1, this.sampleMidrollBids);
            this.adRequestContext.adType = AdRollTypes.MIDROLL;
            return this.aaxManager.fetchBids(this.adRequestContext).then(() => {
                assert.equal(this.analytics.trackEvent.callCount, 2, 'sent two spade events');

                const [eventName, payload] = this.analytics.trackEvent.firstCall.args;
                assert.equal(eventName, AdSpadeEvents.AAX_AD_AUCTION, 'first event is auction');
                // eslint-disable-next-line camelcase
                assert.equal(payload.roll_type, AdRollTypes.MIDROLL, 'first event correct roll_type');
                assert.equal(payload.slotID, this.sampleMidrollBids.slotID, 'first event correct slotID');

                const [eventName2, payload2] = this.analytics.trackEvent.secondCall.args;
                assert.equal(eventName2, AdSpadeEvents.AAX_AD_AUCTION_RESPONSE, 'second event is response');
                // eslint-disable-next-line camelcase
                assert.equal(payload2.roll_type, AdRollTypes.MIDROLL, 'second event correct roll_type');
                assert.equal(payload2.slotID, this.sampleMidrollBids.slotID, 'second event correct slotID');
                // eslint-disable-next-line camelcase
                assert.equal(payload2.aax_latency, this._dateNowDiff, 'second event latency correct');
                assert.equal(payload2.amznbid, this.sampleMidrollBids[0].amznbid, 'second event amznbid correct');
                assert.equal(payload2.amzniid, this.sampleMidrollBids[0].amzniid, 'second event amzniid correct');
                assert.equal(payload2.qsParams, this.sampleMidrollBids[0].qsParams, 'second event qsParams correct');
            });
        });

        QUnit.test('should send spade events on failed fetchBids', function(assert) {
            this.fetchBidsStub.callsArgWith(1, []);
            this.adRequestContext.adType = AdRollTypes.MIDROLL;
            return this.aaxManager.fetchBids(this.adRequestContext).then(() => {
                assert.equal(this.analytics.trackEvent.callCount, 2, 'sent two spade events');

                const [eventName, payload] = this.analytics.trackEvent.firstCall.args;
                assert.equal(eventName, AdSpadeEvents.AAX_AD_AUCTION, 'first event is auction');
                // eslint-disable-next-line camelcase
                assert.equal(payload.roll_type, AdRollTypes.MIDROLL, 'first event correct roll_type');
                assert.equal(payload.slotID, this.sampleMidrollBids.slotID, 'first event correct slotID');

                const [eventName2, payload2] = this.analytics.trackEvent.secondCall.args;
                assert.equal(eventName2, AdSpadeEvents.AAX_AD_AUCTION_ERROR, 'second event is error');
                // eslint-disable-next-line camelcase
                assert.equal(payload2.roll_type, AdRollTypes.MIDROLL, 'second event correct roll_type');
                assert.equal(payload2.slotID, this.sampleMidrollBids.slotID, 'second event correct slotID');
                // eslint-disable-next-line camelcase
                assert.equal(payload2.aax_latency, this._dateNowDiff, 'second event latency correct');
            });
        });
    });
});
