import sinon from 'sinon';
import { unitTest } from 'tests/utils/module';
import { setAnalyticsTracker } from 'actions/analytics-tracker';
import { init as initStore } from 'state';
import { AdsRequestContext } from 'ads/adscontext';
import { IMA_PROVIDER } from 'ads/ima-manager';
import * as PlayerType from 'util/player-type';
import { PREROLL } from 'ads/ima-tags';
import { initializeAdSpadeEvent, sendAdSpadeEvent } from 'ads/ads-spade';

/**
 * Helper for generating IMA style events
 */
function createAdEvent(adMock) {
    return {
        getAd() {
            return {
                getAdId: () => adMock.adId,
                getAdSystem: () => adMock.adSystem,
                getCreativeId: () => adMock.creativeId,
                getTitle: () => adMock.title,
                getDuration: () => adMock.duration,
                getWrapperAdIds: () => [adMock.wrapperAdIds],
                getWrapperAdSystems: () => [adMock.wrapperAdSystems],
                getWrapperCreativeIds: () => [adMock.creativeIds],
            };
        },
        getError() {
            return {
                getMessage: () => adMock.error.message,
                getErrorCode: () => adMock.error.code,
                getType: () => adMock.error.type,
            };
        },
        getUserRequestContext() {
            return adMock.adRequestContext;
        },
    };
}

unitTest('ads | spade', function(hooks) {
    hooks.beforeEach(function() {
        this.analytics = {
            trackEvent: sinon.spy(),
        };
        this.store = initStore();
        this.adRequestContext = new AdsRequestContext({
            adType: PREROLL,
            duration: Math.floor(Math.random() * 30 + 30),
            state: {
                ads: {
                    adblock: false,
                },
                adsManager: {
                    sdk: 'sdkname',
                },
                env: {
                    playerType: PlayerType.PLAYER_POPOUT,
                },
                window: {
                    document: {
                        referrer: `https://${QUnit.config.current.testId.toLowerCase()}.othersite.com`,
                    },
                    location: {
                        href: 'https://player.twitch.tv',
                    },
                    top: {
                        location: {
                            hostname: 'media.curse.com',
                        },
                    },
                },
                stream: {},
            },
            lastAdDisplay: 0,
            userInfo: {
                turbo: null,
                login: 'mrwiggles',
            },
            channelAPIInfo: {
                prerolls: true,
                postrolls: true,
            },
            channelInfo: {
                name: `channel_${QUnit.config.current.testId}`,
                game: `Game ${QUnit.config.current.testId}`,
                mature: true,
                partner: true,
            },
            viewerInfo: {
                chansub: null,
                // eslint-disable-next-line camelcase
                has_ad_free_subscription: null,
            },
            communitiesInfo: {
                communities: [],
            },
            channelAdProperties: {
                /* eslint-disable camelcase */
                valid_responses: {
                    vod_ads_enabled: true,
                },
                /* eslint-enable camelcase */
            },
        });

        this.adEvent = createAdEvent({
            adId: 'ad_123',
            adSystem: 'GDFP',
            creativeId: '456',
            title: 'LUL Ad',
            duration: 30,
            wrapperAdIds: 'ad_wrapper_123',
            wrapperAdSystems: 'Wrapper GDFP',
            wrapperCreativeIds: '789',
            error: {},
        });

        this.store.dispatch(setAnalyticsTracker(this.analytics));
    });

    QUnit.test('can initialize ad spade event', function(assert) {
        const spadeData = initializeAdSpadeEvent(this.adRequestContext, this.adEvent.getAd());
        assert.deepEqual(spadeData, {
            /* eslint-disable camelcase */
            ad_session_id: this.adRequestContext.adSessionId,
            roll_type: this.adRequestContext.adType,
            time_break: this.adRequestContext.duration,
            twitch_correlator: this.adRequestContext.twitchCorrelator,
            adblock: this.adRequestContext.adblock,
            ad_id: this.adEvent.getAd().getAdId(),
            ad_system: this.adEvent.getAd().getAdSystem(),
            creative_id: this.adEvent.getAd().getCreativeId(),
            title: this.adEvent.getAd().getTitle(),
            duration: this.adEvent.getAd().getDuration(),
            wrapper_ad_id: this.adEvent.getAd().getWrapperAdIds()[0],
            wrapper_ad_system: this.adEvent.getAd().getWrapperAdSystems()[0],
            wrapper_creative_id: this.adEvent.getAd().getWrapperCreativeIds()[0],
            /* eslint-enable camelcase */
        });
    });

    QUnit.test('can send ad spade event', function(assert) {
        const spadeData = initializeAdSpadeEvent(this.adRequestContext, this.adEvent.getAd());
        sendAdSpadeEvent(this.store, IMA_PROVIDER, 'test event name', spadeData);

        assert.equal(this.analytics.trackEvent.callCount, 1);
    });
});
