import sinon from 'sinon';
import assign from 'lodash/assign';
import merge from 'lodash/merge';
import omit from 'lodash/omit';
import { unitTest } from 'tests/utils/module';
import { AdsRequestContext } from 'ads/adscontext';
import * as PlayerType from 'util/player-type';
import { PREROLL, MIDROLL, POSTROLL } from 'ads/ima-tags';
import { VODTwitchContentStream } from 'stream/twitch-vod';
import { LiveTwitchContentStream } from 'stream/twitch-live';
import { PLATFORM_WEB } from 'state/env';

unitTest('ads | adscontext', function() {
    QUnit.module('AdsRequestContext', function(hooks) {
        hooks.beforeEach(function() {
            this.baseParams = {
                adType: PREROLL,
                duration: Math.floor(Math.random() * 1e3),
                state: {
                    ads: {
                        adblock: true,
                        vodMidrollArchiveSetting: '',
                    },
                    adsManager: {
                        sdk: 'sdkname',
                    },
                    env: {
                        playerType: PlayerType.PLAYER_POPOUT,
                    },
                    window: {
                        document: {
                            referrer: '',
                        },
                        location: {
                            href: '',
                        },
                        top: {
                            location: {
                                hostname: 'media.curse.com',
                            },
                        },
                    },
                    stream: {},
                    playback: {
                        duration: '',
                    },
                },
                lastAdDisplay: 0,
                userInfo: {},
                channelInfo: {},
                channelAPIInfo: {},
                viewerInfo: {},
                communitiesInfo: {
                    communities: [],
                },
                channelAdProperties: {
                    /* eslint-disable camelcase */
                    valid_responses: {
                        vod_ads_enabled: true,
                    },
                    /* eslint-enable camelcase */
                },
            };

            sinon.stub(Date, 'now').returns(parseInt(QUnit.config.current.testId, 36));
        });

        hooks.afterEach(function() {
            Date.now.restore();
        });

        QUnit.test('adds login, contentType', function(assert) {
            const login = 'mrwiggles';
            const contentType = 'type';

            const adsRequestContext = new AdsRequestContext(merge({}, this.baseParams, {
                userInfo: {
                    login,
                },
                state: {
                    stream: {
                        contentType,
                    },
                },
            }));
            assert.equal(adsRequestContext.login, login);
            assert.equal(adsRequestContext.contentType, contentType);
        });

        QUnit.test('generates an appropriate value for an ad session ID', function(assert) {
            const adsRequestContext = new AdsRequestContext(this.baseParams);

            assert.equal(adsRequestContext.adSessionId.length, 30);
            assert.ok(/^[a-zA-Z0-9]{30}$/.test(adsRequestContext.adSessionId));
        });

        QUnit.test('uses the store value for `vodMidrollArchiveSetting`', function(assert) {
            [true, false].forEach(vodMidrollArchiveSettingResult => {
                const adsRequestContext = new AdsRequestContext(merge({}, this.baseParams, {
                    state: {
                        ads: {
                            vodMidrollArchiveSetting: vodMidrollArchiveSettingResult,
                        },
                    },
                }));

                assert.equal(adsRequestContext.vodMidrollArchiveSetting, vodMidrollArchiveSettingResult);
            });
        });

        QUnit.test('uses the store value for `adblock`', function(assert) {
            [true, false].forEach(adblockResult => {
                const adsRequestContext = new AdsRequestContext(merge({}, this.baseParams, {
                    state: {
                        ads: {
                            adblock: adblockResult,
                        },
                    },
                }));

                assert.equal(adsRequestContext.adblock, adblockResult);
            });
        });

        QUnit.test('uses the given ad type', function(assert) {
            [PREROLL, MIDROLL, POSTROLL].forEach(adType => {
                const adsRequestContext = new AdsRequestContext(assign({}, this.baseParams, {
                    adType,
                }));

                assert.equal(adsRequestContext.adType, adType);
            });
        });

        QUnit.test('uses the channel name from the given API response', function(assert) {
            const channelName = `channel_${QUnit.config.current.testId}`;
            const displayName = `display_${QUnit.config.current.testId}`;

            const adsRequestContext = new AdsRequestContext(assign({}, this.baseParams, {
                channelInfo: {
                    name: channelName,
                    // eslint-disable-next-line camelcase
                    display_name: displayName,
                },
            }));

            assert.equal(adsRequestContext.channel, channelName);
        });

        QUnit.test('uses the community ids from the given API response', function(assert) {
            const adsRequestContext = new AdsRequestContext(assign({}, this.baseParams, {
                communitiesInfo: {
                    communities: [{ _id: '12345' }, { _id: '54321' }],
                },
            }));

            assert.equal(adsRequestContext.communityIds, '12345,54321');
        });

        QUnit.test('community id is null from the given API response', function(assert) {
            const adsRequestContext = new AdsRequestContext(assign({}, this.baseParams, {
                communitiesInfo: {
                    communities: [],
                },
            }));

            assert.equal(adsRequestContext.communityIds, null);
        });

        QUnit.test('uses the chansub token from the given API response', function(assert) {
            [null, `chansub_${QUnit.config.current.testId}`].forEach(token => {
                const adsRequestContext = new AdsRequestContext(assign({}, this.baseParams, {
                    viewerInfo: {
                        chansub: token,
                    },
                }));

                assert.equal(adsRequestContext.chansubToken, token);
            });
        });

        QUnit.test('uses the given duration', function(assert) {
            const duration = Math.floor(Math.random() * 1e3 + 30);

            const adsRequestContext = new AdsRequestContext(assign({}, this.baseParams, {
                duration,
            }));

            assert.equal(adsRequestContext.duration, duration);
        });

        QUnit.test('uses the game from the given API response', function(assert) {
            const game = `game${QUnit.config.current.testId}`;

            const adsRequestContext = new AdsRequestContext(assign({}, this.baseParams, {
                channelInfo: {
                    game,
                },
            }));

            assert.equal(adsRequestContext.game, game);
        });

        QUnit.test('uses lowercase game stripped of special characters', function(assert) {
            const game = `Game:^*&@! ${QUnit.config.current.testId}`;
            const expectedGame = `game_${QUnit.config.current.testId}`;
            const adsRequestContext = new AdsRequestContext(assign({}, this.baseParams, {
                channelInfo: {
                    game,
                },
            }));

            assert.equal(adsRequestContext.game, expectedGame);
        });

        QUnit.test('uses the subscription\'s ad-free status from the given API response', function(assert) {
            [
                {
                    response: true,
                    value: true,
                },
                {
                    response: false,
                    value: false,
                },
                {
                    // sadly, the API can return `null` for what should be a boolean field
                    response: null,
                    value: false,
                },
            ].forEach(({ response, value }) => {
                const adsRequestContext = new AdsRequestContext(assign({}, this.baseParams, {
                    viewerInfo: {
                        // eslint-disable-next-line camelcase
                        has_ad_free_subscription: response,
                    },
                }));

                assert.equal(adsRequestContext.hasAdFreeSubscription, value);
            });
        });

        QUnit.module('kruxId', function() {
            QUnit.test('safely defaults to `null` if Krux is not available', function(assert) {
                const adsRequestContext = new AdsRequestContext(merge({}, this.baseParams, {
                    state: {
                        window: omit(this.baseParams.state.window, ['Krux']),
                    },
                }));

                assert.equal(adsRequestContext.kruxId, null);
            });

            QUnit.test('uses the provided id', function(assert) {
                const kruxId = `krux_${QUnit.config.current.testId.toLowerCase()}`;
                const adsRequestContext = new AdsRequestContext(merge({}, this.baseParams, {
                    state: {
                        window: {
                            Krux: {
                                user: kruxId,
                            },
                        },
                    },
                }));

                assert.equal(adsRequestContext.kruxId, kruxId);
            });

            QUnit.test('defaults to an empty string if Krux is available without an ID', function(assert) {
                const adsRequestContext = new AdsRequestContext(merge({}, this.baseParams, {
                    state: {
                        window: {
                            Krux: {
                                user: null,
                            },
                        },
                    },
                }));

                assert.equal(adsRequestContext.kruxId, '');
            });
        });

        QUnit.test('uses the given last ad display time', function(assert) {
            const lastAdDisplay = Math.floor(Math.random() * 1e8);
            const adsRequestContext = new AdsRequestContext(assign({}, this.baseParams, {
                lastAdDisplay,
            }));

            assert.equal(adsRequestContext.lastAdDisplay, lastAdDisplay);
        });

        QUnit.test('uses the channel\'s mature setting from the API response', function(assert) {
            [true, false].forEach(mature => {
                const adsRequestContext = new AdsRequestContext(assign({}, this.baseParams, {
                    channelInfo: {
                        mature,
                    },
                }));

                assert.equal(adsRequestContext.mature, mature);
            });
        });

        QUnit.test('uses the channel\'s partnered status from the API response', function(assert) {
            [true, false].forEach(partner => {
                const adsRequestContext = new AdsRequestContext(assign({}, this.baseParams, {
                    channelInfo: {
                        partner,
                    },
                }));

                assert.equal(adsRequestContext.partner, partner);
            });
        });

        QUnit.test('uses the provided player type', function(assert) {
            [
                PlayerType.PLAYER_SITE,
                PlayerType.PLAYER_EMBED,
                PlayerType.PLAYER_POPOUT,
                PlayerType.PLAYER_FRONTPAGE,
                PlayerType.PLAYER_CREATIVE,
                PlayerType.PLAYER_FACEBOOK,
                PlayerType.PLAYER_FEED,
            ].forEach(playerType => {
                const adsRequestContext = new AdsRequestContext(merge({}, this.baseParams, {
                    state: {
                        env: {
                            playerType,
                        },
                    },
                }));

                assert.equal(adsRequestContext.playerType, playerType);
            });
        });

        QUnit.test('adds the isMobileLocation', function(assert) {
            const isMobileLocation = 'https://m.twitch.tv/monstertruck';
            const adsRequestContext = new AdsRequestContext(merge({}, this.baseParams, {
                state: {
                    env: {
                        isMobileLocation,
                    },
                },
            }));

            assert.equal(adsRequestContext.isMobileLocation, isMobileLocation);
        });

        QUnit.test('adds the platform', function(assert) {
            const platform = PLATFORM_WEB;
            const adsRequestContext = new AdsRequestContext(merge({}, this.baseParams, {
                state: {
                    env: {
                        platform,
                    },
                },
            }));

            assert.equal(adsRequestContext.platform, platform);
        });

        QUnit.test('uses the preroll and postroll configuration from the API response', function(assert) {
            [
                [true, true],
                [true, false],
                [false, true],
                [false, false],
            ].forEach(([prerolls, postrolls]) => {
                const adsRequestContext = new AdsRequestContext(assign({}, this.baseParams, {
                    channelAPIInfo: {
                        prerolls,
                        postrolls,
                    },
                }));

                assert.equal(adsRequestContext.prerollsEnabled, prerolls);
                assert.equal(adsRequestContext.postrollsEnabled, postrolls);
            });
        });

        QUnit.test('uses the document referrer as the embedUrl', function(assert) {
            const referrer = `https://${QUnit.config.current.testId.toLowerCase()}.wherever.lol`;
            const adsRequestContext = new AdsRequestContext(merge({}, this.baseParams, {
                state: {
                    window: {
                        document: {
                            referrer,
                        },
                    },
                },
            }));

            assert.equal(adsRequestContext.embedUrl, referrer);
        });

        QUnit.test('includes the time at which the request context was created', function(assert) {
            const adsRequestContext = new AdsRequestContext(this.baseParams);

            assert.equal(adsRequestContext.requestTime, Date.now());
        });

        QUnit.test('uses the turbo token from the user info API response', function(assert) {
            [
                `token_${QUnit.config.current.testId.toLowerCase()}`,
                null,
            ].forEach(turbo => {
                const adsRequestContext = new AdsRequestContext(assign({}, this.baseParams, {
                    userInfo: {
                        turbo,
                    },
                }));

                assert.equal(adsRequestContext.turboToken, turbo);
            });
        });

        QUnit.test('looks up the current location as the url', function(assert) {
            const url = `https://twitch.tv/${QUnit.config.current.testId.toLowerCase()}`;
            const adsRequestContext = new AdsRequestContext(merge({}, this.baseParams, {
                state: {
                    window: {
                        location: {
                            href: url,
                        },
                    },
                },
            }));

            assert.equal(adsRequestContext.url, url);
        });

        QUnit.test('includes VOD info if the stream is a VOD', function(assert) {
            const vodID = 'v12345';
            const streamMetadata = {
                name: 'VOD#@&_   title',
                type: 'archive',
            };
            const expectedName = 'vod_title';
            const playback = {
                duration: 240,
            };

            const adsRequestContext = new AdsRequestContext(merge({}, this.baseParams, {
                state: {
                    stream: new VODTwitchContentStream(vodID),
                    streamMetadata,
                    playback,
                },
            }));
            assert.equal(adsRequestContext.vod.name, expectedName);
        });

        QUnit.test('VOD name is properly formatted', function(assert) {
            const vodID = 'v12345';
            const expectedName = 'some_vod_title';
            const streamMetadata = {
                name: 'Some VOD #%##@#@ Title',
                type: 'archive',
            };
            const playback = {
                duration: 240,
            };

            const adsRequestContext = new AdsRequestContext(merge({}, this.baseParams, {
                state: {
                    stream: new VODTwitchContentStream(vodID),
                    streamMetadata,
                    playback,
                },
            }));

            assert.deepEqual(adsRequestContext.vod, {
                id: vodID,
                name: expectedName,
                type: streamMetadata.type,
                duration: playback.duration,
            });
        });

        QUnit.test('does not include VOD info if the stream is not a VOD', function(assert) {
            const channelName = 'twitch';
            const adsRequestContext = new AdsRequestContext(merge({}, this.baseParams, {
                state: {
                    stream: new LiveTwitchContentStream(channelName),
                },
            }));

            assert.deepEqual(adsRequestContext.vod, {
                id: '',
                name: '',
                type: '',
                duration: -1,
            });
        });
    });
});
