import assign from 'lodash/assign';
import sinon from 'sinon';
import { unitTest } from 'tests/utils/module';
import { init as initStore } from 'state';
import { PLATFORM_MOBILE_WEB, PLATFORM_WEB } from 'state/env';
import { setWindow } from 'actions/window';
import { setAnalyticsTracker } from 'actions/analytics-tracker';
import { DecliningAdsManager } from 'ads/declining-manager';
import { AdsRequestContext } from 'ads/adscontext';
import { PREROLL, MIDROLL, POSTROLL } from 'ads/ima-tags';
import * as PlayerType from 'util/player-type';
import { buildFakeWindow } from 'tests/fakes/window.fake';
import { ARCHIVE } from 'vod-midroll-manager';
import { ID3_AD } from 'video';
import { CONTENT_MODE_VOD } from 'stream/twitch-vod';

unitTest('ads | declining-manager', function() {
    QUnit.module('DecliningAdsManager', function(hooks) {
        hooks.beforeEach(function() {
            this.containedAdManager = {
                requestAds: sinon.spy(),
                pause: sinon.spy(),
                play: sinon.spy(),
                setVolume: sinon.spy(),
                setMuted: sinon.spy(),
            };
            this.window = buildFakeWindow({
                document: {
                    referrer: `http://${QUnit.config.current.testId.toLowerCase()}.othersite.com`,
                },
                location: {
                    href: `http://player.twitch.tv/?channel=${QUnit.config.current.testId}`,
                },
                top: {
                    location: {
                        hostname: 'media.curse.com',
                    },
                },
            });
            this.analytics = {
                trackEvent: sinon.spy(),
            };
            this.store = initStore();
            this.store.dispatch(setWindow(this.window));
            this.store.dispatch(setAnalyticsTracker(this.analytics));
            this.adRequestContext = createRequestContext(this.window);
        });

        function assertTrackEvent(context, assert, payload) {
            assert.equal(context.analytics.trackEvent.callCount, 1, 'should call `trackEvent` exactly once');

            const actual = context.analytics.trackEvent.firstCall.args;
            const expected = ['video_ad_request_declined', trackEventPayload(payload, context.adRequestContext)];
            assert.deepEqual(actual, expected, 'should be called with the correct arguments');
        }

        QUnit.test('should decline with `reason_turbo` if the user has a turbo token', function(assert) {
            const adManager = new DecliningAdsManager(this.store, this.containedAdManager, this.analytics);
            this.adRequestContext.turboToken = `token_${QUnit.config.current.testId}`;

            adManager.requestAds(this.adRequestContext);
            assert.equal(this.containedAdManager.requestAds.callCount, 0, 'should not call the ad manager');
            assertTrackEvent(this, assert,
                {
                    // eslint-disable-next-line
                    reason_turbo: true,
                }
            );
        });

        // eslint-disable-next-line max-len
        QUnit.test('should decline preroll ad with `reason_vod_creator` if the user is watching their own VOD', function(assert) {
            const adManager = new DecliningAdsManager(this.store, this.containedAdManager, this.analytics);

            this.adRequestContext.login = QUnit.config.current.testId;
            this.adRequestContext.channel = QUnit.config.current.testId;
            this.adRequestContext.contentType = 'vod';
            this.adRequestContext.vod.duration = 180;
            this.adRequestContext.adType = PREROLL;
            adManager.requestAds(this.adRequestContext);
            assert.equal(this.containedAdManager.requestAds.callCount, 0, 'should not call the ad manager');
            assertTrackEvent(this, assert,
                {
                    // eslint-disable-next-line
                    reason_vod_creator: true,
                }
            );
        });

        // eslint-disable-next-line max-len
        QUnit.test('should decline preroll ad with `reason_short_vod` if the user is watching a VOD that is < 60 seconds', function(assert) {
            const adManager = new DecliningAdsManager(this.store, this.containedAdManager, this.analytics);

            this.adRequestContext.channel = QUnit.config.current.testId;
            this.adRequestContext.contentType = 'vod';
            this.adRequestContext.adType = PREROLL;
            this.adRequestContext.vod.duration = 59;

            adManager.requestAds(this.adRequestContext);
            assert.equal(this.containedAdManager.requestAds.callCount, 0, 'should not call the ad manager');
            assertTrackEvent(this, assert,
                {
                    // eslint-disable-next-line
                    reason_short_vod: true,
                }
            );
        });

        // eslint-disable-next-line max-len
        QUnit.test('should not decline preroll ad with `reason_short_vod` if the user is watching a VOD that >= 60 seconds', function(assert) {
            const adManager = new DecliningAdsManager(this.store, this.containedAdManager, this.analytics);

            this.adRequestContext.channel = QUnit.config.current.testId;
            this.adRequestContext.contentType = 'vod';
            this.adRequestContext.adType = PREROLL;
            this.adRequestContext.vod.duration = 60;

            adManager.requestAds(this.adRequestContext);
            assert.equal(this.containedAdManager.requestAds.callCount, 1,
                'should call the ad manager exactly once');
            assert.equal(this.analytics.trackEvent.callCount, 0, 'should never call `trackEvent`');
        });

        QUnit.test('should decline preroll ad with `reason_vod_midroll` if criteria is met', function(assert) {
            const adManager = new DecliningAdsManager(this.store, this.containedAdManager, this.analytics);

            this.adRequestContext.channel = QUnit.config.current.testId;
            this.adRequestContext.contentType = CONTENT_MODE_VOD;
            this.adRequestContext.adType = MIDROLL;
            this.adRequestContext.requestType = ID3_AD;
            this.adRequestContext.vod.type = ARCHIVE;
            this.adRequestContext.vodMidrollArchiveSetting = 'test';

            adManager.requestAds(this.adRequestContext);
            assert.equal(this.containedAdManager.requestAds.callCount, 0, 'should not call the ad manager');
            assertTrackEvent(this, assert,
                {
                    // eslint-disable-next-line
                    reason_vod_midroll: true,
                }
            );
        });

        [PREROLL, MIDROLL, POSTROLL].forEach(adType => {
            // eslint-disable-next-line
            QUnit.test(`should decline ${adType} ad with reason_vod_ads_disabled if vod ads disabled is met`, function(assert) {
                const adManager = new DecliningAdsManager(this.store, this.containedAdManager, this.analytics);

                this.adRequestContext.channel = QUnit.config.current.testId;
                this.adRequestContext.contentType = CONTENT_MODE_VOD;
                this.adRequestContext.adType = adType;
                this.adRequestContext.vodAdsEnabled = false;

                adManager.requestAds(this.adRequestContext);
                assert.equal(this.containedAdManager.requestAds.callCount, 0, 'should not call the ad manager');
                assertTrackEvent(this, assert,
                    {
                        // eslint-disable-next-line
                        reason_vod_ads_disabled: true,
                    }
                );
            });
        });

        // Skipping this test. We are going to deprecate the experiment
        // associated with this decline reason for now and investigate this
        // further before re-enabling it.
        QUnit.skip('should decline with `already_seen_rolltype` if the user has seen this rolltype', function(assert) {
            const adManager = new DecliningAdsManager(this.store, this.containedAdManager, this.analytics);

            adManager.requestAds(this.adRequestContext);
            this.adRequestContext.requestTime = Date.now();
            this.adRequestContext.lastAdDisplay = 0;

            adManager.requestAds(this.adRequestContext);
            assert.equal(this.containedAdManager.requestAds.callCount, 1,
                'should call the ad manager exactly once');
            assertTrackEvent(this, assert, {
                // eslint-disable-next-line
                reason_already_seen_rolltype: true,
            });
        });

        QUnit.module('when the user has recently seen an ad', function(hooks) {
            hooks.beforeEach(function() {
                this.adRequestContext.requestTime = Date.now();
                this.adRequestContext.lastAdDisplay = this.adRequestContext.requestTime - 1000;
            });

            QUnit.test('should decline with `reason_ratelimit` for prerolls', function(assert) {
                const adManager = new DecliningAdsManager(this.store, this.containedAdManager, this.analytics);
                this.adRequestContext.adType = PREROLL;

                adManager.requestAds(this.adRequestContext);
                assert.equal(this.containedAdManager.requestAds.callCount, 0, 'should not call the ad manager');
                assertTrackEvent(this, assert, {
                    // eslint-disable-next-line
                    reason_ratelimit: true,
                });
            });

            QUnit.test('should not decline for prerolls if forced is true in context', function(assert) {
                const adManager = new DecliningAdsManager(this.store, this.containedAdManager, this.analytics);
                this.adRequestContext.adType = PREROLL;
                this.adRequestContext.forced = true;

                adManager.requestAds(this.adRequestContext);
                assert.equal(this.containedAdManager.requestAds.callCount, 1, 'should call the ad manager');
                assert.equal(this.analytics.trackEvent.callCount, 0, 'should not call `trackEvent`');
            });

            QUnit.test('should not decline for midrolls if forced is true in context', function(assert) {
                const adManager = new DecliningAdsManager(this.store, this.containedAdManager, this.analytics);
                this.adRequestContext.adType = MIDROLL;
                this.adRequestContext.forced = true;

                adManager.requestAds(this.adRequestContext);
                assert.equal(this.containedAdManager.requestAds.callCount, 1, 'should call the ad manager');
                assert.equal(this.analytics.trackEvent.callCount, 0, 'should not call `trackEvent`');
            });

            [MIDROLL, POSTROLL].forEach(adType => {
                QUnit.test(`should not decline ${adType}`, function(assert) {
                    this.adRequestContext.adType = adType;
                    const adManager = new DecliningAdsManager(this.store, this.containedAdManager, this.analytics);

                    adManager.requestAds(this.adRequestContext);
                    assert.equal(this.analytics.trackEvent.callCount, 0);
                    assert.ok(this.containedAdManager.requestAds.calledWith(this.adRequestContext));
                    assert.equal(this.containedAdManager.requestAds.callCount, 1);
                });
            });
        });

        QUnit.test('should not decline prerolls if the user has not seen one recently', function(assert) {
            this.adRequestContext.requestTime = Date.now();
            this.adRequestContext.lastAdDisplay = this.adRequestContext.requestTime - 15 * 60 * 1000;

            const adManager = new DecliningAdsManager(this.store, this.containedAdManager, this.analytics);

            adManager.requestAds(this.adRequestContext);
            assert.equal(this.analytics.trackEvent.callCount, 0);
            assert.ok(this.containedAdManager.requestAds.calledWith(this.adRequestContext));
            assert.equal(this.containedAdManager.requestAds.callCount, 1);
        });

        [PREROLL, POSTROLL].forEach(adType => {
            QUnit.test(`should decline ${adType} when ${adType}s are disabled`, function(assert) {
                this.adRequestContext.adType = adType;
                this.adRequestContext.prerollsEnabled = (adType !== PREROLL);
                this.adRequestContext.postrollsEnabled = (adType !== POSTROLL);

                const adManager = new DecliningAdsManager(this.store, this.containedAdManager, this.analytics);

                adManager.requestAds(this.adRequestContext);
                assert.equal(this.containedAdManager.requestAds.callCount, 0, 'should not call the ad manager');
                assertTrackEvent(this, assert, {
                    // eslint-disable-next-line
                    reason_channeladfree: true,
                });
            });

            QUnit.test(`should not decline ${adType} when the other is disabled`, function(assert) {
                this.adRequestContext.adType = adType;
                this.adRequestContext.prerollsEnabled = (adType === PREROLL);
                this.adRequestContext.postrollsEnabled = (adType === POSTROLL);

                const adManager = new DecliningAdsManager(this.store, this.containedAdManager, this.analytics);

                adManager.requestAds(this.adRequestContext);
                assert.equal(this.containedAdManager.requestAds.callCount, 1);
                assert.ok(this.containedAdManager.requestAds.calledWith(this.adRequestContext));
                assert.equal(this.analytics.trackEvent.callCount, 0);
            });
        });

        // eslint-disable-next-line
        QUnit.test('should decline with `reason_raid` if experiment=yes, preroll and parameter referrer=digest', function(assert) {
            const adManager = new DecliningAdsManager(this.store, this.containedAdManager, this.analytics);

            this.adRequestContext.adType = PREROLL;
            this.adRequestContext.raid = true;
            this.adRequestContext.leviathanExpGroup = 'yes';

            adManager.requestAds(this.adRequestContext);
            assert.equal(this.containedAdManager.requestAds.callCount, 0, 'should not call the ad manager');
            assertTrackEvent(this, assert, {
                // eslint-disable-next-line
                reason_raid: true,
            });
        });

        // eslint-disable-next-line max-len
        QUnit.test('should not decline with `raid` if preroll and parameter referrer=raid is passed and experiment=no', function(assert) {
            const adManager = new DecliningAdsManager(this.store, this.containedAdManager, this.analytics);

            this.adRequestContext.adType = PREROLL;
            this.adRequestContext.raid = true;
            this.adRequestContext.leviathanExpGroup = 'no';

            adManager.requestAds(this.adRequestContext);
            assert.equal(this.containedAdManager.requestAds.callCount, 1, 'should call the ad manager');
        });

        // eslint-disable-next-line max-len
        QUnit.test('should not decline with `raid` if midroll and parameter referrer=raid is passed', function(assert) {
            const adManager = new DecliningAdsManager(this.store, this.containedAdManager, this.analytics);

            this.adRequestContext.adType = MIDROLL;
            this.adRequestContext.raid = true;
            this.adRequestContext.leviathanExpGroup = 'no';

            adManager.requestAds(this.adRequestContext);
            assert.equal(this.containedAdManager.requestAds.callCount, 1, 'should call the ad manager');
        });

        QUnit.test('should decline with `embed_promo` if embed, vod and promo are true', function(assert) {
            const adManager = new DecliningAdsManager(this.store, this.containedAdManager, this.analytics);

            this.adRequestContext.adType = MIDROLL;
            this.adRequestContext.contentType = 'vod';
            this.adRequestContext.playerType = 'embed';
            this.adRequestContext.embedPromo = true;

            adManager.requestAds(this.adRequestContext);
            assert.equal(this.containedAdManager.requestAds.callCount, 0, 'should not call the ad manager');
            assertTrackEvent(this, assert, {
                // eslint-disable-next-line
                reason_embed_promo: true,
            });
        });

        QUnit.test('should not decline with `embed_promo` if promo is false', function(assert) {
            const adManager = new DecliningAdsManager(this.store, this.containedAdManager, this.analytics);

            this.adRequestContext.adType = MIDROLL;
            this.adRequestContext.contentType = 'vod';
            this.adRequestContext.playerType = 'embed';
            this.adRequestContext.embedPromo = false;

            adManager.requestAds(this.adRequestContext);
            assert.equal(this.containedAdManager.requestAds.callCount, 1, 'should call the ad manager');
        });

        QUnit.test('should decline with `reason_mobile_web_upsell` if mobile web on mobile', function(assert) {
            const adManager = new DecliningAdsManager(this.store, this.containedAdManager, this.analytics);

            this.adRequestContext.isMobileLocation = true;
            this.adRequestContext.platform = PLATFORM_MOBILE_WEB;

            adManager.requestAds(this.adRequestContext);
            assert.equal(this.containedAdManager.requestAds.callCount, 0, 'should not call the ad manager');
            assertTrackEvent(this, assert, {
                // eslint-disable-next-line
                reason_mobile_web_upsell: true,
            });
        });

        QUnit.test('should not decline ads if not on mobile web', function(assert) {
            this.adRequestContext.isMobileLocation = false;
            this.adRequestContext.platform = PLATFORM_MOBILE_WEB;

            const adManager = new DecliningAdsManager(this.store, this.containedAdManager, this.analytics);

            adManager.requestAds(this.adRequestContext);
            assert.equal(this.analytics.trackEvent.callCount, 0);
            assert.ok(this.containedAdManager.requestAds.calledWith(this.adRequestContext));
            assert.equal(this.containedAdManager.requestAds.callCount, 1);
        });

        QUnit.test('should not decline ads if mobile web not on mobile', function(assert) {
            this.adRequestContext.isMobileLocation = true;
            this.adRequestContext.platform = PLATFORM_WEB;

            const adManager = new DecliningAdsManager(this.store, this.containedAdManager, this.analytics);

            adManager.requestAds(this.adRequestContext);
            assert.equal(this.analytics.trackEvent.callCount, 0);
            assert.ok(this.containedAdManager.requestAds.calledWith(this.adRequestContext));
            assert.equal(this.containedAdManager.requestAds.callCount, 1);
        });

        [
            {
                playerType: PlayerType.PLAYER_SITE,
                reason: null,
            },
            {
                playerType: PlayerType.PLAYER_EMBED,
                reason: null,
            },
            {
                playerType: PlayerType.PLAYER_POPOUT,
                reason: null,
            },
            {
                playerType: PlayerType.PLAYER_FRONTPAGE,
                reason: 'reason_frontpage',
            },
            {
                playerType: PlayerType.PLAYER_CREATIVE,
                reason: 'reason_creative_player',
            },
            {
                playerType: PlayerType.PLAYER_FACEBOOK,
                reason: 'reason_facebook',
            },
            {
                playerType: PlayerType.PLAYER_HIGHLIGHTER,
                reason: 'reason_highlighter',
            },
            {
                playerType: PlayerType.PLAYER_DASHBOARD,
                reason: 'reason_dashboard',
            },
            {
                playerType: PlayerType.PLAYER_ANIMATED_THUMBNAILS,
                reason: 'reason_animated_thumbnails',
            },
        ].forEach(({ playerType, reason }) => {
            const decline = (reason === null ? 'not decline' : `decline with \`${reason}\``);

            QUnit.test(`should ${decline} when the player is on ${playerType}`, function(assert) {
                this.adRequestContext.playerType = playerType;

                const adManager = new DecliningAdsManager(this.store, this.containedAdManager, this.analytics);

                adManager.requestAds(this.adRequestContext);
                assert.equal(this.containedAdManager.requestAds.callCount, (reason === null ? 1 : 0));
                assert.equal(this.analytics.trackEvent.callCount, (reason === null ? 0 : 1));

                if (reason !== null) {
                    assert.equal(this.containedAdManager.requestAds.callCount, 0, 'should not call the ad manager');
                    assertTrackEvent(this, assert, { [reason]: true });
                } else {
                    assert.ok(this.containedAdManager.requestAds.calledWith(this.adRequestContext));
                }
            });
        });

        [
            {
                adType: PREROLL,
                decline: true,
            },
            {
                adType: MIDROLL,
                decline: false,
            },
            {
                adType: POSTROLL,
                decline: true,
            },
        ].forEach(({ adType, decline }) => {
            const declineStr = (decline ? 'decline' : 'not decline');
            QUnit.skip(`should ${declineStr} when multiple ${adType} ads are requested`, function(assert) {
                this.adRequestContext.adType = adType;

                const adManager = new DecliningAdsManager(this.store, this.containedAdManager, this.analytics);

                adManager.requestAds(this.adRequestContext);

                const secondAdRequestContext = createRequestContext(this.window);
                secondAdRequestContext.adType = adType;

                adManager.requestAds(this.adRequestContext);
                assert.equal(this.containedAdManager.requestAds.callCount, (decline ? 1 : 2));
                assert.equal(this.analytics.trackEvent.callCount, (decline ? 1 : 0));
                assert.ok(this.containedAdManager.requestAds.calledWith(this.adRequestContext));

                if (decline) {
                    assert.ok(this.analytics.trackEvent.calledWith(
                        'video_ad_request_declined',
                        trackEventPayload(
                            {
                                // eslint-disable-next-line
                                reason_already_seen_rolltype: true,
                            },
                            secondAdRequestContext
                        )
                    ));
                } else {
                    assert.ok(this.containedAdManager.requestAds.calledWith(secondAdRequestContext));
                }
            });

            QUnit.test(`should not decline when ${adType}s are requested on a different stream`, function(assert) {
                this.adRequestContext.adType = adType;

                const adManager = new DecliningAdsManager(this.store, this.containedAdManager, this.analytics);

                adManager.requestAds(this.adRequestContext);
                const secondAdRequestContext = createRequestContext(this.window);
                secondAdRequestContext.adType = adType;
                secondAdRequestContext.channel = `otherchannel_${QUnit.config.current.testId}`;

                adManager.requestAds(secondAdRequestContext);
                assert.equal(this.analytics.trackEvent.callCount, 0);
                assert.equal(this.containedAdManager.requestAds.callCount, 2);
                assert.ok(this.containedAdManager.requestAds.calledWith(this.adRequestContext));
                assert.ok(this.containedAdManager.requestAds.calledWith(secondAdRequestContext));
            });
        });

        [true, false, 'unknown'].forEach(function(adblockState) {
            const expectedVal = adblockState === 'unknown' ? null : adblockState;

            // eslint-disable-next-line max-len
            QUnit.test(`should set the adblock param to ${expectedVal} when the adblock state is ${adblockState}`, function(assert) {
                this.adRequestContext.adblock = adblockState;
                this.adRequestContext.playerType =  PlayerType.PLAYER_FRONTPAGE;

                const adManager = new DecliningAdsManager(this.store, this.containedAdManager, this.analytics);

                adManager.requestAds(this.adRequestContext);
                assert.equal(this.containedAdManager.requestAds.callCount, 0, 'should not call the ad manager');
                assertTrackEvent(this, assert, {
                    adblock: expectedVal,
                    // eslint-disable-next-line
                    reason_frontpage: true,
                });
            });
        });

        QUnit.test('pauses the contained ad manager when paused', function(assert) {
            const adManager = new DecliningAdsManager(this.store, this.containedAdManager, this.analytics);

            adManager.pause();

            assert.equal(this.containedAdManager.pause.callCount, 1);
        });

        QUnit.test('resumes the contained ad manager when play is called', function(assert) {
            const adManager = new DecliningAdsManager(this.store, this.containedAdManager, this.analytics);

            adManager.play();

            assert.equal(this.containedAdManager.play.callCount, 1);
        });

        QUnit.test('sets the volume of the contained ad manager when volume is set', function(assert) {
            const adManager = new DecliningAdsManager(this.store, this.containedAdManager, this.analytics);

            const volume = Math.random();
            adManager.setVolume(volume);

            assert.equal(this.containedAdManager.setVolume.callCount, 1);
            assert.ok(this.containedAdManager.setVolume.calledWith(volume));
        });

        QUnit.test('sets the contained ad manager to muted when muted', function(assert) {
            const adManager = new DecliningAdsManager(this.store, this.containedAdManager, this.analytics);

            adManager.setMuted(true);

            assert.equal(this.containedAdManager.setMuted.callCount, 1);
            assert.ok(this.containedAdManager.setMuted.calledWith(true));

            adManager.setMuted(false);
            assert.equal(this.containedAdManager.setMuted.callCount, 2);
            assert.equal(this.containedAdManager.setMuted.secondCall.args[0], false);
        });

        QUnit.test('proxies the contained ad manager `paused` property', function(assert) {
            this.containedAdManager.paused = true;

            const adManager = new DecliningAdsManager(this.store, this.containedAdManager, this.analytics);

            assert.equal(adManager.paused, true);

            this.containedAdManager.paused = false;

            assert.equal(adManager.paused, false);
        });

        QUnit.test('proxies the contained ad manager `sdk` property', function(assert) {
            this.containedAdManager.sdk = 'html5';

            const adManager = new DecliningAdsManager(this.store, this.containedAdManager, this.analytics);

            assert.equal(adManager.sdk, this.containedAdManager.sdk);

            this.containedAdManager.sdk = 'flash';

            assert.equal(adManager.sdk, this.containedAdManager.sdk);
        });
    });
});

function createRequestContext(window) {
    return new AdsRequestContext({
        adType: PREROLL,
        duration: Math.floor(Math.random() * 30 + 30),
        state: {
            ads: {
                adblock: false,
            },
            adsManager: {
                sdk: 'sdkname',
            },
            env: {
                playerType: PlayerType.PLAYER_POPOUT,
            },
            window,
            stream: {},
        },
        lastAdDisplay: 0,
        userInfo: {
            turbo: null,
        },
        channelAPIInfo: {
            prerolls: true,
            postrolls: true,
        },
        channelInfo: {
            name: `channel_${QUnit.config.current.testId}`,
            game: `Game ${QUnit.config.current.testId}`,
            mature: true,
            partner: true,
        },
        viewerInfo: {
            chansub: null,
            // eslint-disable-next-line camelcase
            has_ad_free_subscription: null,
        },
        communitiesInfo: {
            communities: [],
        },
        channelAdProperties: {
            /* eslint-disable camelcase */
            valid_responses: {
                vod_ads_enabled: true,
            },
            /* eslint-enable camelcase */
        },
        raid: false,
    });
}

function trackEventPayload(customization, adRequestContext) {
    return assign(
        {
            /* eslint-disable camelcase */
            ad_session_id: adRequestContext.adSessionId,
            roll_type: adRequestContext.adType,
            time_break: adRequestContext.duration,
            twitch_correlator: adRequestContext.twitchCorrelator,
            provider: 'ima',
            adblock: false,
            // Commenting this reason out. We are going to deprecate the experiment
            // associated with this reason for now and investigate this
            // further before enabling it.

            // reason_already_seen_rolltype: false,
            reason_channeladfree: false,
            reason_channelsub: false,
            reason_creative_player: false,
            reason_facebook: false,
            reason_frontpage: false,
            reason_highlighter: false,
            reason_dashboard: false,
            reason_animated_thumbnails: false,
            reason_mobile_web_upsell: false,
            reason_ratelimit: false,
            reason_short_vod: false,
            reason_turbo: false,
            reason_vod_creator: false,
            reason_raid: false,
            reason_vod_midroll: false,
            reason_embed_promo: false,
            reason_vod_ads_disabled: false,
            /* eslint-enable camelcase */
        },
        customization
    );
}
