import sinon from 'sinon';
import { unitTest } from 'tests/utils/module';
import { buildIMATags, MIDROLL } from 'ads/ima-tags';
import { AdsRequestContext } from 'ads/adscontext';
import { FlashManager } from 'ads/flash-manager';
import { init as initStore } from 'state';
import * as PlayerType from 'util/player-type';
import { AdContentTypes, AdRollTypes, setCurrentAdMetadata } from 'actions/ads';
import * as AdSpadeEvents from 'analytics/spade-events';
import * as AdEvents from 'ads/advertisement-event';
import { FLASH_AD_EVENTS } from 'backend/flash';
import { VODTwitchContentStream } from 'stream/twitch-vod';
import { LiveTwitchContentStream } from 'stream/twitch-live';
import { ACTION_SET_STREAM } from 'actions/stream';
import { ACTION_SET_STREAMMETADATA } from 'actions/stream-metadata';
import { setAnalyticsTracker } from 'actions/analytics-tracker';

unitTest('ads | flash-manager', function(hooks) {
    hooks.beforeEach(function() {
        this.backend = {
            requestAdFill: sinon.spy(),
            play: sinon.spy(),
            pause: sinon.spy(),
            setVolume: sinon.spy(),
            setMuted: sinon.spy(),
            getPaused: sinon.spy(),
            getMuted: sinon.spy(),
            addEventListener: (eventName, callback) => {
                this.backendEventListeners[eventName] = callback;
            },
        };

        this.backendEventListeners = {};

        this.store = initStore();
    });

    QUnit.test('should have the "flash" sdk', function(assert) {
        const flashManager = new FlashManager(this.backend, this.store);
        assert.equal(flashManager.sdk, 'flash');
    });

    QUnit.module('for spade ad events', function(hooks) {
        hooks.beforeEach(function() {
            this.analytics = {
                trackEvent: sinon.spy(),
            };
            this.store.dispatch(setAnalyticsTracker(this.analytics));
        });

        QUnit.module('when on a VOD, add vod properties to', function(hooks) {
            hooks.beforeEach(function() {
                this.vodStreamMetadata = {
                    type: 'upload',
                };
                this.videoID = 'v12345';
                this.vodStream = new VODTwitchContentStream(this.videoID);

                this.store.dispatch({
                    type: ACTION_SET_STREAM,
                    stream: this.vodStream,
                });
                this.store.dispatch({
                    type: ACTION_SET_STREAMMETADATA,
                    streamMetadata: this.vodStreamMetadata,
                });
            });

            QUnit.test('AD_IMPRESSION spade event sent on AD_IMPRESSION from backend', function(assert) {
                // eslint-disable-next-line no-unused-vars
                const flashManager = new FlashManager(this.backend, this.store);
                this.backendEventListeners[FLASH_AD_EVENTS.AD_IMPRESSION]({});

                assert.equal(this.analytics.trackEvent.callCount, 1);

                const [eventName, payload] = this.analytics.trackEvent.firstCall.args;

                assert.equal(eventName, AdSpadeEvents.AD_IMPRESSION);
                assert.equal(payload.vod_id, this.videoID);
                assert.equal(payload.vod_type, this.vodStreamMetadata.type);
            });

            // eslint-disable-next-line max-len
            QUnit.test('AD_IMPRESSION_COMPLETE spade event sent on AD_IMPRESSION_COMPLETE from backend', function(assert) {
                // eslint-disable-next-line no-unused-vars
                const flashManager = new FlashManager(this.backend, this.store);
                this.backendEventListeners[FLASH_AD_EVENTS.AD_IMPRESSION_COMPLETE]({});

                assert.equal(this.analytics.trackEvent.callCount, 1);

                const [eventName, payload] = this.analytics.trackEvent.firstCall.args;

                assert.equal(eventName, AdSpadeEvents.AD_IMPRESSION_COMPLETE);
                assert.equal(payload.vod_id, this.videoID);
                assert.equal(payload.vod_type, this.vodStreamMetadata.type);
            });

            QUnit.test('AD_REQUEST spade event sent on AD_REQUEST from backend', function(assert) {
                // eslint-disable-next-line no-unused-vars
                const flashManager = new FlashManager(this.backend, this.store);
                this.backendEventListeners[FLASH_AD_EVENTS.AD_REQUEST]({});

                assert.equal(this.analytics.trackEvent.callCount, 1);

                const [eventName, payload] = this.analytics.trackEvent.firstCall.args;

                assert.equal(eventName, AdSpadeEvents.AD_REQUEST);
                assert.equal(payload.vod_id, this.videoID);
                assert.equal(payload.vod_type, this.vodStreamMetadata.type);
            });

            QUnit.test('AD_REQUEST_DECLINED spade event sent on AD_REQUEST_DECLINED from backend', function(assert) {
                // eslint-disable-next-line no-unused-vars
                const flashManager = new FlashManager(this.backend, this.store);
                this.backendEventListeners[FLASH_AD_EVENTS.AD_REQUEST_DECLINED]({});

                assert.equal(this.analytics.trackEvent.callCount, 1);

                const [eventName, payload] = this.analytics.trackEvent.firstCall.args;

                assert.equal(eventName, AdSpadeEvents.AD_REQUEST_DECLINED);
                assert.equal(payload.vod_id, this.videoID);
                assert.equal(payload.vod_type, this.vodStreamMetadata.type);
            });

            QUnit.test('AD_REQUEST_RESPONSE spade event sent on AD_REQUEST_RESPONSE from backend', function(assert) {
                // eslint-disable-next-line no-unused-vars
                const flashManager = new FlashManager(this.backend, this.store);
                this.backendEventListeners[FLASH_AD_EVENTS.AD_REQUEST_RESPONSE]({});

                assert.equal(this.analytics.trackEvent.callCount, 1);

                const [eventName, payload] = this.analytics.trackEvent.firstCall.args;

                assert.equal(eventName, AdSpadeEvents.AD_REQUEST_RESPONSE);
                assert.equal(payload.vod_id, this.videoID);
                assert.equal(payload.vod_type, this.vodStreamMetadata.type);
            });

            QUnit.test('AD_REQUEST_ERROR spade event sent on AD_REQUEST_ERROR from backend', function(assert) {
                // eslint-disable-next-line no-unused-vars
                const flashManager = new FlashManager(this.backend, this.store);
                this.backendEventListeners[FLASH_AD_EVENTS.AD_REQUEST_ERROR]({});

                assert.equal(this.analytics.trackEvent.callCount, 1);

                const [eventName, payload] = this.analytics.trackEvent.firstCall.args;

                assert.equal(eventName, AdSpadeEvents.AD_REQUEST_ERROR);
                assert.equal(payload.vod_id, this.videoID);
                assert.equal(payload.vod_type, this.vodStreamMetadata.type);
            });

            QUnit.test('AD_ERROR spade event sent on AD_ERROR from backend', function(assert) {
                // eslint-disable-next-line no-unused-vars
                const flashManager = new FlashManager(this.backend, this.store);
                this.backendEventListeners[FLASH_AD_EVENTS.AD_ERROR]({});

                assert.equal(this.analytics.trackEvent.callCount, 1);

                const [eventName, payload] = this.analytics.trackEvent.firstCall.args;

                assert.equal(eventName, AdSpadeEvents.AD_ERROR);
                assert.equal(payload.vod_id, this.videoID);
                assert.equal(payload.vod_type, this.vodStreamMetadata.type);
            });

            QUnit.test('AD_LOADED spade event sent on AD_LOADED from backend', function(assert) {
                // eslint-disable-next-line no-unused-vars
                const flashManager = new FlashManager(this.backend, this.store);
                this.backendEventListeners[FLASH_AD_EVENTS.AD_LOADED]({});

                assert.equal(this.analytics.trackEvent.callCount, 1);

                const [eventName, payload] = this.analytics.trackEvent.firstCall.args;

                assert.equal(eventName, AdSpadeEvents.AD_LOADED);
                assert.equal(payload.vod_id, this.videoID);
                assert.equal(payload.vod_type, this.vodStreamMetadata.type);
            });
        });

        QUnit.module('when not on a VOD do not add vod properties on', function(hooks) {
            hooks.beforeEach(function() {
                this.store.dispatch({
                    type: ACTION_SET_STREAM,
                    stream: new LiveTwitchContentStream(),
                });
            });

            QUnit.test('AD_IMPRESSION spade event sent on AD_IMPRESSION from backend', function(assert) {
                // eslint-disable-next-line no-unused-vars
                const flashManager = new FlashManager(this.backend, this.store);
                this.backendEventListeners[FLASH_AD_EVENTS.AD_IMPRESSION]({});

                assert.equal(this.analytics.trackEvent.callCount, 1);

                const [eventName, payload] = this.analytics.trackEvent.firstCall.args;

                assert.equal(eventName, AdSpadeEvents.AD_IMPRESSION);
                assert.ok(!payload.hasOwnProperty('vod_id'));
                assert.ok(!payload.hasOwnProperty('vod_type'));
            });

            // eslint-disable-next-line max-len
            QUnit.test('AD_IMPRESSION_COMPLETE spade event sent on AD_IMPRESSION_COMPLETE from backend', function(assert) {
                // eslint-disable-next-line no-unused-vars
                const flashManager = new FlashManager(this.backend, this.store);
                this.backendEventListeners[FLASH_AD_EVENTS.AD_IMPRESSION_COMPLETE]({});

                assert.equal(this.analytics.trackEvent.callCount, 1);

                const [eventName, payload] = this.analytics.trackEvent.firstCall.args;

                assert.equal(eventName, AdSpadeEvents.AD_IMPRESSION_COMPLETE);
                assert.ok(!payload.hasOwnProperty('vod_id'));
                assert.ok(!payload.hasOwnProperty('vod_type'));
            });

            QUnit.test('AD_REQUEST spade event sent on AD_REQUEST from backend', function(assert) {
                // eslint-disable-next-line no-unused-vars
                const flashManager = new FlashManager(this.backend, this.store);
                this.backendEventListeners[FLASH_AD_EVENTS.AD_REQUEST]({});

                assert.equal(this.analytics.trackEvent.callCount, 1);

                const [eventName, payload] = this.analytics.trackEvent.firstCall.args;

                assert.equal(eventName, AdSpadeEvents.AD_REQUEST);
                assert.ok(!payload.hasOwnProperty('vod_id'));
                assert.ok(!payload.hasOwnProperty('vod_type'));
            });

            QUnit.test('AD_REQUEST_DECLINED spade event sent on AD_REQUEST_DECLINED from backend', function(assert) {
                // eslint-disable-next-line no-unused-vars
                const flashManager = new FlashManager(this.backend, this.store);
                this.backendEventListeners[FLASH_AD_EVENTS.AD_REQUEST_DECLINED]({});

                assert.equal(this.analytics.trackEvent.callCount, 1);

                const [eventName, payload] = this.analytics.trackEvent.firstCall.args;

                assert.equal(eventName, AdSpadeEvents.AD_REQUEST_DECLINED);
                assert.ok(!payload.hasOwnProperty('vod_id'));
                assert.ok(!payload.hasOwnProperty('vod_type'));
            });

            QUnit.test('AD_REQUEST_RESPONSE spade event sent on AD_REQUEST_RESPONSE from backend', function(assert) {
                // eslint-disable-next-line no-unused-vars
                const flashManager = new FlashManager(this.backend, this.store);
                this.backendEventListeners[FLASH_AD_EVENTS.AD_REQUEST_RESPONSE]({});

                assert.equal(this.analytics.trackEvent.callCount, 1);

                const [eventName, payload] = this.analytics.trackEvent.firstCall.args;

                assert.equal(eventName, AdSpadeEvents.AD_REQUEST_RESPONSE);
                assert.ok(!payload.hasOwnProperty('vod_id'));
                assert.ok(!payload.hasOwnProperty('vod_type'));
            });

            QUnit.test('AD_REQUEST_ERROR spade event sent on AD_REQUEST_ERROR from backend', function(assert) {
                // eslint-disable-next-line no-unused-vars
                const flashManager = new FlashManager(this.backend, this.store);
                this.backendEventListeners[FLASH_AD_EVENTS.AD_REQUEST_ERROR]({});

                assert.equal(this.analytics.trackEvent.callCount, 1);

                const [eventName, payload] = this.analytics.trackEvent.firstCall.args;

                assert.equal(eventName, AdSpadeEvents.AD_REQUEST_ERROR);
                assert.ok(!payload.hasOwnProperty('vod_id'));
                assert.ok(!payload.hasOwnProperty('vod_type'));
            });

            QUnit.test('AD_ERROR spade event sent on AD_ERROR from backend', function(assert) {
                // eslint-disable-next-line no-unused-vars
                const flashManager = new FlashManager(this.backend, this.store);
                this.backendEventListeners[FLASH_AD_EVENTS.AD_ERROR]({});

                assert.equal(this.analytics.trackEvent.callCount, 1);

                const [eventName, payload] = this.analytics.trackEvent.firstCall.args;

                assert.equal(eventName, AdSpadeEvents.AD_ERROR);
                assert.ok(!payload.hasOwnProperty('vod_id'));
                assert.ok(!payload.hasOwnProperty('vod_type'));
            });

            QUnit.test('AD_LOADED spade event sent on AD_LOADED from backend', function(assert) {
                // eslint-disable-next-line no-unused-vars
                const flashManager = new FlashManager(this.backend, this.store);
                this.backendEventListeners[FLASH_AD_EVENTS.AD_LOADED]({});

                assert.equal(this.analytics.trackEvent.callCount, 1);

                const [eventName, payload] = this.analytics.trackEvent.firstCall.args;

                assert.equal(eventName, AdSpadeEvents.AD_LOADED);
                assert.ok(!payload.hasOwnProperty('vod_id'));
                assert.ok(!payload.hasOwnProperty('vod_type'));
            });
        });
    });

    QUnit.module('sets ads.currentMetadata to', function() {
        QUnit.test('contentType.IMA and rollType.PREROLL on FLASH_AD_EVENTS.AD_START from backend', function(assert) {
            // eslint-disable-next-line no-unused-vars
            const flashManager = new FlashManager(this.backend, this.store);
            assert.equal(this.store.getState().ads.currentMetadata.contentType, AdContentTypes.NONE);
            assert.equal(this.store.getState().ads.currentMetadata.rollType, AdRollTypes.NONE);

            this.backendEventListeners[FLASH_AD_EVENTS.AD_START]({
                // eslint-disable-next-line camelcase
                roll_type: 'preroll',
            });

            assert.equal(this.store.getState().ads.currentMetadata.contentType, AdContentTypes.IMA);
            assert.equal(this.store.getState().ads.currentMetadata.rollType, AdRollTypes.PREROLL);
        });

        QUnit.test('contentType.NONE and rollType.NONE on FLASH_AD_EVENTS.AD_END from backend', function(assert) {
            // eslint-disable-next-line no-unused-vars
            const flashManager = new FlashManager(this.backend, this.store);
            this.store.dispatch(setCurrentAdMetadata({
                contentType: AdContentTypes.IMA,
                rollType: AdRollTypes.PREROLL,
            }));
            assert.equal(this.store.getState().ads.currentMetadata.contentType, AdContentTypes.IMA);
            assert.equal(this.store.getState().ads.currentMetadata.rollType, AdRollTypes.PREROLL);

            this.backendEventListeners[FLASH_AD_EVENTS.AD_END]({
                // eslint-disable-next-line camelcase
                roll_type: 'preroll',
            });

            assert.equal(this.store.getState().ads.currentMetadata.contentType, AdContentTypes.NONE);
            assert.equal(this.store.getState().ads.currentMetadata.rollType, AdRollTypes.NONE);
        });
    });

    QUnit.module('emits', function() {
        QUnit.test('AD_START on FLASH_AD_EVENTS.AD_START from backend', function(assert) {
            const flashManager = new FlashManager(this.backend, this.store);
            const listenerSpy = sinon.spy();
            const eventPayload = {
                // eslint-disable-next-line camelcase
                roll_type: 'preroll',
            };

            flashManager.addEventListener(AdEvents.AD_START, listenerSpy);
            this.backendEventListeners[FLASH_AD_EVENTS.AD_START](eventPayload);

            const [payload] = listenerSpy.firstCall.args;
            assert.equal(listenerSpy.callCount, 1);
            assert.deepEqual(payload, {
                // eslint-disable-next-line camelcase
                roll_type: eventPayload.roll_type,
            });
        });

        QUnit.test('AD_END on FLASH_AD_EVENTS.AD_END from backend', function(assert) {
            const flashManager = new FlashManager(this.backend, this.store);
            const listenerSpy = sinon.spy();
            const eventPayload = {
                // eslint-disable-next-line camelcase
                roll_type: 'preroll',
            };

            flashManager.addEventListener(AdEvents.AD_END, listenerSpy);
            this.backendEventListeners[FLASH_AD_EVENTS.AD_END](eventPayload);

            const [payload] = listenerSpy.firstCall.args;
            assert.equal(listenerSpy.callCount, 1);
            assert.deepEqual(payload, {
                // eslint-disable-next-line camelcase
                roll_type: eventPayload.roll_type,
            });
        });

        QUnit.test('COMPANION_RENDERED on FLASH_AD_EVENTS.COMPANION_RENDERED from backend', function(assert) {
            const flashManager = new FlashManager(this.backend, this.store);
            const listenerSpy = sinon.spy();

            flashManager.addEventListener(AdEvents.COMPANION_RENDERED, listenerSpy);
            this.backendEventListeners[FLASH_AD_EVENTS.COMPANION_RENDERED]();
            assert.equal(1, listenerSpy.callCount);
        });

        QUnit.test('AD_IMPRESSION on FLASH_AD_EVENTS.AD_IMPRESSION from backend', function(assert) {
            const flashManager = new FlashManager(this.backend, this.store);
            const listenerSpy = sinon.spy();
            const eventPayload = {
                // eslint-disable-next-line camelcase
                time_break: 30,
            };

            flashManager.addEventListener(AdEvents.AD_IMPRESSION, listenerSpy);
            this.backendEventListeners[FLASH_AD_EVENTS.AD_IMPRESSION](eventPayload);

            const [payload] = listenerSpy.firstCall.args;
            assert.equal(listenerSpy.callCount, 1);
            assert.deepEqual(payload, {
                // eslint-disable-next-line camelcase
                time_break: eventPayload.time_break,
            });
        });

        QUnit.test('AD_IMPRESSION_COMPLETE on FLASH_AD_EVENTS.AD_IMPRESSION_COMPLETE from backend', function(assert) {
            const flashManager = new FlashManager(this.backend, this.store);
            const listenerSpy = sinon.spy();

            flashManager.addEventListener(AdEvents.AD_IMPRESSION_COMPLETE, listenerSpy);
            this.backendEventListeners[FLASH_AD_EVENTS.AD_IMPRESSION_COMPLETE]();

            assert.equal(1, listenerSpy.callCount);
        });
    });

    QUnit.module('requestAds', function(hooks) {
        hooks.beforeEach(function() {
            this.adsRequestContext = new AdsRequestContext({
                adtype: MIDROLL,
                duration: Math.floor(Math.random() * 30 + 30),
                state: {
                    ads: {
                        adblock: false,
                    },
                    adsManager: {
                        sdk: 'sdkname',
                    },
                    env: {
                        playerType: PlayerType.PLAYER_POPOUT,
                    },
                    window: {
                        document: {
                            referrer: `https://${QUnit.config.current.testId.toLowerCase()}.othersite.com`,
                        },
                        location: {
                            href: 'https://player.twitch.tv',
                        },
                        top: {
                            location: {
                                hostname: 'media.curse.com',
                            },
                        },
                    },
                    stream: {},
                },
                lastaddisplay: 0,
                userInfo: {
                    turbo: null,
                },
                channelAPIInfo: {
                    prerolls: true,
                    postrolls: true,
                },
                channelInfo: {
                    name: `channel_${QUnit.config.current.testId}`,
                    game: `Game ${QUnit.config.current.testId}`,
                    mature: true,
                    partner: true,
                },
                viewerInfo: {
                    chansub: null,
                    // eslint-disable-next-line camelcase
                    has_ad_free_subscription: null,
                },
                communitiesInfo: {
                    communities: [],
                },
                channelAdProperties: {
                    /* eslint-disable camelcase */
                    valid_responses: {
                        vod_ads_enabled: true,
                    },
                    /* eslint-enable camelcase */
                },
            });

            // freezing date so that IMA tags are comparable
            const now = Date.now();
            sinon.stub(Date, 'now', function() {
                return now;
            });
        });

        hooks.afterEach(function() {
            Date.now.restore();
        });

        QUnit.test('should request an ad fill from the backend, passing the ad tag URL', function(assert) {
            const flashManager = new FlashManager(
                this.backend,
                this.store
            );

            const expectedIMATags = buildIMATags(this.adsRequestContext);
            const { duration, adType } = this.adsRequestContext;
            flashManager.requestAds(this.adsRequestContext);
            assert.ok(this.backend.requestAdFill.calledWith(duration, adType, expectedIMATags));
        });
    });

    QUnit.module('when an ad is playing', function(hooks) {
        hooks.beforeEach(function() {
            this.store.getState = () => {
                return {
                    ads: {
                        current: {
                            contentType: AdContentTypes.IMA,
                            rollType: AdRollTypes.PREROLL,
                        },
                    },
                };
            };

            this.flashManager = new FlashManager(
                this.backend,
                this.store
            );
        });

        QUnit.test('play should ask the backend to play', function(assert) {
            this.flashManager.play();
            assert.ok(this.backend.play.called);
        });

        QUnit.test('pause should ask the backend to pause', function(assert) {
            this.flashManager.pause();
            assert.ok(this.backend.pause.called);
        });

        QUnit.test('setVolume should ask the backend to set the volume', function(assert) {
            const vol = 50;
            this.flashManager.setVolume(vol);
            assert.ok(this.backend.setVolume.calledWith(vol));
        });

        QUnit.test('setMuted should ask the backend to mute the player', function(assert) {
            this.flashManager.setMuted(true);
            assert.ok(this.backend.setMuted.calledWith(true));
        });

        QUnit.test('paused should defer to the backend', function(assert) {
            // eslint-disable-next-line no-unused-vars
            const paused = this.flashManager.paused;
            assert.ok(this.backend.getPaused.called);
        });

        QUnit.test('muted should defer to the backend', function(assert) {
            // eslint-disable-next-line no-unused-vars
            const muted = this.flashManager.muted;
            assert.ok(this.backend.getMuted.called);
        });
    });

    QUnit.module('when there is no ad showing', function(hooks) {
        hooks.beforeEach(function() {
            this.store.getState = () => {
                return {
                    ads: {
                        current: {
                            contentType: AdContentTypes.NONE,
                            rollType: AdRollTypes.NONE,
                        },
                    },
                };
            };

            this.flashManager = new FlashManager(
                this.backend,
                this.store
            );
        });

        QUnit.test('play should do nothing', function(assert) {
            this.flashManager.play();
            assert.notOk(this.backend.play.called);
        });

        QUnit.test('pause should do nothing', function(assert) {
            this.flashManager.pause();
            assert.notOk(this.backend.pause.called);
        });

        QUnit.test('setVolume should do nothing', function(assert) {
            this.flashManager.setVolume(50);
            assert.notOk(this.backend.setVolume.called);
        });

        QUnit.test('setMuted should do nothing', function(assert) {
            this.flashManager.setMuted(true);
            assert.notOk(this.backend.setMuted.called);
        });

        QUnit.test('paused should defer to the backend', function(assert) {
            // eslint-disable-next-line no-unused-vars
            const paused = this.flashManager.paused;
            assert.ok(this.backend.getPaused.called);
        });

        QUnit.test('muted should defer to the backend', function(assert) {
            // eslint-disable-next-line no-unused-vars
            const muted = this.flashManager.muted;
            assert.ok(this.backend.getMuted.called);
        });
    });
});

