import sinon from 'sinon';
import { unitTest } from 'tests/utils/module';
import EventEmitter from 'event-emitter';
import { setWindow } from 'actions/window';
import { ACTION_SET_STREAM } from 'actions/stream';
import { ACTION_SET_STREAMMETADATA } from 'actions/stream-metadata';
import { volumeChanged, playerMuted } from 'actions/playback';
import { setCurrentAdMetadata, AdContentTypes, AdRollTypes } from 'actions/ads';
import { setAnalyticsTracker } from 'actions/analytics-tracker';
import { setExperiments } from 'actions/experiments';
import { ACTION_AD_PLAY, ACTION_AD_PLAYING, ACTION_AD_PAUSE } from 'actions/event-emitter';
import { videoAPILoaded } from 'actions/video-api';
import { init as initStore } from 'state';
import { FakeGoogle } from 'tests/fakes/google.fake.js';
import { AdsRequestContext } from 'ads/adscontext';
import { IMAManager, IMA_PROVIDER, AD_BITRATE_MAX, AD_BITRATE_MIN } from 'ads/ima-manager';
import * as PlayerType from 'util/player-type';
import { PREROLL } from 'ads/ima-tags';
import { buildFakeWindow } from 'tests/fakes/window.fake';
import { VODTwitchContentStream } from 'stream/twitch-vod';
import { LiveTwitchContentStream } from 'stream/twitch-live';
import * as AdEvents from 'ads/advertisement-event';
import * as PlayerDimensionsActions from 'actions/player-dimensions';
import { ACTION_AD_COMPANION_EVENT } from 'actions/embed-event-emitter';
import pick from 'lodash/pick';
import { AAX_AD_AUCTION_ERROR } from 'analytics/spade-events';
import { waitFor } from 'tests/utils/waitFor';
/**
 * Helper for generating IMA style events
 */
function createAdEvent(adMock) {
    return {
        getAd() {
            return {
                getAdId: () => adMock.adId,
                getAdSystem: () => adMock.adSystem,
                getCreativeId: () => adMock.creativeId,
                getTitle: () => adMock.title,
                getDuration: () => adMock.duration,
                getWrapperAdIds: () => [adMock.wrapperAdIds],
                getWrapperAdSystems: () => [adMock.wrapperAdSystems],
                getWrapperCreativeIds: () => [adMock.creativeIds],
            };
        },
        getError() {
            return {
                getMessage: () => adMock.error.message,
                getErrorCode: () => adMock.error.code,
                getType: () => adMock.error.type,
            };
        },
        getUserRequestContext() {
            return adMock.adRequestContext;
        },
    };
}

unitTest('ads | ima-manager', function(hooks) {
    hooks.beforeEach(function() {
        this.videoContainer = window.document.createElement('div');
        this.videoContainer.id = 'root';
        const $videoEl = window.document.createElement('video');
        this.videoContainer.appendChild($videoEl);
        window.document.getElementById('qunit-fixture').appendChild(this.videoContainer);

        this.backend = {
            addEventListener: sinon.spy(),
            getCurrentTime: sinon.stub().returns(0),
            getDuration: sinon.stub().returns(Infinity),
            setVolume: sinon.spy(),
            setMuted: sinon.spy(),
            automatedPause: sinon.spy(),
            pause: sinon.spy(),
            play: sinon.spy(),
            getEnded: sinon.stub().returns(false),
            getStats: sinon.stub().returns({ playbackRate: AD_BITRATE_MAX }),
        };

        const nowStub = sinon.stub();
        this._dateNowDiff = 100;
        nowStub.onCall(0).returns(100);
        nowStub.onCall(1).returns(100 + this._dateNowDiff);
        this.window = buildFakeWindow({
            Date: {
                now: nowStub,
            },
        });
        this.analytics = {
            trackEvent: sinon.spy(),
        };
        this.store = initStore();
        const fakeExperiments = {
            get() {
                return Promise.resolve('');
            },
        };
        this.store.dispatch(setExperiments(fakeExperiments));
        this.store.dispatch(setWindow(this.window));
        this.store.dispatch(setAnalyticsTracker(this.analytics));
        this.store.dispatch(videoAPILoaded(this.backend));

        this.adRequestContext = new AdsRequestContext({
            adType: PREROLL,
            duration: Math.floor(Math.random() * 30 + 30),
            state: {
                ads: {
                    adblock: false,
                },
                adsManager: {
                    sdk: 'sdkname',
                },
                env: {
                    playerType: PlayerType.PLAYER_POPOUT,
                },
                window: {
                    document: {
                        referrer: `https://${QUnit.config.current.testId.toLowerCase()}.othersite.com`,
                    },
                    location: {
                        href: 'https://player.twitch.tv',
                    },
                    top: {
                        location: {
                            hostname: 'media.curse.com',
                        },
                    },
                },
                stream: {},
            },
            lastAdDisplay: 0,
            userInfo: {
                turbo: null,
                login: 'mrwiggles',
            },
            channelAPIInfo: {
                prerolls: true,
                postrolls: true,
            },
            channelInfo: {
                name: `channel_${QUnit.config.current.testId}`,
                game: `Game ${QUnit.config.current.testId}`,
                mature: true,
                partner: true,
            },
            viewerInfo: {
                chansub: null,
                // eslint-disable-next-line camelcase
                has_ad_free_subscription: null,
            },
            communitiesInfo: {
                communities: [],
            },
            channelAdProperties: {
                /* eslint-disable camelcase */
                valid_responses: {
                    vod_ads_enabled: true,
                },
                /* eslint-enable camelcase */
            },
        });

        this.vodStreamMetadata = {
            type: 'upload',
        };
        this.videoID = 'v12345';
        this.vodStream = new VODTwitchContentStream(this.videoID);
    });

    QUnit.module('IMAManager', function() {
        QUnit.test('should set VPAID mode to `insecure`', function(assert) {
            new IMAManager(this.videoContainer, this.backend, this.store);

            assert.ok(this.window.google.ima.settings.setVpaidMode.calledWith(
                this.window.google.ima.ImaSdkSettings.VpaidMode.INSECURE
            ));
        });

        QUnit.test('should create a display container', function(assert) {
            new IMAManager(this.videoContainer, this.backend, this.store);

            const displayContainer = this.videoContainer.getElementsByClassName('js-ima-ads-container')[0];

            assert.ok(this.window.google.ima.AdDisplayContainer.calledWithNew());
            assert.ok(this.window.google.ima.AdDisplayContainer.calledWith(displayContainer));
        });

        QUnit.test('should create an ads loader', function(assert) {
            new IMAManager(this.videoContainer, this.backend, this.store);

            const displayContainer = this.window.google.ima.AdDisplayContainer.returnValues[0];

            assert.equal(this.window.google.ima.AdsLoader.callCount, 1);
            assert.ok(this.window.google.ima.AdsLoader.calledWithNew());
            assert.ok(this.window.google.ima.AdsLoader.calledWith(displayContainer));
        });

        QUnit.test('should have the "html5" sdk', function(assert) {
            const imaManager = new IMAManager(this.videoContainer, this.backend, this.store);
            assert.equal(imaManager.sdk, 'html5');
        });

        QUnit.module('destroy', function() {
            QUnit.test('should destroy the ads loader', function(assert) {
                const imaManager = new IMAManager(this.videoContainer, this.backend, this.store);

                imaManager.destroy();

                const adsLoader = this.window.google.ima.AdsLoader.returnValues[0];

                assert.equal(adsLoader.destroy.callCount, 1);
            });

            QUnit.test('should destroy the ads display container', function(assert) {
                const imaManager = new IMAManager(this.videoContainer, this.backend, this.store);

                imaManager.destroy();

                const adsDisplayContainer = this.window.google.ima.AdDisplayContainer.returnValues[0];

                assert.equal(adsDisplayContainer.destroy.callCount, 1);
            });

            QUnit.test('destroy unsubs all items on store', function(assert) {
                const imaManager = new IMAManager(this.videoContainer, this.backend, this.store);
                sinon.spy(imaManager._currentAdsManager, 'resize');
                imaManager.destroy();

                this.store.dispatch({
                    type: PlayerDimensionsActions.ACTION_UPDATE_PLAYER_DIMENSIONS,
                    playerDimensions: {
                        width: 20,
                        height: 10,
                    },
                });

                assert.equal(imaManager._currentAdsManager.resize.callCount, 0);
                imaManager._currentAdsManager.resize.restore();
            });
        });

        QUnit.module('requestAds', function() {
            QUnit.module('when `google` has loaded', function(hooks) {
                hooks.beforeEach(function() {
                    this.window.google = new FakeGoogle();

                    this.imaManager = new IMAManager(
                        this.videoContainer,
                        this.backend,
                        this.store
                    );
                    this.aaxFetchBidStub = sinon.stub(this.imaManager._aaxManager, 'fetchBids');
                    this.aaxFetchBidStub.returns(Promise.resolve([]));
                });

                hooks.afterEach(function() {
                    this.aaxFetchBidStub.restore();
                });

                QUnit.test('should immediately send a `video_ad_request` event', function(assert) {
                    return this.imaManager.requestAds(this.adRequestContext).then(() => {
                        assert.equal(this.analytics.trackEvent.callCount, 1);

                        const [eventName, payload] = this.analytics.trackEvent.firstCall.args;

                        assert.equal(eventName, 'video_ad_request');
                        assert.deepEqual(payload, {
                            /* eslint-disable camelcase */
                            ad_session_id: this.adRequestContext.adSessionId,
                            adblock: this.adRequestContext.adblock,
                            provider: 'ima',
                            roll_type: 'preroll',
                            time_break: this.adRequestContext.duration,
                            twitch_correlator: this.adRequestContext.twitchCorrelator,
                            /* eslint-enable camelcase */
                        });
                    });
                });

                // eslint-disable-next-line max-len
                QUnit.test('should immediately send a `video_ad_request` event with `ads_opt_in` property if opted in', function(assert) {
                    this.adRequestContext.adWhitelist = true;
                    return this.imaManager.requestAds(this.adRequestContext).then(() => {
                        assert.equal(this.analytics.trackEvent.callCount, 1);

                        const [eventName, payload] = this.analytics.trackEvent.firstCall.args;

                        assert.equal(eventName, 'video_ad_request');
                        assert.deepEqual(payload, {
                            /* eslint-disable camelcase */
                            ad_session_id: this.adRequestContext.adSessionId,
                            adblock: this.adRequestContext.adblock,
                            ads_opt_in: true,
                            provider: 'ima',
                            roll_type: 'preroll',
                            time_break: this.adRequestContext.duration,
                            twitch_correlator: this.adRequestContext.twitchCorrelator,
                            /* eslint-enable camelcase */
                        });
                    });
                });

                // eslint-disable-next-line max-len
                QUnit.test('should immediately send a `video_ad_request` event with vod properties if on a vod', function(assert) {
                    this.store.dispatch({
                        type: ACTION_SET_STREAM,
                        stream: this.vodStream,
                    });
                    this.store.dispatch({
                        type: ACTION_SET_STREAMMETADATA,
                        streamMetadata: this.vodStreamMetadata,
                    });

                    return this.imaManager.requestAds(this.adRequestContext).then(() => {
                        assert.equal(this.analytics.trackEvent.callCount, 1);

                        const [eventName, payload] = this.analytics.trackEvent.firstCall.args;

                        assert.equal(eventName, 'video_ad_request');
                        assert.deepEqual(payload, {
                            /* eslint-disable camelcase */
                            ad_session_id: this.adRequestContext.adSessionId,
                            adblock: this.adRequestContext.adblock,
                            provider: 'ima',
                            roll_type: 'preroll',
                            time_break: this.adRequestContext.duration,
                            twitch_correlator: this.adRequestContext.twitchCorrelator,
                            vod_id: this.videoID,
                            vod_type: this.vodStreamMetadata.type,
                            /* eslint-enable camelcase */
                        });
                    });
                });

                QUnit.test('should return immediately if ad already playing', function(assert) {
                    const spy = sinon.spy(this.imaManager._adsLoader, 'requestAds');
                    const stub = sinon.stub(this.imaManager._currentAdsManager, 'getRemainingTime');
                    stub.returns(1);

                    this.imaManager.requestAds(this.adRequestContext);

                    assert.equal(spy.callCount, 0);
                    assert.equal(this.analytics.trackEvent.callCount, 0);
                    spy.restore();
                    stub.restore();
                });

                QUnit.test('should still call ima requestAds on aax exception', function(assert) {
                    const testError = new Error('TestError');
                    sinon.stub(this.imaManager._aaxManager, '_fetchPrerollBids').returns(new Promise(() => {
                        throw testError;
                    }));
                    const spy = sinon.spy(this.imaManager._adsLoader, 'requestAds');

                    return this.imaManager.requestAds(this.adRequestContext).then(() => {
                        assert.equal(spy.callCount, 1);
                    });
                });
            });

            QUnit.module('when `google` has not loaded', function(hooks) {
                hooks.beforeEach(function() {
                    delete this.window.google;

                    this.imaManager = new IMAManager(
                        this.videoContainer,
                        this.backend,
                        this.store
                    );
                    this.aaxFetchBidStub = sinon.stub(this.imaManager._aaxManager, 'fetchBids');
                    this.aaxFetchBidStub.returns(Promise.resolve([]));
                });

                hooks.afterEach(function() {
                    this.aaxFetchBidStub.restore();
                });

                // eslint-disable-next-line max-len
                QUnit.test('should send a `video_ad_request` event followed by `video_ad_request_error`', function(assert) {
                    delete this.window.google;

                    return this.imaManager.requestAds(this.adRequestContext).then(() => {
                        assert.equal(this.analytics.trackEvent.callCount, 2);

                        const [eventName1, payload1] = this.analytics.trackEvent.firstCall.args;
                        const [eventName2, payload2] = this.analytics.trackEvent.secondCall.args;

                        assert.equal(eventName1, 'video_ad_request');
                        assert.deepEqual(payload1, {
                            /* eslint-disable camelcase */
                            ad_session_id: this.adRequestContext.adSessionId,
                            adblock: this.adRequestContext.adblock,
                            provider: 'ima',
                            roll_type: 'preroll',
                            time_break: this.adRequestContext.duration,
                            twitch_correlator: this.adRequestContext.twitchCorrelator,
                            /* eslint-enable camelcase */
                        });

                        assert.equal(eventName2, 'video_ad_request_error');
                        assert.ok(typeof payload2.reason === 'string');
                        delete payload2.reason;

                        assert.deepEqual(payload2, {
                            /* eslint-disable camelcase */
                            ad_session_id: this.adRequestContext.adSessionId,
                            adblock: this.adRequestContext.adblock,
                            provider: 'ima',
                            roll_type: 'preroll',
                            time_break: this.adRequestContext.duration,
                            twitch_correlator: this.adRequestContext.twitchCorrelator,
                            /* eslint-enable camelcase */
                        });
                    });
                });

                // eslint-disable-next-line max-len
                QUnit.test('should send a `video_ad_request` event followed by `video_ad_request_error` with vod properties if on a vod', function(assert) {
                    delete this.window.google;

                    this.store.dispatch({
                        type: ACTION_SET_STREAM,
                        stream: this.vodStream,
                    });
                    this.store.dispatch({
                        type: ACTION_SET_STREAMMETADATA,
                        streamMetadata: this.vodStreamMetadata,
                    });

                    return this.imaManager.requestAds(this.adRequestContext).then(() => {
                        assert.equal(this.analytics.trackEvent.callCount, 2);

                        const [eventName1, payload1] = this.analytics.trackEvent.firstCall.args;
                        const [eventName2, payload2] = this.analytics.trackEvent.secondCall.args;

                        assert.equal(eventName1, 'video_ad_request');
                        assert.deepEqual(payload1, {
                            /* eslint-disable camelcase */
                            ad_session_id: this.adRequestContext.adSessionId,
                            adblock: this.adRequestContext.adblock,
                            provider: 'ima',
                            roll_type: 'preroll',
                            time_break: this.adRequestContext.duration,
                            twitch_correlator: this.adRequestContext.twitchCorrelator,
                            vod_id: this.videoID,
                            vod_type: this.vodStreamMetadata.type,
                            /* eslint-enable camelcase */
                        });

                        assert.equal(eventName2, 'video_ad_request_error');
                        assert.ok(typeof payload2.reason === 'string');
                        delete payload2.reason;

                        assert.deepEqual(payload2, {
                            /* eslint-disable camelcase */
                            ad_session_id: this.adRequestContext.adSessionId,
                            adblock: this.adRequestContext.adblock,
                            provider: 'ima',
                            roll_type: 'preroll',
                            time_break: this.adRequestContext.duration,
                            twitch_correlator: this.adRequestContext.twitchCorrelator,
                            vod_id: this.videoID,
                            vod_type: this.vodStreamMetadata.type,
                            /* eslint-enable camelcase */
                        });
                    });
                });

                QUnit.test('should send an AAX_AD_AUCTION_ERROR on aax exception', function(assert) {
                    const testError = new Error('TestError');
                    this.aaxFetchBidStub.returns(new Promise(() => {
                        throw testError;
                    }));

                    return this.imaManager.requestAds(this.adRequestContext).then(() => {
                        const [eventName, payload] = this.analytics.trackEvent.firstCall.args;
                        assert.equal(eventName, AAX_AD_AUCTION_ERROR);

                        assert.deepEqual(payload, {
                            /* eslint-disable camelcase */
                            ad_session_id: this.adRequestContext.adSessionId,
                            adblock: this.adRequestContext.adblock,
                            provider: 'ima',
                            roll_type: 'preroll',
                            time_break: this.adRequestContext.duration,
                            twitch_correlator: this.adRequestContext.twitchCorrelator,
                            reason: testError.message,
                            /* eslint-enable camelcase */
                        });
                    });
                });
            });

            QUnit.module('when creativeId set', function(hooks) {
                hooks.beforeEach(function() {
                    this.window.google = new FakeGoogle();

                    this.imaManager = new IMAManager(
                        this.videoContainer,
                        this.backend,
                        this.store
                    );

                    this.adRequestContext.creativeId = 1222333444555;

                    sinon.spy(this.imaManager, '_requestAdsInternal');
                    this.requestAdsStub = sinon.stub(this.imaManager._adsLoader, 'requestAds').returns();
                    this.getVastXmlResponseStub = sinon.stub(this.imaManager, '_getVastXmlResponse');
                });

                hooks.afterEach(function() {
                    this.requestAdsStub.restore();
                    this.getVastXmlResponseStub.restore();
                });

                QUnit.test('when fetch vast url succeeds, ads loader has adsResponse', function(assert) {
                    const vastResponse = 'vastResponse';
                    this.getVastXmlResponseStub.returns(Promise.resolve(vastResponse));

                    return this.imaManager.requestAds(this.adRequestContext).then(() => {
                        assert.equal(this.imaManager._requestAdsInternal.callCount, 0);
                        assert.equal(this.requestAdsStub.callCount, 1);

                        const adsRequest = this.requestAdsStub.firstCall.args[0];
                        assert.equal(adsRequest.adTagUrl, '');
                        assert.equal(adsRequest.adsResponse, vastResponse);
                    });
                });

                QUnit.test('when fetch fetching vast url fails, ads loader has no adsResponse', function(assert) {
                    this.getVastXmlResponseStub.returns(Promise.reject(new Error('error')));

                    return this.imaManager.requestAds(this.adRequestContext).then(() => {
                        assert.equal(this.imaManager._requestAdsInternal.callCount, 1);
                        assert.equal(this.requestAdsStub.callCount, 1);

                        const adsRequest = this.requestAdsStub.firstCall.args[0];
                        assert.ok(!adsRequest.adsResponse);
                    });
                });
            });
        });

        QUnit.module('when there is an ad error', function(hooks) {
            hooks.beforeEach(function() {
                this.window = buildFakeWindow({
                    setInterval: sinon.stub().returns(parseInt(QUnit.config.current.testId, 36)),
                    clearInterval: sinon.spy(),
                });

                this.store.dispatch(setWindow(this.window));

                this.adErrorEvent = createAdEvent({
                    adId: 'ad_123',
                    adSystem: 'GDFP',
                    creativeId: '456',
                    title: 'LUL Ad',
                    duration: 30,
                    wrapperAdIds: 'ad_wrapper_123',
                    wrapperAdSystems: 'Wrapper GDFP',
                    wrapperCreativeIds: '789',
                    error: {
                        message: 'Ad error event 123',
                        code: 123,
                        type: 'AD_PLAY',
                    },
                    adRequestContext: this.adRequestContext,
                });

                this.imaManager = new IMAManager(
                    this.videoContainer,
                    this.backend,
                    this.store
                );

                this.imaManager._currentAdsManager = {
                    destroy() {},
                    setVolume() {},
                    setMuted() {},
                };
            });

            QUnit.module('after ads have begun playback', function() {
                QUnit.test('sets ad content in statestore to NONE', function(assert) {
                    this.imaManager._onContentPauseRequested(this.adRequestContext);
                    assert.notEqual(this.store.getState().ads.currentMetadata.contentType, AdContentTypes.NONE);
                    assert.notEqual(this.store.getState().ads.currentMetadata.rollType, AdRollTypes.NONE);
                    this.imaManager._onAdError(this.adErrorEvent);
                    assert.equal(this.store.getState().ads.currentMetadata.contentType, AdContentTypes.NONE);
                    assert.equal(this.store.getState().ads.currentMetadata.rollType, AdRollTypes.NONE);
                });

                QUnit.test('maintains volume that is set during the ads', function(assert) {
                    const VOLUME = 0.123;
                    const MUTED = true;

                    this.imaManager._onContentPauseRequested(this.adRequestContext);
                    this.store.dispatch(volumeChanged(VOLUME));
                    this.store.dispatch(playerMuted(MUTED));
                    this.backend.setMuted.reset();
                    this.backend.setVolume.reset();
                    this.imaManager._onAdError(this.adErrorEvent);

                    assert.equal(this.backend.setMuted.firstCall.args[0], MUTED, 'should set backend muted');
                    assert.equal(this.backend.setVolume.firstCall.args[0], VOLUME, 'should set backend volume');
                });

                QUnit.test('resumes stream if it is a VOD and has not ended', function(assert) {
                    this.store.dispatch({
                        type: ACTION_SET_STREAM,
                        stream: new VODTwitchContentStream(),
                    });

                    this.imaManager._onContentPauseRequested(this.adRequestContext);
                    this.backend.play.reset();
                    this.imaManager._onAdError(this.adErrorEvent);
                    return waitFor(() => this.backend.play.called).then(() => {
                        assert.equal(this.backend.play.callCount, 1, 'should call backend play once');
                    });
                });
            });

            QUnit.test('should send a `video_ad_error` tracking event', function(assert) {
                this.imaManager._onAdError(this.adErrorEvent);

                assert.equal(this.analytics.trackEvent.callCount, 1);

                const [eventName, payload] = this.analytics.trackEvent.firstCall.args;

                assert.equal(eventName, 'video_ad_error');
                assert.deepEqual(payload, {
                    /* eslint-disable camelcase */
                    ad_session_id: this.adRequestContext.adSessionId,
                    provider: 'ima',
                    roll_type: this.adRequestContext.adType,
                    reason: this.adErrorEvent.getError().getMessage(),
                    time_break: this.adRequestContext.duration,
                    twitch_correlator: this.adRequestContext.twitchCorrelator,
                    adblock: this.adRequestContext.adblock,
                    error_code: this.adErrorEvent.getError().getErrorCode(),
                    error_type: this.adErrorEvent.getError().getType(),
                    /* eslint-enable camelcase */
                });
            });

            // eslint-disable-next-line max-len
            QUnit.test('should send a `video_ad_error` tracking event with vod properties if on a vod', function(assert) {
                this.store.dispatch({
                    type: ACTION_SET_STREAM,
                    stream: this.vodStream,
                });
                this.store.dispatch({
                    type: ACTION_SET_STREAMMETADATA,
                    streamMetadata: this.vodStreamMetadata,
                });

                this.imaManager._onAdError(this.adErrorEvent);

                assert.equal(this.analytics.trackEvent.callCount, 1);

                const [eventName, payload] = this.analytics.trackEvent.firstCall.args;

                assert.equal(eventName, 'video_ad_error');
                assert.deepEqual(payload, {
                    /* eslint-disable camelcase */
                    ad_session_id: this.adRequestContext.adSessionId,
                    provider: 'ima',
                    roll_type: this.adRequestContext.adType,
                    reason: this.adErrorEvent.getError().getMessage(),
                    time_break: this.adRequestContext.duration,
                    twitch_correlator: this.adRequestContext.twitchCorrelator,
                    vod_id: this.videoID,
                    vod_type: this.vodStreamMetadata.type,
                    adblock: this.adRequestContext.adblock,
                    error_code: this.adErrorEvent.getError().getErrorCode(),
                    error_type: this.adErrorEvent.getError().getType(),
                    /* eslint-enable camelcase */
                });
            });
        });

        QUnit.module('when an ads manager loads', function(hooks) {
            hooks.beforeEach(function() {
                this._adEventsEmitter = new EventEmitter();
                this.adsManagerLoadedEvent = {
                    getUserRequestContext: () => this.adRequestContext,
                    getAdsManager: sinon.stub().returns({
                        addEventListener: (event, cb) => {
                            this._adEventsEmitter.on(event, cb);
                        },
                        init: sinon.spy(),
                        setVolume: sinon.spy(),
                        start: sinon.spy(),
                    }),
                    getAds() {
                        return null;
                    },
                };

                this.imaManager = new IMAManager(
                    this.videoContainer,
                    this.backend,
                    this.store
                );
            });

            QUnit.test('should cap ad bitrate if playback rate above or equal to AD_BITRATE_MAX', function(assert) {
                const highBitrate = 5000;
                this.backend.getStats = sinon.stub().returns({ playbackRate: highBitrate });
                sinon.spy(this.imaManager, '_getTargetAdBitrate');
                this.imaManager._onAdsManagerLoaded(this.adsManagerLoadedEvent);
                assert.equal(this.imaManager._getTargetAdBitrate.callCount, 1);
                assert.equal(this.imaManager._getTargetAdBitrate.returnValues[0], AD_BITRATE_MAX);
            });

            QUnit.test('should set ad bitrate to 80% of playback rate', function(assert) {
                const lowBitrate = 1000;
                this.backend.getStats = sinon.stub().returns({ playbackRate: lowBitrate });
                sinon.spy(this.imaManager, '_getTargetAdBitrate');
                this.imaManager._onAdsManagerLoaded(this.adsManagerLoadedEvent);
                assert.equal(this.imaManager._getTargetAdBitrate.callCount, 1);
                assert.equal(this.imaManager._getTargetAdBitrate.returnValues[0], lowBitrate * 0.8);
            });

            QUnit.test('should set minimum bitrate if playback rate under AD_BITRATE_MIN', function(assert) {
                const lowBitrate = 400;
                this.backend.getStats = sinon.stub().returns({ playbackRate: lowBitrate });
                sinon.spy(this.imaManager, '_getTargetAdBitrate');
                this.imaManager._onAdsManagerLoaded(this.adsManagerLoadedEvent);
                assert.equal(this.imaManager._getTargetAdBitrate.callCount, 1);
                assert.equal(this.imaManager._getTargetAdBitrate.returnValues[0], AD_BITRATE_MIN);
            });

            QUnit.test('should emit play event when ad starts', function(assert) {
                this.imaManager._onAdsManagerLoaded(this.adsManagerLoadedEvent);
                sinon.spy(this.store, 'dispatch');

                const mockAdStartEvent = {
                    getAd: () => ({
                        getCompanionAds: () => [],
                    }),
                };
                this._adEventsEmitter.emit('start', mockAdStartEvent);

                assert.equal(this.store.dispatch.callCount, 1);
                assert.deepEqual(this.store.dispatch.firstCall.args[0], {
                    type: ACTION_AD_PLAY,
                });
            });

            QUnit.test('should emit ad companion events for embed api when ad companion exist', function(assert) {
                this.imaManager._onAdsManagerLoaded(this.adsManagerLoadedEvent);
                sinon.spy(this.store, 'dispatch');

                const mockAdStartEvent = {
                    getAd: () => ({
                        getCompanionAds: () => ['adCompanion'],
                    }),
                };
                this._adEventsEmitter.emit('start', mockAdStartEvent);

                assert.deepEqual(this.store.dispatch.firstCall.args[0], {
                    type: ACTION_AD_COMPANION_EVENT,
                    event: AdEvents.COMPANION_RENDERED,
                    data: { provider: IMA_PROVIDER },
                });
            });

            QUnit.test('should emit playing event when ad resumes', function(assert) {
                this.imaManager._onAdsManagerLoaded(this.adsManagerLoadedEvent);
                sinon.spy(this.store, 'dispatch');
                this._adEventsEmitter.emit('resume');
                assert.equal(this.store.dispatch.callCount, 1);
                assert.deepEqual(this.store.dispatch.firstCall.args[0], {
                    type: ACTION_AD_PLAYING,
                });
            });

            QUnit.test('should emit pause event when ad pauses', function(assert) {
                this.imaManager._onAdsManagerLoaded(this.adsManagerLoadedEvent);
                sinon.spy(this.store, 'dispatch');
                this._adEventsEmitter.emit('pause');
                assert.equal(this.store.dispatch.callCount, 1);
                assert.deepEqual(this.store.dispatch.firstCall.args[0], {
                    type: ACTION_AD_PAUSE,
                });
            });

            QUnit.test('should send a `video_ad_request_response` tracking event', function(assert) {
                this.imaManager._onAdsManagerLoaded(this.adsManagerLoadedEvent);

                assert.equal(this.analytics.trackEvent.callCount, 1);

                const [eventName, payload] = this.analytics.trackEvent.firstCall.args;

                assert.equal(eventName, 'video_ad_request_response');
                assert.deepEqual(payload, {
                    /* eslint-disable camelcase */
                    ad_session_id: this.adRequestContext.adSessionId,
                    provider: 'ima',
                    roll_type: this.adRequestContext.adType,
                    time_break: this.adRequestContext.duration,
                    twitch_correlator: this.adRequestContext.twitchCorrelator,
                    adblock: this.adRequestContext.adblock,
                    /* eslint-enable camelcase */
                });
            });

            // eslint-disable-next-line max-len
            QUnit.test('should send a `video_ad_request_response` tracking event with vod properties if on a vod', function(assert) {
                this.store.dispatch({
                    type: ACTION_SET_STREAM,
                    stream: this.vodStream,
                });
                this.store.dispatch({
                    type: ACTION_SET_STREAMMETADATA,
                    streamMetadata: this.vodStreamMetadata,
                });

                this.imaManager._onAdsManagerLoaded(this.adsManagerLoadedEvent);

                assert.equal(this.analytics.trackEvent.callCount, 1);

                const [eventName, payload] = this.analytics.trackEvent.firstCall.args;

                assert.equal(eventName, 'video_ad_request_response');
                assert.deepEqual(payload, {
                    /* eslint-disable camelcase */
                    ad_session_id: this.adRequestContext.adSessionId,
                    provider: 'ima',
                    roll_type: this.adRequestContext.adType,
                    time_break: this.adRequestContext.duration,
                    twitch_correlator: this.adRequestContext.twitchCorrelator,
                    vod_id: this.videoID,
                    vod_type: this.vodStreamMetadata.type,
                    adblock: this.adRequestContext.adblock,
                    /* eslint-enable camelcase */
                });
            });
        });

        QUnit.module('when an ad is loaded', function(hooks) {
            hooks.beforeEach(function() {
                this.imaManager = new IMAManager(
                    this.videoContainer,
                    this.backend,
                    this.store
                );

                this.imaManager._currentAdsManager = {
                    destroy() {},
                    setVolume: sinon.spy(),
                };

                this.adEvent = createAdEvent({
                    adId: 'ad_123',
                    adSystem: 'GDFP',
                    creativeId: '456',
                    title: 'LUL Ad',
                    duration: 30,
                    wrapperAdIds: 'ad_wrapper_123',
                    wrapperAdSystems: 'Wrapper GDFP',
                    wrapperCreativeIds: '789',
                    error: {},
                });
            });

            QUnit.test('should respect the player volume setting when not muted', function(assert) {
                const VOLUME = 0.456;
                const UNMUTED = false;

                this.store.dispatch(volumeChanged(VOLUME));
                this.store.dispatch(playerMuted(UNMUTED));
                this.imaManager._currentAdsManager.setVolume.reset();
                // Load ad
                this.imaManager._onAdLoaded(this.adEvent, this.adRequestContext);
                assert.equal(
                    this.imaManager._currentAdsManager.setVolume.firstCall.args[0],
                    VOLUME,
                    'should set the volume'
                );
                assert.equal(this.imaManager._currentAdsManager.setVolume.callCount, 1);
            });

            QUnit.test('should set the volume to 0 when muted', function(assert) {
                const VOLUME = 0.456;
                const MUTED = true;

                this.store.dispatch(volumeChanged(VOLUME));
                this.store.dispatch(playerMuted(MUTED));
                this.imaManager._currentAdsManager.setVolume.reset();
                // Load ad
                this.imaManager._onAdLoaded(this.adEvent, this.adRequestContext);
                assert.equal(
                    this.imaManager._currentAdsManager.setVolume.firstCall.args[0],
                    0,
                    'should set the volume to 0'
                );
                assert.equal(this.imaManager._currentAdsManager.setVolume.callCount, 1);
            });
        });

        QUnit.module('when a wrapper ad is returned', function(hooks) {
            hooks.beforeEach(function() {
                this.adEvent = createAdEvent({
                    adId: 'ad_123',
                    adSystem: 'GDFP',
                    creativeId: '456',
                    title: 'LUL Ad',
                    duration: 30,
                    wrapperAdIds: 'ad_wrapper_123',
                    wrapperAdSystems: 'Wrapper GDFP',
                    wrapperCreativeIds: '789',
                    error: {},
                });
                this.imaManager = new IMAManager(
                    this.videoContainer,
                    this.backend,
                    this.store
                );
            });
        });

        QUnit.module('when ads manager requests for content to be paused', function(hooks) {
            hooks.beforeEach(function() {
                this.window = buildFakeWindow({
                    setInterval: () => {},
                    clearInterval: () => {},
                });
                this.store.dispatch(setWindow(this.window));

                this.imaManager = new IMAManager(
                    this.videoContainer,
                    this.backend,
                    this.store
                );

                this.imaManager._currentAdsManager = {
                    destroy() {},
                    setVolume: sinon.spy(),
                };
            });

            QUnit.test('emit an AD_START event', function(assert) {
                const listenerSpy = sinon.spy();
                this.imaManager.addEventListener(AdEvents.AD_START, listenerSpy);

                this.imaManager._onContentPauseRequested(this.adRequestContext);

                const [payload] = listenerSpy.firstCall.args;
                assert.equal(listenerSpy.callCount, 1);
                assert.deepEqual(payload, {
                    // eslint-disable-next-line camelcase
                    roll_type: this.adRequestContext.adType,
                });
            });

            QUnit.test('sets ad content in statestore to IMA', function(assert) {
                assert.equal(this.store.getState().ads.currentMetadata.contentType, AdContentTypes.NONE);
                assert.equal(this.store.getState().ads.currentMetadata.rollType, AdRollTypes.NONE);
                this.imaManager._onContentPauseRequested(this.adRequestContext);
                assert.equal(this.store.getState().ads.currentMetadata.contentType, AdContentTypes.IMA);
                assert.equal(this.store.getState().ads.currentMetadata.rollType, AdRollTypes.PREROLL);
            });

            QUnit.test('live content is muted', function(assert) {
                this.store.dispatch({
                    type: ACTION_SET_STREAM,
                    stream: new LiveTwitchContentStream(),
                });

                const VOLUME = 0.456;
                const UNMUTED = false;

                this.store.dispatch(volumeChanged(VOLUME));
                this.store.dispatch(playerMuted(UNMUTED));
                this.backend.setMuted.reset();
                // Start ad
                this.imaManager._onContentPauseRequested(this.adRequestContext);
                assert.equal(this.backend.setMuted.firstCall.args[0], true, 'set backend to muted');
            });

            QUnit.test('the ads volume is set to the player volume if not muted', function(assert) {
                const VOLUME = 0.456;
                const UNMUTED = false;

                this.store.dispatch(volumeChanged(VOLUME));
                this.store.dispatch(playerMuted(UNMUTED));
                this.imaManager._currentAdsManager.setVolume.reset();
                // Start ad
                this.imaManager._onContentPauseRequested(this.adRequestContext);
                assert.equal(this.imaManager._currentAdsManager.setVolume.firstCall.args[0], VOLUME);
            });

            QUnit.test('the ads volume is set to 0 if the player volume is muted', function(assert) {
                const VOLUME = 0.456;
                const MUTED = true;

                this.store.dispatch(volumeChanged(VOLUME));
                this.store.dispatch(playerMuted(MUTED));
                this.imaManager._currentAdsManager.setVolume.reset();
                // Start ad
                this.imaManager._onContentPauseRequested(this.adRequestContext);
                assert.equal(this.imaManager._currentAdsManager.setVolume.firstCall.args[0], 0);
            });

            QUnit.test('vod content is paused', function(assert) {
                this.store.dispatch({
                    type: ACTION_SET_STREAM,
                    stream: new VODTwitchContentStream(),
                });

                this.backend.automatedPause.reset();
                // Start ad
                this.imaManager._onContentPauseRequested(this.adRequestContext);
                return waitFor(() => this.backend.automatedPause.called).then(() => {
                    assert.equal(this.backend.automatedPause.callCount, 1, 'backend automatedPause is called once');
                });
            });
        });

        QUnit.module('when an ad impression is registered with DFP', function(hooks) {
            hooks.beforeEach(function() {
                this.adEvent = createAdEvent({
                    adId: 'ad_123',
                    adSystem: 'GDFP',
                    creativeId: '456',
                    title: 'LUL Ad',
                    duration: 30,
                    wrapperAdIds: 'ad_wrapper_123',
                    wrapperAdSystems: 'Wrapper GDFP',
                    wrapperCreativeIds: '789',
                    error: {},
                });
                this.imaManager = new IMAManager(
                    this.videoContainer,
                    this.backend,
                    this.store
                );

                this.aaxFetchBidStub = sinon.stub(this.imaManager._aaxManager, 'fetchBids');
                this.aaxFetchBidStub.returns(Promise.resolve([]));

                return this.imaManager.requestAds(this.adRequestContext).then(() => {
                    this.analytics.trackEvent.reset();
                });
            });

            hooks.afterEach(function() {
                this.aaxFetchBidStub.restore();
            });

            QUnit.test('should emit an AD_IMPRESSION event', function(assert) {
                const listenerSpy = sinon.spy();
                this.imaManager.addEventListener(AdEvents.AD_IMPRESSION, listenerSpy);

                this.imaManager._onAdImpression(this.adEvent, this.adRequestContext);

                const [payload] = listenerSpy.firstCall.args;
                assert.equal(listenerSpy.callCount, 1);
                assert.deepEqual(payload, {
                    // eslint-disable-next-line camelcase
                    time_break: this.adRequestContext.duration,
                });
            });

            QUnit.test('should send a `video_ad_impression` tracking event', function(assert) {
                this.imaManager._onAdImpression(this.adEvent, this.adRequestContext);

                assert.equal(this.analytics.trackEvent.callCount, 1);

                const [eventName, payload] = this.analytics.trackEvent.firstCall.args;

                assert.equal(eventName, 'video_ad_impression');

                const expectedResult = {
                    /* eslint-disable camelcase */
                    ad_session_id: this.adRequestContext.adSessionId,
                    ad_id: this.adEvent.getAd().getAdId(),
                    adblock: this.adRequestContext.adblock,
                    provider: 'ima',
                    roll_type: this.adRequestContext.adType,
                    time_break: this.adRequestContext.duration,
                    twitch_correlator: this.adRequestContext.twitchCorrelator,
                    ad_system: this.adEvent.getAd().getAdSystem(),
                    creative_id: this.adEvent.getAd().getCreativeId(),
                    title: this.adEvent.getAd().getTitle(),
                    /* eslint-enable camelcase */
                };

                // eslint-disable-next-line max-len
                const pickedPayload = pick(payload, ['ad_session_id', 'ad_id', 'adblock', 'provider', 'roll_type', 'time_break', 'twitch_correlator', 'ad_system', 'creative_id', 'title']);
                assert.deepEqual(pickedPayload, expectedResult);
                // eslint-disable-next-line max-len
                assert.equal(payload.request_to_impression_latency, this._dateNowDiff, 'The latency measurement should be 1 based on fake Date defined in hooks.beforeEach');
            });

            // eslint-disable-next-line max-len
            QUnit.test('should send a `video_ad_impression` tracking event with vod properties if on a vod', function(assert) {
                this.store.dispatch({
                    type: ACTION_SET_STREAM,
                    stream: this.vodStream,
                });
                this.store.dispatch({
                    type: ACTION_SET_STREAMMETADATA,
                    streamMetadata: this.vodStreamMetadata,
                });

                this.imaManager._onAdImpression(this.adEvent, this.adRequestContext);

                assert.equal(this.analytics.trackEvent.callCount, 1);

                const [eventName, payload] = this.analytics.trackEvent.firstCall.args;

                assert.equal(eventName, 'video_ad_impression');

                const expectedResult = {
                    /* eslint-disable camelcase */
                    ad_session_id: this.adRequestContext.adSessionId,
                    ad_id: this.adEvent.getAd().getAdId(),
                    adblock: this.adRequestContext.adblock,
                    provider: 'ima',
                    roll_type: this.adRequestContext.adType,
                    vod_id: this.videoID,
                    vod_type: this.vodStreamMetadata.type,
                    time_break: this.adRequestContext.duration,
                    twitch_correlator: this.adRequestContext.twitchCorrelator,
                    ad_system: this.adEvent.getAd().getAdSystem(),
                    creative_id: this.adEvent.getAd().getCreativeId(),
                    title: this.adEvent.getAd().getTitle(),
                    /* eslint-enable camelcase */
                };

                // eslint-disable-next-line max-len
                const pickedPayload = pick(payload, ['ad_session_id', 'ad_id', 'adblock', 'provider', 'roll_type', 'vod_id', 'vod_type', 'time_break', 'twitch_correlator', 'ad_system', 'creative_id', 'title']);
                assert.deepEqual(pickedPayload, expectedResult);
                // eslint-disable-next-line max-len
                assert.equal(payload.request_to_impression_latency, this._dateNowDiff, 'The latency measurement should be 1 based on fake Date defined in hooks.beforeEach');
            });
        });

        QUnit.module('when an ad completes playing', function(hooks) {
            hooks.beforeEach(function() {
                this.window = buildFakeWindow({
                    setInterval: sinon.stub().returns(parseInt(QUnit.config.current.testId, 36)),
                    clearInterval: sinon.spy(),
                });
                this.store.dispatch(setWindow(this.window));

                this.imaManager = new IMAManager(
                    this.videoContainer,
                    this.backend,
                    this.store
                );

                this.adEvent = createAdEvent({
                    adId: 'ad_123',
                    adSystem: 'GDFP',
                    creativeId: '456',
                    title: 'LUL Ad',
                    duration: 30,
                    wrapperAdIds: 'ad_wrapper_123',
                    wrapperAdSystems: 'Wrapper GDFP',
                    wrapperCreativeIds: '789',
                    error: {},
                });
            });

            QUnit.test('should emit an AD_IMPRESSION_COMPLETE event', function(assert) {
                const listenerSpy = sinon.spy();
                this.imaManager.addEventListener(AdEvents.AD_IMPRESSION_COMPLETE, listenerSpy);

                this.imaManager._onAdEnded(this.adEvent, this.adRequestContext);

                assert.equal(listenerSpy.callCount, 1);
            });

            QUnit.test('should send a `video_ad_impression_complete` tracking event', function(assert) {
                this.imaManager._onAdEnded(this.adEvent, this.adRequestContext);

                assert.equal(this.analytics.trackEvent.callCount, 1);

                const [eventName, payload] = this.analytics.trackEvent.firstCall.args;

                assert.equal(eventName, 'video_ad_impression_complete');
                assert.deepEqual(payload, {
                    /* eslint-disable camelcase */
                    ad_session_id: this.adRequestContext.adSessionId,
                    ad_id: this.adEvent.getAd().getAdId(),
                    adblock: this.adRequestContext.adblock,
                    duration: this.adEvent.getAd().getDuration(),
                    provider: 'ima',
                    roll_type: this.adRequestContext.adType,
                    time_break: this.adRequestContext.duration,
                    twitch_correlator: this.adRequestContext.twitchCorrelator,
                    ad_system: this.adEvent.getAd().getAdSystem(),
                    creative_id: this.adEvent.getAd().getCreativeId(),
                    title: this.adEvent.getAd().getTitle(),
                    wrapper_ad_id: this.adEvent.getAd().getWrapperAdIds()[0],
                    wrapper_ad_system: this.adEvent.getAd().getWrapperAdSystems()[0],
                    wrapper_creative_id: this.adEvent.getAd().getWrapperCreativeIds()[0],
                    /* eslint-enable camelcase */
                });
            });

            // eslint-disable-next-line max-len
            QUnit.test('should send a `video_ad_impression_complete` tracking event with vod properties if on a vod', function(assert) {
                this.store.dispatch({
                    type: ACTION_SET_STREAM,
                    stream: this.vodStream,
                });
                this.store.dispatch({
                    type: ACTION_SET_STREAMMETADATA,
                    streamMetadata: this.vodStreamMetadata,
                });

                this.imaManager._onAdEnded(this.adEvent, this.adRequestContext);

                assert.equal(this.analytics.trackEvent.callCount, 1);

                const [eventName, payload] = this.analytics.trackEvent.firstCall.args;

                assert.equal(eventName, 'video_ad_impression_complete');
                assert.deepEqual(payload, {
                    /* eslint-disable camelcase */
                    ad_session_id: this.adRequestContext.adSessionId,
                    ad_id: this.adEvent.getAd().getAdId(),
                    adblock: this.adRequestContext.adblock,
                    duration: this.adEvent.getAd().getDuration(),
                    provider: 'ima',
                    roll_type: this.adRequestContext.adType,
                    vod_id: this.videoID,
                    vod_type: this.vodStreamMetadata.type,
                    time_break: this.adRequestContext.duration,
                    twitch_correlator: this.adRequestContext.twitchCorrelator,
                    ad_system: this.adEvent.getAd().getAdSystem(),
                    creative_id: this.adEvent.getAd().getCreativeId(),
                    title: this.adEvent.getAd().getTitle(),
                    wrapper_ad_id: this.adEvent.getAd().getWrapperAdIds()[0],
                    wrapper_ad_system: this.adEvent.getAd().getWrapperAdSystems()[0],
                    wrapper_creative_id: this.adEvent.getAd().getWrapperCreativeIds()[0],
                    /* eslint-enable camelcase */
                });
            });

            QUnit.module('And the ad pod has played through', function() {
                QUnit.test('emit AD_END', function(assert) {
                    const listenerSpy = sinon.spy();
                    this.imaManager.addEventListener(AdEvents.AD_END, listenerSpy);

                    this.store.dispatch(setCurrentAdMetadata({
                        contentType: AdContentTypes.IMA,
                        rollType: AdRollTypes.PREROLL,
                    }));

                    this.imaManager._onContentResumeRequested(this.adRequestContext);

                    const [payload] = listenerSpy.firstCall.args;
                    assert.equal(listenerSpy.callCount, 1);
                    assert.deepEqual(payload, {
                        // eslint-disable-next-line camelcase
                        roll_type: this.adRequestContext.adType,
                    });
                });

                QUnit.test('sets ad content in statestore to NONE', function(assert) {
                    this.store.dispatch(setCurrentAdMetadata({
                        contentType: AdContentTypes.IMA,
                        rollType: AdRollTypes.PREROLL,
                    }));

                    assert.equal(this.store.getState().ads.currentMetadata.contentType, AdContentTypes.IMA);
                    assert.equal(this.store.getState().ads.currentMetadata.rollType, AdRollTypes.PREROLL);
                    this.imaManager._onContentResumeRequested(this.adRequestContext);
                    assert.equal(this.store.getState().ads.currentMetadata.contentType, AdContentTypes.NONE);
                    assert.equal(this.store.getState().ads.currentMetadata.rollType, AdRollTypes.NONE);
                });

                QUnit.test('set backend volume to what was set during the ads', function(assert) {
                    const OLD_VOLUME = 0.123;
                    const NEW_VOLUME = 0.456;

                    const OLD_MUTED = false;
                    const NEW_MUTED = true;

                    this.store.dispatch(volumeChanged(OLD_VOLUME));
                    this.store.dispatch(playerMuted(OLD_MUTED));

                    this.imaManager._onContentPauseRequested(this.adRequestContext);
                    this.store.dispatch(volumeChanged(NEW_VOLUME));
                    this.store.dispatch(playerMuted(NEW_MUTED));
                    this.backend.setMuted.reset();
                    this.backend.setVolume.reset();
                    this.imaManager._onContentResumeRequested(this.adRequestContext);

                    assert.equal(this.backend.setMuted.firstCall.args[0], NEW_MUTED, 'should set new backend muted');
                    assert.equal(this.backend.setVolume.firstCall.args[0], NEW_VOLUME, 'should set new backend volume');
                });

                QUnit.test('resumes stream if it is a VOD and has not ended', function(assert) {
                    this.store.dispatch({
                        type: ACTION_SET_STREAM,
                        stream: new VODTwitchContentStream(),
                    });

                    this.imaManager._onContentPauseRequested(this.adRequestContext);
                    this.backend.play.reset();
                    this.imaManager._onContentResumeRequested(this.adRequestContext);
                    return waitFor(() => this.backend.play.called).then(() => {
                        assert.equal(this.backend.play.callCount, 1, 'should call backend play once');
                    });
                });
            });
        });

        QUnit.module('when the parent container gets resized during ad', function() {
            // eslint-disable-next-line max-statements
            QUnit.test('should resize the ad', function(assert) {
                const testIntervalId = 98375;
                // position is required for offset[Width|Height] calculation
                const $qunitFixture = window.document.getElementById('qunit-fixture');
                $qunitFixture.insertAdjacentHTML('beforeend', `
                    <div id='main-container' style='position:absolute;width:10px;height:5px;'>
                        <div id='video-container'>
                            <video></video>
                        </div>
                    </div>
                `);
                this.window = buildFakeWindow({
                    // Can't pass actual setInterval/clearInterval without Illegal Invocation errors
                    setInterval: sinon.stub().returns(testIntervalId),
                    clearInterval: sinon.spy(),
                });
                this.store.dispatch(setWindow(this.window));

                this.videoContainer = $qunitFixture.querySelector('#video-container');
                const imaManager = new IMAManager(
                    this.videoContainer,
                    this.backend,
                    this.store
                );

                sinon.spy(imaManager._currentAdsManager, 'resize');
                this.store.dispatch({
                    type: PlayerDimensionsActions.ACTION_UPDATE_PLAYER_DIMENSIONS,
                    playerDimensions: {
                        width: 10,
                        height: 5,
                    },
                });
                const playerDimensions = this.store.getState().playerDimensions;

                assert.equal(playerDimensions.width, 10);
                assert.equal(playerDimensions.height, 5);

                assert.ok(imaManager._currentAdsManager.resize.called);
                assert.equal(imaManager._currentAdsManager.resize.callCount, 1, 'should call resize');
                assert.ok(imaManager._currentAdsManager.resize.calledWith(10, 5));
                imaManager._currentAdsManager.resize.restore();
            });
        });
    });
});
