import sinon from 'sinon';
import { unitTest } from 'tests/utils/module';
import { buildIMATags, PREROLL, MIDROLL,
         POSTROLL } from 'ads/ima-tags';
import { AdsRequestContext } from 'ads/adscontext';
import * as PlayerType from 'util/player-type';
import { parseUri } from 'util/parseuri';
import { CONTENT_MODE_VOD } from 'stream/twitch-vod';
import { CONTENT_MODE_LIVE } from 'stream/twitch-live';
import { PLATFORM_WEB, PLATFORM_MOBILE_WEB } from 'state/env';

unitTest('ads | ima-tags', function(hooks) {
    hooks.beforeEach(function() {
        this.adsRequestContext = new AdsRequestContext({
            adType: PREROLL,
            duration: Math.floor(Math.random() * 1e3) + 30,
            sdk: 'sdkname',
            state: {
                ads: {
                    adblock: false,
                },
                adsManager: {
                    sdk: 'sdkname',
                },
                env: {
                    playerType: PlayerType.PLAYER_POPOUT,
                },
                stream: {},
                window: {
                    document: {
                        referrer: 'othersite.com',
                    },
                    location: {
                        href: 'https://player.twitch.tv',
                    },
                    top: {
                        location: {
                            hostname: 'media.curse.com',
                        },
                    },
                },
            },
            userInfo: {
                turbo: null,
            },
            channelAPIInfo: {
                prerolls: true,
                postrolls: true,
            },
            channelInfo: {
                _id: `${QUnit.config.current.testId.toLowerCase()}`,
                name: `channel_${QUnit.config.current.testId.toLowerCase()}`,
                game: `Game ${QUnit.config.current.testId}`,
                mature: true,
                partner: false,
            },
            viewerInfo: {
                // eslint-disable-next-line camelcase
                has_ad_free_subscription: null,
            },
            communitiesInfo: {
                communities: [{ _id: '12345' }, { _id: '54321' }],
            },
            channelAdProperties: {
                /* eslint-disable camelcase */
                valid_responses: {
                    vod_ads_enabled: true,
                },
                /* eslint-enable camelcase */
            },
            padtimeExpGroup: 'experiment_group',
            skippableExpGroup: 'experiment_group',
        });
    });

    QUnit.module('buildIMATags', function(hooks) {
        hooks.beforeEach(function() {
            sinon.stub(Date, 'now').returns(parseInt(QUnit.config.current.testId, 36));
        });

        hooks.afterEach(function() {
            Date.now.restore();
        });

        QUnit.test('ensure returned tag contains basic parameters', function(assert) {
            const adTagUrl = buildIMATags(this.adsRequestContext);
            const url = parseUri(adTagUrl);

            assert.equal(url.protocol, '', 'should be protocol relative');
            assert.equal(url.host, 'pubads.g.doubleclick.net');
            assert.equal(url.path, '/gampad/live/ads');
            assert.equal(
                url.queryKey.iu,
                `/3576121/twitch.web/client/desktop/live/${this.adsRequestContext.channelId}`
            );
            assert.equal(url.queryKey.ius_szs, '300x250');
            assert.equal(url.queryKey.sz, '640x480');
            assert.equal(url.queryKey.impl, 's');
            assert.equal(url.queryKey.gdfp_req, '1');
            assert.equal(url.queryKey.env, 'vp');
            assert.equal(url.queryKey.output, 'xml_vast3');
            assert.equal(url.queryKey.url, this.adsRequestContext.url);
            assert.equal(url.queryKey.unviewed_position_start, '1');
            assert.equal(url.queryKey.correlator, Date.now());
            assert.ok(url.queryKey.hasOwnProperty('cust_params'));
        });

        QUnit.test('VOD uses the standard DFP endpoint', function(assert) {
            this.adsRequestContext.contentType = CONTENT_MODE_VOD;

            const adTagUrl = buildIMATags(this.adsRequestContext);
            const url = parseUri(adTagUrl);

            assert.equal(url.path, '/gampad/ads');
        });

        QUnit.test('LIVE uses the live DFP endpoint', function(assert) {
            this.adsRequestContext.contentType = CONTENT_MODE_LIVE;

            const adTagUrl = buildIMATags(this.adsRequestContext);
            const url = parseUri(adTagUrl);

            assert.equal(url.path, '/gampad/live/ads');
        });

        QUnit.test('custom parameters uses `partner` field from request context', function(assert) {
            [true, false].forEach(partner => {
                this.adsRequestContext.partner = partner;

                const adTagUrl = parseUri(buildIMATags(this.adsRequestContext));
                const params = parseCustParams(adTagUrl.queryKey.cust_params);

                assert.equal(params.partner, String(partner));
            });
        });

        QUnit.test('custom parameters uses `game` field from request context', function(assert) {
            const adTagUrl = parseUri(buildIMATags(this.adsRequestContext));
            const params = parseCustParams(adTagUrl.queryKey.cust_params);

            assert.equal(params.game, this.adsRequestContext.game.toLowerCase().split(' ').join('_'));
        });

        QUnit.test('custom parameters uses `channel` field from request context', function(assert) {
            const adTagUrl = parseUri(buildIMATags(this.adsRequestContext));
            const params = parseCustParams(adTagUrl.queryKey.cust_params);

            assert.equal(params.chan, this.adsRequestContext.channel);
        });

        QUnit.test('custom parameters reports embed player separately', function(assert) {
            [
                PlayerType.PLAYER_SITE,
                PlayerType.PLAYER_EMBED,
                PlayerType.PLAYER_POPOUT,
                PlayerType.PLAYER_FRONTPAGE,
                PlayerType.PLAYER_FEED,
                PlayerType.PLAYER_CREATIVE,
                PlayerType.PLAYER_FACEBOOK,
            ].forEach(playerType => {
                this.adsRequestContext.playerType = playerType;

                const adTagUrl = parseUri(buildIMATags(this.adsRequestContext));
                const params = parseCustParams(adTagUrl.queryKey.cust_params);

                assert.equal(
                    params.embed,
                    String(this.adsRequestContext.playerType === PlayerType.PLAYER_EMBED)
                );
            });
        });

        QUnit.test('custom parameters uses `platform` field from request context', function(assert) {
            [
                PLATFORM_WEB,
                PLATFORM_MOBILE_WEB,
            ].forEach(platform => {
                this.adsRequestContext.platform = platform;

                const adTagUrl = parseUri(buildIMATags(this.adsRequestContext));
                const params = parseCustParams(adTagUrl.queryKey.cust_params);

                assert.equal(params.platform, platform);
            });
        });

        QUnit.test('custom parameters uses `playerType` field from request context', function(assert) {
            [
                PlayerType.PLAYER_SITE,
                PlayerType.PLAYER_EMBED,
                PlayerType.PLAYER_POPOUT,
                PlayerType.PLAYER_FRONTPAGE,
                PlayerType.PLAYER_FEED,
                PlayerType.PLAYER_CREATIVE,
                PlayerType.PLAYER_FACEBOOK,
            ].forEach(playerType => {
                this.adsRequestContext.playerType = playerType;

                const adTagUrl = parseUri(buildIMATags(this.adsRequestContext));
                const params = parseCustParams(adTagUrl.queryKey.cust_params);

                assert.equal(params.playerType, playerType);
            });
        });

        QUnit.test('custom parameters uses `mature` field from request context', function(assert) {
            [true, false].forEach(mature => {
                this.adsRequestContext.mature = mature;

                const adTagUrl = parseUri(buildIMATags(this.adsRequestContext));
                const params = parseCustParams(adTagUrl.queryKey.cust_params);

                assert.equal(params.mature, String(mature));
            });
        });

        QUnit.test('custom parameters converts roll type to DFP value', function(assert) {
            [
                {
                    rollType: PREROLL,
                    value: '1',
                },
                {
                    rollType: MIDROLL,
                    value: '2',
                },
                {
                    rollType: POSTROLL,
                    value: '3',
                },
            ].forEach(({ rollType, value }) => {
                this.adsRequestContext.adType = rollType;

                const adTagUrl = parseUri(buildIMATags(this.adsRequestContext));
                const params = parseCustParams(adTagUrl.queryKey.cust_params);

                assert.equal(params.pos, value);
            });
        });

        QUnit.test('custom parameters uses `duration` field from request context', function(assert) {
            this.adsRequestContext.duration = Math.floor(Math.random() * 30) + 30;

            const adTagUrl = parseUri(buildIMATags(this.adsRequestContext));
            const params = parseCustParams(adTagUrl.queryKey.cust_params);

            assert.equal(params.timebreak, this.adsRequestContext.duration);
        });

        QUnit.test('custom parameters uses `adblock` field from request context', function(assert) {
            [
                {
                    adblock: true,
                    value: 'true',
                },
                {
                    adblock: false,
                    value: 'false',
                },
                {
                    adblock: (void 0),
                    value: 'unknown',
                },
            ].forEach(({ adblock, value }) => {
                this.adsRequestContext.adblock = adblock;

                const adTagUrl = parseUri(buildIMATags(this.adsRequestContext));
                const params = parseCustParams(adTagUrl.queryKey.cust_params);

                assert.equal(params.adblock, value);
            });
        });

        QUnit.test('custom parameters uses `sdk` field from request context', function(assert) {
            const adTagUrl = parseUri(buildIMATags(this.adsRequestContext));
            const params = parseCustParams(adTagUrl.queryKey.cust_params);

            assert.equal(params.sdk, this.adsRequestContext.sdk);
        });

        QUnit.test('custom parameters include `deviceId` field from request context', function(assert) {
            const adTagUrl = parseUri(buildIMATags(this.adsRequestContext));
            const params = parseCustParams(adTagUrl.queryKey.cust_params);

            assert.equal(params.deviceId, this.adsRequestContext.deviceId.toLowerCase());
        });

        QUnit.test('includes `kruxId` only if Krux is available', function(assert) {
            [
                {
                    kruxId: null,
                    value: (void 0),
                },
                {
                    kruxId: `krux_${QUnit.config.current.testId.toLowerCase()}`,
                    value: `krux_${QUnit.config.current.testId.toLowerCase()}`,
                },
            ].forEach(({ kruxId, value }) => {
                this.adsRequestContext.kruxId = kruxId;

                const adTagUrl = parseUri(buildIMATags(this.adsRequestContext));
                const params = parseCustParams(adTagUrl.queryKey.cust_params);

                if (value === (void 0)) {
                    assert.notOk(params.hasOwnProperty('kuid'));
                } else {
                    assert.equal(params.kuid, value);
                }
            });
        });

        QUnit.test('includes `communities` only if communities are available', function(assert) {
            [
                {
                    communityIds: null,
                },
                {
                    communityIds: '12345',
                },
            ].forEach(({ communityIds }) => {
                this.adsRequestContext.communityIds = communityIds;

                const adTagUrl = parseUri(buildIMATags(this.adsRequestContext));
                const params = parseCustParams(adTagUrl.queryKey.cust_params);

                if (communityIds === null) {
                    assert.notOk(params.hasOwnProperty('communityIds'));
                } else {
                    assert.equal(params.community, communityIds);
                }
            });
        });

        QUnit.test('includes `embed_url` if the player is an embed', function(assert) {
            [
                PlayerType.PLAYER_SITE,
                PlayerType.PLAYER_EMBED,
                PlayerType.PLAYER_POPOUT,
                PlayerType.PLAYER_FRONTPAGE,
                PlayerType.PLAYER_FEED,
                PlayerType.PLAYER_CREATIVE,
                PlayerType.PLAYER_FACEBOOK,
            ].forEach(playerType => {
                this.adsRequestContext.playerType = playerType;

                const adTagUrl = parseUri(buildIMATags(this.adsRequestContext));
                const params = parseCustParams(adTagUrl.queryKey.cust_params);

                if (playerType === PlayerType.PLAYER_EMBED) {
                    assert.equal(params.embed_url, 'othersite.com');
                } else {
                    assert.notOk(params.hasOwnProperty('embed_url'));
                }
            });
        });

        QUnit.test('includes VOD attributes if the context stream is a VOD', function(assert) {
            this.adsRequestContext.contentType = CONTENT_MODE_VOD;
            this.adsRequestContext.vod.id = '12345';
            this.adsRequestContext.vod.type = 'archive';
            this.adsRequestContext.vod.name = 'Some VOD name';

            const adTagUrl = parseUri(buildIMATags(this.adsRequestContext));
            const params = parseCustParams(adTagUrl.queryKey.cust_params);

            assert.equal(params.vod_type, this.adsRequestContext.vod.type);
            assert.equal(params.vod_name, 'some_vod_name');
            assert.equal(params.vod_id, this.adsRequestContext.vod.id);
        });

        QUnit.test('does not include VOD attributes if the context stream is live', function(assert) {
            this.adsRequestContext.contentType = CONTENT_MODE_LIVE;

            const adTagUrl = parseUri(buildIMATags(this.adsRequestContext));
            const params = parseCustParams(adTagUrl.queryKey.cust_params);

            assert.ok(!params.hasOwnProperty('vod_type'));
            assert.ok(!params.hasOwnProperty('vod_name'));
            assert.ok(!params.hasOwnProperty('vod_id'));
        });

        QUnit.test('includes prime ad experiment group', function(assert) {
            const adTagUrl = parseUri(buildIMATags(this.adsRequestContext));
            const params = parseCustParams(adTagUrl.queryKey.cust_params);

            assert.equal(params.padtime, this.adsRequestContext.padtime);
        });

        QUnit.test('custom parameters include `skip` if not control', function(assert) {
            this.adsRequestContext.skippableExpGroup = 'exp_group';

            const adTagUrl = parseUri(buildIMATags(this.adsRequestContext));
            const params = parseCustParams(adTagUrl.queryKey.cust_params);

            assert.equal(params.skip, 'exp_group');
        });

        QUnit.test('custom parameters do not include `skip` if control', function(assert) {
            this.adsRequestContext.skippableExpGroup = 'control';

            const adTagUrl = parseUri(buildIMATags(this.adsRequestContext));
            const params = parseCustParams(adTagUrl.queryKey.cust_params);

            assert.equal(params.skip, null);
        });
    });
});

function parseCustParams(params) {
    return params.split('%26').reduce((m, kv) => {
        const [k, v] = kv.split('%3D');
        // eslint-disable-next-line no-param-reassign
        m[k] = v;
        return m;
    }, {});
}
