import sinon from 'sinon';
import extend from 'lodash/extend';
import defaultsDeep from 'lodash/defaultsDeep';
import * as UniqueId from 'util/unique-id';
import {
    Analytics, removeVariantName, REBROADCAST_TYPE,
    BROADCAST_PLATFORM_WATCH_PARTY,
    BROADCAST_PLATFORM_WATCH_PARTY_RERUN,
    BROADCAST_PLATFORM_WATCH_PARTY_PREMIERE } from 'analytics/analytics';
import { init as initStore } from 'state';
import { setCaptionsEnabled, setCaptionsPreset } from 'actions/captions';
import * as CollectionActions from 'actions/collection';
import { setExperiments } from 'actions/experiments';
import { ACTION_SET_STREAMMETADATA } from 'actions/stream-metadata';
import { setAnalyticsTracker } from 'actions/analytics-tracker';
import { setWindow } from 'actions/window';
import { incrementQualityChangeCount } from 'actions/analytics';
import { ACTION_SET_MINIPLAYER_MODE } from 'actions/ui';
import { ACTION_ERROR } from 'actions/error';
import { ACTION_SET_STREAM } from 'actions/stream';
import { setFullScreen } from 'actions/screen-mode';
import { ACTION_SET_WATCHPARTY_VOD } from 'actions/watch-party';
import { setManifestInfo } from 'actions/manifest-info';
import { setCommunitiesData, joinCommunitiesIds } from 'actions/communities';
import { ACTION_SET_PLAYER_TYPE } from 'actions/env';
import { VODTwitchContentStream } from 'stream/twitch-vod';
import { ProvidedContentStream } from 'stream/provided';
import { PLAYER_POPOUT, PLAYER_SITE, PLAYER_SITE_MINI } from 'util/player-type';
import { waitFor } from 'tests/utils/waitFor';
import { unitTest } from 'tests/utils/module';
import { buildFakeWindow } from 'tests/fakes/window.fake';
import { TEST_COLLECTION } from 'tests/fixtures/collection';
import EventEmitter from 'events';
import { ABS_STREAM_FORMAT_CHANGE } from 'backend/events/twitch-event';
import { updatePlaybackState } from 'actions/playback';
import { PLAYING, ENDED } from 'backend/events/media-event';
import { TIER_1_EVENTS, VIDEO_PLAYBACK_ERROR } from 'analytics/events';
import { emitAdPlayingEvent } from 'actions/event-emitter';
import { ACTION_SET_AD_METADATA, AdContentTypes, AdRollTypes } from 'actions/ads';
import Errors from 'errors';

const COMMUNITIES_FAKE_DATA = {
    communities: [
        {
            _id: '999-888-777-666-555',
            name: 'purpleGames',
            summary: 'purple games',
            description: 'games that have the purple color',
            rules: 'only purple allowed',
        },
        {
            _id: '111-222-333-444-555',
            name: 'violetGames',
            summary: 'violet games',
            description: 'games that have the violet color',
            rules: 'only violet allowed',
        },
    ],
};
const COMMUNITIES_FAKE_IDS = joinCommunitiesIds(COMMUNITIES_FAKE_DATA.communities);
const FAKE_UPDATE_COLLECTION_SESSION_TIME_ACTION = 'fake update session timestamp';

QUnit.module('removeVariantName', function() {
    const AUTO_VARIANT = 'Auto';

    function testRemoveVariant(str, expected) {
        QUnit.test(`should get the variant '${expected}' from '${str}'`, function(assert) {
            const actual = removeVariantName(str);
            assert.equal(actual, expected);
        });
    }

    testRemoveVariant('Auto (abs)', AUTO_VARIANT);
    testRemoveVariant('Auto (123fsd.@#@#$)', AUTO_VARIANT);
    testRemoveVariant('medium', 'medium');
    testRemoveVariant(' Auto (blah)', AUTO_VARIANT);
    testRemoveVariant(' Auto (blah)         ', AUTO_VARIANT);
    testRemoveVariant('blah (123)', 'blah');
    testRemoveVariant(undefined, 'undefined');
    testRemoveVariant('', '');
    testRemoveVariant(null, 'null');
});

unitTest('analytics | analytics', function(hooks) {
    hooks.beforeEach(function() {
        const events = new EventEmitter();
        this.player = {
            addEventListener(eventName, callback) {
                events.on(eventName, callback);
            },
            emit(event, payload) {
                events.emit(event, payload);
            },
            getNetworkProfile() {
                return {
                    filter() {
                        return [{
                            startTime: parseInt(QUnit.config.current.testId, 36),
                            firstByteLatency: 15,
                            downloadDuration: 2,
                            currBufferLength: 6,
                            reasonCode: 1,
                            currQuality: 'Medium',
                        }];
                    },
                };
            },
            getVideoInfo() {
                return {};
            },
            getVolume() {},
            getMuted() {},
            getTheatre() {},
            getChannel() {},
            getVideo() {},
            getVersion() {},
            getBackend() {},
            getSeeking() {},
            getCurrentTime() {
                return 1000;
            },
        };
        this.tracker = {
            eventHash: {},
            setProperty: sinon.spy(),
            setProperties: sinon.spy(),
            trackEvent(eventName, properties) {
                this.eventHash[eventName] = properties;
            },
            trackEvents: sinon.spy(),
        };
        this.state = {
            isFullScreen() {},
        };

        this.store = initStore();
        this.options = {};
        this.window = buildFakeWindow();

        this.currentBroadcastID = parseInt(QUnit.config.current.testId, 36);

        const fakeExperiments = {
            get() {
                return Promise.resolve({});
            },
        };
        this.store.dispatch(setExperiments(fakeExperiments));
        this.store.dispatch(setCaptionsEnabled(true));
        this.store.dispatch(setAnalyticsTracker(this.tracker));
        this.store.dispatch(setWindow(this.window));
        this.store.dispatch(setManifestInfo({
            // eslint-disable-next-line camelcase
            broadcast_id: this.currentBroadcastID,
        }));
        this.store.dispatch({
            type: ACTION_SET_STREAMMETADATA,
            streamMetadata: {
                broadcastID: this.currentBroadcastID,
            },
        });
        this.api.setLoggedIn(true);

        this.api.expectUserInfo({
            login: 'monstercat',
            turbo: true,
        });

        this.api.expectKrakenUserInfo({
            _id: parseInt(QUnit.config.current.testId, 36),
        });

        // During init of the analytics object, 3 calls are made: oAuth token, userInfo, and krakenUserInfo
        // For tests that instantiate the object, use with waitFor to ensure tests don't finish before
        // the base hooks restore the sinon fakeserver.
        this.waitForAnalytics = testsFn => {
            return waitFor(() => this.api.calls().matched.length === 3).then(testsFn);
        };

        this.analytics = new Analytics(this.player, this.tracker, this.state, this.store, this.options);
    });

    hooks.afterEach(function() {
        this.analytics.destroy();
    });

    QUnit.test('onABSFormatChange tracks abs_stream_format_change when event from player is fired', function(assert) {
        const mockABSPayload = {
            stream_format_current: 'medium', // eslint-disable-line camelcase
            stream_format_previous: 'low2', // eslint-disable-line camelcase
        };
        this.player.emit(ABS_STREAM_FORMAT_CHANGE, mockABSPayload);
        const expected = extend(mockABSPayload, {
            minutes_logged: this.analytics.minutesWatchedTimer.totalMinutes, // eslint-disable-line camelcase
            // eslint-disable-next-line camelcase
            manifest_broadcast_id: this.currentBroadcastID,
            // eslint-disable-next-line camelcase
            broadcast_id: this.currentBroadcastID,
        });
        const result = this.tracker.eventHash[ABS_STREAM_FORMAT_CHANGE];
        assert.deepEqual(result, expected, 'minutes_logged is added to abs payload');
    });

    QUnit.module('initProperties', function() {
        const DEFAULT_PROPERTIES = {
            login: 'monstercat',
            turbo: true,
            staff: false,
            content_id: '', // eslint-disable-line camelcase
            customer_id: '', // eslint-disable-line camelcase
        };

        function testProperties(properties) {
            QUnit.test(`should set the correct properties for ${JSON.stringify(properties)}`, function(assert) {
                const UUID = parseInt(QUnit.config.current.testId, 36);

                this.expectedProperties = defaultsDeep(properties, DEFAULT_PROPERTIES);
                // eslint-disable-next-line camelcase
                this.expectedProperties.user_id = UUID;

                // Need to restore as these are mocked out in the beforeEach
                // but need to be overriden in this test
                this.api.clearFakeResponses();

                this.api.setLoggedIn(true);

                this.api.expectUserInfo({
                    login: this.expectedProperties.login,
                    turbo: this.expectedProperties.turbo,
                });

                this.api.expectKrakenUserInfo({
                    type: properties.staff ? 'staff' : 'user',
                    // eslint-disable-next-line camelcase
                    _id: UUID,
                });

                const spy = this.tracker.setProperties;
                this.analytics.initProperties();

                const setPropertiesCallArguments = spy.getCalls().map(c => c.args[0]);

                return Promise.all(setPropertiesCallArguments).then(results => {
                    const allProps = extend({}, ...results);
                    assert.deepEqual(allProps, this.expectedProperties);
                });
            });
        }

        testProperties({ staff: true });
        testProperties({ staff: false });
    });

    QUnit.module('onLoadStart', function() {
        const CHANNEL = 'monstercat';
        const CHANNEL_ID = 123;
        const GAME = 'testGame';
        const PARTNER = false;
        const DEFAULT_BROADCASTER = 'unknown_rtmp';

        QUnit.test('destroys and creates a new MW timer', function(assert) {
            const MWTimerSpy = sinon.spy(this.analytics.minutesWatchedTimer, 'destroy');
            const initMWTimerSpy = sinon.spy(this.analytics, 'initMinutesWatchedTimer');
            this.analytics.onLoadStart();

            assert.ok(MWTimerSpy.called, 'should call destroy() on MinutesWatched object');
            assert.ok(initMWTimerSpy, 'should create a new MinutesWatched object');
        });

        QUnit.test('resets hasPlayed/buffer state', function(assert) {
            this.analytics.hasPlayed = true;
            this.analytics.bufferEmptyCount = 1;
            this.analytics.bufferEmptyStartTime = 1000;
            this.analytics.onLoadStart();

            assert.equal(this.analytics.hasPlayed, false, 'should reset hasPlayed to false');
            assert.equal(this.analytics.bufferEmptyCount, 0, 'should reset bufferEmptyCount to 0');
            assert.equal(this.analytics.bufferEmptyStartTime, null, 'should reset bufferEmptyStartTime to null');
        });

        QUnit.module('vods', function(hooks) {
            const VIDEO = 'v123456';
            const VIDEO_BROADCAST_TYPE = 'archive';
            hooks.beforeEach(function() {
                sinon.stub(this.player, 'getVideo').returns(VIDEO);
                this.oldMinutesWatchedTimer = this.analytics.minutesWatchedTimer;
                this.tracker.setProperties.reset();
                this.analytics.onLoadStart();
                this.player.getVideo.restore();

                this.api.expectVideoInfo(VIDEO, {
                    _id: VIDEO,
                    channel: {
                        name: CHANNEL,
                    },
                    game: GAME,
                });

                this.api.expectChannelInfo(CHANNEL, {
                    _id: CHANNEL_ID,
                    partner: PARTNER,
                });
            });

            QUnit.test('initialize new minutesWatchedTimer on onLoadStart for vods', function(assert) {
                assert.notEqual(this.oldMinutesWatchedTimer, this.analytics.minutesWatchedTimer);
            });

            QUnit.test('correct properties are set for vods', function(assert) {
                /* eslint-disable camelcase */
                return waitFor(() => this.tracker.setProperties.calledTwice).then(() => {
                    assert.equal(this.tracker.setProperties.callCount, 2);
                    this.tracker.setProperties.firstCall.args[0].then(obj => {
                        assert.deepEqual(obj, {
                            partner: PARTNER,
                        });
                    });
                    this.tracker.setProperties.secondCall.args[0].then(obj => {
                        assert.deepEqual(obj, {
                            channel: CHANNEL,
                            channel_id: CHANNEL_ID,
                            vod: VIDEO,
                            vod_id: VIDEO,
                            vod_type: VIDEO_BROADCAST_TYPE,
                            game: GAME,
                            live: false,
                            content_mode: 'vod',
                        });
                    });
                });
                /* eslint-enable camelcase */
            });
        });

        QUnit.module('streams', function(hooks) {
            hooks.beforeEach(function() {
                sinon.stub(this.player, 'getChannel').returns(CHANNEL);
                this.oldMinutesWatchedTimer = this.analytics.minutesWatchedTimer;
                this.tracker.setProperties.reset();
                this.analytics.onLoadStart();
                this.player.getChannel.restore();

                this.api.expectChannelInfo(CHANNEL, {
                    _id: CHANNEL_ID,
                    game: GAME,
                    partner: PARTNER,
                });

                this.api.expectCommunitiesInfo(CHANNEL_ID, COMMUNITIES_FAKE_DATA);
            });

            QUnit.test('initialize new minutesWatchedTimer on onLoadStart for streams', function(assert) {
                assert.notEqual(this.oldMinutesWatchedTimer, this.analytics.minutesWatchedTimer);
            });

            QUnit.test('correct properties are set for live streams', function(assert) {
                /* eslint-disable camelcase */
                this.api.expectChannelAPIInfo(CHANNEL, {
                    broadcaster_software: DEFAULT_BROADCASTER,
                });

                return waitFor(() => this.tracker.setProperties.called).then(() => {
                    assert.equal(this.tracker.setProperties.callCount, 1);
                    return this.tracker.setProperties.firstCall.args[0].then(obj => {
                        assert.deepEqual(obj, {
                            broadcaster_software: DEFAULT_BROADCASTER,
                            channel: CHANNEL,
                            channel_id: CHANNEL_ID,
                            game: GAME,
                            live: true,
                            content_mode: 'live',
                            partner: PARTNER,
                            vod_type: undefined,
                        });
                    });
                });
                /* eslint-enable camelcase */
            });

            QUnit.test('correct properties are set for watch parties', function(assert) {
                /* eslint-disable camelcase */
                this.api.expectChannelAPIInfo(CHANNEL, {
                    broadcaster_software: BROADCAST_PLATFORM_WATCH_PARTY,
                });

                return waitFor(() => this.tracker.setProperties.called).then(() => {
                    assert.equal(this.tracker.setProperties.callCount, 1);
                    return this.tracker.setProperties.firstCall.args[0].then(obj => {
                        assert.deepEqual(obj, {
                            broadcaster_software: BROADCAST_PLATFORM_WATCH_PARTY,
                            channel: CHANNEL,
                            channel_id: CHANNEL_ID,
                            game: GAME,
                            live: true,
                            content_mode: 'vodcast',
                            partner: PARTNER,
                            vod_type: undefined,
                        });
                    });
                });
                /* eslint-enable camelcase */
            });

            QUnit.test('correct properties are set for premieres (a type of watch party)', function(assert) {
                /* eslint-disable camelcase */
                this.api.expectChannelAPIInfo(CHANNEL, {
                    broadcaster_software: BROADCAST_PLATFORM_WATCH_PARTY_PREMIERE,
                });

                return waitFor(() => this.tracker.setProperties.called).then(() => {
                    assert.equal(this.tracker.setProperties.callCount, 1);
                    return this.tracker.setProperties.firstCall.args[0].then(obj => {
                        assert.deepEqual(obj, {
                            broadcaster_software: BROADCAST_PLATFORM_WATCH_PARTY_PREMIERE,
                            channel: CHANNEL,
                            channel_id: CHANNEL_ID,
                            game: GAME,
                            live: true,
                            content_mode: 'vodcast',
                            partner: PARTNER,
                            vod_type: undefined,
                        });
                    });
                });
                /* eslint-enable camelcase */
            });

            QUnit.test('correct properties are set for reruns (a type of watch party)', function(assert) {
                /* eslint-disable camelcase */
                this.api.expectChannelAPIInfo(CHANNEL, {
                    broadcaster_software: BROADCAST_PLATFORM_WATCH_PARTY_RERUN,
                });

                return waitFor(() => this.tracker.setProperties.called).then(() => {
                    assert.equal(this.tracker.setProperties.callCount, 1);
                    return this.tracker.setProperties.firstCall.args[0].then(obj => {
                        assert.deepEqual(obj, {
                            broadcaster_software: BROADCAST_PLATFORM_WATCH_PARTY_RERUN,
                            channel: CHANNEL,
                            channel_id: CHANNEL_ID,
                            game: GAME,
                            live: true,
                            content_mode: 'vodcast',
                            partner: PARTNER,
                            vod_type: undefined,
                        });
                    });
                });
                /* eslint-enable camelcase */
            });

            QUnit.test('correct properties are set for rebroadcasts', function(assert) {
                /* eslint-disable camelcase */
                this.api.expectChannelAPIInfo(CHANNEL, {
                    broadcaster_software: REBROADCAST_TYPE,
                });

                return waitFor(() => this.tracker.setProperties.called).then(() => {
                    assert.equal(this.tracker.setProperties.callCount, 1);
                    this.tracker.setProperties.firstCall.args[0].then(obj => {
                        assert.deepEqual(obj, {
                            broadcaster_software: REBROADCAST_TYPE,
                            channel: CHANNEL,
                            channel_id: CHANNEL_ID,
                            game: GAME,
                            live: false,
                            content_mode: 'vod',
                            partner: PARTNER,
                            vod_type: undefined,
                        });
                    });
                });
                /* eslint-enable camelcase */
            });

            QUnit.test('setCommunitiesData is dispatched', function(assert) {
                this.api.expectChannelAPIInfo(CHANNEL);

                sinon.spy(this.store, 'dispatch');

                const expectedAction = setCommunitiesData(COMMUNITIES_FAKE_DATA);

                return waitFor(() => this.tracker.setProperties.called).then(() => {
                    assert.equal(this.tracker.setProperties.callCount, 1);
                    return this.tracker.setProperties.firstCall.args[0].then(() => {
                        assert.equal(this.store.dispatch.callCount, 1);
                        assert.deepEqual(this.store.dispatch.firstCall.args[0], expectedAction);
                    });
                });
            });
        });
    });

    QUnit.module('minute-watched', function() {
        QUnit.test('minutes logged property resets to 0 after onLoadStart', function(assert) {
            return this.waitForAnalytics(() => {
                this.analytics.minutesWatchedTimer.totalMinutes = 1;
                this.analytics.minutesWatchedTimer._events.trigger('minute');
                assert.equal(this.tracker.eventHash[TIER_1_EVENTS.MINUTE_WATCHED].minutes_logged, 1);

                this.analytics.minutesWatchedTimer.totalMinutes = 5;
                this.analytics.minutesWatchedTimer._events.trigger('minute');
                assert.equal(this.tracker.eventHash[TIER_1_EVENTS.MINUTE_WATCHED].minutes_logged, 5);

                this.analytics.onLoadStart();
                this.analytics.minutesWatchedTimer._events.trigger('minute');
                assert.equal(this.tracker.eventHash[TIER_1_EVENTS.MINUTE_WATCHED].minutes_logged, 0);
            });
        });

        QUnit.test('minutes watched timer stops on onPause', function(assert) {
            return this.waitForAnalytics(() => {
                const MWTimerSpy = sinon.spy(this.analytics.minutesWatchedTimer, 'stop');
                this.analytics.onPause();
                assert.ok(MWTimerSpy.called, 'should call stop() on MinutesWatched object');
            });
        });

        QUnit.test('minutes watched timer is not stopped during player initiated ads', function(assert) {
            return this.waitForAnalytics(() => {
                const VIDEO = 'v123456';
                const MWTimerSpy = sinon.spy(this.analytics.minutesWatchedTimer, 'stop');
                this.store.dispatch({
                    type: ACTION_SET_STREAM,
                    stream: new VODTwitchContentStream(VIDEO),
                });
                this.store.dispatch(emitAdPlayingEvent());
                this.store.dispatch({
                    type: ACTION_SET_AD_METADATA,
                    currentMetadata: {
                        contentType: AdContentTypes.IMA,
                        rollType: AdRollTypes.PREROLL,
                    },
                });
                this.analytics.onPause();
                assert.ok(MWTimerSpy.notCalled, 'should not call stop() on MinutesWatched object');
            });
        });

        QUnit.test('tracks whether captions are enabled', function(assert) {
            return this.waitForAnalytics(() => {
                assert.equal(this.tracker.eventHash[TIER_1_EVENTS.MINUTE_WATCHED], undefined);

                this.analytics.minutesWatchedTimer._events.trigger('minute');
                assert.equal(this.tracker.eventHash[TIER_1_EVENTS.MINUTE_WATCHED].captions_enabled, true);
            });
        });

        QUnit.test('contains broadcast_id', function(assert) {
            return this.waitForAnalytics(() => {
                assert.equal(this.tracker.eventHash[TIER_1_EVENTS.MINUTE_WATCHED], undefined);

                this.analytics.minutesWatchedTimer._events.trigger('minute');
                assert.equal(
                    this.tracker.eventHash[TIER_1_EVENTS.MINUTE_WATCHED].broadcast_id,
                    this.store.getState().streamMetadata.broadcastID
                );
            });
        });

        QUnit.test('contains transcoder_type', function(assert) {
            const transcodestack = QUnit.config.current.testId;
            this.store.dispatch(setManifestInfo({ transcodestack }));
            return this.waitForAnalytics(() => {
                assert.equal(this.tracker.eventHash[TIER_1_EVENTS.MINUTE_WATCHED], undefined);

                this.analytics.minutesWatchedTimer._events.trigger('minute');
                assert.equal(
                    this.tracker.eventHash[TIER_1_EVENTS.MINUTE_WATCHED].transcoder_type,
                    transcodestack
                );
            });
        });

        QUnit.test('contains language', function(assert) {
            return this.waitForAnalytics(() => {
                assert.equal(this.tracker.eventHash[TIER_1_EVENTS.MINUTE_WATCHED], undefined);

                this.analytics.minutesWatchedTimer._events.trigger('minute');
                assert.equal(
                    this.tracker.eventHash[TIER_1_EVENTS.MINUTE_WATCHED].language,
                    this.store.getState().lang.langCode
                );
            });
        });

        QUnit.test('contains community_ids', function(assert) {
            this.store.dispatch(setCommunitiesData(COMMUNITIES_FAKE_DATA));
            return this.waitForAnalytics(() => {
                assert.equal(this.tracker.eventHash[TIER_1_EVENTS.MINUTE_WATCHED], undefined);

                this.analytics.minutesWatchedTimer._events.trigger('minute');
                assert.equal(
                    this.tracker.eventHash[TIER_1_EVENTS.MINUTE_WATCHED].community_ids,
                    COMMUNITIES_FAKE_IDS
                );
            });
        });

        QUnit.test('tracks number of quality change count, and resets it', function(assert) {
            return this.waitForAnalytics(() => {
                assert.equal(this.tracker.eventHash[TIER_1_EVENTS.MINUTE_WATCHED], undefined);
                this.store.dispatch(incrementQualityChangeCount());
                this.store.dispatch(incrementQualityChangeCount());
                this.analytics.minutesWatchedTimer._events.trigger('minute');
                assert.equal(this.tracker.eventHash[TIER_1_EVENTS.MINUTE_WATCHED].quality_change_count, 2);
                assert.equal(this.store.getState().analytics.qualityChangeCount, 0);
            });
        });

        QUnit.test('sets player_size_mode to popout when player is popout', function(assert) {
            return this.waitForAnalytics(() => {
                this.options.player = PLAYER_POPOUT;
                this.analytics.minutesWatchedTimer._events.trigger('minute');
                assert.equal(this.tracker.eventHash[TIER_1_EVENTS.MINUTE_WATCHED].player_size_mode, 'popout');
            });
        });

        QUnit.test('sets player_size_mode to fullscreen when player is fullscreen', function(assert) {
            return this.waitForAnalytics(() => {
                this.store.dispatch(setFullScreen(true));
                this.analytics.minutesWatchedTimer._events.trigger('minute');
                assert.equal(this.tracker.eventHash[TIER_1_EVENTS.MINUTE_WATCHED].player_size_mode, 'fullscreen');
            });
        });

        QUnit.test('sets player_size_mode to theatre when player is in theatre mode', function(assert) {
            return this.waitForAnalytics(() => {
                this.player.getTheatre = () => true;
                this.analytics.minutesWatchedTimer._events.trigger('minute');
                assert.equal(this.tracker.eventHash[TIER_1_EVENTS.MINUTE_WATCHED].player_size_mode, 'theatre mode');
            });
        });

        QUnit.test('sets player_size_mode to fullscreen when fullscreen and popout', function(assert) {
            return this.waitForAnalytics(() => {
                this.options.player = PLAYER_POPOUT;
                this.store.dispatch(setFullScreen(true));
                this.analytics.minutesWatchedTimer._events.trigger('minute');
                assert.equal(this.tracker.eventHash[TIER_1_EVENTS.MINUTE_WATCHED].player_size_mode, 'fullscreen');
            });
        });

        QUnit.test('sets player_size_mode to fullscreen when fullscreen and theatre', function(assert) {
            return this.waitForAnalytics(() => {
                this.player.getTheatre = () => true;
                this.store.dispatch(setFullScreen(true));
                this.analytics.minutesWatchedTimer._events.trigger('minute');
                assert.equal(this.tracker.eventHash[TIER_1_EVENTS.MINUTE_WATCHED].player_size_mode, 'fullscreen');
            });
        });

        QUnit.test('sets player_size_mode to mini when mini', function(assert) {
            return this.waitForAnalytics(() => {
                this.store.dispatch({
                    type: ACTION_SET_MINIPLAYER_MODE,
                    value: true,
                });

                this.analytics.minutesWatchedTimer._events.trigger('minute');
                assert.equal(this.tracker.eventHash[TIER_1_EVENTS.MINUTE_WATCHED].player_size_mode, 'mini');
            });
        });

        QUnit.test('tracks autoplayed', function(assert) {
            return this.waitForAnalytics(() => {
                this.options.autoplay = true;
                this.analytics.minutesWatchedTimer._events.trigger('minute');
                assert.equal(this.tracker.eventHash[TIER_1_EVENTS.MINUTE_WATCHED].autoplayed, this.options.autoplay);

                this.options.autoplay = false;
                this.analytics.minutesWatchedTimer._events.trigger('minute');
                assert.equal(this.tracker.eventHash[TIER_1_EVENTS.MINUTE_WATCHED].autoplayed, this.options.autoplay);
            });
        });

        QUnit.module('when playing a collection', function(hooks) {
            hooks.beforeEach(function() {
                sinon.stub(CollectionActions, 'updateCollectionSessionTimestamp').returns({
                    type: FAKE_UPDATE_COLLECTION_SESSION_TIME_ACTION,
                });
                sinon.stub(UniqueId, 'generate').returns('a collection session id');
                sinon.spy(this.store, 'dispatch');
            });

            hooks.afterEach(function() {
                CollectionActions.updateCollectionSessionTimestamp.restore();
                UniqueId.generate.restore();
                this.store.dispatch.restore();
            });

            QUnit.test('contains collection fields when playing a collection', function(assert) {
                return this.waitForAnalytics(() => {
                    this.analytics.onLoadStart();

                    const video = TEST_COLLECTION.items[1];
                    this.store.dispatch(CollectionActions.setCollection(TEST_COLLECTION));
                    this.store.dispatch({
                        type: ACTION_SET_STREAM,
                        stream: new VODTwitchContentStream(`v${video.item_id}`),
                    });

                    this.analytics.minutesWatchedTimer.totalMinutes = 1;
                    this.analytics.minutesWatchedTimer._events.trigger('minute');

                    const mwEvent = this.tracker.eventHash[TIER_1_EVENTS.MINUTE_WATCHED];
                    assert.equal(
                        mwEvent.collection_session_id,
                        'a collection session id',
                        'mw reports the correct session id'
                    );
                    assert.equal(
                        mwEvent.collection_id,
                        TEST_COLLECTION._id,
                        'mw reports the correct collection id'
                    );
                    assert.equal(
                        mwEvent.collection_item_position,
                        1,
                        'mw reports the correct item position'
                    );
                });
            });

            QUnit.test('dispatches an action to update the session store collection timestamp', function(assert) {
                return this.waitForAnalytics(() => {
                    this.analytics.onLoadStart();

                    const video = TEST_COLLECTION.items[1];
                    this.store.dispatch(CollectionActions.setCollection(TEST_COLLECTION));
                    this.store.dispatch({
                        type: ACTION_SET_STREAM,
                        stream: new VODTwitchContentStream(`v${video.item_id}`),
                    });
                    this.store.dispatch.reset();

                    this.analytics.minutesWatchedTimer.totalMinutes = 1;
                    this.analytics.minutesWatchedTimer._events.trigger('minute');

                    const [fakeUpdateTimeAction] = this.store.dispatch.firstCall.args;
                    assert.deepEqual(
                        fakeUpdateTimeAction,
                        { type: FAKE_UPDATE_COLLECTION_SESSION_TIME_ACTION },
                        'update collection session timestamp action is dispatched'
                    );
                });
            });
        });
    });

    QUnit.test('on provided stream change, contentId and customerId are set correctly', function(assert) {
        return this.waitForAnalytics(() => {
            const contentId = 'acontentid';
            const customerId = 'acustomerid';

            this.tracker.setProperties.reset();
            this.store.dispatch({
                type: ACTION_SET_STREAM,
                stream: new ProvidedContentStream({
                    customerId,
                    contentId,
                }),
            });

            assert.equal(this.tracker.setProperties.callCount, 1, 'tracking properties are set');
            assert.deepEqual(this.tracker.setProperties.firstCall.args[0],{
                content_id: contentId, // eslint-disable-line camelcase
                customer_id: customerId, // eslint-disable-line camelcase
            }, 'content_id and customer_id property set to provided content stream properties');
        });
    });

    QUnit.test('when not a provided stream change, contentId and customerId are set to empty', function(assert) {
        return this.waitForAnalytics(() => {
            this.tracker.setProperties.reset();
            this.store.dispatch({
                type: ACTION_SET_STREAM,
                stream: new VODTwitchContentStream('a video id'),
            });

            assert.equal(this.tracker.setProperties.callCount, 1, 'tracking properties are set');
            assert.deepEqual(this.tracker.setProperties.firstCall.args[0],{
                content_id: '', // eslint-disable-line camelcase
                customer_id: '', // eslint-disable-line camelcase
            }, 'content_id and customer_id property set to empty strings on non provided content stream');
        });
    });

    QUnit.test('reports `is_https` === `true` when videoInfo.segment_protocol is https', function(assert) {
        return this.waitForAnalytics(() => {
            const eventName = 'arbitrary_event';

            sinon.stub(this.player, 'getVideoInfo').returns({
                // eslint-disable-next-line camelcase
                segment_protocol: 'https',
            });

            this.analytics.trackEvent(eventName, {});

            assert.equal(this.tracker.eventHash[eventName].is_https, true);
        });
    });

    QUnit.test('reports `is_https` === `false` when videoInfo.segment_protocol is http', function(assert) {
        return this.waitForAnalytics(() => {
            const eventName = 'arbitrary_event';

            sinon.stub(this.player, 'getVideoInfo').returns({
                // eslint-disable-next-line camelcase
                segment_protocol: 'http',
            });

            this.analytics.trackEvent(eventName, {});

            assert.equal(this.tracker.eventHash[eventName].is_https, false);
        });
    });

    QUnit.module('network_profile', function(hooks) {
        hooks.beforeEach(function() {
            const fakeExperiments = {
                get() {
                    return Promise.resolve('yes'); // enable network_profile experiments for this module
                },
            };
            this.analytics.destroy();
            this.store.dispatch(setExperiments(fakeExperiments));
            this.analytics = new Analytics(this.player, this.tracker, this.state, this.store, this.options);
        });

        QUnit.test('contains time field', function(assert) {
            this.analytics.minutesWatchedTimer._events.trigger('minute');

            return waitFor(() => this.tracker.trackEvents.called).then(() => {
                const streamData = this.player.getNetworkProfile().filter()[0];
                const correctTime = streamData.startTime + streamData.firstByteLatency + streamData.downloadDuration;
                assert.ok(this.tracker.trackEvents.calledOnce);
                assert.ok(this.tracker.trackEvents.firstCall.calledWith([{
                    event: 'network_profile',
                    properties: sinon.match.has('time', correctTime),
                }]));
            });
        });

        QUnit.test('networkProfile object has valid well-defined fields', function(assert) {
            const networkProfile = this.player.getNetworkProfile().filter()[0];
            // eslint-disable-next-line max-len
            const correctTime = networkProfile.startTime + networkProfile.firstByteLatency + networkProfile.downloadDuration;

            const expected = [{
                event: 'network_profile',
                properties: {
                    /* eslint-disable camelcase */
                    time: correctTime,
                    time_to_first_byte: networkProfile.firstByteLatency,
                    download_duration: networkProfile.downloadDuration,
                    curr_buffer_length: networkProfile.currBufferLength,
                    reason_code: networkProfile.reasonCode,
                    curr_quality: networkProfile.currQuality,
                    bytes: undefined,
                    duration: undefined,
                    /* eslint-enable camelcase */
                },
            }];

            this.analytics.minutesWatchedTimer._events.trigger('minute');

            return waitFor(() => this.tracker.trackEvents.called).then(() => {
                assert.ok(this.tracker.trackEvents.calledOnce);
                const [result] = this.tracker.trackEvents.args[0];
                assert.deepEqual(result, expected);
            });
        });
    });

    QUnit.module('closed-captioning presets', function() {
        QUnit.test('tracks when pre-sets are selected, and which presets', function(assert) {
            return this.waitForAnalytics(() => {
                assert.equal(this.tracker.eventHash.player_caption_preset, undefined);

                this.store.dispatch(setCaptionsPreset('myPreset'));
                assert.equal(this.tracker.eventHash.player_caption_preset.captions_preset, 'myPreset');
            });
        });
    });

    QUnit.module('destroy', function() {
        QUnit.test('destroys minutes-watched timer', function(assert) {
            return this.waitForAnalytics(() => {
                sinon.spy(this.analytics.minutesWatchedTimer, 'destroy');

                this.analytics.destroy();
                assert.ok(this.analytics.minutesWatchedTimer.destroy.called);
            });
        });

        QUnit.test('destroys the Comscore analytics module', function(assert) {
            return this.waitForAnalytics(() => {
                sinon.spy(this.analytics.comscore, 'destroy');

                this.analytics.destroy();

                assert.equal(this.analytics.comscore.destroy.callCount, 1);
            });
        });

        QUnit.test('clears all store subscriptions', function(assert) {
            return this.waitForAnalytics(() => {
                sinon.spy(this.analytics, 'trackEvent');
                this.store.dispatch(setCaptionsPreset('yellow-on-black'));

                assert.equal(this.analytics.trackEvent.callCount, 1);

                this.analytics.destroy();

                this.store.dispatch(setCaptionsPreset('white-on-black'));
                assert.equal(this.analytics.trackEvent.callCount, 1);
            });
        });
    });

    QUnit.module('track event', function(hooks) {
        hooks.beforeEach(function() {
            sinon.spy(this.analytics, 'trackEvent');
            sinon.spy(this.analytics.tracker, 'trackEvent');
        });

        hooks.afterEach(function() {
            this.analytics.trackEvent.restore();
            this.analytics.tracker.trackEvent.restore();
        });

        QUnit.module('video_init', function() {
            QUnit.test('is fired on backend init', function(assert) {
                return this.waitForAnalytics(() => {
                    this.analytics.onBackendInit();
                    assert.ok(this.analytics.tracker.trackEvent.calledWith('video_init'));
                });
            });
        });

        QUnit.module('video_end', function() {
            QUnit.test('is fired when playback ends', function(assert) {
                this.store.dispatch(updatePlaybackState(PLAYING));

                return this.waitForAnalytics(() => {
                    this.store.dispatch(updatePlaybackState(ENDED));
                    assert.ok(this.analytics.tracker.trackEvent.calledWith('video_end'));
                });
            });
        });

        QUnit.module('Tier 1 Event VIDEO_PLAY', function() {
            QUnit.test('is fired on play with time_since_load_start property', function(assert) {
                return this.waitForAnalytics(() => {
                    this.analytics.onPlaying();

                    assert.ok(this.analytics.trackEvent.calledWith(TIER_1_EVENTS.VIDEO_PLAY, sinon.match({
                    // eslint-disable-next-line camelcase
                        time_since_load_start: sinon.match.number,
                    })));
                });
            });

            QUnit.test('contains community_ids', function(assert) {
                this.store.dispatch(setCommunitiesData(COMMUNITIES_FAKE_DATA));
                return this.waitForAnalytics(() => {
                    this.analytics.onPlaying();
                    assert.equal(
                        this.tracker.eventHash[TIER_1_EVENTS.VIDEO_PLAY].community_ids,
                        COMMUNITIES_FAKE_IDS
                    );
                });
            });

            QUnit.test('contains autoplayed set to false if options.autoplay is false', function(assert) {
                return this.waitForAnalytics(() => {
                    this.options.autoplay = false;
                    this.analytics.onPlaying();
                    assert.equal(
                        this.tracker.eventHash[TIER_1_EVENTS.VIDEO_PLAY].autoplayed,
                        this.options.autoplay
                    );
                });
            });

            QUnit.test('contains autoplayed set to true if options.autoplay is true', function(assert) {
                return this.waitForAnalytics(() => {
                    this.options.autoplay = false;
                    this.analytics.onPlaying();
                    assert.equal(
                        this.tracker.eventHash[TIER_1_EVENTS.VIDEO_PLAY].autoplayed,
                        this.options.autoplay
                    );
                });
            });

            QUnit.test('contains broadcast_id', function(assert) {
                return this.waitForAnalytics(() => {
                    this.analytics.onPlaying();
                    assert.equal(
                        this.tracker.eventHash[TIER_1_EVENTS.VIDEO_PLAY].broadcast_id,
                        this.store.getState().streamMetadata.broadcastID,
                        'has broadcastId'
                    );
                });
            });

            QUnit.test('contains collection fields when playing a collection', function(assert) {
                return this.waitForAnalytics(() => {
                    const video = TEST_COLLECTION.items[1];
                    this.store.dispatch(CollectionActions.setCollection(TEST_COLLECTION));
                    this.store.dispatch({
                        type: ACTION_SET_STREAM,
                        stream: new VODTwitchContentStream(`v${video.item_id}`),
                    });

                    this.analytics.onPlaying();

                    const videoPlayEvent = this.tracker.eventHash[TIER_1_EVENTS.VIDEO_PLAY];
                    assert.equal(
                        videoPlayEvent.collection_id,
                        TEST_COLLECTION._id,
                        'video-play reports the correct collection id'
                    );
                    assert.equal(
                        videoPlayEvent.collection_item_position,
                        1,
                        'video-play reports the correct item position'
                    );
                });
            });
        });

        QUnit.module('Tier 1 Event BUFFER_EMPTY', function(hooks) {
            // bufferEmptyCount is set to 0 on onLoadStart
            hooks.beforeEach(function() {
                this.analytics.bufferEmptyCount = 0;
            });

            hooks.afterEach(function() {
                this.analytics.bufferEmptyCount = 0;
            });

            QUnit.test('is fired on waiting with buffer_empty_count property', function(assert) {
                return this.waitForAnalytics(() => {
                    this.analytics.onWaiting();
                    assert.ok(this.analytics.trackEvent.calledWith(TIER_1_EVENTS.BUFFER_EMPTY, sinon.match({
                        // eslint-disable-next-line camelcase
                        buffer_empty_count: sinon.match.number,
                    })));
                });
            });

            QUnit.test('contains community_ids, broadcast_id, manifest_broadcast_id', function(assert) {
                this.store.dispatch(setCommunitiesData(COMMUNITIES_FAKE_DATA));
                return this.waitForAnalytics(() => {
                    this.analytics.onWaiting();
                    assert.equal(
                        this.tracker.eventHash[TIER_1_EVENTS.BUFFER_EMPTY].community_ids,
                        COMMUNITIES_FAKE_IDS
                    );

                    assert.equal(
                        this.tracker.eventHash[TIER_1_EVENTS.BUFFER_EMPTY].broadcast_id,
                        this.currentBroadcastID
                    );

                    assert.equal(
                        this.tracker.eventHash[TIER_1_EVENTS.BUFFER_EMPTY].manifest_broadcast_id,
                        this.currentBroadcastID
                    );
                });
            });
        });

        QUnit.module('Tier 1 Event BUFFER_REFILL', function(hooks) {
            // bufferEmptyStartTime is set at the same time as buffer-empty event
            hooks.beforeEach(function() {
                this.analytics.bufferEmptyStartTime = 0;
            });

            hooks.afterEach(function() {
                this.analytics.bufferEmptyStartTime = null;
            });

            QUnit.test('is fired on play after a buffer-empty event with buffer_time property', function(assert) {
                return this.waitForAnalytics(() => {
                    this.analytics.onPlaying();
                    assert.ok(this.analytics.trackEvent.calledWith(TIER_1_EVENTS.BUFFER_REFILL, sinon.match({
                        // eslint-disable-next-line camelcase
                        buffering_time: sinon.match.number,
                    })));
                });
            });

            QUnit.test('contains broadcast_id', function(assert) {
                this.store.dispatch({
                    type: ACTION_SET_STREAMMETADATA,
                    streamMetadata: {
                        broadcastID: QUnit.config.current.testId,
                    },
                });

                return this.waitForAnalytics(() => {
                    this.analytics.onPlaying();
                    assert.ok(this.analytics.trackEvent.calledWith(TIER_1_EVENTS.BUFFER_REFILL, sinon.match({
                        // eslint-disable-next-line camelcase
                        broadcast_id: QUnit.config.current.testId,
                    })));
                });
            });

            QUnit.test('contains community_ids', function(assert) {
                this.store.dispatch(setCommunitiesData(COMMUNITIES_FAKE_DATA));
                return this.waitForAnalytics(() => {
                    this.analytics.onPlaying();
                    assert.equal(
                        this.tracker.eventHash[TIER_1_EVENTS.BUFFER_REFILL].community_ids,
                        COMMUNITIES_FAKE_IDS
                    );
                });
            });
        });

        QUnit.module('vod_seek', function(hooks) {
            hooks.beforeEach(function() {
                this.analytics.isSeekInProgress = true;
                this.analytics.timeStampBeforeSeek = 100;
                this.analytics.lastSeekTime = 2000;
                sinon.stub(Date, 'now').returns(3000);
            });

            hooks.afterEach(function() {
                this.analytics.isSeekInProgress = false;
                this.analytics.timeStampBeforeSeek = 0;
                this.analytics.lastSeekTime = null;
                Date.now.restore();
            });

            QUnit.test('is fired on can-play when seeking with appropriately calculated properties', function(assert) {
                return this.waitForAnalytics(() => {
                    this.analytics.onCanPlay();
                    assert.ok(this.analytics.trackEvent.calledWith('vod_seek', sinon.match({
                    /* eslint-disable camelcase */
                        timestamp_departed: 100,
                        timestamp_target: 1000,
                        time_spent_seeking: ((3000 - 2000) / 1000),
                    /* eslint-enable camelcase */
                    })));
                });
            });
        });

        QUnit.module('VIDEO_PLAYBACK_ERROR', function() {
            QUnit.test('is fired on new error in store', function(assert) {
                return this.waitForAnalytics(() => {
                    const errorCode = Errors.CODES.ABORTED;
                    this.store.dispatch({
                        type: ACTION_ERROR,
                        code: errorCode,
                    });
                    assert.ok(this.analytics.trackEvent.calledWith(VIDEO_PLAYBACK_ERROR, sinon.match({
                        code: errorCode,
                        reason: Errors.getMessage(errorCode),
                    })), 'video playback error tracking event sent');
                });
            });

            QUnit.test('is fired if error is already in store when analytics is instantiated', function(assert) {
                const errorCode = Errors.CODES.ABORTED;
                this.store.dispatch({
                    type: ACTION_ERROR,
                    code: errorCode,
                });

                return this.waitForAnalytics(() => {
                    assert.ok(this.analytics.trackEvent.calledWith(VIDEO_PLAYBACK_ERROR, sinon.match({
                        code: errorCode,
                        reason: Errors.getMessage(errorCode),
                    })), 'video playback error tracking event sent');
                });
            });
        });
    });

    QUnit.module('comscore', function(hooks) {
        hooks.beforeEach(function() {
            this.window.COMSCORE = {
                beacon: sinon.spy(),
            };
        });

        QUnit.test('fires a player and video beacon on first play', function(assert) {
            return this.waitForAnalytics(() => {
                this.analytics.onPlaying();

                assert.equal(this.window.COMSCORE.beacon.callCount, 2);
                assert.notDeepEqual(
                    this.window.COMSCORE.beacon.firstCall.args[0],
                    this.window.COMSCORE.beacon.firstCall.args[1]
                );
            });
        });

        QUnit.test('does not fire a beacon on the second play of the same stream', function(assert) {
            return this.waitForAnalytics(() => {
                this.analytics.onPlaying();
                this.analytics.onPlaying();

                assert.equal(this.window.COMSCORE.beacon.callCount, 2);
            });
        });

        QUnit.test('fires a beacon on the first play of a newly loaded stream', function(assert) {
            return this.waitForAnalytics(() => {
                this.analytics.onPlaying();
                assert.equal(this.window.COMSCORE.beacon.callCount, 2);

                this.analytics.onLoadStart();
                this.analytics.onPlaying();
                assert.equal(this.window.COMSCORE.beacon.callCount, 4);
            });
        });
    });

    QUnit.module('countess', function() {
        QUnit.test('for a VOD on Playing event emits a video view count', function(assert) {
            const countessTrackVodViewSpy = sinon.spy(this.analytics.countessTracker, 'trackVODView');
            const videoID = 'v12345667';
            const channelName = 'channel';

            this.store.dispatch({
                type: ACTION_SET_STREAM,
                stream: new VODTwitchContentStream(videoID),
            });

            const videoInfo = this.api.expectVideoInfo(videoID, {
                channel: {
                    name: channelName,
                },
            });

            this.api.expectChannelInfo(videoInfo.channel.name, {
                name: videoInfo.channel.name,
            });

            this.analytics.onPlaying();

            return waitFor(() => countessTrackVodViewSpy.called).then(() => {
                assert.ok(countessTrackVodViewSpy.calledWith(sinon.match(videoInfo)));
            });
        });
    });

    QUnit.module('onMiniChange when env is PLAYER_SITE', function(hooks) {
        hooks.beforeEach(function() {
            this.store.dispatch({
                type: ACTION_SET_PLAYER_TYPE,
                playerType: PLAYER_SITE,
            });
        });

        QUnit.test('when the isMini state changes to true', function(assert) {
            this.store.dispatch({
                type: ACTION_SET_MINIPLAYER_MODE,
                value: true,
            });
            // eslint-disable-next-line max-len
            assert.ok(this.tracker.setProperty.calledWith('player', PLAYER_SITE_MINI), 'sets player to PLAYER_SITE_MINI');
        });

        QUnit.test('when the isMini state changes to false', function(assert) {
            this.store.dispatch({
                type: ACTION_SET_MINIPLAYER_MODE,
                value: true,
            });

            this.store.dispatch({
                type: ACTION_SET_MINIPLAYER_MODE,
                value: false,
            });

            assert.ok(this.tracker.setProperty.calledWith('player', PLAYER_SITE), 'sets player to PLAYER_SITE');
        });
    });

    QUnit.module('onMiniChange when env is not PLAYER_SITE', function() {
        QUnit.test('when isMini changes', function(assert) {
            this.store.dispatch({
                type: ACTION_SET_MINIPLAYER_MODE,
                value: true,
            });

            assert.notOk(this.tracker.setProperty.called, 'it does not set the player attribute');
        });
    });

    QUnit.module('onWatchPartyVodChange', function() {
        QUnit.test('when watchParty.vodId changes', function(assert) {
            const countessTrackVodViewSpy = sinon.spy(this.analytics.countessTracker, 'trackVODView');

            this.store.dispatch({
                type: ACTION_SET_WATCHPARTY_VOD,
                pubsubVodInfo: {
                    vodId: 'new_vod_id',
                    broadcastType: 'new type',
                    incrementViewCountURL: 'fake-url-for-incrementing',
                },
            });

            return waitFor(() => countessTrackVodViewSpy.called).then(() => {
                assert.ok(countessTrackVodViewSpy.calledWith(sinon.match({
                    increment_view_count_url: 'fake-url-for-incrementing', // eslint-disable-line camelcase
                })));
                assert.ok(
                    this.tracker.setProperty.calledWith('vod_type', 'new type'), 'vod_type should be set on VOD change'
                );
            });
        });

        QUnit.test('when watchParty.vodId changes to null it is not fired', function(assert) {
            const countessTrackVodViewSpy = sinon.spy(this.analytics.countessTracker, 'trackVODView');

            this.store.dispatch({
                type: ACTION_SET_WATCHPARTY_VOD,
                pubsubVodInfo: {
                    vodId: null,
                    incrementViewCountURL: 'fake-url-for-incrementing',
                },
            });

            assert.ok(countessTrackVodViewSpy.notCalled, 'it should not increment view count for a null vod id');
        });

        QUnit.test('when setting the same vod ID it should not fire', function(assert) {
            const countessTrackVodViewSpy = sinon.spy(this.analytics.countessTracker, 'trackVODView');

            this.store.dispatch({
                type: ACTION_SET_WATCHPARTY_VOD,
                pubsubVodInfo: {
                    vodId: '1',
                    incrementViewCountURL: 'fake-url-for-incrementing',
                },
            });

            return waitFor(() => countessTrackVodViewSpy.called).then(() => {
                // Fires for initial set
                assert.ok(countessTrackVodViewSpy.calledWith(sinon.match({
                    increment_view_count_url: 'fake-url-for-incrementing', // eslint-disable-line camelcase
                })));

                countessTrackVodViewSpy.reset();

                this.store.dispatch({
                    type: ACTION_SET_WATCHPARTY_VOD,
                    pubsubVodInfo: {
                        vodId: '1',
                        incrementViewCountURL: 'fake-url-for-incrementing',
                        foo: '',
                    },
                });

                assert.ok(countessTrackVodViewSpy.notCalled, 'it should not increment view count for the same vod id');
            });
        });
    });

    QUnit.module('onWatchPartyChange', function() {
        QUnit.test('when watchParty.watchPartyId changes', function(assert) {
            const countessTrackViewSpy = sinon.spy(this.analytics.countessTracker, 'trackView');

            this.store.dispatch({
                type: ACTION_SET_WATCHPARTY_VOD,
                pubsubVodInfo: {
                    watchPartyId: 'new_watch_party_id',
                },
            });

            return waitFor(() => countessTrackViewSpy.called).then(() => {
                assert.ok(countessTrackViewSpy.calledWith('watch_party', 'new_watch_party_id'));
            });
        });

        QUnit.test('when watchParty.watchPartyId is null it is not fired', function(assert) {
            const countessTrackViewSpy = sinon.spy(this.analytics.countessTracker, 'trackView');

            this.store.dispatch({
                type: ACTION_SET_WATCHPARTY_VOD,
                pubsubVodInfo: {
                    watchPartyId: null,
                },
            });

            assert.ok(countessTrackViewSpy.notCalled, 'it should not increment view count for a null watch party id');
        });

        QUnit.test('when setting the same watch party ID it should not fire', function(assert) {
            const countessTrackViewSpy = sinon.spy(this.analytics.countessTracker, 'trackView');

            this.store.dispatch({
                type: ACTION_SET_WATCHPARTY_VOD,
                pubsubVodInfo: {
                    watchPartyId: '2',
                },
            });

            return waitFor(() => countessTrackViewSpy.called).then(() => {
                // Fires for initial set
                assert.ok(countessTrackViewSpy.calledWith('watch_party', '2'));

                countessTrackViewSpy.reset();

                this.store.dispatch({
                    type: ACTION_SET_WATCHPARTY_VOD,
                    pubsubVodInfo: {
                        watchPartyId: '2',
                        foo: '',
                    },
                });

                assert.ok(countessTrackViewSpy.notCalled, 'it should not increment view count for the same ID');
            });
        });
    });
});
