import each from 'lodash/each';
import padStart from 'lodash/padStart';
import sinon from 'sinon';
import Comscore from 'tests/fakes/comscore.fake.js';
import { init as initStore } from 'state';
import * as StreamActions from 'actions/stream';
import * as StreamMetadataActions from 'actions/stream-metadata';
import { setCurrentAdMetadata, clearCurrentAdMetadata, AdContentTypes,
         AdRollTypes } from 'actions/ads';
import { setWindow } from 'actions/window';
import { CONTENT_MODE_LIVE, LiveTwitchContentStream } from 'stream/twitch-live';
import { CONTENT_MODE_VOD, VODTwitchContentStream } from 'stream/twitch-vod';
import { unitTest } from 'tests/utils/module';
import { emitEvent } from 'tests/utils/events';
import { waitFor } from 'tests/utils/waitFor';
import { ComscoreAnalytics } from 'analytics/comscore';
import { buildFakeWindow } from 'tests/fakes/window.fake';
import * as AdEvents from 'ads/advertisement-event';

unitTest('analytics/comscore', function(hooks) {
    hooks.beforeEach(function() {
        this.analytics = {
        };
        this.player = {
            addEventListener: sinon.spy(),
        };
        this.window = buildFakeWindow();

        this.store = initStore();
        this.store.dispatch(setWindow(this.window));

        this.streams = {
            [CONTENT_MODE_LIVE]: new LiveTwitchContentStream(
                `channel_${QUnit.config.current.testId.toLowerCase()}`,
                Promise.resolve(null),
                Promise.resolve(null)
            ),
            [CONTENT_MODE_VOD]: new VODTwitchContentStream(
                `v${parseInt(QUnit.config.current.testId, 36)}`,
                Promise.resolve(null),
                Promise.resolve(null)
            ),
        };

        this.api.setLoggedIn(false);

        this.comscoreAnalytics = new ComscoreAnalytics(this.analytics, this.player, this.store);

        // Reset Comscore fake spies
        const { comscore } = this.store.getState();
        for (const prop in comscore.streamingTag) {
            comscore.streamingTag[prop].reset();
        }
    });

    const GAME_GENRES = {
        Creative: 'Creative',
        Gaming: 'Gaming',
        Poker: 'Poker',
        Music: 'Music',
        [`Game ${Number(Math.random() * 1e9).toString(36)}`]: 'Gaming',
    };

    const TEST_CASES = [
        {
            contentMode: CONTENT_MODE_LIVE,
            partner: true,
            mediaType: Comscore.StreamingTag.ContentType.Live,
            advertisements: {
                preroll: Comscore.StreamingTag.AdType.LinearLive,
                midroll: Comscore.StreamingTag.AdType.LinearLive,
                postroll: Comscore.StreamingTag.AdType.LinearLive,
            },
        },
        {
            contentMode: CONTENT_MODE_LIVE,
            partner: false,
            mediaType: Comscore.StreamingTag.ContentType.UserGeneratedLive,
            advertisements: {
                preroll: Comscore.StreamingTag.AdType.LinearLive,
                midroll: Comscore.StreamingTag.AdType.LinearLive,
                postroll: Comscore.StreamingTag.AdType.LinearLive,
            },
        },
        {
            contentMode: CONTENT_MODE_VOD,
            duration: 30,
            partner: true,
            mediaType: Comscore.StreamingTag.ContentType.ShortFormOnDemand,
            advertisements: {
                preroll: Comscore.StreamingTag.AdType.LinearOnDemandPreRoll,
                midroll: Comscore.StreamingTag.AdType.LinearOnDemandMidRoll,
                postroll: Comscore.StreamingTag.AdType.LinearOnDemandPostRoll,
            },
        },
        {
            contentMode: CONTENT_MODE_VOD,
            duration: 30,
            partner: false,
            mediaType: Comscore.StreamingTag.ContentType.UserGeneratedShortFormOnDemand,
            advertisements: {
                preroll: Comscore.StreamingTag.AdType.LinearOnDemandPreRoll,
                midroll: Comscore.StreamingTag.AdType.LinearOnDemandMidRoll,
                postroll: Comscore.StreamingTag.AdType.LinearOnDemandPostRoll,
            },
        },
        {
            contentMode: CONTENT_MODE_VOD,
            duration: 3600,
            partner: true,
            mediaType: Comscore.StreamingTag.ContentType.LongFormOnDemand,
            advertisements: {
                preroll: Comscore.StreamingTag.AdType.LinearOnDemandPreRoll,
                midroll: Comscore.StreamingTag.AdType.LinearOnDemandMidRoll,
                postroll: Comscore.StreamingTag.AdType.LinearOnDemandPostRoll,
            },
        },
        {
            contentMode: CONTENT_MODE_VOD,
            duration: 3600,
            partner: false,
            mediaType: Comscore.StreamingTag.ContentType.UserGeneratedLongFormOnDemand,
            advertisements: {
                preroll: Comscore.StreamingTag.AdType.LinearOnDemandPreRoll,
                midroll: Comscore.StreamingTag.AdType.LinearOnDemandMidRoll,
                postroll: Comscore.StreamingTag.AdType.LinearOnDemandPostRoll,
            },
        },
    ];

    QUnit.test('calls `stop()` before the page unloads', function(assert) {
        const { comscore } = this.store.getState();

        assert.equal(
            this.window.addEventListener.withArgs(
                'beforeunload',
                sinon.match({
                    handleEvent: sinon.match.func,
                })
            ).callCount,
            1
        );

        emitEvent(this.window, 'beforeunload');

        assert.equal(comscore.streamingTag.stop.callCount, 1);
    });

    QUnit.test('calls `stop()` when the analytics module is destroyed', function(assert) {
        const { comscore } = this.store.getState();

        this.comscoreAnalytics.destroy();

        assert.equal(comscore.streamingTag.stop.callCount, 1);
    });

    TEST_CASES.forEach(function(testCase) {
        const partnerType = (testCase.partner ? 'a partnered' : 'an unpartnered');
        const contentDesc = (function(contentMode, duration) {
            switch (contentMode) {
            case CONTENT_MODE_LIVE:
                return 'live';
            case CONTENT_MODE_VOD:
                if (duration > 600) {
                    return 'long VOD';
                }
                return 'short VOD';
            }
        })(testCase.contentMode, testCase.duration);

        QUnit.module(`when watching ${contentDesc} content from ${partnerType} broadcaster`, function(hooks) {
            hooks.beforeEach(function() {
                this.store.dispatch({
                    type: StreamActions.ACTION_SET_STREAM,
                    stream: this.streams[testCase.contentMode],
                });
            });

            QUnit.module('when advertisements are showing', function(hooks) {
                hooks.beforeEach(function() {
                    this.store.dispatch(setCurrentAdMetadata({
                        contentType: AdContentTypes.IMA,
                        rollType: AdRollTypes.PREROLL,
                    }));

                    // TODO need a better way of testing asynchronous callbacks in event
                    // listeners; there's no return value to key off, and no
                    // call to waitFor
                    sinon.spy(Promise, 'all');
                });

                hooks.afterEach(function() {
                    Promise.all.restore();
                });

                QUnit.test('should not call `playVideoContentPart` when content begins playing', function(assert) {
                    emitEvent(this.player, 'playing');

                    assert.equal(Promise.all.callCount, 0);
                });

                QUnit.test('should not call `stop` when content is paused', function(assert) {
                    const { comscore } = this.store.getState();

                    emitEvent(this.player, 'pause');

                    assert.equal(comscore.streamingTag.stop.callCount, 0);
                });
            });

            QUnit.module('when advertisements are not showing', function(hooks) {
                hooks.beforeEach(function() {
                    this.store.dispatch(clearCurrentAdMetadata());
                });

                each(GAME_GENRES, (genre, game) => {
                    const gameName = (game === genre ? game : 'arbitrary game');
                    QUnit.test(`calls \`playVideoContentPart\` when ${gameName} content plays`, function(assert) {
                        const now = new Date();
                        this.store.dispatch({
                            type: StreamMetadataActions.ACTION_SET_STREAMMETADATA,
                            streamMetadata: {
                                broadcastID: parseInt(QUnit.config.current.testId, 36),
                                channel: {
                                    displayName: `Channel ${QUnit.config.current.testId}`,
                                    name: '',
                                    partner: testCase.partner,
                                    status: `Episode ${parseInt(QUnit.config.current.testId, 36)}`,
                                },
                                // eslint-disable-next-line max-len
                                createdAt: new Date(now.getUTCFullYear(), now.getUTCMonth(), now.getUTCDate()).getTime(),
                                game,
                                preview: {
                                    template: '',
                                },
                                viewers: parseInt(QUnit.config.current.testId, 36),
                            },
                        });

                        /* eslint-disable camelcase */
                        this.channelInfo = {
                            name: 'channelName',
                            display_name: 'displayName',
                        };

                        if (testCase.contentMode === CONTENT_MODE_VOD) {
                            this.videoInfo = this.api.expectVideoInfo(this.streams[testCase.contentMode].videoId, {
                                game,
                                channel: this.channelInfo,
                                length: testCase.duration,
                                partner: testCase.partner,
                            });
                        }
                        this.channelInfo = this.api.expectChannelInfo(this.channelInfo.name, {
                            name: this.channelInfo.name,
                            display_name: this.channelInfo.display_name,
                            partner: testCase.partner,
                        });
                        /* eslint-enable camelcase */

                        const { comscore } = this.store.getState();

                        emitEvent(this.player, 'playing');

                        return waitFor(() => comscore.streamingTag.playVideoContentPart.firstCall).
                            then(call => {
                                const metadata = {
                                    /* eslint-disable camelcase */
                                    ns_st_pu: '*null',
                                    ns_st_pr: '*null',
                                    ns_st_sn: '*null',
                                    ns_st_en: '*null',
                                    ns_st_ge: genre,
                                    ns_st_ti: '*null',
                                    ns_st_ia: '0',
                                    ns_st_tdt: '*null',
                                    c3: 'TWITCH',
                                    c4: 'twitch.tv',
                                    c6: '*null',
                                    /* eslint-enable */
                                };
                                if (testCase.contentMode === CONTENT_MODE_LIVE) {
                                    const { streamMetadata } = this.store.getState();
                                    /* eslint-disable camelcase */
                                    metadata.ns_st_ci = String(streamMetadata.broadcastID);
                                    metadata.ns_st_cl = '*null';
                                    metadata.ns_st_st = streamMetadata.channel.displayName;
                                    metadata.ns_st_ep = streamMetadata.channel.status;
                                    metadata.ns_st_ddt = toComscoreDate(now);
                                    metadata.ns_st_ce = '1';
                                    /* eslint-enable */
                                } else {
                                    /* eslint-disable camelcase */
                                    metadata.ns_st_ci = String(this.videoInfo.broadcast_id);
                                    metadata.ns_st_cl = String(this.videoInfo.length * 1000);
                                    metadata.ns_st_st = this.videoInfo.channel.display_name;
                                    metadata.ns_st_ep = this.videoInfo.title;
                                    metadata.ns_st_ddt = toComscoreDate(new Date(this.videoInfo.recorded_at));
                                    metadata.ns_st_ce = '1';
                                    /* eslint-enable */
                                }
                                assert.deepEqual(call.args[0], metadata);
                                assert.equal(call.args[1], testCase.mediaType);
                            });
                    });
                });
            });

            QUnit.test('should call `stop` when content is paused', function(assert) {
                const { comscore } = this.store.getState();

                emitEvent(this.player, 'pause');

                assert.equal(comscore.streamingTag.stop.callCount, 1);
            });

            QUnit.module('when the current content has ever played', function(hooks) {
                hooks.beforeEach(function() {
                    this.analytics.hasPlayed = true;
                });

                QUnit.module('when the player is paused', function(hooks) {
                    hooks.beforeEach(function() {
                        this.player.paused = true;
                    });

                    QUnit.test('should not call `stop` when a VOD begins seeking to a new playhead', function(assert) {
                        const { comscore } = this.store.getState();

                        emitEvent(this.player, 'seeking');

                        assert.equal(comscore.streamingTag.stop.callCount, 0);
                    });
                });

                QUnit.module('when the player is playing', function(hooks) {
                    hooks.beforeEach(function() {
                        this.player.paused = false;
                    });

                    QUnit.test('should call `stop` when a VOD begins seeking to a new playhead', function(assert) {
                        const { comscore } = this.store.getState();

                        emitEvent(this.player, 'seeking');

                        assert.equal(comscore.streamingTag.stop.callCount, 1);
                    });

                    QUnit.module('when the content reaches its end', function(hooks) {
                        hooks.beforeEach(function() {
                            emitEvent(this.player, 'ended');
                            sinon.spy(this.store, 'dispatch');
                        });

                        QUnit.test('should reset tag when a VOD begins seeking to a new playhead', function(assert) {
                            assert.ok(this.comscoreAnalytics.ended);
                            emitEvent(this.player, 'seeking');
                            assert.ok(!this.comscoreAnalytics.ended);
                            this.store.dispatch.calledWithMatch({
                                type: 'reset tag',
                            });
                        });
                    });
                });
            });

            QUnit.module('when the current content has never played', function(hooks) {
                hooks.beforeEach(function() {
                    this.analytics.hasPlayed = false;
                });

                QUnit.test('should not call `stop` when a VOD begins seeking to a new playhead', function(assert) {
                    const { comscore } = this.store.getState();

                    emitEvent(this.player, 'seeking');

                    assert.equal(comscore.streamingTag.stop.callCount, 0);
                });
            });

            QUnit.test('should call `stop` when content reaches its end', function(assert) {
                const { comscore } = this.store.getState();

                emitEvent(this.player, 'ended');

                assert.equal(comscore.streamingTag.stop.callCount, 1);
            });

            each(testCase.advertisements, (mediaType, rollType) => {
                QUnit.test(`should call \`playVideoAdvertisement\` when a ${rollType} plays`, function(assert) {
                    const { comscore } = this.store.getState();
                    const duration = Math.floor(parseInt(QUnit.config.current.testId, 36) / 1000);

                    // TODO: when VP-845 fixed, remove this block--ended will fire after postroll.
                    if (rollType === 'postroll') {
                        emitEvent(this.player, 'ended');
                    }

                    emitEvent(this.player, AdEvents.AD_START, {
                        /* eslint-disable camelcase */
                        roll_type: rollType,
                        time_break: duration,
                        /* eslint-enable */
                    });
                    emitEvent(this.player, AdEvents.AD_IMPRESSION, {
                        /* eslint-disable camelcase */
                        roll_type: rollType,
                        time_break: duration,
                        /* eslint-enable */
                    });

                    assert.ok(comscore.streamingTag.stop.calledBefore(comscore.streamingTag.playVideoAdvertisement));
                    assert.deepEqual(comscore.streamingTag.playVideoAdvertisement.firstCall.args, [
                        {
                            // eslint-disable-next-line camelcase
                            ns_st_cl: String(duration * 1000),
                        },
                        mediaType,
                    ]);
                });

                /* eslint-disable max-len */
                QUnit.test(`should ${rollType === 'postroll' ? 'not ' : ''}call \`stop()\` at the beginning of a ${rollType} ad pod`, function(assert) {
                    const { comscore } = this.store.getState();
                    const duration = Math.floor(parseInt(QUnit.config.current.testId, 36) / 1000);

                    emitEvent(this.player, AdEvents.AD_START, {
                        /* eslint-disable camelcase */
                        roll_type: rollType,
                        time_break: duration,
                        /* eslint-enable */
                    });

                    emitEvent(this.player, AdEvents.AD_IMPRESSION, {
                        /* eslint-disable camelcase */
                        roll_type: rollType,
                        time_break: duration,
                        /* eslint-enable */
                    });

                    // TODO: when VP-845 fixed, this will need fixing--ended will not be called before postroll
                    if (rollType === 'postroll') {
                        assert.ok(!comscore.streamingTag.stop.called);
                    } else {
                        assert.ok(comscore.streamingTag.stop.calledOnce);
                    }
                });

                /* eslint-disable max-len */
                QUnit.test(`should ${rollType === 'postroll' ? 'not ' : ''}call \`playVideoContentPart\` at the end of a ${rollType} ad pod`, function(assert) {
                    const { comscore } = this.store.getState();
                    const duration = Math.floor(parseInt(QUnit.config.current.testId, 36) / 1000);
                    const now = new Date();

                    this.store.dispatch({
                        type: StreamMetadataActions.ACTION_SET_STREAMMETADATA,
                        streamMetadata: {
                            broadcastID: parseInt(QUnit.config.current.testId, 36),
                            channel: {
                                displayName: `Channel ${QUnit.config.current.testId}`,
                                name: '',
                                partner: testCase.partner,
                                status: `Episode ${parseInt(QUnit.config.current.testId, 36)}`,
                            },
                            createdAt: new Date(now.getUTCFullYear(), now.getUTCMonth(), now.getUTCDate()).getTime(),
                            game: 'arbitrary game',
                            preview: {
                                template: '',
                            },
                            viewers: parseInt(QUnit.config.current.testId, 36),
                        },
                    });

                    this.channelInfo = {
                        /* eslint-disable camelcase */
                        name: 'channelName',
                        display_name: 'displayName',
                        /* eslint-enable */
                    };
                    if (testCase.contentMode === CONTENT_MODE_VOD) {
                        this.videoInfo = this.api.expectVideoInfo(this.streams[testCase.contentMode].videoId, {
                            game: 'arbitrary game',
                            channel: this.channelInfo,
                            length: testCase.duration,
                            partner: testCase.partner,
                        });
                    }

                    /* eslint-disable camelcase */
                    this.channelInfo = this.api.expectChannelInfo(this.channelInfo.name, {
                        name: this.channelInfo.name,
                        display_name: this.channelInfo.display_name,
                        partner: testCase.partner,
                    });
                    /* eslint-enable camelcase */

                    emitEvent(this.player, AdEvents.AD_IMPRESSION_COMPLETE, {
                        /* eslint-disable camelcase */
                        roll_type: rollType,
                        time_break: duration,
                        /* eslint-enable */
                    });

                    emitEvent(this.player, AdEvents.AD_END, {
                        /* eslint-disable camelcase */
                        roll_type: rollType,
                        time_break: duration,
                        /* eslint-enable */
                    });

                    if (rollType === 'postroll') {
                        assert.ok(!comscore.streamingTag.playVideoContentPart.called);
                    } else {
                        return waitFor(() => comscore.streamingTag.playVideoContentPart.called).
                            then(() => {
                                assert.ok(comscore.streamingTag.playVideoContentPart.calledOnce);
                            });
                    }
                });

                /* eslint-disable max-len */
                QUnit.test('should call `playVideoAdvertisement()` and `stop()` in correct sequence for each ad impression', function(assert) {
                    const { comscore } = this.store.getState();
                    const duration = Math.floor(parseInt(QUnit.config.current.testId, 36) / 1000);

                    emitEvent(this.player, AdEvents.AD_IMPRESSION, {
                        /* eslint-disable camelcase */
                        roll_type: rollType,
                        time_break: duration,
                        /* eslint-enable */
                    });

                    assert.ok(comscore.streamingTag.playVideoAdvertisement.calledOnce);

                    emitEvent(this.player, AdEvents.AD_IMPRESSION_COMPLETE, {
                        /* eslint-disable camelcase */
                        roll_type: rollType,
                        time_break: duration,
                        /* eslint-enable */
                    });

                    assert.ok(comscore.streamingTag.stop.calledOnce);

                    emitEvent(this.player, AdEvents.AD_IMPRESSION, {
                        /* eslint-disable camelcase */
                        roll_type: rollType,
                        time_break: duration,
                        /* eslint-enable */
                    });

                    assert.ok(comscore.streamingTag.playVideoAdvertisement.calledTwice);

                    emitEvent(this.player, AdEvents.AD_IMPRESSION_COMPLETE, {
                        /* eslint-disable camelcase */
                        roll_type: rollType,
                        time_break: duration,
                        /* eslint-enable */
                    });

                    assert.ok(comscore.streamingTag.stop.calledTwice);
                });
            });

            QUnit.test('should call `stop` when an advertisement stops playing', function(assert) {
                const { comscore } = this.store.getState();

                emitEvent(this.player, AdEvents.AD_IMPRESSION_COMPLETE);
                emitEvent(this.player, AdEvents.AD_END);

                assert.equal(comscore.streamingTag.stop.callCount, 1);
            });
        });
    });
});

function toComscoreDate(date) {
    const year = date.getUTCFullYear();
    const month = padStart(date.getUTCMonth() + 1, 2, '0');
    const day = padStart(date.getUTCDate(), 2, '0');
    return `${year}-${month}-${day}`;
}
