import { FirstPartyClient } from 'analytics/first-party';
import { init as initStore } from 'state';
import { TIER_1_EVENTS, VIDEO_PLAYBACK_ERROR } from 'analytics/events';
import { ACTION_FIRST_PARTY_ANALYTIC_EVENT } from 'actions/embed-event-emitter';
import sinon from 'sinon';

QUnit.module('analytics | first-party', function(hooks) {
    hooks.beforeEach(function() {
        this.store = initStore();
        this.dispatchSpy = sinon.stub(this.store, 'dispatch');
        this.firstPartyClient = new FirstPartyClient(this.store);
    });

    // eslint-disable-next-line max-len
    QUnit.test('on minute watched event dispatches emit first party events action with limited payload', function(assert) {
        const minuteWatchedEvent = {
            event: TIER_1_EVENTS.MINUTE_WATCHED,
            /* eslint-disable camelcase */
            properties: {
                time: 123567,
                device_id: 'a device id',
                play_session_id: 'a session id',
                customer_id: 'a customer id',
                content_id: 'a content id',
                extraProperty: 'an extra property',
            },
            /* eslint-enable camelcase */
        };

        assert.equal(this.dispatchSpy.callCount, 0);

        this.firstPartyClient.trackEvents([minuteWatchedEvent]);

        assert.equal(this.dispatchSpy.callCount, 1);

        const dispatchedAction = this.dispatchSpy.firstCall.args[0];

        assert.equal(dispatchedAction.type, ACTION_FIRST_PARTY_ANALYTIC_EVENT);
        assert.equal(dispatchedAction.event, TIER_1_EVENTS.MINUTE_WATCHED);
        /* eslint-disable camelcase */
        assert.deepEqual(dispatchedAction.data, {
            time: 123567,
            device_id: 'a device id',
            play_session_id: 'a session id',
            customer_id: 'a customer id',
            content_id: 'a content id',
        });
        /* eslint-enable camelcase */
    });

    QUnit.test('on video play event dispatches emit first party events action with limited payload', function(assert) {
        const videoPlayEvent = {
            event: TIER_1_EVENTS.VIDEO_PLAY,
            /* eslint-disable camelcase */
            properties: {
                time: 123567,
                device_id: 'a device id',
                play_session_id: 'a session id',
                customer_id: 'a customer id',
                content_id: 'a content id',
                time_since_load_start: 124566,
                extraProperty: 'an extra property',
            },
            /* eslint-enable camelcase */
        };

        assert.equal(this.dispatchSpy.callCount, 0);

        this.firstPartyClient.trackEvents([videoPlayEvent]);

        assert.equal(this.dispatchSpy.callCount, 1);

        const dispatchedAction = this.dispatchSpy.firstCall.args[0];

        assert.equal(dispatchedAction.type, ACTION_FIRST_PARTY_ANALYTIC_EVENT);
        assert.equal(dispatchedAction.event, TIER_1_EVENTS.VIDEO_PLAY);
        /* eslint-disable camelcase */
        assert.deepEqual(dispatchedAction.data, {
            time: 123567,
            device_id: 'a device id',
            play_session_id: 'a session id',
            time_since_load_start: 124566,
            customer_id: 'a customer id',
            content_id: 'a content id',
        });
        /* eslint-enable camelcase */
    });

    // eslint-disable-next-line max-len
    QUnit.test('on buffer empty event dispatches emit first party events action with limited payload', function(assert) {
        const bufferEmptyEvent = {
            event: TIER_1_EVENTS.BUFFER_EMPTY,
            /* eslint-disable camelcase */
            properties: {
                time: 123567,
                device_id: 'a device id',
                play_session_id: 'a session id',
                customer_id: 'a customer id',
                content_id: 'a content id',
                buffer_empty_count: 123566,
                extraProperty: 'an extra property',
            },
            /* eslint-enable camelcase */
        };

        assert.equal(this.dispatchSpy.callCount, 0);

        this.firstPartyClient.trackEvents([bufferEmptyEvent]);

        assert.equal(this.dispatchSpy.callCount, 1);

        const dispatchedAction = this.dispatchSpy.firstCall.args[0];

        assert.equal(dispatchedAction.type, ACTION_FIRST_PARTY_ANALYTIC_EVENT);
        assert.equal(dispatchedAction.event, TIER_1_EVENTS.BUFFER_EMPTY);
        /* eslint-disable camelcase */
        assert.deepEqual(dispatchedAction.data, {
            time: 123567,
            device_id: 'a device id',
            play_session_id: 'a session id',
            buffer_empty_count: 123566,
            customer_id: 'a customer id',
            content_id: 'a content id',
        });
        /* eslint-enable camelcase */
    });

    QUnit.test('on video error event dispatches emit first party events action with limited payload', function(assert) {
        const videoErrorEvent = {
            event: VIDEO_PLAYBACK_ERROR,
            /* eslint-disable camelcase */
            properties: {
                time: 123567,
                device_id: 'a device id',
                play_session_id: 'a session id',
                customer_id: 'a customer id',
                content_id: 'a content id',
                reason: 'a reason',
                code: 12345,
                extraProperty: 'an extra property',
            },
            /* eslint-enable camelcase */
        };

        assert.equal(this.dispatchSpy.callCount, 0);

        this.firstPartyClient.trackEvents([videoErrorEvent]);

        assert.equal(this.dispatchSpy.callCount, 1);

        const dispatchedAction = this.dispatchSpy.firstCall.args[0];

        assert.equal(dispatchedAction.type, ACTION_FIRST_PARTY_ANALYTIC_EVENT);
        assert.equal(dispatchedAction.event, VIDEO_PLAYBACK_ERROR);
        /* eslint-disable camelcase */
        assert.deepEqual(dispatchedAction.data, {
            time: 123567,
            device_id: 'a device id',
            play_session_id: 'a session id',
            customer_id: 'a customer id',
            content_id: 'a content id',
            reason: 'a reason',
            code: 12345,
        });
        /* eslint-enable camelcase */
    });

    QUnit.test('filters out non first party analytic events on trackEvents', function(assert) {
        const firstPartyEvent = {
            event: VIDEO_PLAYBACK_ERROR,
            properties: {
                time: 123567,
            },
        };

        const nonFirstPartyEvent = {
            event: TIER_1_EVENTS.BUFFER_REFILL,
            /* eslint-disable camelcase */
            properties: {
                time: 123567,
            },
            /* eslint-enable camelcase */
        };

        assert.equal(this.dispatchSpy.callCount, 0);

        this.firstPartyClient.trackEvents([firstPartyEvent, nonFirstPartyEvent]);

        assert.equal(this.dispatchSpy.callCount, 1);

        const dispatchedAction = this.dispatchSpy.firstCall.args[0];

        assert.equal(dispatchedAction.type, ACTION_FIRST_PARTY_ANALYTIC_EVENT);
        assert.equal(dispatchedAction.event, VIDEO_PLAYBACK_ERROR);
        assert.deepEqual(dispatchedAction.data, {
            time: 123567,
        });
    });
});
