import { LatencyTracker, murmurSegmentSampler } from 'analytics/latency';

import { setAnalyticsTracker } from 'actions/analytics-tracker';
import { setCurrentQuality } from 'actions/quality';
import { ACTION_SET_STREAM } from 'actions/stream';
import * as TwitchEvents from 'backend/events/twitch-event';
import { init as initStore } from 'state';
import { LiveTwitchContentStream } from 'stream/twitch-live';
import { VODTwitchContentStream } from 'stream/twitch-vod';

QUnit.module('analytics | latency', function(hooks) {
    const channel = 'lirik';
    const quality = 'chunked';
    const segment = 'index-0000002731-LJKI.ts';

    const segmentKey = 'lirik,chunked,index-0000002731-LJKI.ts';

    hooks.beforeEach(function() {
        this.analytics = new FakeAnalytics();
        this.player = new FakePlayer();

        this.store = initStore();
        this.store.dispatch(setAnalyticsTracker(new FakeAnalytics()));
        this.store.dispatch(setStream(channel, true));
        this.store.dispatch(setCurrentQuality(quality));

        /* eslint-disable camelcase */
        this.segmentChangePayload = {
            name: segment,
            broadcaster_send: 1,
            ingest_receive: 2,
            ingest_send: 3,
            transcode_receive: 4,
            transcode_send: 5,
            playback_start: 6,
        };
        /* eslint-enable camelcase */

        this.latencyTracker = new LatencyTracker(this.analytics, 1.0, this.player, this.store);
    });

    QUnit.test('sets up channel and quality based on state store', function(assert) {
        // Send a segment change:
        this.player.emit(TwitchEvents.SEGMENT_CHANGE, this.segmentChangePayload);

        // Did we get one event with the proper channel and quality?
        assert.equal(this.analytics.tracked.length, 1, 'exactly one tracking event should have been sent');
        let event = this.analytics.tracked[0];
        assert.equal(event.properties.segment, segmentKey);

        // Switch channel and quality:
        this.store.dispatch(setStream('monstercat', true));
        this.store.dispatch(setCurrentQuality('low'));

        // Send a new segment change:
        /* eslint-disable camelcase */
        this.player.emit(TwitchEvents.SEGMENT_CHANGE, {
            name: 'segment2.ts',
            broadcaster_send: 1,
            ingest_receive: 2,
            ingest_send: 3,
            transcode_receive: 4,
            transcode_send: 5,
            playback_start: 6,
        });
        /* eslint-enable camelcase */

        // Did we get an event with the proper channel and quality?
        assert.equal(this.analytics.tracked.length, 2, 'exactly two tracking event should have been sent');
        event = this.analytics.tracked[1];
        assert.equal(event.properties.segment, 'monstercat,low,segment2.ts');
    });

    QUnit.test('sends a tracking event when it receives a segment change', function(assert) {
        assert.equal(this.analytics.tracked.length, 0, 'mock analytics started with events already tracked');
        this.player.emit(TwitchEvents.SEGMENT_CHANGE, this.segmentChangePayload);
        assert.equal(this.analytics.tracked.length, 1, 'exactly one tracking event should have been sent');
    });

    QUnit.test('matches the sampling scheme used in git-aws.internal.justin.tv/video/logster', function(assert) {
        this.latencyTracker._sample = murmurSegmentSampler(0.5);

        // This is a set of golden test outputs. If the sampler is set
        // to 0.5 pct, then logster would give the 'want' result when
        // deciding whether to include the given data in sampling.
        const tests = [
            {
                channel: 'monstercat',
                quality: 'low',
                segment: 'index-0000002729-dY31.ts',
                want: true,
            },
            {
                channel: 'monstercat',
                quality: 'low',
                segment: 'index-0000002730-V8dR.ts',
                want: false,
            },
            {
                channel: 'monstercat',
                quality: 'low',
                segment: 'index-0000002731-LJKI.ts',
                want: true,
            },
            {
                channel: 'monstercat',
                quality: 'low',
                segment: 'index-0000002732-gdV9.ts',
                want: true,
            },
            {
                channel: 'monstercat',
                quality: 'low',
                segment: 'index-0000002733-4QPQ.ts',
                want: false,
            },
            {
                channel: 'monstercat',
                quality: 'mobile',
                segment: 'index-0000002758-gs57.ts',
                want: false,
            },
            {
                channel: 'monstercat',
                quality: 'mobile',
                segment: 'index-0000002759-mGGk.ts',
                want: false,
            },
            {
                channel: 'monstercat',
                quality: 'mobile',
                segment: 'index-0000002760-4966.ts',
                want: true,
            },
        ];

        tests.forEach(testcase => {
            this.store.dispatch(setStream(testcase.channel, true));
            this.store.dispatch(setCurrentQuality(testcase.quality));
            const have = this.latencyTracker._sample(testcase.segment);
            assert.equal(have, testcase.want, `expected sampling to match logster. case: ${testcase}`);
        });
    });

    QUnit.test('tracked data matches expectations', function(assert) {
        // Guarantee that the event will pass sampling.
        this.latencyTracker._sample = function() {
            return true;
        };
        // Trigger the event.
        this.latencyTracker._onSegmentChange(this.segmentChangePayload);

        assert.equal(this.analytics.tracked.length, 1, 'exactly one event should have been sent');
        const event = this.analytics.tracked[0];
        assert.equal(event.name, 'latency_report_playback', 'event name is not correct');

        /* eslint-disable camelcase */
        const expectedProps = {
            segment: segmentKey,
            broadcaster_send: 1,
            ingest_receive: 2,
            ingest_send: 3,
            transcode_receive: 4,
            transcode_send: 5,
            playback_start: 6,
        };
        /* eslint-enable camelcase */
        assert.deepEqual(event.properties, expectedProps, 'event properties are not correct');
    });

    QUnit.test('downcases the channel name', function(assert) {
        setStream('LiRiK', true);
        const have = this.latencyTracker._segmentName(segment);
        assert.equal(have, segmentKey);
    });

    QUnit.test('obeys the sampling function', function(assert) {
        // Set sampler to only allow a specific segment
        this.latencyTracker._sample = function(segment) {
            return segment === `${channel},${quality},pass`;
        };

        assert.equal(this.analytics.tracked.length, 0);

        this.player.emit(TwitchEvents.SEGMENT_CHANGE, { name: 'dontpass' });
        assert.equal(this.analytics.tracked.length, 0);

        this.player.emit(TwitchEvents.SEGMENT_CHANGE, { name: 'pass' });
        assert.equal(this.analytics.tracked.length, 1);
    });

    QUnit.test('murmur sampler works', function(assert) {
        assert.equal(
            murmurSegmentSampler(1.0)('anything'),
            true,
            'should always pass when rate is 1.0'
        );
        assert.equal(
            murmurSegmentSampler(0.0)('anything'),
            false,
            'should always fail when rate is 0.0'
        );
    });

    QUnit.test('does not send events for non-live streams', function(assert) {
        // Start with a VOD stream.
        this.store.dispatch(setStream('whatever', false));

        // Trigger a segment
        this.player.emit(TwitchEvents.SEGMENT_CHANGE, this.segmentChangePayload);

        // We shouldn't have sent any events
        assert.equal(this.analytics.tracked.length, 0);

        // Switch to live and trigger a segment. Now, we should have one tracked event.
        this.store.dispatch(setStream('lirik', true));
        this.player.emit(TwitchEvents.SEGMENT_CHANGE, this.segmentChangePayload);
        assert.equal(this.analytics.tracked.length, 1);

        // Switch back to VOD, it should go back to being untracked
        this.store.dispatch(setStream('whatever', false));
        this.player.emit(TwitchEvents.SEGMENT_CHANGE, this.segmentChangePayload);
        assert.equal(this.analytics.tracked.length, 1);
    });
});

class FakeAnalytics {
    constructor() {
        this.tracked = [];
    }

    trackEvent(name, props) {
        this.tracked.push({
            name: name,
            properties: props,
        });
    }
}

class FakePlayer {
    constructor() {
        this.listeners = {};
    }

    addEventListener(eventName, handler) {
        if (this.listeners[eventName]) {
            this.listeners[eventName].push(handler);
        } else {
            this.listeners[eventName] = [handler];
        }
    }

    emit(eventName, payload) {
        this.listeners[eventName].forEach(function(handler) {
            handler(payload);
        });
    }
}

// Mock the setChannel/setVideo action because the real versions do
// some API calls. We just want to get the channel name recorded and
// set up a live or VOD stream.
function setStream(name, live) {
    let stream;
    if (live) {
        stream = new LiveTwitchContentStream(name, 'oauthtoken', 'usherparams');
    } else {
        stream = new VODTwitchContentStream(name, 'oauthtoken', 'usherparams');
    }

    return function(dispatch) {
        dispatch({
            type: ACTION_SET_STREAM,
            stream: stream,
        });
    };
}
