import sinon from 'sinon';
import { buildAudienceVerification, buildSlicer, MoatAnalytics } from 'analytics/moat.js';
import { setWindow } from 'actions/window';
import { init as initStore } from 'state';
import { buildFakeWindow } from 'tests/fakes/window.fake';

QUnit.module('analytics | moat', function(hooks) {
    hooks.beforeEach(function() {
        this.store = initStore();
        this.window = buildFakeWindow({
            google: {
                ima: {
                    ViewMode: {
                        NORMAL: 'normal',
                    },
                },
            },
        });

        this.store.dispatch(setWindow(this.window));
    });

    QUnit.test('MoatAnalytics passes expected values to initMoatTracking', function(assert) {
        const moatTrackingSpy = sinon.spy();
        const adContainer = {};
        const systemUnderTest = new MoatAnalytics(adContainer, this.store);
        systemUnderTest._moatTracking = moatTrackingSpy;

        const adsManager = {};
        const adRequestContext = {
            channel: 'channel',
            channelId: '1234',
            game: 'game',
        };
        systemUnderTest.trackAd(adsManager, adRequestContext);

        const expectedIds = {
            partnerCode: 'twitchvpaid2756206246',
            viewMode: this.window.google.ima.ViewMode.NORMAL,
            slicer1: 'www.twitch.tv',
            slicer2: buildSlicer(adRequestContext),
            zMoatChannel: adRequestContext.channel,
            zMoatChannelID: adRequestContext.channelId,
            zMoatGame: adRequestContext.game,
        };

        assert.ok(moatTrackingSpy.calledOnce);
        const [actualInstance, actualIds, actualContainer] = moatTrackingSpy.firstCall.args;

        assert.equal(actualInstance, adsManager, 'should have same ads manager instance');
        assert.deepEqual(actualIds, expectedIds);
        assert.equal(actualContainer, adContainer, 'should have same ad container instance');
    });

    QUnit.test('trackAd() swallows exceptions', function(assert) {
        const adContainer = {};
        const systemUnderTest = new MoatAnalytics(adContainer, this.store);
        const initMoatStub = sinon.stub().throws();
        let exceptionNotCalled = true;
        systemUnderTest._moatTracking = initMoatStub;

        const adsManager = {};
        const adRequestContext = {
            channel: 'channel',
            channelId: '1234',
            game: 'game',
        };

        try {
            systemUnderTest.trackAd(adsManager, adRequestContext);
        } catch (e) {
            exceptionNotCalled = false;
        }

        assert.equal(initMoatStub.callCount, 1, 'should call initMoatTracking() exactly once');
        assert.ok(exceptionNotCalled, 'trackAd() should swallow possible exceptions from Moat');
    });

    QUnit.test('buildSlicer no channel, no game', function(assert) {
        const adRequestContext = { };
        assert.equal(buildSlicer(adRequestContext), 'unknown');
    });

    QUnit.test('buildSlicer channel, no game', function(assert) {
        const adRequestContext = { channel: 'channel' };
        assert.equal(buildSlicer(adRequestContext), 'channel');
    });

    QUnit.test('buildSlicer channel and game', function(assert) {
        const adRequestContext = {
            channel: 'channel',
            game: 'game',
        };
        assert.equal(buildSlicer(adRequestContext), 'channel + game');
    });

    QUnit.test('buildSlicer no channel, game', function(assert) {
        const adRequestContext = {
            game: 'game',
        };
        assert.equal(buildSlicer(adRequestContext), 'unknown + game');
    });

    QUnit.test('buildAudienceVerification returns the correct verification object', function(assert) {
        const adRequestContext = {
            channel: 'channel',
            channelId: '1234',
            game: 'game',
        };

        const expectedVerificationObject = {
            zMoatChannel: adRequestContext.channel,
            zMoatChannelID: adRequestContext.channelId,
            zMoatGame: adRequestContext.game,
        };

        assert.deepEqual(buildAudienceVerification(adRequestContext), expectedVerificationObject);
    });
});
