import { SpadeClient } from 'analytics/spade';
import { unitTest } from 'tests/utils/module';
import sinon from 'sinon';

const UNTRUSTED_MINUTE_WATCHED = 'x_untrusted_minute-watched_filtered';
const TRUSTED_HOST = '//trustedhost.com';
const UNTRUSTED_HOST = '//untrustedhost.com';

unitTest('analytics | spade', function(hooks) {
    hooks.beforeEach(function() {
        this.spadeClient = new SpadeClient({
            trustedURI: TRUSTED_HOST,
            untrustedHost: UNTRUSTED_HOST,
        });
    });

    QUnit.test('trackEvent works with a payload that has a character out of range of 8bit ascii', function(assert) {
        // eslint-disable-next-line max-len
        const expectedRequestBody = 'data=eyJldmVudCI6ImV2ZW50TmFtZSIsInByb3BlcnRpZXMiOnsiY3JhenlTdHJpbmciOiLinJMgw6AgbGEgbW9kZSJ9fQ%3D%3D';

        this.api.mock(TRUSTED_HOST, 200, {
            method: 'POST',
        });

        try {
            this.spadeClient.trackEvent('eventName', { crazyString: '✓ à la mode' });
        } catch (e) {
            assert.ok(false);
        }

        assert.equal(this.api.calls().matched.length, 1);
        assert.equal(this.api.lastCall()[1].body, expectedRequestBody);
    });

    QUnit.test('minute-watched event is modified and sent to untrusted host: trackEvent', function(assert) {
        sinon.spy(this.spadeClient, '_sendToUntrustedHost');

        this.api.mock(TRUSTED_HOST, 200, {
            method: 'POST',
        });
        this.api.mock(UNTRUSTED_HOST, 200, {
            method: 'POST',
        });

        try {
            this.spadeClient.trackEvent('minute-watched', {});
        } catch (e) {
            assert.ok(false);
        }

        const sendToUntrustedHostArgs = this.spadeClient._sendToUntrustedHost.firstCall.args[0];
        assert.equal(sendToUntrustedHostArgs.event, UNTRUSTED_MINUTE_WATCHED);

        assert.equal(this.api.calls().matched.length, 2);
        assert.equal(this.api.lastCall()[0], UNTRUSTED_HOST);
    });

    QUnit.test('minute-watched event is modified and sent to untrusted host: trackEvents', function(assert) {
        sinon.spy(this.spadeClient, '_sendToUntrustedHost');
        const mockMWEvent = {
            event: 'minute-watched',
            properties: {},
        };

        this.api.mock(TRUSTED_HOST, 200, {
            method: 'POST',
        });
        this.api.mock(UNTRUSTED_HOST, 200, {
            method: 'POST',
        });

        try {
            this.spadeClient.trackEvents([mockMWEvent]);
        } catch (e) {
            assert.ok(false);
        }

        const sendToUntrustedHostArgs = this.spadeClient._sendToUntrustedHost.firstCall.args[0];
        assert.equal(sendToUntrustedHostArgs.event, UNTRUSTED_MINUTE_WATCHED);

        assert.equal(this.api.calls().matched.length, 2);
        assert.equal(this.api.lastCall()[0], UNTRUSTED_HOST);
    });
});
