import sinon from 'sinon';
import without from 'lodash/without';
import { AnalyticsTracker } from 'analytics/tracker';
import { init as initStore } from 'state';
import { DEFAULT_ENVIRONMENT } from 'state/env';
import * as AnalyticsActions from 'actions/analytics';
import { ACTION_SET_PLAYER_TYPE } from 'actions/env';
import { ACTION_SET_TRACKING_PROPERTIES } from 'actions/tracking';
import * as PlayerType from 'util/player-type';
import { waitFor } from 'tests/utils/waitFor';
import { unitTest } from 'tests/utils/module';
import { SpadeClient } from 'analytics/spade';
import { MixpanelClient } from 'analytics/mixpanel';
import { FirstPartyClient } from 'analytics/first-party';

const FAKE_SET_TRACKING_CLIENTS_ACTION = 'a fake set tracking ';

unitTest('analytics | tracker', function() {
    QUnit.module('sets tracking clients', function(hooks) {
        hooks.beforeEach(function() {
            this.store = initStore();
            sinon.stub(AnalyticsActions, 'setTrackingClients', clients => ({
                type: FAKE_SET_TRACKING_CLIENTS_ACTION,
                clients,
            }));
            sinon.spy(this.store, 'dispatch');

            this.testSetClients = (playerType, expectedClients, assert) => {
                this.store.dispatch({
                    type: ACTION_SET_PLAYER_TYPE,
                    playerType,
                });

                this.store.dispatch.reset();
                this.tracker = new AnalyticsTracker(this.store, {});
                assert.equal(this.store.dispatch.callCount, 1, 'dispatch called once');

                const setClientAction = this.store.dispatch.firstCall.args[0];
                const { type, clients: resultClients } = setClientAction;

                assert.equal(
                    type,
                    FAKE_SET_TRACKING_CLIENTS_ACTION,
                    `dispatched set client action for playerType ${playerType}`
                );
                assert.equal(
                    resultClients.length,
                    expectedClients.length,
                    `dispatched correct number of clients for playerType ${playerType}`);
                expectedClients.forEach(expectedClient => {
                    assert.ok(
                        resultClients.some(c => c instanceof expectedClient),
                        `set ${expectedClient} for playerType: ${playerType}`
                    );
                });
            };
        });

        hooks.afterEach(function() {
            AnalyticsActions.setTrackingClients.restore();
        });

        // eslint-disable-next-line max-len
        QUnit.test('to include Mixpanel, Spade, and First Party Analytics clients for Amazon VSE player type', function(assert) {
            const expectedClients = [MixpanelClient, SpadeClient, FirstPartyClient];
            this.testSetClients(PlayerType.PLAYER_AMAZON_VSE, expectedClients, assert);
        });

        // eslint-disable-next-line max-len
        QUnit.test('to include Mixpanel, Spade, and First Party Analytics clients for Amazon Live player type', function(assert) {
            const expectedClients = [MixpanelClient, SpadeClient, FirstPartyClient];
            this.testSetClients(PlayerType.PLAYER_AMAZON_LIVE, expectedClients, assert);
        });

        QUnit.test('to include Mixpanel, Spade clients for all other player types', function(assert) {
            const expectedClients = [MixpanelClient, SpadeClient];
            const {
                ALL_PLAYER_TYPES,
                PLAYER_AMAZON_VSE,
                PLAYER_AMAZON_LIVE,
            } = PlayerType;
            const otherPlayerTypes = without(ALL_PLAYER_TYPES, PLAYER_AMAZON_VSE, PLAYER_AMAZON_LIVE);

            otherPlayerTypes.forEach(playerType => {
                this.testSetClients(playerType, expectedClients, assert);
            });
        });
    });

    QUnit.module('global properties', function(hooks) {
        hooks.beforeEach(function() {
            this.store = initStore();
            this.getPlayerTypeStub = sinon.stub(PlayerType, 'getPlayerType').returns(PlayerType.PLAYER_SITE);
        });

        hooks.afterEach(function() {
            this.store = initStore();
            this.getPlayerTypeStub.restore();
        });

        QUnit.test('browser info is in each event', function(assert) {
            const bowser = require('bowser');

            this.tracker = new AnalyticsTracker(this.store, {});
            this.trackingClient = {
                trackEvents: sinon.spy(),
            };
            this.store.dispatch(AnalyticsActions.setTrackingClients([this.trackingClient]));

            this.tracker.trackEvents([
                {
                    event: 'dummy',
                    properties: {},
                },
            ]);

            return waitFor(() => this.trackingClient.trackEvents.called).then(() => {
                assert.equal(this.trackingClient.trackEvents.callCount, 1, 'should call trackEvents once');
                const browserFamily = bowser.name;
                const browserVer = bowser.version;
                const osName = bowser.osname;
                const osVer = bowser.osversion;
                const { properties } = this.trackingClient.trackEvents.firstCall.args[0][0];
                assert.equal(properties.browser_family, browserFamily, 'includes browser family');
                assert.equal(properties.browser_version, browserVer, 'includes browser version');
                assert.equal(properties.os_name, osName, 'includes os name');
                assert.equal(properties.os_version, osVer, 'includes os version');
            });
        });

        QUnit.test('sets client_app to null when player type is not embed', function(assert) {
            this.getPlayerTypeStub.returns(PlayerType.PLAYER_EMBED);

            this.store.dispatch({
                type: ACTION_SET_PLAYER_TYPE,
                playerType: PlayerType.PLAYER_EMBED,
            });
            this.tracker = new AnalyticsTracker(this.store, {});
            this.trackingClient = {
                trackEvents: sinon.spy(),
            };
            this.store.dispatch(AnalyticsActions.setTrackingClients([this.trackingClient]));

            this.tracker.trackEvents([
                {
                    event: 'dummy',
                    properties: {},
                },
            ]);
            return waitFor(() => this.trackingClient.trackEvents.called).then(() => {
                assert.equal(this.trackingClient.trackEvents.callCount, 1, 'should call trackEvents once');
                assert.ok(this.trackingClient.trackEvents.firstCall.calledWith([{
                    event: 'dummy',
                    properties: sinon.match.has('client_app', null),
                }]), 'should set the client_app to null');
            });
        });
        QUnit.test(`sets client_app to 'ember' when player type is ${PlayerType.PLAYER_SITE}`, function(assert) {
            this.store.dispatch({
                type: ACTION_SET_PLAYER_TYPE,
                playerType: PlayerType.PLAYER_SITE,
            });

            this.tracker = new AnalyticsTracker(this.store, {});
            this.trackingClient = {
                trackEvents: sinon.spy(),
            };
            this.store.dispatch(AnalyticsActions.setTrackingClients([this.trackingClient]));

            this.tracker.trackEvents([
                {
                    event: 'dummy',
                    properties: {},
                },
            ]);
            return waitFor(() => this.trackingClient.trackEvents.called).then(() => {
                assert.equal(this.trackingClient.trackEvents.callCount, 1, 'should call trackEvents once');
                assert.ok(this.trackingClient.trackEvents.firstCall.calledWith([{
                    event: 'dummy',
                    properties: sinon.match.has('client_app', 'ember'),
                }]), 'should set the client_app to ember');
            });
        });
    });

    QUnit.module('trackEvents', function(hooks) {
        hooks.beforeEach(function() {
            this.store = initStore();
            this.tracker = new AnalyticsTracker(this.store, {});
            this.trackingClient = {
                trackEvents: sinon.spy(),
            };
            this.store.dispatch(AnalyticsActions.setTrackingClients([this.trackingClient]));
        });

        QUnit.test('gives events without times a timestamp', function(assert) {
            this.tracker.trackEvents([
                {
                    event: 'dummy',
                    properties: {},
                },
            ]);  // start with properties.time = undefined
            return waitFor(() => this.trackingClient.trackEvents.called).then(() => {
                assert.ok(this.trackingClient.trackEvents.calledOnce);
                assert.ok(this.trackingClient.trackEvents.firstCall.calledWith([{
                    event: 'dummy',
                    properties: sinon.match.has('time'),
                }]));
            });
        });

        QUnit.test(`should set the platform=${PlayerType.PLAYER_CURSE} when player type is 'curse'`, function(assert) {
            this.store.dispatch({
                type: ACTION_SET_PLAYER_TYPE,
                playerType: PlayerType.PLAYER_CURSE,
            });
            this.tracker.trackEvents([
                {
                    event: 'dummy',
                    properties: {},
                },
            ]);
            return waitFor(() => this.trackingClient.trackEvents.called).then(() => {
                assert.equal(this.trackingClient.trackEvents.callCount, 1, 'should call trackEvents once');
                assert.ok(this.trackingClient.trackEvents.firstCall.calledWith([{
                    event: 'dummy',
                    properties: sinon.match.has('platform', PlayerType.PLAYER_CURSE),
                }]), 'should set the platform type as curse');
            });
        });

        QUnit.test(`should set the platform=${DEFAULT_ENVIRONMENT.platform} when player type is invalid`,
            function(assert) {
                this.store.dispatch({
                    type: 'hello world',
                    playerType: PlayerType.PLAYER_CURSE,
                });
                this.tracker.trackEvents([
                    {
                        event: 'dummy',
                        properties: {},
                    },
                ]);
                return waitFor(() => this.trackingClient.trackEvents.called).then(() => {
                    assert.equal(this.trackingClient.trackEvents.callCount, 1, 'should call trackEvents once');
                    assert.ok(this.trackingClient.trackEvents.firstCall.calledWith([{
                        event: 'dummy',
                        properties: sinon.match.has('platform', DEFAULT_ENVIRONMENT.platform),
                    }]), 'should show the default platorm');
                });
            });

        QUnit.test('does not change the timestamp of events with timestamps', function(assert) {
            const fakeTimestamp = parseInt(QUnit.config.current.testId, 36);
            this.tracker.trackEvents([
                {
                    event: 'dummy',
                    properties: {
                        time: fakeTimestamp,
                    },
                },
            ]);
            return waitFor(() => this.trackingClient.trackEvents.called).then(() => {
                assert.ok(this.trackingClient.trackEvents.calledOnce);
                assert.ok(this.trackingClient.trackEvents.firstCall.calledWith([{
                    event: 'dummy',
                    properties: sinon.match.has('time', fakeTimestamp),
                }]));
            });
        });

        QUnit.test('brings in properties that were set with setProperties', function(assert) {
            this.tracker.setProperties({
                property1: 'value1',
                property2: 'value2',
            });

            this.tracker.trackEvents([
                {
                    event: 'dummy',
                    properties: {
                        time: 12,
                    },
                },
            ]);
            return waitFor(() => this.trackingClient.trackEvents.called).then(() => {
                assert.ok(this.trackingClient.trackEvents.calledOnce);
                assert.ok(this.trackingClient.trackEvents.firstCall.calledWith([sinon.match({
                    event: 'dummy',
                    properties: {
                        time: 12,
                        property1: 'value1',
                        property2: 'value2',
                    },
                })]));
            });
        });

        QUnit.test('trackEvent is called with properties set in store', function(assert) {
            const properties = {
                // eslint-disable-next-line camelcase
                host_channel: 'awesome',
            };
            this.store.dispatch({
                type: ACTION_SET_TRACKING_PROPERTIES,
                properties,
            });

            this.tracker.trackEvents([
                {
                    event: 'dummy',
                },
            ]);

            return waitFor(() => this.trackingClient.trackEvents.called).then(() => {
                assert.ok(this.trackingClient.trackEvents.firstCall.calledWith([sinon.match({
                    event: 'dummy',
                    properties,
                })]));
            });
        });
    });
});
