import sinon from 'sinon';
import { AutoSuggestNotificationManager, WEEK, TEN_MINUTES,
         AUTO_SUGGEST_NOTIFICATION_TIMESTAMP, MAX_BUFFER_EMPTY_COUNT,
         AUTO_SUGGEST_NOTIFICATION_CLOSED_COUNT } from 'auto-suggest-notification-manager';
import { ACTION_ADD_NOTIFICATION } from 'actions/notifications';
import { init as initStore } from 'state';
import { unitTest } from '../../utils/module';
import { AutoSuggestNotification } from 'ui/components/notifications/auto-suggest-notification-component';
import { ACTION_SET_WINDOW } from 'actions/window';
import { localStore } from 'tests/fakes/local-store.fake';

const playback = {
    bufferEmpties: 1,
};

const ONE_MIN = 60000;
const TWO_MINS = 120000;

unitTest('auto-suggest-notification-manager', function(hooks) {
    hooks.beforeEach(function() {
        this.store = initStore();
        this.autoSuggestNotificationManager = new AutoSuggestNotificationManager(this.store);

        this.setWindowTime = time => {
            this.store.dispatch({
                type: ACTION_SET_WINDOW,
                window: {
                    navigator: {},
                    document: {},
                    Date: () => ({
                        getTime: () => time,
                    }),
                },
            });
        };
    });

    // eslint-disable-next-line max-len
    QUnit.test('dispatches addNotification if it has been over a week since the last seen notification', function(assert) {
        sinon.spy(this.store, 'dispatch');
        sinon.stub(this.autoSuggestNotificationManager, '_thirdBufferEmptyInTenMinutes').returns(true);
        sinon.stub(this.autoSuggestNotificationManager, '_hasNotSeenAutoSuggestNotificationThisWeek').returns(true);

        this.autoSuggestNotificationManager._trackBufferEmpty({ playback });

        assert.ok(this.store.dispatch.called);
        assert.ok(this.store.dispatch.calledOnce);
        assert.ok(this.store.dispatch.calledWith({
            type: ACTION_ADD_NOTIFICATION,
            notification: AutoSuggestNotification,
        }));
    });

    // eslint-disable-next-line max-len
    QUnit.test('does not dispatch addNotification if it has not been over a week since the last seen notification', function(assert) {
        sinon.spy(this.store, 'dispatch');

        sinon.stub(this.autoSuggestNotificationManager, '_thirdBufferEmptyInTenMinutes').returns(true);
        sinon.stub(this.autoSuggestNotificationManager, '_hasNotSeenAutoSuggestNotificationThisWeek').returns(false);

        this.autoSuggestNotificationManager._trackBufferEmpty({ playback });

        assert.notOk(this.store.dispatch.called);
        assert.ok(this.store.dispatch.neverCalledWith({
            type: ACTION_ADD_NOTIFICATION,
            notification: AutoSuggestNotification,
        }));
    });

    QUnit.test('thirdBufferEmptyInTenMinutes returns true if max buffer-empty found within ten mins', function(assert) {
        this.autoSuggestNotificationManager._previousBufferTimestamp = ONE_MIN;
        this.setWindowTime(TWO_MINS);
        this.autoSuggestNotificationManager._currentBufferEmptyCount = MAX_BUFFER_EMPTY_COUNT;

        const result = this.autoSuggestNotificationManager._thirdBufferEmptyInTenMinutes();
        assert.ok(result);
        assert.equal(this.autoSuggestNotificationManager._previousBufferTimestamp, TWO_MINS);
        assert.equal(this.autoSuggestNotificationManager._currentBufferEmptyCount, 0);
    });

    QUnit.test('thirdBufferEmptyInTenMinutes returns false if its been over ten mins', function(assert) {
        this.autoSuggestNotificationManager._previousBufferTimestamp = ONE_MIN;
        this.setWindowTime(TEN_MINUTES + TWO_MINS);
        this.autoSuggestNotificationManager._currentBufferEmptyCount = 2;

        const result = this.autoSuggestNotificationManager._thirdBufferEmptyInTenMinutes();
        assert.notOk(result);
        assert.equal(this.autoSuggestNotificationManager._previousBufferTimestamp, TEN_MINUTES + TWO_MINS);
        assert.equal(this.autoSuggestNotificationManager._currentBufferEmptyCount, 0);
    });

    QUnit.test('_hasNotSeenAutoSuggestNotificationThisWeek returns true if its been a week', function(assert) {
        this.setWindowTime(WEEK + WEEK + TEN_MINUTES);
        localStore.set(AUTO_SUGGEST_NOTIFICATION_TIMESTAMP, WEEK);

        const result = this.autoSuggestNotificationManager._hasNotSeenAutoSuggestNotificationThisWeek();
        assert.ok(result);
    });

    QUnit.test('_hasNotSeenAutoSuggestNotificationThisWeek returns false if it has not been a week', function(assert) {
        this.setWindowTime(TEN_MINUTES + TWO_MINS);

        localStore.set(AUTO_SUGGEST_NOTIFICATION_TIMESTAMP, TEN_MINUTES);

        const result = this.autoSuggestNotificationManager._hasNotSeenAutoSuggestNotificationThisWeek();
        assert.notOk(result);
    });

    QUnit.test('if localStore AUTO_SUGGEST_NOTIFICATION_CLOSED_COUNT is greater than two', function(assert) {
        sinon.spy(this.store, 'dispatch');

        localStore.set(AUTO_SUGGEST_NOTIFICATION_CLOSED_COUNT, 2);
        this.autoSuggestNotificationManager._showAutoSuggestNotification();

        assert.notOk(this.store.dispatch.called);
        assert.ok(this.store.dispatch.neverCalledWith({
            type: ACTION_ADD_NOTIFICATION,
            notification: AutoSuggestNotification,
        }));
    });

    QUnit.test('if localStore AUTO_SUGGEST_NOTIFICATION_CLOSED_COUNT is less than two', function(assert) {
        sinon.spy(this.store, 'dispatch');

        localStore.set(AUTO_SUGGEST_NOTIFICATION_CLOSED_COUNT, 1);
        this.autoSuggestNotificationManager._showAutoSuggestNotification();

        assert.ok(this.store.dispatch.called);
        assert.ok(this.store.dispatch.calledWith({
            type: ACTION_ADD_NOTIFICATION,
            notification: AutoSuggestNotification,
        }));
    });
});
