import { unitTest } from 'tests/utils/module';
import sinon from 'sinon';
import { init as initStore } from 'state';
import * as MediaEvents from 'backend/events/media-event';
import { BackendClip, BACKEND_CLIP } from 'backend/clip';

function createVideoBackend(store, options = {}) {
    return new BackendClip(options, store);
}

unitTest('backend | clip', function(hooks) {
    hooks.beforeEach(function() {
        this.store = initStore();
    });

    QUnit.module('on construction', function() {
        QUnit.test('should create a video tag', function(assert) {
            const backend = createVideoBackend(this.store);
            assert.equal(backend._video.nodeName, 'VIDEO');
        });

        QUnit.test('should have preload set to auto, and type to video/mp4', function(assert) {
            const backend = createVideoBackend(this.store);
            assert.equal(backend._video.preload, 'auto');
            assert.equal(backend._video.type, 'video/mp4');
        });

        QUnit.test('should add autoplay prop if true', function(assert) {
            const backend = createVideoBackend(this.store, { autoplay: true });
            assert.equal(backend._video.autoplay, true);
        });

        QUnit.test('should return correct backend name', function(assert) {
            const backend = createVideoBackend(this.store);
            assert.equal(backend.getBackend(), BACKEND_CLIP);
        });
    });

    QUnit.module('playback methods', function() {
        QUnit.test('setSrc should set a src and call .load', function(assert) {
            const backend = createVideoBackend(this.store, { autoplay: true });
            sinon.spy(backend._video, 'load');
            backend.setSrc(QUnit.config.current.testId);

            assert.equal(backend._video.load.callCount, 1);
            assert.equal(backend._src, QUnit.config.current.testId);
        });

        QUnit.test('.load NOOP if no src is set', function(assert) {
            const backend = createVideoBackend(this.store, { autoplay: true });
            sinon.spy(backend._video, 'load');

            backend.load();
            assert.equal(backend._video.load.callCount, 0);
        });

        QUnit.test('play should call play on videotag', function(assert) {
            const backend = createVideoBackend(this.store, { autoplay: true });
            sinon.spy(backend._video, 'play');
            backend.play();
            assert.equal(backend._video.play.callCount, 1);
        });

        QUnit.test('pause should call pause on videotag', function(assert) {
            const backend = createVideoBackend(this.store, { autoplay: true });
            sinon.spy(backend._video, 'pause');
            backend.pause();
            assert.equal(backend._video.pause.callCount, 1);
        });
    });

    QUnit.module('method passthroughs', function() {
        function testGetters(videoProperty, backendMethod) {
            QUnit.test(`calling ${backendMethod} should return this._video.${videoProperty}`, function(assert) {
                const backend = createVideoBackend(this.store);
                const actualValue = backend._video[videoProperty];
                assert.strictEqual(
                    backend[backendMethod](),
                    actualValue,
                    `${backendMethod} returns this._video.${videoProperty}`
                );
            });
        }

        testGetters('paused', 'getPaused');
        testGetters('seeking', 'getSeeking');
        testGetters('volume', 'getVolume');
        testGetters('currentTime', 'getCurrentTime');
        testGetters('playbackRate', 'getPlaybackRate');
        testGetters('ended', 'getEnded');
        testGetters('muted', 'getMuted');

        QUnit.test('calling getPlayed should return this._video.played', function(assert) {
            const backend = createVideoBackend(this.store);
            const actualValue = backend._video.played;
            assert.deepEqual(
                backend.getPlayed(),
                actualValue
            );
        });

        QUnit.test('calling getDuration should return this._video.duration', function(assert) {
            const backend = createVideoBackend(this.store);
            const actualValue = backend._video.duration;
            assert.deepEqual(
                backend.getDuration(),
                actualValue
            );
        });

        function testSetters(videoProperty, value, backendMethod) {
            // eslint-disable-next-line max-len
            QUnit.test(`calling ${backendMethod} with ${value} should set this._video.${videoProperty}`, function(assert) {
                const backend = createVideoBackend(this.store);
                backend[backendMethod](value);

                assert.equal(backend._video[videoProperty], value, `${videoProperty} is set to ${value}`);
            });
        }

        testSetters('volume', 0.5, 'setVolume');
        testSetters('muted', true, 'setMuted');
        testSetters('src', 'https://random.url.mp4/', 'setSrc');
        testSetters('currentTime', 12314, 'setCurrentTime');
        testSetters('playbackRate', 2, 'setPlaybackRate');
        testSetters('loop', true, 'setLoop');
    });

    QUnit.module('events', function() {
        function testEventPropagation(eventName) {
            QUnit.test(`should propagate ${eventName} event`, function(assert) {
                const backend = createVideoBackend(this.store);
                const event = new Event(eventName);
                const spy = sinon.spy();

                backend.addEventListener(eventName, spy);
                backend._video.dispatchEvent(event);

                assert.equal(spy.callCount, 1, `${eventName} propagates up from video tag`);
            });
        }

        testEventPropagation(MediaEvents.CAN_PLAY_THROUGH);
        testEventPropagation(MediaEvents.CAN_PLAY);
        testEventPropagation(MediaEvents.PLAYING);
        testEventPropagation(MediaEvents.ENDED);
        testEventPropagation(MediaEvents.LOADED_METADATA);
        testEventPropagation(MediaEvents.PAUSE);
        testEventPropagation(MediaEvents.PLAY);
        testEventPropagation(MediaEvents.DURATION_CHANGE);
        testEventPropagation(MediaEvents.SEEKING);
        testEventPropagation(MediaEvents.SEEKED);
        testEventPropagation(MediaEvents.RATE_CHANGE);
        testEventPropagation(MediaEvents.WAITING);
        testEventPropagation(MediaEvents.ERROR);
        testEventPropagation(MediaEvents.LOADSTART);
    });
});
