import * as ReadyState from 'backend/state/ready-state';
import * as TwitchEvents from 'backend/events/twitch-event';
import * as MediaEvents from 'backend/events/media-event';
import * as PlayerType from 'util/player-type';
import FakeMediaPlayer from 'tests/fakes/mediaplayer.fake.js';
import sinon from 'sinon';
import { stringify } from 'query-string';
import { MEDIAPLAYER_VERSION_CONTROL, FAST_BREAD } from 'experiments';
import { PREROLL_TIMEOUT } from 'ads/surestream/consts';
import { ACTION_SET_EXPERIMENTS } from 'actions/experiments';
import { ACTION_SET_STREAM } from 'actions/stream';
import * as MediaPlayer from 'backend/mediaplayer';
import { LiveTwitchContentStream, CONTENT_MODE_LIVE } from 'stream/twitch-live';
import { VODTwitchContentStream, CONTENT_MODE_VOD } from 'stream/twitch-vod';
import { VIDEO_PLAY_MASTER_MANIFEST } from 'analytics/analytics';
import { PLAYBACK_ERROR } from 'analytics/spade-events';
import { init as initStore } from 'state';
import { unitTest } from 'tests/utils/module';
import { usherHost as USHER_HOST } from 'settings';
import { waitFor } from 'tests/utils/waitFor';
import { resetPlaySession } from 'actions/analytics';
import { setWindow } from 'actions/window';
import { setAutoplay } from 'actions/playback';
import { setManifestInfo } from 'actions/manifest-info';
import round from 'lodash/round';
import forEach from 'lodash/forEach';
import size from 'lodash/size';
import { setAnalyticsTracker, ACTION_TRACK_EVENT } from 'actions/analytics-tracker';
import { buildFakeWindow } from 'tests/fakes/window.fake';
import PlayerCoreLoader from 'tests/fakes/player-core-loader.fake';
import * as ErrorActions from 'actions/error';
import { setOnline } from 'actions/online';
import Errors from 'errors';

function setStream(store, stream) {
    store.dispatch({
        type: ACTION_SET_STREAM,
        stream,
    });
}

const FAKE_SET_ERROR_ACTION = 'fake set error action';

// eslint-disable-next-line max-statements
unitTest('backend | mediaplayer', function(hooks) {
    hooks.beforeEach(function() {
        this.options = {
            autoplay: true,
            muted: false,
        };

        this.store = initStore();
        this.createExperiment = options => {
            return {
                get: uuid => {
                    switch (uuid) {
                    case MEDIAPLAYER_VERSION_CONTROL:
                        return Promise.resolve(options.mediaplayer);
                    case FAST_BREAD:
                        return Promise.resolve('control');
                    }
                },
            };
        };

        this.store.dispatch({
            type: ACTION_SET_EXPERIMENTS,
            experiments: this.createExperiment({
                mediaplayer: QUnit.config.testId,
                vpQOS: 'no',

            }),
        });

        this.backend = new MediaPlayer.BackendMediaPlayer(this.options, this.store);

        this.oauthToken = `oauth_${QUnit.config.current.testId}`;
        this.usherParams = {
            godlike: true,
        };
        this.accessTokenParams = {
            // eslint-disable-next-line camelcase
            need_https: false,
            adblock: true,
            platform: 'web-test',
            playerType: PlayerType.PLAYER_SITE,
        };
        this.experimentSettings = {
            transcode: Promise.resolve('regular'),
        };
        this.api.setLoggedIn(true);
        this.api.expectUserInfo();

        sinon.spy(PlayerCoreLoader, 'loadMediaPlayer');
        sinon.stub(ErrorActions, 'setError', code => ({
            type: FAKE_SET_ERROR_ACTION,
            code,
        }));
    });

    hooks.afterEach(function() {
        this.backend.destroy();
        // Restore because PlayerCoreLoader is a singleton object.
        PlayerCoreLoader.loadMediaPlayer.restore();
        ErrorActions.setError.restore();
    });

    QUnit.test('can init MediaPlayer', function(assert) {
        assert.expect(0);
        this.backend.loadMediaPlayer();
        return waitFor(() => this.backend._mediaPlayer);
    });

    QUnit.test('on init sets muted attribute on video element if options.muted is true', function(assert) {
        const backend = new MediaPlayer.BackendMediaPlayer({ muted: true }, this.store);
        return waitFor(() => backend._mediaPlayer).
            then(() => {
                const videoElement = backend._mediaPlayer.getHTMLVideoElement();
                assert.ok(videoElement.hasAttribute('muted'), 'video element has `muted` attribute');
            });
    });

    QUnit.test('on init sets playinline attributes on video element if options.playsinline is true', function(assert) {
        const backend = new MediaPlayer.BackendMediaPlayer({ playsinline: true }, this.store);
        return waitFor(() => backend._mediaPlayer).
            then(() => {
                const videoElement = backend._mediaPlayer.getHTMLVideoElement();
                assert.ok(videoElement.hasAttribute('playsinline'), 'video element has `playsinline` attr');
                assert.ok(videoElement.hasAttribute('webkit-playsinline'), 'video tag has `webkit-playsinline` attr');
            });
    });

    QUnit.test('passes the right experiment value to PlayerCoreLoader', function(assert) {
        return this.backend.loadMediaPlayer().then(() => {
            assert.ok(PlayerCoreLoader.loadMediaPlayer.calledWith({
                value: QUnit.config.testId,
                logLevel: 'error',
                latencyValue: 'control',
            }), 'Player Core Loader called with appropriate experiment value');
        });
    });

    QUnit.test(`Mediaplayer should emit ${TwitchEvents.PLAYER_INIT} once`, function(assert) {
        const backend = new MediaPlayer.BackendMediaPlayer({}, this.store);
        sinon.spy(backend._eventEmitter, 'emit');
        return waitFor(() => backend._eventEmitter.emit.calledOnce).
        then(() => {
            assert.ok(backend._eventEmitter.emit.calledOnce);
            assert.ok(backend._eventEmitter.emit.calledWith(TwitchEvents.PLAYER_INIT));
        });
    });

    QUnit.test('`getVersion` returns empty string', function(assert) {
        const backend = new MediaPlayer.BackendMediaPlayer({}, this.store);
        assert.equal(backend.getVersion(), '');
    });

    QUnit.module('after initialize', function(hooks) {
        hooks.beforeEach(function() {
            this.makeDelayedStream = (contentType, contentId, streamUrl, ms) => {
                const delay = ms => new Promise(resolve => setTimeout(resolve, ms));
                return {
                    contentType,
                    channel: contentType === CONTENT_MODE_LIVE ? contentId : null,
                    videoId: contentType === CONTENT_MODE_VOD ? contentId : null,
                    streamUrl: delay(ms).
                        then(() => streamUrl),
                };
            };
        });
        QUnit.test('is listening to expected events', function(assert) {
            this.backend.loadMediaPlayer();
            return waitFor(() => this.backend._mediaPlayer).
                then(() => {
                    const spy = this.backend._mediaPlayer.addEventListener;
                    // eslint-disable-next-line max-len
                    assert.equal(spy.args.length, size(FakeMediaPlayer.PlayerEvent) + size(FakeMediaPlayer.MetadataEvent) + size(FakeMediaPlayer.PlayerState));
                    forEach(FakeMediaPlayer.PlayerEvent, eventType => {
                        assert.ok(spy.withArgs(eventType).calledOnce, `should call ${eventType} once`);
                    });
                });
        });

        QUnit.test('If src is set make sure mediaplayer.load is called', function(assert) {
            this.backend.setSrc('notarealsrc.m3u8');
            this.backend.loadMediaPlayer();
            return waitFor(() => this.backend._mediaPlayer).
                then(() => {
                    sinon.spy(this.backend._mediaPlayer, 'load');
                    this.backend.load();
                    assert.ok(this.backend._mediaPlayer.load.calledOnce);
                });
        });

        QUnit.test('can get backend ID', function(assert) {
            assert.equal(this.backend.getBackend(), MediaPlayer.BACKEND_MEDIA_PLAYER);
        });

        QUnit.test('can get ready state', function(assert) {
            assert.equal(this.backend.getReadyState(), ReadyState.HAVE_NOTHING);
        });

        QUnit.test('can set source', function(assert) {
            const sourceTest = '1234567890';
            this.backend.setSrc(sourceTest);
            return waitFor(() => this.backend._mediaPlayer).
                then(() => {
                    assert.equal(this.backend.getSrc(), sourceTest);
                });
        });

        QUnit.test('setChannel only calls setSrc with requested stream', function(assert) {
            sinon.spy(this.backend, 'setSrc');

            const FIRST_CHANNEL = 'firstchannel';
            const FIRST_STREAM_URL = `streamUrl_${FIRST_CHANNEL}`;
            const SECOND_CHANNEL = 'secondchannel';
            const SECOND_STREAM_URL = `streamUrl_${SECOND_CHANNEL}`;

            const firstStream = this.makeDelayedStream(CONTENT_MODE_LIVE, FIRST_CHANNEL, FIRST_STREAM_URL, 30);
            const secondStream = this.makeDelayedStream(CONTENT_MODE_LIVE, SECOND_CHANNEL, SECOND_STREAM_URL, 15);

            setStream(this.store, firstStream);
            this.backend.setChannel(FIRST_CHANNEL, firstStream);
            setStream(this.store, secondStream);
            this.backend.setChannel(SECOND_CHANNEL, secondStream);

            return waitFor(() => this.backend.setSrc.called).
                then(() => {
                    const [result] = this.backend.setSrc.args[0];
                    assert.equal(result, SECOND_STREAM_URL);
                });
        });

        QUnit.test('setVideo only calls setSrc with requested stream', function(assert) {
            sinon.spy(this.backend, 'setSrc');

            const FIRST_VOD_ID = 'firstvod';
            const FIRST_STREAM_URL = `streamUrl_${FIRST_VOD_ID}`;
            const SECOND_VOD_ID = 'secondvod';
            const SECOND_STREAM_URL = `streamUrl_${SECOND_VOD_ID}`;

            const firstStream = this.makeDelayedStream(CONTENT_MODE_VOD, FIRST_VOD_ID, FIRST_STREAM_URL, 30);
            const secondStream = this.makeDelayedStream(CONTENT_MODE_VOD, SECOND_VOD_ID, SECOND_STREAM_URL, 15);

            setStream(this.store, firstStream);
            this.backend.setVideo(FIRST_VOD_ID, firstStream);
            setStream(this.store, secondStream);
            this.backend.setVideo(SECOND_VOD_ID, secondStream);

            return waitFor(() => this.backend.setSrc.called).
                then(() => {
                    const [result] = this.backend.setSrc.args[0];
                    assert.equal(result, SECOND_STREAM_URL);
                });
        });

        QUnit.test('can set channel', function(assert) {
            const channel = 'monstercat';
            const oauthTokenPromise = Promise.resolve(this.oauthToken);
            // eslint-disable-next-line max-len
            const accessToken = this.api.expectChannelAccessToken(channel.toLowerCase(), {
                canAccess: true,
                oauthToken: this.oauthToken,
                accessTokenParams: this.accessTokenParams,
            });
            const stream = new LiveTwitchContentStream(
                channel,
                oauthTokenPromise,
                this.usherParams,
                this.accessTokenParams,
                this.experimentSettings
            );
            setStream(this.store, stream);

            sinon.spy(this.backend, 'setSrc');
            this.backend.loadMediaPlayer();
            return waitFor(() => this.backend._mediaPlayer).
                then(() => {
                    this.backend.setChannel(channel, stream);
                    return oauthTokenPromise;
                }).then(() => {
                    return waitFor(() => this.backend.setSrc.called);
                }).then(() => {
                    const params = {
                        token: accessToken.token,
                        sig: accessToken.sig,
                        // eslint-disable-next-line camelcase
                        allow_source: true,
                        godlike: true,
                        /* eslint-disable camelcase */
                        fast_bread: false,
                        baking_bread: false,
                        baking_brownies: false,
                        baking_brownies_timeout: PREROLL_TIMEOUT,
                        /* eslint-disable camelcase */
                    };

                    // eslint-disable-next-line max-len
                    const usherUrl = `${USHER_HOST}/api/channel/hls/${channel}.m3u8?${stringify(params)}`;
                    assert.equal(this.backend.setSrc.firstCall.args[0], usherUrl);
                });
        });

        QUnit.test('can set video', function(assert) {
            const vodId = 'v1234567890';
            const oauthTokenPromise = Promise.resolve(this.oauthToken);
            const stream = new VODTwitchContentStream(
                vodId,
                oauthTokenPromise,
                this.usherParams,
                this.accessTokenParams,
                this.experimentSettings
            );
            setStream(this.store, stream);
            // eslint-disable-next-line max-len
            const accessToken = this.api.expectVodAccessToken(vodId.toLowerCase(), {
                canAccess: true,
                oauthToken: this.oauthToken,
                accessTokenParams: this.accessTokenParams,
            });

            sinon.spy(this.backend, 'setSrc');
            this.backend.loadMediaPlayer();
            return waitFor(() => this.backend._mediaPlayer).
                then(() => {
                    this.backend.setVideo(vodId, stream);
                    return oauthTokenPromise;
                }).then(() => {
                    return waitFor(() => this.backend.setSrc.called);
                }).then(() => {
                    const params = {
                        nauth: accessToken.token,
                        nauthsig: accessToken.sig,
                        // eslint-disable-next-line camelcase
                        allow_source: true,
                        godlike: true,
                        /* eslint-disable camelcase */
                        fast_bread: false,
                        baking_bread: false,
                        baking_brownies: false,
                        baking_brownies_timeout: PREROLL_TIMEOUT,
                        /* eslint-disable camelcase */
                    };
                    // eslint-disable-next-line max-len
                    const usherUrl = `${USHER_HOST}/vod/${vodId.substring(1)}.m3u8?${stringify(params)}`;
                    assert.equal(this.backend.setSrc.firstCall.args[0], usherUrl);
                });
        });

        QUnit.module('after parsing the master manifest', function(hooks) {
            hooks.beforeEach(function() {
                this.backend.loadMediaPlayer();
                /* eslint-disable camelcase */
                this.manifestInfo = {
                    ABS: 'false',
                    CLUSTER: 'sfo01',
                    'MANIFEST-CLUSTER': 'sfo01',
                    'MANIFEST-NODE': 'video-edge.sfo01',
                    'MANIFEST-NODE-TYPE': 'legacy',
                    NODE: 'video-edge.sfo01',
                    'SERVING-ID': 'abcd',
                    'STREAM-TIME': 5,
                    'USER-IP': '127.0.0.1',
                    FUTURE: true,
                };

                this.normalizedManifestInfo = {
                    abs: 'false',
                    cluster: 'sfo01',
                    manifest_cluster: 'sfo01',
                    manifest_node: 'video-edge.sfo01',
                    manifest_node_type: 'legacy',
                    node: 'video-edge.sfo01',
                    serving_id: 'abcd',
                    stream_time: 5,
                    user_ip: '127.0.0.1',
                    future: true,
                };
                /* eslint-enable camelcase */

                this.fakeTracker = {
                    trackEvent: sinon.spy(),
                };
                this.store.dispatch(setAnalyticsTracker(this.fakeTracker));

                // Reset playSessionStartTime for calculating
                sinon.stub(Date, 'now').onFirstCall().returns(0);
                this.store.dispatch(resetPlaySession());

                this.dateNow = parseInt(QUnit.config.current.testId, 36);
                Date.now.returns(this.dateNow);
            });

            hooks.afterEach(function() {
                Date.now.restore();
            });

            QUnit.test('send a `video_play_master_manifest` tracking event', function(assert) {
                return waitFor(() => this.backend._mediaPlayer).then(() => {
                    this.backend._mediaPlayer.emit(FakeMediaPlayer.PlayerState.READY);
                    assert.ok(this.fakeTracker.trackEvent.calledWith(VIDEO_PLAY_MASTER_MANIFEST, {
                        // eslint-disable-next-line camelcase
                        time_since_load_start: this.dateNow,
                    }));
                });
            });

            QUnit.test('emit a manifest info event with normalized manifest', function(assert) {
                return waitFor(() => this.backend._mediaPlayer).then(() => {
                    this.backend._mediaPlayer._setManifestInfo(this.manifestInfo);
                    const listener = sinon.spy();
                    this.backend.addEventListener(TwitchEvents.MANIFEST_EXTRA_INFO, listener);
                    this.backend._mediaPlayer.emit(FakeMediaPlayer.PlayerState.READY, this.manifestInfo);
                    assert.ok(listener.calledWith(this.normalizedManifestInfo));
                });
            });
        });

        QUnit.module('emit loaded metadata event', function() {
            QUnit.test('on the first mp duration change event after a mp ready event is fired', function(assert) {
                return waitFor(() => this.backend._mediaPlayer).then(() => {
                    const listener = sinon.spy();
                    this.backend.addEventListener(MediaEvents.LOADED_METADATA, listener);

                    this.backend._mediaPlayer.emit(FakeMediaPlayer.PlayerState.READY);
                    assert.equal(listener.callCount, 0, 'loaded metadata not emitted');

                    this.backend._mediaPlayer.emit(FakeMediaPlayer.PlayerEvent.DURATION_CHANGED);
                    assert.equal(listener.callCount, 1, 'loaded metadata emitted');
                });
            });

            // eslint-disable-next-line max-len
            QUnit.test('on subsequent mp duration change events after the first mp duration change event is fired after a mp ready event', function(assert) {
                return waitFor(() => this.backend._mediaPlayer).then(() => {
                    const listener = sinon.spy();
                    this.backend.addEventListener(MediaEvents.LOADED_METADATA, listener);

                    this.backend._mediaPlayer.emit(FakeMediaPlayer.PlayerState.READY);
                    this.backend._mediaPlayer.emit(FakeMediaPlayer.PlayerEvent.DURATION_CHANGED);

                    listener.reset();

                    this.backend._mediaPlayer.emit(FakeMediaPlayer.PlayerEvent.DURATION_CHANGED);
                    assert.equal(listener.callCount, 0, 'loaded metadata not emitted');
                });
            });

            QUnit.test('on repeat firings of an mp ready event and then a mp duration change event', function(assert) {
                return waitFor(() => this.backend._mediaPlayer).then(() => {
                    const listener = sinon.spy();
                    this.backend.addEventListener(MediaEvents.LOADED_METADATA, listener);

                    this.backend._mediaPlayer.emit(FakeMediaPlayer.PlayerState.READY);
                    this.backend._mediaPlayer.emit(FakeMediaPlayer.PlayerEvent.DURATION_CHANGED);
                    assert.equal(listener.callCount, 1, 'loaded metadata emitted');

                    listener.reset();

                    this.backend._mediaPlayer.emit(FakeMediaPlayer.PlayerState.READY);
                    this.backend._mediaPlayer.emit(FakeMediaPlayer.PlayerEvent.DURATION_CHANGED);
                    assert.equal(listener.callCount, 1, 'loaded metadata emitted');
                });
            });
        });
    });

    QUnit.module('playback control and lookups', function() {
        QUnit.test('can change volume', function(assert) {
            const volume = 0.6;
            this.backend.setVolume(volume);
            return waitFor(() => this.backend._mediaPlayer).then(() => {
                assert.equal(this.backend._mediaPlayer.getVolume(), volume);
            });
        });

        QUnit.test('can mute and unmute volume which sets and unsets video muted attribute', function(assert) {
            return waitFor(() => this.backend._mediaPlayer).then(() => {
                const videoElement = this.backend._mediaPlayer.getHTMLVideoElement();

                assert.notOk(this.backend._mediaPlayer.isMuted(), 'initially backend should not be muted');
                assert.notOk(videoElement.hasAttribute('muted'), 'initially video tag should not have `muted` attr');

                this.backend.setMuted(true);
                assert.ok(this.backend._mediaPlayer.isMuted(), 'backend should be muted');
                assert.ok(videoElement.hasAttribute('muted'), 'video tag should have `muted` attr');

                this.backend.setMuted(false);
                assert.notOk(this.backend._mediaPlayer.isMuted(), 'backend should not be muted');
                assert.notOk(videoElement.hasAttribute('muted'), 'video tag should not have `muted` attr');
            });
        });

        QUnit.test('can lookup seeking status', function(assert) {
            assert.equal(this.backend.getSeeking(), false);
        });

        QUnit.test('can lookup buffered property', function(assert) {
            assert.equal(this.backend.getBuffered().length, 0);
        });
    });

    QUnit.test('when playing a vod, call `video.play`, not setChannel', function(assert) {
        setStream(this.store, new VODTwitchContentStream(
            'monstercat',
            Promise.resolve(this.oauthToken),
            this.usherParams,
            this.accessTokenParams
        ));
        this.backend.loadMediaPlayer();
        return waitFor(() => this.backend._mediaPlayer).
            then(() => {
                sinon.spy(this.backend, 'setChannel');
                sinon.spy(this.backend._mediaPlayer, 'play');
                this.backend.play();
                assert.notOk(this.backend.setChannel.called);
                assert.ok(this.backend._mediaPlayer.play.called);
            });
    });

    QUnit.module('onSegmentChanged', function() {
        QUnit.test('fires SEGMENT_CHANGE event if TOFN tag exists', function(assert) {
            const segmentName = 'thisisasegment.ts';
            const data = [{
                id: 'TOFN',
                info: [segmentName],
            }];
            const handler = sinon.spy();

            this.backend.loadMediaPlayer();
            return waitFor(() => this.backend._mediaPlayer).
                then(() => {
                    this.backend.addEventListener(TwitchEvents.SEGMENT_CHANGE, handler);
                    this.backend._mediaPlayer.emit(FakeMediaPlayer.MetadataEvent.ID3, data);
                    assert.equal(handler.callCount, 1);
                    assert.deepEqual(handler.firstCall.args[0], { name: segmentName });
                });
        });

        QUnit.test('fires MIDROLL_REQUESTED event if commercial is requested', function(assert) {
            const info = {
                cmd: 'commercial',
                length: 60,
            };
            const data = [{
                id: 'TXXX',
                info: [JSON.stringify(info)],
            }];
            const handler = sinon.spy();

            this.backend.loadMediaPlayer();
            return waitFor(() => this.backend._mediaPlayer).
                then(() => {
                    this.backend.addEventListener(TwitchEvents.MIDROLL_REQUESTED, handler);
                    this.backend._mediaPlayer.emit(FakeMediaPlayer.MetadataEvent.ID3, data);
                    assert.equal(handler.callCount, 1);
                    assert.deepEqual(handler.firstCall.args[0], { duration: info.length });
                });
        });

        QUnit.test('fires STITCHED_AD_START and STITCHED_AD_END for stitched ads', function(assert) {
            const startHandler = sinon.spy();
            const endHandler = sinon.spy();

            const data = {
                URL: 'test',
            };

            this.backend.loadMediaPlayer();
            return waitFor(() => this.backend._mediaPlayer).
                then(() => {
                    this.backend.addEventListener(TwitchEvents.STITCHED_AD_START, startHandler);
                    this.backend.addEventListener(TwitchEvents.STITCHED_AD_END, endHandler);
                    this.backend._mediaPlayer.emit(FakeMediaPlayer.MetadataEvent.SPLICE_OUT, data);
                    this.backend._mediaPlayer.emit(FakeMediaPlayer.MetadataEvent.SPLICE_IN);
                    assert.equal(startHandler.callCount, 1);
                    assert.equal(endHandler.callCount, 1);
                    assert.deepEqual(startHandler.firstCall.args[0], data);
                });
        });
    });

    QUnit.module('on destroy', function() {
        QUnit.test('calls destroy on MediaPlayer', function(assert) {
            this.backend.loadMediaPlayer();
            return waitFor(() => this.backend._mediaPlayer).
                then(() => {
                    const spy = sinon.spy(this.backend._mediaPlayer, 'delete');
                    this.backend.destroy();
                    assert.ok(spy.called);
                });
        });
    });

    QUnit.test('setQuality calls setQuality in mediaplayer', function(assert) {
        const quality = 'medium';

        this.backend.loadMediaPlayer();
        return waitFor(() => this.backend._mediaPlayer).
            then(() => {
                const allQualities = this.backend._mediaPlayer.getQualities();
                sinon.spy(this.backend._mediaPlayer, 'setQuality');
                this.backend.setQuality(quality);
                assert.ok(this.backend._mediaPlayer.setQuality.called);
                // The fake mediaplayer.fake.js has medium on second index after source and high
                assert.deepEqual(this.backend._mediaPlayer.setQuality.firstCall.args[0], allQualities[2]);
            });
    });

    QUnit.test('calling `load` with autoplay calls video.play', function(assert) {
        this.backend.setSrc('notarealsrc.m3u8');
        this.backend.loadMediaPlayer();
        return waitFor(() => this.backend._mediaPlayer).
            then(() => {
                sinon.spy(this.backend._mediaPlayer, 'play');
                this.backend.load();
                assert.ok(this.backend._mediaPlayer.play.called);
            });
    });

    QUnit.test('when setting a channel, call video.play if autoplay is set', function(assert) {
        const channel = 'monstercat';
        const stream = new LiveTwitchContentStream(
            channel,
            Promise.resolve(this.oauthToken),
            this.usherParams,
            this.accessTokenParams,
            this.experimentSettings
        );
        setStream(this.store, stream);

        this.api.expectChannelAccessToken(channel.toLowerCase(), {
            canAccess: true,
            oauthToken: this.oauthToken,
            accessTokenParams: this.accessTokenParams,
        });

        this.backend.loadMediaPlayer();
        return waitFor(() => this.backend._mediaPlayer).
            then(() => {
                sinon.spy(this.backend._mediaPlayer, 'play');
                assert.notOk(this.backend._mediaPlayer.play.called);
                this.backend.setChannel(channel, stream);
                return waitFor(() => this.backend._mediaPlayer.play.called);
            });
    });

    QUnit.test('when setting a channel with autoplay true, fire the MediaEvent.PLAY', function(assert) {
        const channel = 'monstercat';
        const stream = new LiveTwitchContentStream(
            channel,
            Promise.resolve(this.oauthToken),
            this.usherParams,
            this.accessTokenParams,
            this.experimentSettings
        );
        setStream(this.store, stream);

        this.api.expectChannelAccessToken(channel.toLowerCase(), {
            canAccess: true,
            oauthToken: this.oauthToken,
            accessTokenParams: this.accessTokenParams,
        });

        const listener = sinon.spy();
        this.backend.addEventListener(MediaEvents.PLAY, listener);
        this.backend.loadMediaPlayer();
        return waitFor(() => this.backend._mediaPlayer).
            then(() => {
                this.backend.setChannel(channel, stream);
                return waitFor(() => this.backend._src).
                    then(() => {
                        assert.ok(listener.called, 'MediaEvent.PLAY event should be fired');
                    });
            });
    });

    QUnit.test('when setting video on autoplay, fire MediaEvent.PLAY event', function(assert) {
        const vodId = 'v1234567890';
        const oauthTokenPromise = Promise.resolve(this.oauthToken);
        const stream = new VODTwitchContentStream(
            vodId,
            oauthTokenPromise,
            this.usherParams,
            this.accessTokenParams,
            this.experimentSettings
        );
        setStream(this.store, stream);
        // eslint-disable-next-line max-len
        this.api.expectVodAccessToken(vodId.toLowerCase(), {
            canAccess: true,
            oauthToken: this.oauthToken,
            accessTokenParams: this.accessTokenParams,
        });

        const listener = sinon.spy();
        this.backend.addEventListener(MediaEvents.PLAY, listener);
        this.backend.loadMediaPlayer();
        return waitFor(() => this.backend._mediaPlayer).
            then(() => {
                this.backend.setVideo(vodId, stream);
                return waitFor(() => this.backend._src).
                    then(() => {
                        assert.ok(listener.called, 'MediaEvent.PLAY event should be fired');
                    });
            });
    });

    QUnit.test('when calling play with autoplay false, fire MediaEvent.PLAY event', function(assert) {
        const options =  {
            autoplay: false,
        };
        this.store.dispatch(setAutoplay(options.autoplay));
        const channel = 'monstercat';
        const stream = new LiveTwitchContentStream(
            channel,
            Promise.resolve(this.oauthToken),
            this.usherParams,
            this.accessTokenParams,
            this.experimentSettings
        );
        this.api.expectChannelAccessToken(channel.toLowerCase(), {
            canAccess: true,
            oauthToken: this.oauthToken,
            accessTokenParams: this.accessTokenParams,
        });

        const backend = new MediaPlayer.BackendMediaPlayer(options, this.store);
        const listener = sinon.spy();
        backend.addEventListener(MediaEvents.PLAY, listener);
        backend.loadMediaPlayer();

        return waitFor(() => backend._mediaPlayer).
            then(() => {
                setStream(this.store, stream);
                backend.play();
                assert.ok(listener.called, 'MediaEvent.PLAY event should be fired');
            });
    });

    QUnit.test('when calling play with autoplay false, not call _mediaplayer.play', function(assert) {
        const options =  {
            autoplay: false,
        };
        this.store.dispatch(setAutoplay(options.autoplay));
        const channel = 'monstercat';
        const stream = new LiveTwitchContentStream(
            channel,
            Promise.resolve(this.oauthToken),
            this.usherParams,
            this.accessTokenParams,
            this.experimentSettings
        );
        this.api.expectChannelAccessToken(channel.toLowerCase(), {
            canAccess: true,
            oauthToken: this.oauthToken,
            accessTokenParams: this.accessTokenParams,
        });

        const backend = new MediaPlayer.BackendMediaPlayer(options, this.store);
        backend.loadMediaPlayer();

        return waitFor(() => this.backend._mediaPlayer).
            then(() => {
                sinon.spy(backend._mediaPlayer, 'play');
                setStream(this.store, stream);
                assert.notOk(backend._mediaPlayer.play.called);
                backend.play();
                return waitFor(() => backend._mediaPlayer.play.calledOnce);
            });
    });

    QUnit.module('when MediaPlayer fires a BUFFERING event', function() {
        QUnit.test('paused does not change', function(assert) {
            this.backend.loadMediaPlayer();
            return waitFor(() => this.backend._mediaPlayer).
                then(() => {
                    const prevPaused = this.backend.getPaused();
                    this.backend._mediaPlayer.emit(FakeMediaPlayer.PlayerEvent.REBUFFERING);
                    assert.equal(this.backend.getPaused(), prevPaused);
                });
        });

        QUnit.test('WAITING is emitted', function(assert) {
            this.backend.loadMediaPlayer();
            return waitFor(() => this.backend._mediaPlayer).
                then(() => {
                    const listener = sinon.spy();
                    this.backend.addEventListener(MediaEvents.WAITING, listener);
                    this.backend._mediaPlayer.emit(FakeMediaPlayer.PlayerEvent.REBUFFERING);
                    assert.ok(listener.called);
                });
        });
    });

    QUnit.module('when MediaPlayer fires a PLAYBACK_RATE_CHANGED event', function() {
        QUnit.test('RATE_CHANGE is fired', function(assert) {
            this.backend.loadMediaPlayer();
            return waitFor(() => this.backend._mediaPlayer).
                then(() => {
                    const listener = sinon.spy();
                    this.backend.addEventListener(MediaEvents.RATE_CHANGE, listener);
                    this.backend._mediaPlayer.emit(FakeMediaPlayer.PlayerEvent.PLAYBACK_RATE_CHANGED);
                    assert.ok(listener.called);
                });
        });
    });

    QUnit.module('when MediaPlayer fires a BUFFER_UPDATE event', function() {
        QUnit.test('BUFFER_CHANGE is fired', function(assert) {
            this.backend.loadMediaPlayer();
            return waitFor(() => this.backend._mediaPlayer).
                then(() => {
                    const listener = sinon.spy();
                    this.backend.addEventListener(TwitchEvents.BUFFER_CHANGE, listener);
                    this.backend._mediaPlayer.emit(FakeMediaPlayer.PlayerEvent.BUFFER_UPDATE);
                    assert.ok(listener.called);
                });
        });
    });

    QUnit.module('when MediaPlayer fires a SEEK_COMPLETED event', function() {
        QUnit.test('SEEKED and CAN_PLAY are fired', function(assert) {
            return this.backend.loadMediaPlayer().then(() => {
                const seekedListener = sinon.spy();
                const canPlayListener = sinon.spy();
                this.backend.addEventListener(MediaEvents.SEEKED, seekedListener);
                this.backend.addEventListener(MediaEvents.CAN_PLAY, canPlayListener);
                this.backend._mediaPlayer.emit(FakeMediaPlayer.PlayerEvent.SEEK_COMPLETED);
                assert.ok(seekedListener.called, 'SEEKED event listener was not called');
                assert.ok(canPlayListener.called, 'CAN_PLAY event listener was not called');
            });
        });
    });

    QUnit.module('when MediaPlayer fires a QUALITY_CHANGED event', function() {
        QUnit.test('QUALITY_CHANGE is always fired', function(assert) {
            this.backend.loadMediaPlayer();
            return waitFor(() => this.backend._mediaPlayer).
                then(() => {
                    const expectedQuality = {
                        group: 'chunked',
                        name: 'Source',
                    };
                    sinon.stub(this.backend._mediaPlayer, 'getQuality', () => expectedQuality);

                    const listener = sinon.spy();
                    this.backend.addEventListener(TwitchEvents.QUALITY_CHANGE, listener);
                    this.backend._mediaPlayer.emit(FakeMediaPlayer.PlayerEvent.QUALITY_CHANGED);
                    assert.ok(listener.called);
                    assert.deepEqual(listener.firstCall.args[0], {
                        quality: 'chunked',
                        isAuto: true,
                    });

                    this.backend._mediaPlayer.getQuality.restore();
                });
        });

        QUnit.test('ABS_STREAM_FORMAT_CHANGE is fired if using auto', function(assert) {
            this.backend.loadMediaPlayer();
            return waitFor(() => this.backend._mediaPlayer).
                then(() => {
                    const firstQuality = {
                        group: '480p',
                    };
                    const secondQuality = {
                        group: 'chunked',
                    };
                    const absStats = {
                        foo: 42,
                    };
                    const getQualityStub = sinon.stub(this.backend._mediaPlayer, 'getQuality');
                    sinon.stub(this.backend._mediaPlayer, 'getABSStats', () => absStats);

                    getQualityStub.returns(firstQuality);
                    this.backend._mediaPlayer.emit(FakeMediaPlayer.PlayerEvent.QUALITY_CHANGED);

                    const listener = sinon.spy();
                    this.backend.addEventListener(TwitchEvents.ABS_STREAM_FORMAT_CHANGE, listener);

                    getQualityStub.returns(secondQuality);
                    this.backend._mediaPlayer.emit(FakeMediaPlayer.PlayerEvent.QUALITY_CHANGED);

                    assert.ok(listener.called);
                    assert.deepEqual(listener.firstCall.args[0], Object.assign({
                        /* eslint-disable camelcase */
                        stream_format_previous: firstQuality.group,
                        stream_format_current: secondQuality.group,
                        /* eslint-enable camelcase */
                    }, absStats));

                    this.backend._mediaPlayer.getQuality.restore();
                    this.backend._mediaPlayer.getABSStats.restore();
                });
        });
    });

    QUnit.module('when MediaPlayer fires NOT_SUPPORTED ERROR event', function(hooks) {
        hooks.beforeEach(function() {
            sinon.spy(this.store, 'dispatch');
        });

        hooks.afterEach(function() {
            this.store.dispatch.restore();
        });

        QUnit.test('dispatched FORMAT_NOT_SUPPORTED error code', function(assert) {
            return this.backend.initialize().then(() => {
                this.store.dispatch.reset();
                this.backend._mediaPlayer.emit(FakeMediaPlayer.PlayerEvent.ERROR, {
                    type: FakeMediaPlayer.ErrorType.NOT_SUPPORTED,
                });

                assert.equal(this.store.dispatch.callCount, 2, 'dispatch called once');
                const setErrorAction = this.store.dispatch.secondCall.args[0];

                assert.equal(
                    setErrorAction.type,
                    FAKE_SET_ERROR_ACTION,
                    'dispatched set error action'
                );
                assert.equal(
                    setErrorAction.code,
                    Errors.CODES.FORMAT_NOT_SUPPORTED,
                    'dispatched correct error code'
                );
            });
        });

        QUnit.test('sends PLAYBACK_ERROR FATAL ERROR spade event', function(assert) {
            return this.backend.initialize().then(() => this.backend._mediaPlayer).
                then(() => {
                    this.backend.addEventListener(MediaEvents.ERROR, () => {});
                    this.backend._mediaPlayer.emit(FakeMediaPlayer.PlayerEvent.ERROR, {
                        type: FakeMediaPlayer.ErrorType.NOT_SUPPORTED,
                    });

                    assert.equal(this.store.dispatch.callCount, 2);
                    assert.ok(this.store.dispatch.calledWith({
                        type: ACTION_TRACK_EVENT,
                        eventName: PLAYBACK_ERROR,
                        eventProperties: {
                            // eslint-disable-next-line camelcase
                            playback_error_code: MediaPlayer.FATAL_ERROR_CODE,
                            // eslint-disable-next-line camelcase
                            playback_error_msg: MediaPlayer.PLAYER_FATAL_ERROR,
                        },
                    }));
                });
        });
    });

    QUnit.module('when MediaPlayer fires NETWORK ERROR event', function(hooks) {
        hooks.beforeEach(function() {
            sinon.spy(this.store, 'dispatch');
        });

        hooks.afterEach(function() {
            this.store.dispatch.restore();
        });

        QUnit.test('dispatch NETWORK error code', function(assert) {
            return this.backend.initialize().then(() => {
                this.store.dispatch.reset();
                this.backend._mediaPlayer.emit(FakeMediaPlayer.PlayerEvent.ERROR, {
                    type: FakeMediaPlayer.ErrorType.NETWORK,
                });

                assert.equal(this.store.dispatch.callCount, 2, 'dispatch called once');
                const setErrorAction = this.store.dispatch.secondCall.args[0];

                assert.equal(
                    setErrorAction.type,
                    FAKE_SET_ERROR_ACTION,
                    'dispatched set error action'
                );
                assert.equal(
                    setErrorAction.code,
                    Errors.CODES.NETWORK,
                    'dispatched correct error code'
                );
            });
        });

        QUnit.test('sends PLAYBACK_ERROR FATAL ERROR spade event', function(assert) {
            return this.backend.initialize().then(() => this.backend._mediaPlayer).
                then(() => {
                    this.backend.addEventListener(MediaEvents.ERROR, () => {});
                    this.backend._mediaPlayer.emit(FakeMediaPlayer.PlayerEvent.ERROR, {
                        type: FakeMediaPlayer.ErrorType.NETWORK,
                    });

                    assert.equal(this.store.dispatch.callCount, 2);
                    assert.ok(this.store.dispatch.calledWith({
                        type: ACTION_TRACK_EVENT,
                        eventName: PLAYBACK_ERROR,
                        eventProperties: {
                            // eslint-disable-next-line camelcase
                            playback_error_code: MediaPlayer.FATAL_ERROR_CODE,
                            // eslint-disable-next-line camelcase
                            playback_error_msg: MediaPlayer.PLAYER_FATAL_ERROR,
                        },
                    }));
                });
        });
    });

    QUnit.module('when MediaPlayer fires NETWORK_IO ERROR event', function(hooks) {
        hooks.beforeEach(function() {
            sinon.spy(this.store, 'dispatch');
        });

        hooks.afterEach(function() {
            this.store.dispatch.restore();
        });

        QUnit.test('dispatch NETWORK error code', function(assert) {
            return this.backend.initialize().then(() => {
                this.store.dispatch.reset();
                this.backend._mediaPlayer.emit(FakeMediaPlayer.PlayerEvent.ERROR, {
                    type: FakeMediaPlayer.ErrorType.NETWORK_IO,
                });

                assert.equal(this.store.dispatch.callCount, 2, 'dispatch called once');
                const setErrorAction = this.store.dispatch.secondCall.args[0];

                assert.equal(
                    setErrorAction.type,
                    FAKE_SET_ERROR_ACTION,
                    'dispatched set error action'
                );
                assert.equal(
                    setErrorAction.code,
                    Errors.CODES.NETWORK,
                    'dispatched correct error code'
                );
            });
        });

        QUnit.test('sends PLAYBACK_ERROR FATAL ERROR spade event', function(assert) {
            return this.backend.initialize().then(() => this.backend._mediaPlayer).
                then(() => {
                    this.backend.addEventListener(MediaEvents.ERROR, () => {});
                    this.backend._mediaPlayer.emit(FakeMediaPlayer.PlayerEvent.ERROR, {
                        type: FakeMediaPlayer.ErrorType.NETWORK_IO,
                    });

                    assert.equal(this.store.dispatch.callCount, 2);
                    assert.ok(this.store.dispatch.calledWith({
                        type: ACTION_TRACK_EVENT,
                        eventName: PLAYBACK_ERROR,
                        eventProperties: {
                            // eslint-disable-next-line camelcase
                            playback_error_code: MediaPlayer.FATAL_ERROR_CODE,
                            // eslint-disable-next-line camelcase
                            playback_error_msg: MediaPlayer.PLAYER_FATAL_ERROR,
                        },
                    }));
                });
        });
    });

    QUnit.module('when MediaPlayer fires an ERROR event with the source being decoder', function(hooks) {
        hooks.beforeEach(function() {
            sinon.spy(this.store, 'dispatch');
        });

        hooks.afterEach(function() {
            this.store.dispatch.restore();
        });

        QUnit.test('dispatch DECODE error code', function(assert) {
            return this.backend.initialize().then(() => {
                this.store.dispatch.reset();
                this.backend._mediaPlayer.emit(FakeMediaPlayer.PlayerEvent.ERROR, {
                    type: FakeMediaPlayer.ErrorType.GENERIC,
                    source: FakeMediaPlayer.ErrorSource.DECODER,
                });

                assert.equal(this.store.dispatch.callCount, 2, 'dispatch called once');
                const setErrorAction = this.store.dispatch.secondCall.args[0];

                assert.equal(
                    setErrorAction.type,
                    FAKE_SET_ERROR_ACTION,
                    'dispatched set error action'
                );
                assert.equal(
                    setErrorAction.code,
                    Errors.CODES.DECODE,
                    'dispatched correct error code'
                );
            });
        });

        QUnit.test('sends PLAYBACK_ERROR FATAL ERROR spade event', function(assert) {
            return this.backend.initialize().then(() => this.backend._mediaPlayer).
                then(() => {
                    this.backend.addEventListener(MediaEvents.ERROR, () => {});
                    this.backend._mediaPlayer.emit(FakeMediaPlayer.PlayerEvent.ERROR, {
                        type: FakeMediaPlayer.ErrorType.GENERIC,
                        source: FakeMediaPlayer.ErrorSource.DECODER,
                    });

                    assert.equal(this.store.dispatch.callCount, 2);
                    assert.ok(this.store.dispatch.calledWith({
                        type: ACTION_TRACK_EVENT,
                        eventName: PLAYBACK_ERROR,
                        eventProperties: {
                            // eslint-disable-next-line camelcase
                            playback_error_code: MediaPlayer.FATAL_ERROR_CODE,
                            // eslint-disable-next-line camelcase
                            playback_error_msg: MediaPlayer.PLAYER_FATAL_ERROR,
                        },
                    }));
                });
        });
    });

    QUnit.module('when MediaPlayer fires other ERROR events without source being decoder', function(hooks) {
        hooks.beforeEach(function() {
            sinon.spy(this.store, 'dispatch');
        });

        hooks.afterEach(function() {
            this.store.dispatch.restore();
        });

        QUnit.test('dispatch ABORTED error code', function(assert) {
            return this.backend.initialize().then(() => {
                this.store.dispatch.reset();
                this.backend._mediaPlayer.emit(FakeMediaPlayer.PlayerEvent.ERROR, {
                    type: FakeMediaPlayer.ErrorType.GENERIC,
                });

                assert.equal(this.store.dispatch.callCount, 2, 'dispatch called once');
                const setErrorAction = this.store.dispatch.secondCall.args[0];

                assert.equal(
                    setErrorAction.type,
                    FAKE_SET_ERROR_ACTION,
                    'dispatched set error action'
                );
                assert.equal(
                    setErrorAction.code,
                    Errors.CODES.ABORTED,
                    'dispatched correct error code'
                );
            });
        });

        QUnit.test('sends PLAYBACK_ERROR FATAL ERROR spade event', function(assert) {
            return this.backend.initialize().then(() => this.backend._mediaPlayer).
                then(() => {
                    this.backend.addEventListener(MediaEvents.ERROR, () => {});
                    this.backend._mediaPlayer.emit(FakeMediaPlayer.PlayerEvent.ERROR, {
                        type: FakeMediaPlayer.ErrorType.GENERIC,
                    });

                    assert.equal(this.store.dispatch.callCount, 2);
                    assert.ok(this.store.dispatch.calledWith({
                        type: ACTION_TRACK_EVENT,
                        eventName: PLAYBACK_ERROR,
                        eventProperties: {
                            // eslint-disable-next-line camelcase
                            playback_error_code: MediaPlayer.FATAL_ERROR_CODE,
                            // eslint-disable-next-line camelcase
                            playback_error_msg: MediaPlayer.PLAYER_FATAL_ERROR,
                        },
                    }));
                });
        });
    });

    QUnit.module('when MediaPlayer fires an NOT_AVAILABLE ERROR ', function(hooks) {
        hooks.beforeEach(function() {
            sinon.spy(this.store, 'dispatch');
        });

        hooks.afterEach(function() {
            this.store.dispatch.restore();
        });

        QUnit.test('sends PLAYBACK_ERROR CCU_CAP_REACHED spade event - when CCU_CAP error', function(assert) {
            return this.backend.initialize().then(() => {
                sinon.spy(this.backend, '_onCCUCapReached');
                this.backend._mediaPlayer.emit(FakeMediaPlayer.PlayerEvent.ERROR, {
                    type: FakeMediaPlayer.ErrorType.NOT_AVAILABLE,
                    code: 509,
                });

                assert.equal(this.store.dispatch.callCount, 2, '2 dispatches');

                const { firstCall, secondCall } = this.store.dispatch;

                assert.equal(this.backend._onCCUCapReached.callCount, 1, '_onCCUCapReached called once');

                assert.deepEqual(firstCall.args[0], {
                    type: ACTION_TRACK_EVENT,
                    eventName: PLAYBACK_ERROR,
                    eventProperties: {
                        // eslint-disable-next-line camelcase
                        playback_error_code: Errors.CODES.CCU_CAP_REACHED,
                        // eslint-disable-next-line camelcase
                        playback_error_msg: MediaPlayer.PLAYER_LVS_CCU_CAP_ERROR,
                    },
                }, 'correct tracking event fired');

                assert.deepEqual(secondCall.args[0], {
                    type: FAKE_SET_ERROR_ACTION,
                    code: Errors.CODES.CCU_CAP_REACHED,
                }, 'correct error set in store');
            });
        });

        QUnit.test('sends PLAYBACK_ERROR OFFLINE ERROR spade event - when not CCU_CAP error', function(assert) {
            return this.backend.initialize().then(() => {
                sinon.spy(this.backend, '_onOfflineError');
                this.backend._mediaPlayer.emit(FakeMediaPlayer.PlayerEvent.ERROR, {
                    type: FakeMediaPlayer.ErrorType.NOT_AVAILABLE,
                });

                assert.equal(this.store.dispatch.callCount, 1);
                assert.ok(this.store.dispatch.calledWith({
                    type: ACTION_TRACK_EVENT,
                    eventName: PLAYBACK_ERROR,
                    eventProperties: {
                        // eslint-disable-next-line camelcase
                        playback_error_code: MediaPlayer.OFFLINE_ERROR_CODE,
                        // eslint-disable-next-line camelcase
                        playback_error_msg: MediaPlayer.PLAYER_OFFLINE_ERROR,
                    },
                }));
            });
        });

        QUnit.test('emits OFFLINE and ENDED Event', function(assert) {
            this.backend.loadMediaPlayer();
            return waitFor(() => this.backend._mediaPlayer).
                then(() => {
                    const offlineListener = sinon.spy();
                    const endedListener = sinon.spy();
                    this.backend.addEventListener(MediaEvents.ENDED, endedListener);
                    this.backend.addEventListener(TwitchEvents.OFFLINE, offlineListener);
                    this.backend._mediaPlayer.emit(FakeMediaPlayer.PlayerEvent.ERROR, {
                        type: FakeMediaPlayer.ErrorType.NOT_AVAILABLE,
                    });
                    assert.ok(endedListener.calledOnce, 'ended event emitted');
                    assert.ok(offlineListener.calledOnce, 'offline event emitted');
                });
        });
    });

    QUnit.module('when MediaPlayer fires an authorization error', function(hooks) {
        hooks.beforeEach(function() {
            sinon.spy(this.store, 'dispatch');
            const stream = new LiveTwitchContentStream(
                'monstercat',
                Promise.resolve(this.oauthToken),
                this.usherParams,
                this.accessTokenParams,
                this.experimentSettings
            );

            setStream(this.store, stream);
        });

        hooks.afterEach(function() {
            this.store.dispatch.restore();
        });

        QUnit.test('NO-OP if stream is offline', function(assert) {
            this.store.dispatch(setOnline(false));
            this.store.dispatch.reset();
            return this.backend.initialize().then(() => {
                sinon.spy(this.backend, '_retryStreamLoad');

                this.backend._mediaPlayer.emit(
                    FakeMediaPlayer.PlayerEvent.ERROR,
                    { type: FakeMediaPlayer.ErrorType.AUTHORIZATION }
                );

                assert.equal(this.store.dispatch.callCount, 0, 'no actions are dispatched');
                assert.equal(this.backend._retryStreamLoad.callCount, 0, 'retry not called');
            });
        });

        QUnit.test('attempt to retry stream load on first error', function(assert) {
            return this.backend.initialize().then(() => {
                sinon.spy(this.backend, '_retryStreamLoad');
                this.backend._mediaPlayer.emit(
                    FakeMediaPlayer.PlayerEvent.ERROR,
                    { type: FakeMediaPlayer.ErrorType.AUTHORIZATION }
                );
                assert.ok(this.backend._retryStreamLoad.called);
            });
        });

        QUnit.test('Sends PLAYBACK_ERROR WARN AUTH ERROR spade event', function(assert) {
            return this.backend.initialize().then(() => this.backend._mediaPlayer).
                then(() => {
                    sinon.spy(this.backend, '_retryStreamLoad');
                    this.backend._mediaPlayer.emit(
                        FakeMediaPlayer.PlayerEvent.ERROR,
                        { type: FakeMediaPlayer.ErrorType.AUTHORIZATION }
                    );

                    assert.equal(this.store.dispatch.callCount, 2);
                    assert.ok(this.store.dispatch.calledWith({
                        type: ACTION_TRACK_EVENT,
                        eventName: PLAYBACK_ERROR,
                        eventProperties: {
                            // eslint-disable-next-line camelcase
                            playback_error_code: MediaPlayer.WARN_AUTH_ERROR_CODE,
                            // eslint-disable-next-line camelcase
                            playback_error_msg: MediaPlayer.PLAYER_WARN_AUTH_ERROR,
                        },
                    }));
                });
        });

        QUnit.test('dispatch CONTENT_NOT_AVAILABLE error code', function(assert) {
            return this.backend.initialize().then(() => {
                this.store.dispatch.reset();
                this.backend._mediaPlayer.emit(FakeMediaPlayer.PlayerEvent.ERROR, {
                    type: FakeMediaPlayer.ErrorType.AUTHORIZATION,
                });

                this.store.dispatch.reset();

                this.backend._mediaPlayer.emit(FakeMediaPlayer.PlayerEvent.ERROR, {
                    type: FakeMediaPlayer.ErrorType.AUTHORIZATION,
                });

                const setErrorAction = this.store.dispatch.firstCall.args[0];

                assert.equal(
                    setErrorAction.type,
                    FAKE_SET_ERROR_ACTION,
                    'dispatched set error action'
                );
                assert.equal(
                    setErrorAction.code,
                    Errors.CODES.CONTENT_NOT_AVAILABLE,
                    'dispatched correct error code'
                );
            });
        });

        QUnit.test('Sends PLAYBACK_ERROR FATAL AUTH ERROR spade event', function(assert) {
            return this.backend.initialize().then(() => this.backend._mediaPlayer).
                then(() => {
                    const errorListener = sinon.spy();
                    this.backend.addEventListener(MediaEvents.ERROR, errorListener);
                    this.backend._mediaPlayer.emit(
                        FakeMediaPlayer.PlayerEvent.ERROR,
                        { type: FakeMediaPlayer.ErrorType.AUTHORIZATION }
                    );

                    this.backend._mediaPlayer.emit(
                        FakeMediaPlayer.PlayerEvent.ERROR,
                        { type: FakeMediaPlayer.ErrorType.AUTHORIZATION }
                    );

                    assert.equal(this.store.dispatch.callCount, 4);
                    assert.ok(this.store.dispatch.getCall(3).calledWith({
                        type: ACTION_TRACK_EVENT,
                        eventName: PLAYBACK_ERROR,
                        eventProperties: {
                            // eslint-disable-next-line camelcase
                            playback_error_code: MediaPlayer.FATAL_AUTH_ERROR_CODE,
                            // eslint-disable-next-line camelcase
                            playback_error_msg: MediaPlayer.PLAYER_FATAL_AUTH_ERROR,
                        },
                    }));
                });
        });
    });

    QUnit.test('when getting stats, returns current player stats', function(assert) {
        // expect default stats
        const defaultStats = this.backend.getStats();

        assert.equal(defaultStats.playbackRate, -1);
        assert.equal(defaultStats.fps, -1);
        assert.equal(defaultStats.bufferSize, -1);
        assert.equal(defaultStats.skippedFrames, -1);
        assert.equal(defaultStats.memoryUsage, '0 MB');
        assert.equal(defaultStats.hlsLatencyEncoder, -1);
        assert.equal(defaultStats.hlsLatencyBroadcaster, -1);
        assert.equal(defaultStats.videoResolution, '');
        assert.equal(defaultStats.displayResolution, '');

        // expect real stats
        this.backend.loadMediaPlayer();
        return waitFor(() => this.backend._mediaPlayer).
            // eslint-disable-next-line max-statements
            then(() => {
                const videoStats = {
                    bufferSize: 1.1,
                    displayResolution: 'fakeres',
                    skippedFrames: 2,
                    fps: 3.0013,
                    bitrate: 123456,
                    hlsLatencyBroadcaster: 4124,
                    hlsLatencyEncoder: 5124,
                    memoryUsage: 64,
                    playbackRate: 1,
                    videoWidth: '10',
                    videoHeight: '20',
                    displayWidth: '30',
                    displayHeight: '40',
                };

                // Make sure rounding works
                this.backend._mediaPlayer._setPlaybackStatistics(videoStats);
                const roundedStats = this.backend.getStats();

                assert.equal(roundedStats.playbackRate, 123);
                assert.equal(roundedStats.fps, 3);
                assert.equal(roundedStats.bufferSize, 1);
                assert.equal(roundedStats.skippedFrames, 2);
                assert.equal(roundedStats.memoryUsage, '64 MB');
                assert.equal(roundedStats.hlsLatencyEncoder, 5);
                assert.equal(roundedStats.hlsLatencyBroadcaster, 4);
                assert.equal(roundedStats.videoResolution, '10x20');
                assert.equal(roundedStats.displayResolution, '30x40');
            });
    });

    QUnit.test('when getting video info, return expected object with video info', function(assert) {
        this.backend.loadMediaPlayer();
        return waitFor(() => this.backend._mediaPlayer).
            then(() => {
                const coreInfo = {
                    bufferSize: 1,
                    skippedFrames: 2,
                    fps: 30.1,
                    bitrate: 1500068,
                    hlsLatencyBroadcaster: 4.11,
                    hlsLatencyEncoder: 5.99,
                    memoryUsage: 64,
                    playbackRate: 7,
                    targetDuration: 5,
                    videoWidth: '10',
                    videoHeight: '20',
                    displayWidth: '30',
                    displayHeight: '40',
                };
                const fakeWindow = buildFakeWindow({
                    location: {
                        protocol: 'https:',
                    },
                    clearInterval() {},
                    clearTimeout() {},
                });
                const quality = {
                    group: 'low',
                    name: 'Low',
                    bitrate: 1024,
                };

                /* eslint-disable camelcase */
                const manifestInfo = {
                    broadcastId: '12345',
                    cluster: 'sfo01',
                    manifest_cluster: 'sfo01',
                    manifest_node: 'video-edge.sfo01',
                    manifest_node_type: 'legacy',
                    node: 'video-edge.sfo01',
                    serving_id: 'abcd',
                    stream_time: 5,
                    user_ip: '127.0.0.1',
                    origin: 's3',
                    region: 'UNKNOWN',
                };

                const expectedBandwidth = coreInfo.bitrate / 1000;
                const expectedVideoInfo = {
                    bandwidth: expectedBandwidth,
                    broadcast_id: parseInt(coreInfo.broadcastId, 10),
                    cluster: manifestInfo.cluster,
                    current_bitrate: expectedBandwidth,
                    current_fps: round(coreInfo.fps),
                    current_fps_exact: coreInfo.fps,
                    dropped_frames: coreInfo.skippedFrames,
                    hls_latency_broadcaster: round(coreInfo.hlsLatencyBroadcaster),
                    hls_latency_encoder: round(coreInfo.hlsLatencyEncoder),
                    hls_target_duration: coreInfo.targetDuration,
                    manifest_cluster: manifestInfo.manifest_cluster,
                    manifest_node: manifestInfo.manifest_node,
                    manifest_node_type: manifestInfo.manifest_node_type,
                    node: manifestInfo.node,
                    paused: true,
                    playing: false,
                    segment_protocol: 'https',
                    serving_id: manifestInfo.serving_id,
                    stream_time: this.backend._mediaPlayer.getPosition(),
                    stream_time_offset: parseFloat(manifestInfo.stream_time),
                    totalMemoryNumber: coreInfo.memoryUsage,
                    vid_display_height: coreInfo.displayHeight,
                    vid_display_width: coreInfo.displayWidth,
                    vid_height: coreInfo.videoHeight,
                    vid_width: coreInfo.videoWidth,
                    video_buffer_size: coreInfo.bufferSize,
                    volume: 1,
                    user_ip: manifestInfo.user_ip,
                    vod_cdn_origin: manifestInfo.origin,
                    vod_cdn_region: manifestInfo.region,
                };
                /* eslint-enable camelcase */

                this.backend._mediaPlayer._setPlaybackStatistics(coreInfo);
                this.backend._mediaPlayer.setQuality(quality);
                this.store.dispatch(setWindow(fakeWindow));
                this.store.dispatch(setManifestInfo(manifestInfo));
                assert.deepEqual(this.backend.getVideoInfo(), expectedVideoInfo);
            });
    });

    QUnit.test('getVideoInfo() returns http for `segment_protocol` if on http page', function(assert) {
        const fakeWindow = buildFakeWindow({
            location: {
                protocol: 'http:',
            },
            clearInterval() {},
            clearTimeout() {},
        });

        this.backend.loadMediaPlayer();
        return waitFor(() => this.backend._mediaPlayer).
            then(() => {
                this.backend.setQuality('low');
                this.store.dispatch(setWindow(fakeWindow));
                assert.equal(this.backend.getVideoInfo().segment_protocol, 'http');
            });
    });

    QUnit.test('when getting video info, return expected object with video info', function(assert) {
        this.backend.loadMediaPlayer();
        return waitFor(() => this.backend._mediaPlayer).
            then(() => {
                const coreInfo = {
                    bufferSize: 1,
                    skippedFrames: 2,
                    fps: 29.7,
                    bitrate: 2100068,
                    hlsLatencyBroadcaster: 4.99,
                    hlsLatencyEncoder: 5.11,
                    memoryUsage: 64,
                    playbackRate: 7,
                    targetDuration: 5,
                    videoWidth: '10',
                    videoHeight: '20',
                    displayWidth: '30',
                    displayHeight: '40',
                };
                const fakeWindow = buildFakeWindow({
                    location: {
                        protocol: 'https:',
                    },
                    clearInterval() {},
                    clearTimeout() {},
                });

                const quality = {
                    group: 'low',
                    name: 'Low',
                    bitrate: 1024,
                };

                /* eslint-disable camelcase */
                const manifestInfo = {
                    broadcastId: '12345',
                    cluster: 'sfo01',
                    manifest_cluster: 'sfo01',
                    manifest_node: 'video-edge.sfo01',
                    manifest_node_type: 'legacy',
                    node: 'video-edge.sfo01',
                    serving_id: 'abcd',
                    stream_time: 5,
                    user_ip: '127.0.0.1',
                    origin: 's3',
                    region: 'UNKNOWN',
                };

                const expectedBandwidth = coreInfo.bitrate / 1000;
                const expectedVideoInfo = {
                    bandwidth: expectedBandwidth,
                    broadcast_id: parseInt(coreInfo.broadcastId, 10),
                    cluster: manifestInfo.cluster,
                    current_bitrate: expectedBandwidth,
                    current_fps: round(coreInfo.fps),
                    current_fps_exact: coreInfo.fps,
                    dropped_frames: coreInfo.skippedFrames,
                    hls_latency_broadcaster: round(coreInfo.hlsLatencyBroadcaster),
                    hls_latency_encoder: round(coreInfo.hlsLatencyEncoder),
                    hls_target_duration: coreInfo.targetDuration,
                    manifest_cluster: manifestInfo.manifest_cluster,
                    manifest_node: manifestInfo.manifest_node,
                    manifest_node_type: manifestInfo.manifest_node_type,
                    node: manifestInfo.node,
                    paused: true,
                    playing: false,
                    segment_protocol: 'https',
                    serving_id: manifestInfo.serving_id,
                    stream_time: this.backend._mediaPlayer.getPosition(),
                    stream_time_offset: manifestInfo.stream_time,
                    totalMemoryNumber: coreInfo.memoryUsage,
                    vid_display_height: coreInfo.displayHeight,
                    vid_display_width: coreInfo.displayWidth,
                    vid_height: coreInfo.videoHeight,
                    vid_width: coreInfo.videoWidth,
                    video_buffer_size: coreInfo.bufferSize,
                    volume: 1,
                    user_ip: manifestInfo.user_ip,
                    vod_cdn_origin: manifestInfo.origin,
                    vod_cdn_region: manifestInfo.region,
                };
                /* eslint-enable camelcase */

                this.backend._mediaPlayer._setPlaybackStatistics(coreInfo);
                this.backend._mediaPlayer.setQuality(quality);
                this.store.dispatch(setWindow(fakeWindow));
                this.store.dispatch(setManifestInfo(manifestInfo));
                assert.deepEqual(this.backend.getVideoInfo(), expectedVideoInfo);
            });
    });

    QUnit.test('getVideoInfo() returns http for `segment_protocol` if on http page', function(assert) {
        const fakeWindow = buildFakeWindow({
            location: {
                protocol: 'http:',
            },
            clearInterval() {},
            clearTimeout() {},
        });

        this.backend.loadMediaPlayer();
        return waitFor(() => this.backend._mediaPlayer).
            then(() => {
                this.backend.setQuality('low');
                this.store.dispatch(setWindow(fakeWindow));
                assert.equal(this.backend.getVideoInfo().segment_protocol, 'http');
            });
    });

    // eslint-disable-next-line max-len
    QUnit.test('when CAPTION is fired, current caption data should be saved and returned by getCaption', function(assert) {
        this.backend.loadMediaPlayer();
        return waitFor(() => this.backend._mediaPlayer).
            then(() => {
                const fakeCaptions = {
                    fakeData: ['this', 'is', 'a bunch of', 'caption text'],
                };
                this.backend._mediaPlayer.emit(FakeMediaPlayer.MetadataEvent.CAPTION, fakeCaptions);
                assert.deepEqual(this.backend._currentCaptionData, fakeCaptions);
                assert.deepEqual(this.backend.getCaption(), fakeCaptions);
            });
    });

    // eslint-disable-next-line max-len
    QUnit.test('calling paused() will check for this.mediaplayer.getPlayerState()', function(assert) {
        this.backend.loadMediaPlayer();
        return waitFor(() => this.backend._mediaPlayer).
            then(() => {
                const result = this.backend.getPaused();
                assert.notEqual(result, undefined);
                // eslint-disable-next-line max-len
                assert.equal(result, this.backend._mediaPlayer.getPlayerState() === this.backend._mediaPlayerHandle.PlayerState.IDLE);
            });
    });

    // eslint-disable-next-line max-len
    QUnit.test('calling ended() will return this.core.ended()', function(assert) {
        this.backend.loadMediaPlayer();
        return waitFor(() => this.backend._mediaPlayer).
            then(() => {
                const result = this.backend.getEnded();
                assert.notEqual(result, undefined);
                // eslint-disable-next-line max-len
                assert.equal(result, this.backend._mediaPlayer.getPlayerState() === this.backend._mediaPlayerHandle.PlayerState.ENDED);
            });
    });

    QUnit.test('does add a listener on the backend for WAITING event', function(assert) {
        const listener = sinon.spy();
        this.backend.addEventListener(MediaEvents.WAITING, listener);
        return waitFor(() => this.backend._mediaPlayer).
            then(() => {
                this.backend._eventEmitter.emit(MediaEvents.WAITING);
                assert.ok(listener.called);
            });
    });

    QUnit.test('when seeking video, expect MediaEvent.CAN_PLAY event', function(assert) {
        const vodId = 'v1234567890';
        const oauthTokenPromise = Promise.resolve(this.oauthToken);
        const stream = new VODTwitchContentStream(
            vodId,
            oauthTokenPromise,
            this.usherParams,
            this.accessTokenParams,
            this.experimentSettings
        );
        setStream(this.store, stream);
        // eslint-disable-next-line max-len
        this.api.expectVodAccessToken(vodId.toLowerCase(), {
            canAccess: true,
            oauthToken: this.oauthToken,
            accessTokenParams: this.accessTokenParams,
        });

        const listener = sinon.spy();
        this.backend.addEventListener(MediaEvents.CAN_PLAY, listener);
        this.backend.loadMediaPlayer();
        this.backend.setVideo(vodId, stream);
        return waitFor(() => this.backend._mediaPlayer).
            then(() => {
                this.backend.setCurrentTime(Math.random() * this.backend.getDuration());
                this.backend._mediaPlayer.emit(FakeMediaPlayer.PlayerEvent.SEEK_COMPLETED);
                assert.ok(listener.called, 'MediaEvent.CAN_PLAY event should be fired');
            });
    });
});
