import * as ReadyState from 'backend/state/ready-state';
import * as TwitchEvents from 'backend/events/twitch-event';
import * as MediaEvents from 'backend/events/media-event';
import * as PlayerType from 'util/player-type';
import FakePlayerCore from 'tests/fakes/playercore.fake.js';
import sinon from 'sinon';
import { stringify } from 'query-string';
import { PLAYER_CORE_VER_CONTROL } from 'experiments';
import { PREROLL_TIMEOUT } from 'ads/surestream/consts';
import { ACTION_SET_EXPERIMENTS } from 'actions/experiments';
import { ACTION_SET_STREAM } from 'actions/stream';
import { setOnline } from 'actions/online';
import * as PlayerCore from 'backend/player-core';
import { LiveTwitchContentStream, CONTENT_MODE_LIVE } from 'stream/twitch-live';
import { VODTwitchContentStream, CONTENT_MODE_VOD } from 'stream/twitch-vod';
import { VIDEO_PLAY_MASTER_MANIFEST, VIDEO_PLAY_VARIANT_MANIFEST } from 'analytics/analytics';
import { PLAYBACK_ERROR } from 'analytics/spade-events';
import { init as initStore } from 'state';
import { unitTest } from 'tests/utils/module';
import { usherHost as USHER_HOST } from 'settings';
import { waitFor } from 'tests/utils/waitFor';
import { resetPlaySession } from 'actions/analytics';
import { setWindow } from 'actions/window';
import { setManifestInfo } from 'actions/manifest-info';
import round from 'lodash/round';
import forEach from 'lodash/forEach';
import size from 'lodash/size';
import { setAnalyticsTracker, ACTION_TRACK_EVENT } from 'actions/analytics-tracker';
import { buildFakeWindow } from 'tests/fakes/window.fake';
import PlayerCoreLoader from 'tests/fakes/player-core-loader.fake';
import Errors from 'errors';
import { ACTION_ERROR } from 'actions/error';

function setStream(store, stream) {
    store.dispatch({
        type: ACTION_SET_STREAM,
        stream,
    });
}

// eslint-disable-next-line max-statements
unitTest('backend | player-core', function(hooks) {
    hooks.beforeEach(function() {
        this.options = {
            autoplay: true,
        };

        this.store = initStore();
        this.createExperiment = options => {
            return {
                get: uuid => {
                    switch (uuid) {
                    case PLAYER_CORE_VER_CONTROL:
                        return Promise.resolve(options.playerCore);
                    }
                },
            };
        };

        this.store.dispatch({
            type: ACTION_SET_EXPERIMENTS,
            experiments: this.createExperiment({
                playerCore: '1.0',
                vpQOS: 'no',
            }),
        });

        this.backend = new PlayerCore.BackendPlayerCore(this.options, this.store);

        this.oauthToken = `oauth_${QUnit.config.current.testId}`;
        this.usherParams = {
            godlike: true,
        };
        this.accessTokenParams = {
            // eslint-disable-next-line camelcase
            need_https: false,
            adblock: true,
            platform: 'web-test',
            playerType: PlayerType.PLAYER_SITE,
        };
        this.experimentSettings = {
            transcode: Promise.resolve('regular'),
        };
        this.api.setLoggedIn(true);
        this.api.expectUserInfo();

        sinon.spy(PlayerCoreLoader, 'load');
    });

    hooks.afterEach(function() {
        this.backend.destroy();
        // Restore because PlayerCoreLoader is a singleton object.
        PlayerCoreLoader.load.restore();
    });

    QUnit.test('can set playsinline attribute on video element', function(assert) {
        const options = {
            playsinline: true,
        };
        const backend = new PlayerCore.BackendPlayerCore(options, this.store);
        const $qunitFixture = window.document.getElementById('qunit-fixture');
        backend.attach($qunitFixture);
        assert.ok($qunitFixture.querySelector('video').hasAttribute('webkit-playsinline'));
        assert.ok($qunitFixture.querySelector('video').hasAttribute('playsinline'));
    });

    QUnit.test('can init PlayerCore', function(assert) {
        assert.expect(0);
        this.backend.loadPlayerCore();
        return waitFor(() => this.backend.core);
    });

    QUnit.module('_initPlayerCore', function(hooks) {
        hooks.beforeEach(function() {
            this.playerCore = new PlayerCore.BackendPlayerCore({}, this.store);
            this.fakeCore = function() {};
            this.fakeCore.Event = {};
            this.fakeCore.prototype.addEventListener = sinon.spy();
            this.fakeCore.prototype.attachMedia = sinon.spy();
        });

        QUnit.test(`_initPlayerCore should emit ${TwitchEvents.PLAYER_INIT} once`, function(assert) {
            sinon.spy(this.playerCore.events, 'emit');
            this.playerCore._initPlayerCore(this.fakeCore);
            assert.ok(this.playerCore.events.emit.calledOnce);
            assert.ok(this.playerCore.events.emit.calledWith(TwitchEvents.PLAYER_INIT));
        });
    });

    QUnit.module('before initialize', function() {
        QUnit.test('`getVersion` returns empty string', function(assert) {
            assert.equal(this.backend.getVersion(), '');
        });
    });

    QUnit.module('after initialize', function(hooks) {
        hooks.beforeEach(function() {
            this.makeDelayedStream = (contentType, contentId, streamUrl, ms) => {
                const delay = ms => new Promise(resolve => setTimeout(resolve, ms));
                return {
                    contentType,
                    channel: contentType === CONTENT_MODE_LIVE ? contentId : null,
                    videoId: contentType === CONTENT_MODE_VOD ? contentId : null,
                    streamUrl: delay(ms).
                        then(() => streamUrl),
                };
            };
        });
        QUnit.test('is listening to expected events', function(assert) {
            this.backend.loadPlayerCore();
            return waitFor(() => this.backend.core).
                then(() => {
                    const spy = this.backend.core.addEventListener;
                    assert.equal(spy.args.length, size(FakePlayerCore.Event));
                    forEach(FakePlayerCore.Event, eventType => {
                        assert.ok(spy.withArgs(eventType).calledOnce, `should call ${eventType} once`);
                    });
                });
        });

        QUnit.test('if muted is true in options, sets muted attribute on video element', function(assert) {
            const backend = new PlayerCore.BackendPlayerCore({ muted: true }, this.store);
            assert.ok(backend.video.hasAttribute('muted'));
        });

        QUnit.test('if autoplay is true, call .load', function(assert) {
            this.backend.loadPlayerCore();
            sinon.spy(this.backend, 'load');
            return waitFor(() => this.backend.core).
                then(() => {
                    assert.ok(this.backend.load.called);
                });
        });

        QUnit.test('can get backend ID', function(assert) {
            assert.equal(this.backend.getBackend(), PlayerCore.BACKEND_PLAYER_CORE);
        });

        QUnit.test('can get ready state', function(assert) {
            assert.equal(this.backend.getReadyState(), ReadyState.HAVE_NOTHING);
        });

        QUnit.test('can set source', function(assert) {
            const sourceTest = '1234567890';
            this.backend.setSrc(sourceTest);
            assert.equal(this.backend.getSrc(), sourceTest);
        });

        QUnit.test('setChannel only calls setSrc with requested stream', function(assert) {
            sinon.spy(this.backend, 'setSrc');

            const FIRST_CHANNEL = 'firstchannel';
            const FIRST_STREAM_URL = `streamUrl_${FIRST_CHANNEL}`;
            const SECOND_CHANNEL = 'secondchannel';
            const SECOND_STREAM_URL = `streamUrl_${SECOND_CHANNEL}`;

            const firstStream = this.makeDelayedStream(CONTENT_MODE_LIVE, FIRST_CHANNEL, FIRST_STREAM_URL, 30);
            const secondStream = this.makeDelayedStream(CONTENT_MODE_LIVE, SECOND_CHANNEL, SECOND_STREAM_URL, 15);

            setStream(this.store, firstStream);
            this.backend.setChannel(FIRST_CHANNEL, firstStream);
            setStream(this.store, secondStream);
            this.backend.setChannel(SECOND_CHANNEL, secondStream);

            return waitFor(() => this.backend.setSrc.called).
                then(() => {
                    const [result] = this.backend.setSrc.args[0];
                    assert.equal(result, SECOND_STREAM_URL);
                });
        });

        QUnit.test('setVideo only calls setSrc with requested stream', function(assert) {
            sinon.spy(this.backend, 'setSrc');

            const FIRST_VOD_ID = 'firstvod';
            const FIRST_STREAM_URL = `streamUrl_${FIRST_VOD_ID}`;
            const SECOND_VOD_ID = 'secondvod';
            const SECOND_STREAM_URL = `streamUrl_${SECOND_VOD_ID}`;

            const firstStream = this.makeDelayedStream(CONTENT_MODE_VOD, FIRST_VOD_ID, FIRST_STREAM_URL, 30);
            const secondStream = this.makeDelayedStream(CONTENT_MODE_VOD, SECOND_VOD_ID, SECOND_STREAM_URL, 15);

            setStream(this.store, firstStream);
            this.backend.setVideo(FIRST_VOD_ID, firstStream);
            setStream(this.store, secondStream);
            this.backend.setVideo(SECOND_VOD_ID, secondStream);

            return waitFor(() => this.backend.setSrc.called).
                then(() => {
                    const [result] = this.backend.setSrc.args[0];
                    assert.equal(result, SECOND_STREAM_URL);
                });
        });

        QUnit.test('can set channel', function(assert) {
            const channel = 'monstercat';
            const oauthTokenPromise = Promise.resolve(this.oauthToken);
            // eslint-disable-next-line max-len
            const accessToken = this.api.expectChannelAccessToken(channel.toLowerCase(), {
                canAccess: true,
                oauthToken: this.oauthToken,
                accessTokenParams: this.accessTokenParams,
            });
            const stream = new LiveTwitchContentStream(
                channel,
                oauthTokenPromise,
                this.usherParams,
                this.accessTokenParams,
                this.experimentSettings
            );
            setStream(this.store, stream);

            sinon.spy(this.backend, 'setSrc');
            this.backend.loadPlayerCore();
            return waitFor(() => this.backend.core).
                then(() => {
                    this.backend.setChannel(channel, stream);
                    return oauthTokenPromise;
                }).then(() => {
                    return waitFor(() => this.backend.setSrc.called);
                }).then(() => {
                    const params = {
                        token: accessToken.token,
                        sig: accessToken.sig,
                        // eslint-disable-next-line camelcase
                        allow_source: true,
                        godlike: true,
                        /* eslint-disable camelcase */
                        fast_bread: false,
                        baking_bread: false,
                        baking_brownies: false,
                        baking_brownies_timeout: PREROLL_TIMEOUT,
                        /* eslint-disable camelcase */
                    };

                    // eslint-disable-next-line max-len
                    const usherUrl = `${USHER_HOST}/api/channel/hls/${channel}.m3u8?${stringify(params)}`;
                    assert.equal(this.backend.setSrc.firstCall.args[0], usherUrl);
                });
        });

        QUnit.test('can set video', function(assert) {
            const vodId = 'v1234567890';
            const oauthTokenPromise = Promise.resolve(this.oauthToken);
            const stream = new VODTwitchContentStream(
                vodId,
                oauthTokenPromise,
                this.usherParams,
                this.accessTokenParams,
                this.experimentSettings
            );
            setStream(this.store, stream);
            // eslint-disable-next-line max-len
            const accessToken = this.api.expectVodAccessToken(vodId.toLowerCase(), {
                canAccess: true,
                oauthToken: this.oauthToken,
                accessTokenParams: this.accessTokenParams,
            });

            sinon.spy(this.backend, 'setSrc');
            this.backend.loadPlayerCore();
            return waitFor(() => this.backend.core).
                then(() => {
                    this.backend.setVideo(vodId, stream);
                    return oauthTokenPromise;
                }).then(() => {
                    return waitFor(() => this.backend.setSrc.called);
                }).then(() => {
                    const params = {
                        nauth: accessToken.token,
                        nauthsig: accessToken.sig,
                        // eslint-disable-next-line camelcase
                        allow_source: true,
                        godlike: true,
                        /* eslint-disable camelcase */
                        fast_bread: false,
                        baking_bread: false,
                        baking_brownies: false,
                        baking_brownies_timeout: PREROLL_TIMEOUT,
                        /* eslint-disable camelcase */
                    };
                    // eslint-disable-next-line max-len
                    const usherUrl = `${USHER_HOST}/vod/${vodId.substring(1)}.m3u8?${stringify(params)}`;
                    assert.equal(this.backend.setSrc.firstCall.args[0], usherUrl);
                });
        });

        QUnit.module('after parsing the master manifest', function(hooks) {
            hooks.beforeEach(function() {
                this.backend.loadPlayerCore();
                /* eslint-disable camelcase */
                this.manifestInfo = {
                    ABS: 'false',
                    CLUSTER: 'sfo01',
                    'MANIFEST-CLUSTER': 'sfo01',
                    'MANIFEST-NODE': 'video-edge.sfo01',
                    'MANIFEST-NODE-TYPE': 'legacy',
                    NODE: 'video-edge.sfo01',
                    'SERVING-ID': 'abcd',
                    'STREAM-TIME': 5,
                    'USER-IP': '127.0.0.1',
                };

                this.normalizedManifestInfo = {
                    abs: 'false',
                    cluster: 'sfo01',
                    manifest_cluster: 'sfo01',
                    manifest_node: 'video-edge.sfo01',
                    manifest_node_type: 'legacy',
                    node: 'video-edge.sfo01',
                    serving_id: 'abcd',
                    stream_time: 5,
                    user_ip: '127.0.0.1',
                };
                /* eslint-enable camelcase */

                this.fakeTracker = {
                    trackEvent: sinon.spy(),
                };
                this.store.dispatch(setAnalyticsTracker(this.fakeTracker));

                // Reset playSessionStartTime for calculating
                sinon.stub(Date, 'now').onFirstCall().returns(0);
                this.store.dispatch(resetPlaySession());

                this.dateNow = parseInt(QUnit.config.current.testId, 36);
                Date.now.returns(this.dateNow);
            });

            hooks.afterEach(function() {
                Date.now.restore();
            });

            QUnit.test('send a `video_play_master_manifest` tracking event', function(assert) {
                return waitFor(() => this.backend.core).then(() => {
                    this.backend.core.emit(FakePlayerCore.Event.HLS_MASTER_PARSED, this.manifestInfo);
                    assert.ok(this.fakeTracker.trackEvent.calledWith(VIDEO_PLAY_MASTER_MANIFEST, {
                        // eslint-disable-next-line camelcase
                        time_since_load_start: this.dateNow,
                    }));
                });
            });

            QUnit.test('emit a manifest info event with normalized manifest', function(assert) {
                return waitFor(() => this.backend.core).then(() => {
                    const listener = sinon.spy();
                    this.backend.addEventListener(TwitchEvents.MANIFEST_EXTRA_INFO, listener);
                    this.backend.core.emit(FakePlayerCore.Event.HLS_MASTER_PARSED, this.manifestInfo);
                    assert.ok(listener.calledWith(this.normalizedManifestInfo));
                });
            });
        });

        QUnit.module('after parsing the variant manifest', function(hooks) {
            hooks.beforeEach(function() {
                this.backend.loadPlayerCore();
                this.variantInfo = {
                    duration: 30,
                    ended: false,
                    targetDuration: 5,
                    type: 'LIVE',
                };

                this.fakeTracker = {
                    trackEvent: sinon.spy(),
                };
                this.store.dispatch(setAnalyticsTracker(this.fakeTracker));

                sinon.stub(Date, 'now').onFirstCall().returns(0);
                this.store.dispatch(resetPlaySession());

                this.dateNow = parseInt(QUnit.config.current.testId, 36);
                Date.now.returns(this.dateNow);
            });

            hooks.afterEach(function() {
                Date.now.restore();
            });

            QUnit.test('send a `video_play_variant_manifest` tracking event after a variant manifest event',
                function(assert) {
                    return waitFor(() => this.backend.core).then(() => {
                        this.backend.core.emit(FakePlayerCore.Event.HLS_VARIANT_PARSED, this.variantInfo);
                        assert.deepEqual(
                            this.fakeTracker.trackEvent.firstCall.args,
                            [
                                VIDEO_PLAY_VARIANT_MANIFEST,
                                {
                                    // eslint-disable-next-line camelcase
                                    time_since_load_start: this.dateNow,
                                },
                            ],
                            'variant manifest should be emitted'
                        );
                    });
                }
            );

            QUnit.test('send a `video_play_variant_manifest` tracking event only once per master manifest event',
                function(assert) {
                    return waitFor(() => this.backend.core).then(() => {
                        this.backend.core.emit(FakePlayerCore.Event.HLS_VARIANT_PARSED, this.variantInfo);
                        assert.deepEqual(
                            this.fakeTracker.trackEvent.firstCall.args,
                            [
                                VIDEO_PLAY_VARIANT_MANIFEST,
                                {
                                    // eslint-disable-next-line camelcase
                                    time_since_load_start: this.dateNow,
                                },
                            ],
                            'first variant manifest should be emitted'
                        );
                        this.backend.core.emit(FakePlayerCore.Event.HLS_VARIANT_PARSED, this.variantInfo);
                        assert.equal(this.fakeTracker.trackEvent.callCount, 1, 'should only send event once');
                        this.backend.core.emit(FakePlayerCore.Event.HLS_MASTER_PARSED, {});
                        assert.equal(
                            this.fakeTracker.trackEvent.callCount,
                            2,
                            'master manifest is expected to emit an event'
                        );
                        assert.equal(
                            this.fakeTracker.trackEvent.secondCall.args[0],
                            VIDEO_PLAY_MASTER_MANIFEST,
                            'master manifest event should be emitted'
                        );
                        this.backend.core.emit(FakePlayerCore.Event.HLS_VARIANT_PARSED, this.variantInfo);
                        assert.equal(
                            this.fakeTracker.trackEvent.callCount,
                            3,
                            'should send new event after new master manifest event'
                        );
                        assert.deepEqual(
                            this.fakeTracker.trackEvent.thirdCall.args,
                            [
                                VIDEO_PLAY_VARIANT_MANIFEST,
                                {
                                // eslint-disable-next-line camelcase
                                    time_since_load_start: this.dateNow,
                                },
                            ],
                            'should emit another variant manifest event after new master manifest event'
                        );
                    });
                }
            );
        });
    });

    QUnit.module('playback control and lookups', function() {
        QUnit.test('can change volume', function(assert) {
            const volume = 0.6;
            this.backend.setVolume(volume);
            assert.equal(this.backend.video.volume, volume);
        });

        QUnit.test('can mute and unmute volume', function(assert) {
            assert.equal(this.backend.video.muted, false);
            this.backend.setMuted(true);
            assert.equal(this.backend.video.muted, true);
            this.backend.setMuted(false);
            assert.equal(this.backend.video.muted, false);
        });

        QUnit.test('mute and unmute sets and unsets muted attribute on video tag', function(assert) {
            assert.notOk(this.backend.video.hasAttribute('muted'));
            this.backend.setMuted(true);
            assert.ok(this.backend.video.hasAttribute('muted'));
            this.backend.setMuted(false);
            assert.notOk(this.backend.video.hasAttribute('muted'));
        });

        QUnit.test('can lookup seeking status', function(assert) {
            assert.equal(this.backend.getSeeking(), false);
        });

        QUnit.test('can lookup buffered property', function(assert) {
            assert.equal(this.backend.getBuffered().length, 0);
        });
    });

    QUnit.test('when pausing a live stream, stop is called on player-core', function(assert) {
        setStream(this.store, new LiveTwitchContentStream(
            'monstercat',
            Promise.resolve('oauth'),
            {},
            {}
        ));
        this.backend.loadPlayerCore();
        return waitFor(() => this.backend.core).
            then(() => {
                sinon.spy(this.backend.core, 'stop');
                this.backend.pause();
                assert.ok(this.backend.core.stop.called);
            });
    });

    QUnit.test('when pausing a VOD, stop is not called on player-core', function(assert) {
        setStream(this.store, new VODTwitchContentStream(
            'monstercat',
            Promise.resolve(this.oauthToken),
            this.usherParams,
            this.accessTokenParams
        ));
        this.backend.loadPlayerCore();
        return waitFor(() => this.backend.core).
            then(() => {
                sinon.spy(this.backend.core, 'stop');
                this.backend.pause();
                assert.notOk(this.backend.core.stop.called);
            });
    });

    QUnit.test('when playing a live stream, call `setChannel` then video.play', function(assert) {
        const channel = 'monstercat';
        const stream = new LiveTwitchContentStream(
            channel,
            Promise.resolve(this.oauthToken),
            this.usherParams,
            this.accessTokenParams,
            this.experimentSettings
        );

        this.api.expectChannelAccessToken(channel.toLowerCase(), {
            canAccess: true,
            oauthToken: this.oauthToken,
            accessTokenParams: this.accessTokenParams,
        });

        setStream(this.store, stream);
        this.backend.loadPlayerCore();
        return waitFor(() => this.backend.core).
            then(() => {
                sinon.spy(this.backend, 'setChannel');
                sinon.spy(this.backend.video, 'play');
                this.backend.play();
                assert.notOk(this.backend.video.play.called);
                assert.ok(this.backend.setChannel.called);
                return waitFor(() => this.backend.video.play.called);
            });
    });

    QUnit.test('when playing a vod, call `video.play`, not setChannel', function(assert) {
        setStream(this.store, new VODTwitchContentStream(
            'monstercat',
            Promise.resolve(this.oauthToken),
            this.usherParams,
            this.accessTokenParams
        ));
        this.backend.loadPlayerCore();
        return waitFor(() => this.backend.core).
            then(() => {
                sinon.spy(this.backend, 'setChannel');
                sinon.spy(this.backend.video, 'play');
                this.backend.play();
                assert.notOk(this.backend.setChannel.called);
                assert.ok(this.backend.video.play.called);
            });
    });

    QUnit.module('onSegmentChanged', function() {
        QUnit.test('fires SEGMENT_CHANGE event if TOFN tag exists', function(assert) {
            const segmentName = 'thisisasegment.ts';
            const data = {
                ID3: [
                    {
                        id: 'TOFN',
                        info: [segmentName],
                    },
                ],
            };
            const handler = sinon.spy();

            this.backend.loadPlayerCore();
            return waitFor(() => this.backend.core).
                then(() => {
                    this.backend.addEventListener(TwitchEvents.SEGMENT_CHANGE, handler);
                    this.backend.core.emit(FakePlayerCore.Event.SEGMENT_CHANGED, data);
                    assert.deepEqual(handler.firstCall.args[0], { name: segmentName });
                });
        });

        QUnit.test('fires MIDROLL_REQUESTED event if commercial is requested', function(assert) {
            const info = {
                cmd: 'commercial',
                length: 60,
            };
            const data = {
                ID3: [
                    {
                        id: 'TXXX',
                        info: [JSON.stringify(info)],
                    },
                ],
            };
            const handler = sinon.spy();

            this.backend.loadPlayerCore();
            return waitFor(() => this.backend.core).
                then(() => {
                    this.backend.addEventListener(TwitchEvents.MIDROLL_REQUESTED, handler);
                    this.backend.core.emit(FakePlayerCore.Event.SEGMENT_CHANGED, data);
                    assert.deepEqual(handler.firstCall.args[0], { duration: info.length });
                });
        });

        QUnit.test('fires STITCHED_AD_START and STITCHED_AD_END for stitched ads', function(assert) {
            const startHandler = sinon.spy();
            const endHandler = sinon.spy();

            const data = {
                URL: 'test',
            };

            this.backend.loadPlayerCore();
            return waitFor(() => this.backend.core).
                then(() => {
                    this.backend.addEventListener(TwitchEvents.STITCHED_AD_START, startHandler);
                    this.backend.addEventListener(TwitchEvents.STITCHED_AD_END, endHandler);
                    this.backend.core.emit(FakePlayerCore.Event.SPLICEOUT, data);
                    this.backend.core.emit(FakePlayerCore.Event.SPLICEIN);
                    assert.ok(startHandler.calledOnce);
                    assert.ok(endHandler.calledOnce);
                    assert.deepEqual(startHandler.firstCall.args[0], data);
                });
        });
    });

    QUnit.module('on destroy', function() {
        QUnit.test('calls destroy on playercore', function(assert) {
            this.backend.loadPlayerCore();
            return waitFor(() => this.backend.core).
                then(() => {
                    sinon.spy(this.backend.core, 'pause');
                    sinon.spy(this.backend.core, 'destroy');
                    this.backend.destroy();
                    assert.ok(this.backend.core.pause.called);
                    assert.ok(this.backend.core.destroy.called);
                });
        });
    });

    QUnit.test('setQuality calls setQuality in player-core', function(assert) {
        const quality = 'medium';

        this.backend.loadPlayerCore();
        return waitFor(() => this.backend.core).
            then(() => {
                sinon.spy(this.backend.core, 'setQuality');
                this.backend.setQuality(quality);
                assert.ok(this.backend.core.setQuality.called);
                assert.equal(this.backend.core.setQuality.firstCall.args[0], quality);
            });
    });

    QUnit.test('calling `load` with autoplay calls video.play', function(assert) {
        this.backend.src = 'notarealsrc.m3u8';
        this.backend.loadPlayerCore();
        return waitFor(() => this.backend.core).
            then(() => {
                sinon.spy(this.backend.core, 'play');
                this.backend.load();
                assert.ok(this.backend.core.play.called);
            });
    });

    QUnit.test('when setting a channel, call video.play if autoplay is set', function(assert) {
        const channel = 'monstercat';
        const stream = new LiveTwitchContentStream(
            channel,
            Promise.resolve(this.oauthToken),
            this.usherParams,
            this.accessTokenParams,
            this.experimentSettings
        );
        setStream(this.store, stream);

        this.api.expectChannelAccessToken(channel.toLowerCase(), {
            canAccess: true,
            oauthToken: this.oauthToken,
            accessTokenParams: this.accessTokenParams,
        });

        this.backend.loadPlayerCore();
        return waitFor(() => this.backend.core).
            then(() => {
                sinon.spy(this.backend.core, 'play');
                assert.notOk(this.backend.core.play.called);
                this.backend.setChannel(channel, stream);
                return waitFor(() => this.backend.core.play.called);
            });
    });

    QUnit.test('when calling play with autoplay false, call video.play', function(assert) {
        const options =  {
            autoplay: false,
        };
        const channel = 'monstercat';
        const stream = new LiveTwitchContentStream(
            channel,
            Promise.resolve(this.oauthToken),
            this.usherParams,
            this.accessTokenParams,
            this.experimentSettings
        );
        this.api.expectChannelAccessToken(channel.toLowerCase(), {
            canAccess: true,
            oauthToken: this.oauthToken,
            accessTokenParams: this.accessTokenParams,
        });

        const backend = new PlayerCore.BackendPlayerCore(options, this.store);
        backend.loadPlayerCore();

        return waitFor(() => this.backend.core).
            then(() => {
                sinon.spy(backend.core, 'play');
                setStream(this.store, stream);
                assert.notOk(backend.core.play.called);
                backend.play();
                return waitFor(() => backend.core.play.calledOnce);
            });
    });

    QUnit.module('onVariantSwitchRequested', function() {
        QUnit.test('adds handler to player core\'s segment_changed event', function(assert) {
            const variantSwitchData = {
                previous: 'low',
                current: 'mobile',
                stats: JSON.stringify({
                    /* eslint-disable camelcase */
                    abs_version: '1.0',
                    buffer_level: 8945,
                    estimated_bandwidth: 13886,
                    current_stream_format_bitrate: 1760,
                    reason_code: 2,
                    reason: 'Normal',
                    /* eslint-enable camelcase */
                }),
            };
            this.backend.loadPlayerCore();
            return waitFor(() => this.backend.core).
                then(() => {
                    this.backend.core.addEventListener.reset();
                    this.backend.core.emit(FakePlayerCore.Event.VARIANT_SWITCH_REQUESTED, variantSwitchData);
                    assert.ok(this.backend.core.addEventListener.calledWith(
                        FakePlayerCore.Event.SEGMENT_CHANGED,
                        sinon.match.func
                    ));
                });
        });

        // eslint-disable-next-line max-len
        QUnit.test('If current Quality is auto, emit an abs_stream_format_change event', function(assert) {
            /* eslint-disable camelcase */
            const variantSwitchStats = {
                abs_version: '1.0',
                buffer_level: 8945,
                estimated_bandwidth: 13886,
                current_stream_format_bitrate: 1760,
                reason_code: 2,
                reason: 'Normal',
            };
            const variantSwitchData = {
                previous: 'low',
                current: 'mobile',
                stats: variantSwitchStats,
            };
            const expectedOutput = Object.assign({
                stream_format_current: 'mobile',
                stream_format_previous: 'low',
            }, variantSwitchStats);
            /* eslint-enable camelcase */

            this.backend.getQuality = sinon.stub().returns('auto');
            this.backend.loadPlayerCore();

            return waitFor(() => this.backend.core).
                then(() => {
                    const emitSpy = sinon.spy(this.backend.events, 'emit');
                    this.backend.core.emit(FakePlayerCore.Event.VARIANT_SWITCH_REQUESTED, variantSwitchData);
                    assert.deepEqual(emitSpy.args[0], [
                        TwitchEvents.ABS_STREAM_FORMAT_CHANGE,
                        expectedOutput,
                    ]);
                });
        });

        // eslint-disable-next-line max-len
        QUnit.test('fires quality change event and removes handler on segment with new variant', function(assert) {
            const variantSwitchData = {
                previous: 'low',
                current: 'mobile',
            };

            this.backend.loadPlayerCore();
            return waitFor(() => this.backend.core).
                then(() => {
                    this.backend.core.addEventListener.reset();
                    this.backend.core.emit(FakePlayerCore.Event.VARIANT_SWITCH_REQUESTED, variantSwitchData);
                    assert.ok(this.backend.core.addEventListener.calledWith(
                        FakePlayerCore.Event.SEGMENT_CHANGED,
                        sinon.match.func
                    ));

                    const eventData = {
                        ID3: [],
                        variant: variantSwitchData.current,
                    };
                    const handler = sinon.spy();

                    sinon.spy(this.backend.core, 'removeEventListener');
                    this.backend.addEventListener(TwitchEvents.QUALITY_CHANGE, handler);

                    this.backend.core.emit(FakePlayerCore.Event.SEGMENT_CHANGED, eventData);
                    assert.ok(this.backend.core.removeEventListener.calledWith(
                        FakePlayerCore.Event.SEGMENT_CHANGED,
                        sinon.match.func
                    ));
                    assert.deepEqual(handler.firstCall.args[0], {
                        quality: eventData.variant,
                        isAuto: false,
                    });
                });
        });

        // eslint-disable-next-line max-len
        QUnit.test('if switch requested again before first one completes, existing handler removed', function(assert) {
            const variantSwitchData1 = {
                previous: 'low',
                current: 'mobile',
            };
            const variantSwitchData2 = {
                previous: 'mobile',
                current: 'medium',
            };

            this.backend.loadPlayerCore();
            return waitFor(() => this.backend.core).
                then(() => {
                    this.backend.core.addEventListener.reset();
                    this.backend.core.emit(FakePlayerCore.Event.VARIANT_SWITCH_REQUESTED, variantSwitchData1);

                    assert.ok(this.backend.core.addEventListener.calledWith(
                        FakePlayerCore.Event.SEGMENT_CHANGED,
                        sinon.match.func
                    ));

                    sinon.spy(this.backend.core, 'removeEventListener');
                    this.backend.core.emit(FakePlayerCore.Event.VARIANT_SWITCH_REQUESTED, variantSwitchData2);
                    assert.ok(this.backend.core.removeEventListener.calledWith(
                        FakePlayerCore.Event.SEGMENT_CHANGED,
                        sinon.match.func
                    ));

                    const eventData = {
                        ID3: [],
                        variant: variantSwitchData1.current,
                    };
                    const handler = sinon.spy();

                    this.backend.addEventListener(TwitchEvents.QUALITY_CHANGE, handler);

                    this.backend.core.emit(FakePlayerCore.Event.SEGMENT_CHANGED, eventData);
                    assert.notOk(handler.called);
                });
        });
    });

    QUnit.module('when playercore fires a BUFFERING event', function() {
        QUnit.test('paused does not change', function(assert) {
            this.backend.loadPlayerCore();
            return waitFor(() => this.backend.core).
                then(() => {
                    const prevPaused = this.backend.getPaused();
                    this.backend.core.emit(FakePlayerCore.Event.BUFFERING);
                    assert.equal(this.backend.getPaused(), prevPaused);
                });
        });

        QUnit.test('WAITING is emitted', function(assert) {
            this.backend.loadPlayerCore();
            return waitFor(() => this.backend.core).
                then(() => {
                    const listener = sinon.spy();
                    this.backend.addEventListener(MediaEvents.WAITING, listener);
                    this.backend.core.emit(FakePlayerCore.Event.BUFFERING);
                    assert.ok(listener.called);
                });
        });
    });

    QUnit.module('when playercore fires a FATAL_ERROR event', function() {
        QUnit.test('dispatched the appropriate media error', function(assert) {
            sinon.spy(this.store, 'dispatch');
            this.backend.loadPlayerCore();
            return waitFor(() => this.backend.core).
                then(() => {
                    this.store.dispatch.reset();
                    this.backend.core.emit(FakePlayerCore.Event.FATAL_ERROR, {
                        code: 1,
                    });
                    assert.deepEqual(this.store.dispatch.secondCall.args[0], {
                        type: ACTION_ERROR,
                        code: Errors.CODES.ABORTED,
                    }, 'dispatched aborted error code');

                    this.store.dispatch.reset();
                    this.backend.core.emit(FakePlayerCore.Event.FATAL_ERROR, {
                        code: 2,
                    });
                    assert.deepEqual(this.store.dispatch.secondCall.args[0], {
                        type: ACTION_ERROR,
                        code: Errors.CODES.NETWORK,
                    }, 'dispatched network error code');

                    this.store.dispatch.reset();
                    this.backend.core.emit(FakePlayerCore.Event.FATAL_ERROR, {
                        code: 3,
                    });
                    assert.deepEqual(this.store.dispatch.secondCall.args[0], {
                        type: ACTION_ERROR,
                        code: Errors.CODES.DECODE,
                    }, 'dispatched decode error code');

                    this.store.dispatch.reset();
                    this.backend.core.emit(FakePlayerCore.Event.FATAL_ERROR, {
                        code: 4,
                    });
                    assert.deepEqual(this.store.dispatch.secondCall.args[0], {
                        type: ACTION_ERROR,
                        code: Errors.CODES.FORMAT_NOT_SUPPORTED,
                    }, 'dispatched format not supported error code');
                });
        });

        QUnit.test('Sends PLAYBACK_ERROR spade event', function(assert) {
            sinon.spy(this.store, 'dispatch');
            this.backend.loadPlayerCore();
            return waitFor(() => this.backend.core).
                then(() => {
                    const listener = sinon.spy();
                    this.backend.addEventListener(MediaEvents.ERROR, listener);

                    this.backend.core.emit(FakePlayerCore.Event.FATAL_ERROR, {
                        code: 4,
                    });

                    assert.equal(this.store.dispatch.callCount, 2);
                    assert.ok(this.store.dispatch.calledWith({
                        type: ACTION_TRACK_EVENT,
                        eventName: PLAYBACK_ERROR,
                        eventProperties: {
                            // eslint-disable-next-line camelcase
                            playback_error_code: PlayerCore.FATAL_ERROR_CODE,
                            // eslint-disable-next-line camelcase
                            playback_error_msg: PlayerCore.PLAYER_FATAL_ERROR,
                        },
                    }));
                });
        });
    });

    QUnit.module('when playercore fires an OFFLINE event', function() {
        QUnit.test('Sends PLAYBACK_ERROR spade event', function(assert) {
            sinon.spy(this.store, 'dispatch');
            this.backend.loadPlayerCore();
            return waitFor(() => this.backend.core).
                then(() => {
                    this.backend.core.emit(FakePlayerCore.Event.OFFLINE, {
                        code: 4,
                    });

                    assert.equal(this.store.dispatch.callCount, 1);
                    assert.ok(this.store.dispatch.calledWith({
                        type: ACTION_TRACK_EVENT,
                        eventName: PLAYBACK_ERROR,
                        eventProperties: {
                            // eslint-disable-next-line camelcase
                            playback_error_code: PlayerCore.OFFLINE_ERROR_CODE,
                            // eslint-disable-next-line camelcase
                            playback_error_msg: PlayerCore.PLAYER_OFFLINE_ERROR,
                        },
                    }));
                });
        });
    });

    QUnit.test('when getting stats, returns current player stats', function(assert) {
        this.backend.loadPlayerCore();
        return waitFor(() => this.backend.core).
            // eslint-disable-next-line max-statements
            then(() => {
                const videoStats = {
                    bufferSize: 1.1,
                    displayResolution: 'fakeres',
                    skippedFrames: 2,
                    fps: 3.0013,
                    hlsLatencyBroadcaster: 4124,
                    hlsLatencyEncoder: 5124,
                    memoryUsage: 64,
                    playbackRate: 7.12041,
                    videoResolution: 'fakeres',
                };
                const videoStats2 = {
                    bufferSize: 2,
                    displayResolution: 'fakerres',
                    skippedFrames: 3,
                    fps: undefined,
                    hlsLatencyBroadcaster: undefined,
                    hlsLatencyEncoder: undefined,
                    memoryUsage: 64,
                    playbackRate: undefined,
                    videoResolution: 'fakerres',
                };

                // Make sure rounding works
                this.backend.core._setPlaybackStatistics(videoStats);
                const roundedStats = this.backend.getStats();

                assert.equal(roundedStats.playbackRate, 7.12);
                assert.equal(roundedStats.fps, 3);
                assert.equal(roundedStats.bufferSize, 1);
                assert.equal(roundedStats.skippedFrames, 2);
                assert.equal(roundedStats.memoryUsage, '64 MB');
                assert.equal(roundedStats.hlsLatencyEncoder, 5);
                assert.equal(roundedStats.hlsLatencyBroadcaster, 4);
                assert.equal(roundedStats.videoResolution, 'fakeres');
                assert.equal(roundedStats.displayResolution, 'fakeres');

                // Make sure undefineds are handled gracefully
                this.backend.core._setPlaybackStatistics(videoStats2);
                const roundedStats2 = this.backend.getStats();

                assert.equal(roundedStats2.playbackRate, 0);
                assert.equal(roundedStats2.fps, 0);
                assert.equal(roundedStats2.bufferSize, 2);
                assert.equal(roundedStats2.skippedFrames, 3);
                assert.equal(roundedStats2.memoryUsage, '64 MB');
                assert.equal(roundedStats2.hlsLatencyEncoder, 0);
                assert.equal(roundedStats2.hlsLatencyBroadcaster, 0);
                assert.equal(roundedStats2.videoResolution, 'fakerres');
                assert.equal(roundedStats2.displayResolution, 'fakerres');
            });
    });

    QUnit.test('when getting video info, return expected object with video info', function(assert) {
        this.backend.loadPlayerCore();
        return waitFor(() => this.backend.core).
            then(() => {
                const coreInfo = {
                    broadcastId: '12345',
                    bufferSize: 1,
                    displayResolution: 'fakeres',
                    skippedFrames: 2,
                    fps: 3,
                    hlsLatencyBroadcaster: 4.2,
                    hlsLatencyEncoder: 5.2,
                    memoryUsage: 64,
                    playbackRate: 7,
                    targetDuration: 5,
                    videoResolution: 'fakeres',
                };
                const fakeWindow = buildFakeWindow({
                    location: {
                        protocol: 'https:',
                    },
                    clearInterval() {},
                    clearTimeout() {},
                });
                const quality = 'low';

                /* eslint-disable camelcase */
                const manifestInfo = {
                    cluster: 'sfo01',
                    manifest_cluster: 'sfo01',
                    manifest_node: 'video-edge.sfo01',
                    manifest_node_type: 'legacy',
                    node: 'video-edge.sfo01',
                    serving_id: 'abcd',
                    stream_time: 5,
                    user_ip: '127.0.0.1',
                    origin: 's3',
                    region: 'UNKNOWN',
                };

                const expectedVideoInfo = {
                    bandwidth: coreInfo.playbackRate,
                    broadcast_id: parseInt(coreInfo.broadcastId, 10),
                    cluster: manifestInfo.cluster,
                    current_bitrate: coreInfo.playbackRate,
                    current_fps: round(coreInfo.fps),
                    current_fps_exact: coreInfo.fps,
                    dropped_frames: coreInfo.skippedFrames,
                    hls_latency_broadcaster: round(coreInfo.hlsLatencyBroadcaster),
                    hls_latency_encoder: round(coreInfo.hlsLatencyEncoder),
                    hls_target_duration: coreInfo.targetDuration,
                    manifest_cluster: manifestInfo.manifest_cluster,
                    manifest_node: manifestInfo.manifest_node,
                    manifest_node_type: manifestInfo.manifest_node_type,
                    node: manifestInfo.node,
                    paused: true,
                    playing: false,
                    segment_protocol: 'https',
                    serving_id: manifestInfo.serving_id,
                    stream_time: this.backend.video.currentTime,
                    stream_time_offset: parseFloat(manifestInfo.stream_time),
                    totalMemoryNumber: coreInfo.memoryUsage,
                    vid_display_height: coreInfo.displayHeight,
                    vid_display_width: coreInfo.displayWidth,
                    vid_height: coreInfo.videoHeight,
                    vid_width: coreInfo.videoWidth,
                    video_buffer_size: coreInfo.bufferSize,
                    volume: 1,
                    user_ip: manifestInfo.user_ip,
                    vod_cdn_origin: manifestInfo.origin,
                    vod_cdn_region: manifestInfo.region,
                };
                /* eslint-enable camelcase */

                this.backend.core._setVideoInfo(coreInfo);
                this.backend.core.setQuality(quality);
                this.store.dispatch(setWindow(fakeWindow));
                this.store.dispatch(setManifestInfo(manifestInfo));
                assert.deepEqual(this.backend.getVideoInfo(), expectedVideoInfo);
            });
    });

    QUnit.test('getVideoInfo() returns http for `segment_protocol` if on http page', function(assert) {
        const fakeWindow = buildFakeWindow({
            location: {
                protocol: 'http:',
            },
            clearInterval() {},
            clearTimeout() {},
        });

        this.backend.loadPlayerCore();
        return waitFor(() => this.backend.core).
            then(() => {
                this.store.dispatch(setWindow(fakeWindow));
                assert.equal(this.backend.getVideoInfo().segment_protocol, 'http');
            });
    });

    QUnit.test('when CORE_ANALYTICS is fired, trackEvent should be fired', function(assert) {
        const payload = {
            spadeEventName: 'player-core-error',
            spadeEventData: {
                code: 2001,
                version: '0.5.4',
            },
        };
        const fakeTracker = {
            trackEvent: sinon.spy(),
        };
        this.store.dispatch(setAnalyticsTracker(fakeTracker));
        this.backend.loadPlayerCore();
        return waitFor(() => this.backend.core).
            then(() => {
                this.backend.core.emit(FakePlayerCore.Event.CORE_ANALYTICS, payload);
                assert.ok(fakeTracker.trackEvent.called);
                assert.deepEqual(fakeTracker.trackEvent.firstCall.args[0], payload.spadeEventName);
                assert.deepEqual(fakeTracker.trackEvent.firstCall.args[1], payload.spadeEventData);
            });
    });

    QUnit.test('when getting video info, return expected object with video info', function(assert) {
        this.backend.loadPlayerCore();
        return waitFor(() => this.backend.core).
            then(() => {
                const coreInfo = {
                    broadcastId: '12345',
                    bufferSize: 1,
                    displayResolution: 'fakeres',
                    skippedFrames: 2,
                    fps: 3,
                    hlsLatencyBroadcaster: 4.9999,
                    hlsLatencyEncoder: 5.010101,
                    memoryUsage: 64,
                    playbackRate: 7,
                    targetDuration: 5,
                    videoResolution: 'fakeres',
                };
                const fakeWindow = buildFakeWindow({
                    location: {
                        protocol: 'https:',
                    },
                    clearInterval() {},
                    clearTimeout() {},
                });
                const quality = 'low';

                /* eslint-disable camelcase */
                const manifestInfo = {
                    cluster: 'sfo01',
                    manifest_cluster: 'sfo01',
                    manifest_node: 'video-edge.sfo01',
                    manifest_node_type: 'legacy',
                    node: 'video-edge.sfo01',
                    serving_id: 'abcd',
                    stream_time: 5,
                    user_ip: '127.0.0.1',
                    origin: 's3',
                    region: 'UNKNOWN',
                };

                const expectedVideoInfo = {
                    bandwidth: coreInfo.playbackRate,
                    broadcast_id: parseInt(coreInfo.broadcastId, 10),
                    cluster: manifestInfo.cluster,
                    current_bitrate: coreInfo.playbackRate,
                    current_fps: round(coreInfo.fps),
                    current_fps_exact: coreInfo.fps,
                    dropped_frames: coreInfo.skippedFrames,
                    hls_latency_broadcaster: round(coreInfo.hlsLatencyBroadcaster),
                    hls_latency_encoder: round(coreInfo.hlsLatencyEncoder),
                    hls_target_duration: coreInfo.targetDuration,
                    manifest_cluster: manifestInfo.manifest_cluster,
                    manifest_node: manifestInfo.manifest_node,
                    manifest_node_type: manifestInfo.manifest_node_type,
                    node: manifestInfo.node,
                    paused: true,
                    playing: false,
                    segment_protocol: 'https',
                    serving_id: manifestInfo.serving_id,
                    stream_time: this.backend.video.currentTime,
                    stream_time_offset: manifestInfo.stream_time,
                    totalMemoryNumber: coreInfo.memoryUsage,
                    vid_display_height: coreInfo.displayHeight,
                    vid_display_width: coreInfo.displayWidth,
                    vid_height: coreInfo.videoHeight,
                    vid_width: coreInfo.videoWidth,
                    video_buffer_size: coreInfo.bufferSize,
                    volume: 1,
                    user_ip: manifestInfo.user_ip,
                    vod_cdn_origin: manifestInfo.origin,
                    vod_cdn_region: manifestInfo.region,
                };
                /* eslint-enable camelcase */

                this.backend.core._setVideoInfo(coreInfo);
                this.backend.core.setQuality(quality);
                this.store.dispatch(setWindow(fakeWindow));
                this.store.dispatch(setManifestInfo(manifestInfo));
                assert.deepEqual(this.backend.getVideoInfo(), expectedVideoInfo);
            });
    });

    QUnit.test('getVideoInfo() returns http for `segment_protocol` if on http page', function(assert) {
        const fakeWindow = buildFakeWindow({
            location: {
                protocol: 'http:',
            },
            clearInterval() {},
            clearTimeout() {},
        });

        this.backend.loadPlayerCore();
        return waitFor(() => this.backend.core).
            then(() => {
                this.store.dispatch(setWindow(fakeWindow));
                assert.equal(this.backend.getVideoInfo().segment_protocol, 'http');
            });
    });

    // eslint-disable-next-line max-len
    QUnit.test('when CAPTION is fired, current caption data should be saved and returned by getCaption', function(assert) {
        this.backend.loadPlayerCore();
        return waitFor(() => this.backend.core).
            then(() => {
                const fakeCaptions = {
                    fakeData: ['this', 'is', 'a bunch of', 'caption text'],
                };
                this.backend.core.emit(FakePlayerCore.Event.CAPTION, fakeCaptions);
                assert.deepEqual(this.backend.currentCaptionData, fakeCaptions);
                assert.deepEqual(this.backend.getCaption(), fakeCaptions);
            });
    });

    // eslint-disable-next-line max-len
    QUnit.test('when there is an offline event, we consider the stream ended', function(assert) {
        this.backend.loadPlayerCore();
        return waitFor(() => this.backend.core).
            then(() => {
                const listener = sinon.spy();
                this.backend.addEventListener(MediaEvents.ENDED, listener);
                this.backend.core.emit(FakePlayerCore.Event.OFFLINE);
                assert.ok(this.backend.getEnded() === true);
                assert.ok(listener.called);
            });
    });

    // eslint-disable-next-line max-len
    QUnit.test('calling paused() will return this.core.paused()', function(assert) {
        this.backend.loadPlayerCore();
        return waitFor(() => this.backend.core).
            then(() => {
                const result = this.backend.getPaused();
                assert.notEqual(result, undefined);
                assert.equal(result, this.backend.core.paused());
            });
    });

    // eslint-disable-next-line max-len
    QUnit.test('calling ended() will return this.core.ended()', function(assert) {
        this.backend.loadPlayerCore();
        return waitFor(() => this.backend.core).
            then(() => {
                const result = this.backend.getEnded();
                assert.notEqual(result, undefined);
                assert.equal(result, this.backend.core.ended());
            });
    });

    QUnit.test('does not add a listener on the video tag for WAITING event', function(assert) {
        const listener = sinon.spy();
        this.backend.addEventListener(MediaEvents.WAITING, listener);

        const waitingEvent = new Event(MediaEvents.WAITING);
        this.backend.video.dispatchEvent(waitingEvent);
        assert.notOk(listener.called);
    });

    QUnit.test('does add a listener on the backend for WAITING event', function(assert) {
        const listener = sinon.spy();
        this.backend.addEventListener(MediaEvents.WAITING, listener);
        this.backend.events.emit(MediaEvents.WAITING);
        assert.ok(listener.called);
    });

    QUnit.module('on player core auth error', function(hooks) {
        hooks.beforeEach(function() {
            this.stream = new LiveTwitchContentStream(
                'monstercat',
                Promise.resolve('oauth'),
                {},
                {},
                this.experimentSettings
            );
            setStream(this.store, this.stream);

            this.backend.loadPlayerCore();
            return waitFor(() => this.backend.core);
        });

        QUnit.test('NO-OP if stream is offline', function(assert) {
            sinon.stub(this.backend, '_retryStreamLoad');
            this.store.dispatch(setOnline(false));
            this.backend.core.emit(FakePlayerCore.Event.AUTH_ERROR);
            assert.equal(this.backend._retryStreamLoad.callCount, 0, 'should not retry player-core backend');
        });

        QUnit.test('attempts to retry stream load on auth error from player core', function(assert) {
            sinon.stub(this.backend, '_retryStreamLoad');
            this.backend.core.emit(FakePlayerCore.Event.AUTH_ERROR);
            assert.ok(this.backend._retryStreamLoad.called);
        });

        // eslint-disable-next-line max-len
        QUnit.test('dispatches error code after second auth error without successful stream reload', function(assert) {
            sinon.stub(this.backend, '_retryStreamLoad');
            sinon.spy(this.store, 'dispatch');

            this.backend.core.emit(FakePlayerCore.Event.AUTH_ERROR);
            this.store.dispatch.reset();
            this.backend.core.emit(FakePlayerCore.Event.AUTH_ERROR);
            assert.deepEqual(this.store.dispatch.firstCall.args[0], {
                type: ACTION_ERROR,
                code: Errors.CODES.CONTENT_NOT_AVAILABLE,
            }, 'dispatched content not available error code');
        });

        // eslint-disable-next-line max-len
        QUnit.test('retryStreamLoad calls destroy, initialize, resetNAuthToken, streamUrl, then setSrc', function(assert) {
            sinon.spy(this.backend, 'initialize');
            sinon.spy(this.backend, 'destroy');
            sinon.spy(this.backend, 'setSrc');
            sinon.stub(this.stream, 'resetNAuthToken');
            sinon.stub(this.stream, 'streamUrl', {
                get: () => {
                    return Promise.resolve('A URL');
                },
            });

            this.backend._retryStreamLoad();

            return waitFor(() => this.backend.setSrc.called).
                then(() => {
                    assert.ok(this.backend.destroy.calledOnce);
                    assert.ok(this.backend.initialize.calledAfter(this.backend.destroy));
                    assert.ok(this.backend.initialize.calledOnce);
                    assert.ok(this.stream.resetNAuthToken.calledAfter(this.backend.initialize));
                    assert.ok(this.backend.setSrc.calledAfter(this.stream.resetNAuthToken));
                });
        });
    });
});
