import sinon from 'sinon';
import { unitTest } from 'tests/utils/module';
import { BackendPlayerCore, BACKEND_PLAYER_CORE } from 'backend/player-core';
import { BackendMediaPlayer, BACKEND_MEDIA_PLAYER } from 'backend/mediaplayer';
import { BackendFlash, BACKEND_FLASH } from 'backend/flash';
import { BackendHls, BACKEND_HLS } from 'backend/hls';
import { BackendClip, BACKEND_CLIP } from 'backend/clip';
import { BackendVideoTag, BACKEND_VIDEO_TAG } from 'backend/video-tag';
import { getAvailableBackends } from 'backend/util';
import { ALL_PLAYER_TYPES, CLIPS_PLAYER_TYPES, PLAYER_AMAZON_LIVE,
         PLAYER_AMAZON_VSE } from 'util/player-type';
import difference from 'lodash/difference';

unitTest('backend | util', function() {
    QUnit.module('when all backends are supported by the browser', function(hooks) {
        hooks.beforeEach(function() {
            sinon.stub(BackendPlayerCore, 'canPlay').returns(true);
            sinon.stub(BackendMediaPlayer, 'canPlay').returns(true);
            sinon.stub(BackendFlash, 'canPlay').returns(true);
            sinon.stub(BackendHls, 'canPlay').returns(true);
            sinon.stub(BackendClip, 'canPlay').returns(true);
        });

        hooks.afterEach(function() {
            BackendPlayerCore.canPlay.restore();
            BackendMediaPlayer.canPlay.restore();
            BackendFlash.canPlay.restore();
            BackendHls.canPlay.restore();
            BackendClip.canPlay.restore();
        });

        // eslint-disable-next-line max-len
        QUnit.test('and player type is a clips player type, availableBackends returns clips backend only', function(assert) {
            CLIPS_PLAYER_TYPES.forEach(playerType => {
                assert.deepEqual(
                    getAvailableBackends(playerType),
                    [BACKEND_CLIP],
                    `for playerType: ${playerType}, only clip backend is available`
                );
            });
        });

        // eslint-disable-next-line max-len
        QUnit.test('and player type is amazon live, availableBackends only returns, mediaplayer, player core, hls backends in order', function(assert) {
            const expectedAvailableBackends = [
                BACKEND_MEDIA_PLAYER,
                BACKEND_PLAYER_CORE,
                BACKEND_HLS,
            ];
            assert.deepEqual(
                getAvailableBackends(PLAYER_AMAZON_LIVE),
                expectedAvailableBackends,
                'for amazon live playerType, only mediaplayer, playercore, hls backends is available'
            );
        });

        // eslint-disable-next-line max-len
        QUnit.test('and player type is amazon vse, availableBackends only returns video tag backend', function(assert) {
            const expectedAvailableBackends = [
                BACKEND_VIDEO_TAG,
            ];
            assert.deepEqual(
                getAvailableBackends(PLAYER_AMAZON_VSE),
                expectedAvailableBackends,
                'for amazon vse playerType, only video tag backend'
            );
        });

        // eslint-disable-next-line max-len
        QUnit.test('and player is any other type, availableBackends returns all backends except for clips/video-tag in order', function(assert) {
            const expectedAvailableBackends = [
                BACKEND_MEDIA_PLAYER,
                BACKEND_PLAYER_CORE,
                BACKEND_HLS,
                BACKEND_FLASH,
            ];

            const otherPlayerTypes = difference(
                ALL_PLAYER_TYPES,
                CLIPS_PLAYER_TYPES,
                [PLAYER_AMAZON_LIVE, PLAYER_AMAZON_VSE]
            );

            otherPlayerTypes.forEach(playerType => {
                assert.deepEqual(
                    getAvailableBackends(playerType),
                    expectedAvailableBackends,
                    `for ${playerType} playerType, mediaplayer, player core, hls, flash are available in that order`
                );
            });
        });

        // eslint-disable-next-line max-len
        QUnit.test('and player type is undefined, availableBackends returns all backends except for clips', function(assert) {
            const expectedAvailableBackends = [
                BACKEND_MEDIA_PLAYER,
                BACKEND_PLAYER_CORE,
                BACKEND_HLS,
                BACKEND_FLASH,
            ];

            assert.deepEqual(
                getAvailableBackends(),
                expectedAvailableBackends,
                'for undefined playerType, mediaplayer, player core, hls, flash are available in that order'
            );
        });
    });

    // eslint-disable-next-line max-len
    QUnit.module('when all backends (except video tag based backends) are not supported by the browser', function(hooks) {
        hooks.beforeEach(function() {
            sinon.stub(BackendPlayerCore, 'canPlay').returns(false);
            sinon.stub(BackendMediaPlayer, 'canPlay').returns(false);
            sinon.stub(BackendFlash, 'canPlay').returns(false);
            sinon.stub(BackendHls, 'canPlay').returns(false);
            sinon.stub(BackendClip, 'canPlay').returns(true); // clips backend always supported
            sinon.stub(BackendVideoTag, 'canPlay').returns(true); // video-tag backend always supported
        });

        hooks.afterEach(function() {
            BackendPlayerCore.canPlay.restore();
            BackendMediaPlayer.canPlay.restore();
            BackendFlash.canPlay.restore();
            BackendHls.canPlay.restore();
            BackendClip.canPlay.restore();
            BackendVideoTag.canPlay.restore();
        });

        // eslint-disable-next-line max-len
        QUnit.test('and player is not clips/amazon_vse type, getAvailableBackends returns empty array', function(assert) {
            const notClipsPlayerTypes = difference(ALL_PLAYER_TYPES, CLIPS_PLAYER_TYPES, [PLAYER_AMAZON_VSE]);
            notClipsPlayerTypes.forEach(playerType => {
                assert.deepEqual(
                    getAvailableBackends(playerType),
                    [],
                    `for ${playerType} playerType, returns empty array`
                );
            });
        });

        QUnit.test('and player type is undefined getAvailableBackends returns empty array', function(assert) {
            assert.deepEqual(
                getAvailableBackends(),
                [],
                'for undefined playerType, returns empty array'
            );
        });
    });

    QUnit.module('when some backend are supported by the browser', function(hooks) {
        hooks.beforeEach(function() {
            sinon.stub(BackendPlayerCore, 'canPlay').returns(false);
            sinon.stub(BackendMediaPlayer, 'canPlay').returns(true);
            sinon.stub(BackendFlash, 'canPlay').returns(true);
            sinon.stub(BackendHls, 'canPlay').returns(false);
            sinon.stub(BackendClip, 'canPlay').returns(true); // clips backend always supported
            sinon.stub(BackendVideoTag, 'canPlay').returns(true); // video tag backend always supported
        });

        hooks.afterEach(function() {
            BackendPlayerCore.canPlay.restore();
            BackendMediaPlayer.canPlay.restore();
            BackendFlash.canPlay.restore();
            BackendHls.canPlay.restore();
            BackendClip.canPlay.restore();
            BackendVideoTag.canPlay.restore();
        });

        // eslint-disable-next-line max-len
        QUnit.test('and player is amazon live type, getAvailableBackends return what is available out of hls, mediaplayer, playercore', function(assert) {
            assert.deepEqual(
                getAvailableBackends(PLAYER_AMAZON_LIVE),
                [BACKEND_MEDIA_PLAYER],
                'for amazon live playerType, returns mediaplayer'
            );
        });

        // eslint-disable-next-line max-len
        QUnit.test('and player is not clips/amazon live/amazon vse, getAvailableBackends returns what is available', function(assert) {
            const otherPlayerTypes = difference(
                ALL_PLAYER_TYPES,
                CLIPS_PLAYER_TYPES,
                [PLAYER_AMAZON_LIVE, PLAYER_AMAZON_VSE]
            );
            otherPlayerTypes.forEach(playerType => {
                assert.deepEqual(
                    getAvailableBackends(playerType),
                    [BACKEND_MEDIA_PLAYER, BACKEND_FLASH],
                    `for ${playerType} playerType, returns mediaplayer and flash in order`
                );
            });
        });

        QUnit.test('and player type is undefined getAvailableBackends returns what is available', function(assert) {
            assert.deepEqual(
                getAvailableBackends(),
                [BACKEND_MEDIA_PLAYER, BACKEND_FLASH],
                'for undefined playerType, returns mediaplayer and flash in order'
            );
        });
    });
});
