import {
    ClipGenerator,
    getClipsUrl,
    PRODUCTION_CLIPS_URL,
    STAGING_CLIPS_URL,
    ALPHA_CLIPS_URL,
    CLIPS_URL_COOKIE_KEY,
} from 'clip-generator';
import { ACTION_SET_STREAMMETADATA } from 'actions/stream-metadata';
import { ACTION_SET_MANIFEST_INFO } from 'actions/manifest-info';
import { ACTION_SET_STREAM } from 'actions/stream';
import { ACTION_SET_WINDOW } from 'actions/window';
import { ACTION_RESET_PLAY_SESSION } from 'actions/analytics';
import { init as initStore } from 'state';
import { buildFakeWindow } from 'tests/fakes/window.fake';
import sinon from 'sinon';
import { ACTION_SET_BACKEND } from 'actions/backend';
import { ACTION_SET_PLAYER_BACKEND_TYPE } from 'actions/backend-info';
import { cookie } from 'cookie_js';
import { BACKEND_MEDIA_PLAYER } from 'backend/mediaplayer';
import { BACKEND_PLAYER_CORE } from 'backend/player-core';

function setFakeWindowWithOrigin(context, origin) {
    const fakeWindow = buildFakeWindow({
        location: {
            origin,
        },
        open: context.windowOpenSpy,
    });
    context.setWindow(context.store, fakeWindow);
}

QUnit.module('clip-generator', function(hooks) {
    const roundedOffsetValue = 4322;
    const nonPlayerCoreRoundedOffsetValue = 4384;
    const currentChannelValue = 'twitchmedia';
    const streamTimeValue = 4321.54;
    const currentTimeValue = 70;
    const currentBroadcastIdValue = '12345';
    const playerBackendTypeValue = 'player-core';
    const playSessionIdValue = 'sessionId';

    /* eslint-disable max-statements */
    hooks.beforeEach(function() {
        this.store = initStore();
        this.state = {
            currentTime: currentTimeValue,
        };

        var template = `
            <form class="js-create-clip-form" method="post" action="" target="">
                <input class="js-create-clip-player-backend-type" type="hidden" name="player_backend_type" value="" />
                <input class="js-create-clip-channel" type="hidden" name="channel" value="" />
                <input class="js-create-clip-offset" type="hidden" name="offset" value="" />
                <input class="js-create-clip-broadcast_id" type="hidden" name="broadcast_id" value="" />
                <input class="js-create-clip-vod_id" type="hidden" name="vod_id" value="" />
                <input class="js-create-clip-play_session_id" type="hidden" name="play_session_id" value="" />
            </form>
        `;

        this.root = window.document.getElementById('qunit-fixture');

        this.root.insertAdjacentHTML('beforeend', template);

        this.windowOpenSpy = sinon.spy();

        const streamMetadata = {
            /* eslint-disable camelcase */
            broadcastID: currentBroadcastIdValue,
            channel: {
                displayName: '',
                name: currentChannelValue,
                partner: true,
                status: '',
            },
            createdAt: '',
            game: '',
            preview: {
                template: '',
            },
            viewers: 0,
            /* eslint-enable camelcase */
        };

        const stream = {
            contentType: 'live',
        };

        this.store.dispatch({
            type: ACTION_SET_STREAM,
            stream,
        });

        this.store.dispatch({
            type: ACTION_SET_STREAMMETADATA,
            streamMetadata,
        });

        this.backend = {
            elapsedTime: () => {
                return streamTimeValue;
            },
            getBackend() {
                return playerBackendTypeValue;
            },
        };

        this.store.dispatch({
            type: ACTION_SET_BACKEND,
            backend: this.backend,
        });

        this.store.dispatch({
            type: ACTION_SET_PLAYER_BACKEND_TYPE,
            playerBackendType: playerBackendTypeValue,
        });

        const manifestInfo = {
            /* eslint-disable camelcase */
            abs: false,
            broadcast_id: 0,
            cluster: '',
            manifest_cluster: '',
            manifest_node: '',
            node: '',
            segment_protocol: '',
            server_time: 0,
            serving_id: '',
            stream_time: streamTimeValue,
            user_ip: '',
            /* eslint-enable camelcase */
        };
        this.store.dispatch({
            type: ACTION_SET_MANIFEST_INFO,
            info: manifestInfo,
        });

        this.store.dispatch({
            type: ACTION_RESET_PLAY_SESSION,
            playSessionId: playSessionIdValue,
            playSessionStartTime: Date.now(),
        });

        this.verifyInputAndSubmission = function(assert, offsetValue, backendType) {
            assert.equal(this.root.getElementsByClassName('js-create-clip-player-backend-type')[0].value, backendType);
            assert.equal(this.root.getElementsByClassName('js-create-clip-player-backend-type')[0].value, backendType);
            assert.equal(this.root.getElementsByClassName('js-create-clip-channel')[0].value, currentChannelValue);
            assert.equal(this.root.getElementsByClassName('js-create-clip-offset')[0].value, offsetValue);
            assert.equal(
                this.root.getElementsByClassName('js-create-clip-broadcast_id')[0].value,
                currentBroadcastIdValue
            );
            assert.equal(
                this.root.getElementsByClassName('js-create-clip-play_session_id')[0].value,
                playSessionIdValue
            );
            assert.ok(this.subject._submitForm.called, 'form submitted');
            assert.ok(this.windowOpenSpy.called);
        };

        this.setWindow = function(store, window) {
            this.store.dispatch({
                type: ACTION_SET_WINDOW,
                window,
            });
        };

        this.createSubjectWithOrigin = function(origin) {
            setFakeWindowWithOrigin(this, origin);
            this.subject = new ClipGenerator(this.state, this.store);
            sinon.stub(this.subject, '_submitForm');
        };
    });
    /* eslint-enable max-statements */

    QUnit.test('getClipsUrl should use production given the prod site', function(assert) {
        const customUrl = 'anyurlyouwantttt';
        cookie.set(CLIPS_URL_COOKIE_KEY, customUrl);
        assert.equal(getClipsUrl('https://www.twitch.tv'), PRODUCTION_CLIPS_URL);
        cookie.remove(CLIPS_URL_COOKIE_KEY);
    });

    QUnit.test('getClipsUrl should use alpha given the alpha site', function(assert) {
        const customUrl = 'anyurlyouwantttt';
        cookie.set(CLIPS_URL_COOKIE_KEY, customUrl);
        assert.equal(getClipsUrl('https://betaplayer.twitch.tv'), ALPHA_CLIPS_URL);
        cookie.remove(CLIPS_URL_COOKIE_KEY);
    });

    QUnit.test('getClipsUrl should use custom staging url when there is one given any other url', function(assert) {
        const customUrl = 'anyurlyouwantttt';
        cookie.set(CLIPS_URL_COOKIE_KEY, customUrl);
        assert.equal(getClipsUrl('other'), customUrl);
        cookie.remove(CLIPS_URL_COOKIE_KEY);
    });

    QUnit.test('getClipsUrl should use the default staging when there is no custom given other url', function(assert) {
        cookie.remove(CLIPS_URL_COOKIE_KEY);
        assert.equal(getClipsUrl('other'), STAGING_CLIPS_URL);
    });

    QUnit.test('recording a clip on www.twitch.tv, opens clips generation page on clips.twitch.tv', function(assert) {
        this.createSubjectWithOrigin('https://www.twitch.tv');
        this.subject.hookUpForm(this.root);
        this.subject.recordClip();

        this.verifyInputAndSubmission(assert, roundedOffsetValue, playerBackendTypeValue);
    });

    QUnit.test('recording a clip on player.twitch opens clip generation on clips.twitch.tv', function(assert) {
        this.createSubjectWithOrigin('https://player.twitch.tv');
        this.subject.hookUpForm(this.root);
        assert.equal(
            this.root.getElementsByClassName('js-create-clip-form')[0].getAttribute('action'),
            'https://clips.twitch.tv/clips'
        );
        this.subject.recordClip();

        this.verifyInputAndSubmission(assert, roundedOffsetValue, playerBackendTypeValue);
    });

    QUnit.test('recording a clip on staging, opens clip generation on staging clips', function(assert) {
        this.createSubjectWithOrigin('https://staging-ffa-41-www.dev.us-west2.twitch.tv');
        this.subject.hookUpForm(this.root);

        assert.equal(
            this.root.getElementsByClassName('js-create-clip-form')[0].getAttribute('action'),
            'https://clips-staging.twitch.tv/clips'
        );
        this.subject.recordClip();

        this.verifyInputAndSubmission(assert, roundedOffsetValue, playerBackendTypeValue);
    });

    QUnit.test('recording a clip anywhere else, opens the clips generation page on staging', function(assert) {
        this.createSubjectWithOrigin('https://staging-or-something-www.twitch.tv');
        this.subject.hookUpForm(this.root);
        assert.equal(
            this.root.getElementsByClassName('js-create-clip-form')[0].getAttribute('action'),
            'https://clips-staging.twitch.tv/clips'
        );
        this.subject.recordClip();

        this.verifyInputAndSubmission(assert, roundedOffsetValue, playerBackendTypeValue);
    });

    QUnit.test('recording a clip with a non player-core backend uses different offset', function(assert) {
        const otherBackendType = 'other backend type';
        const otherBackend = {
            elapsedTime: () => {
                return streamTimeValue;
            },
            getBackend() {
                return otherBackendType;
            },
        };

        this.store.dispatch({
            type: ACTION_SET_BACKEND,
            backend: otherBackend,
        });

        this.store.dispatch({
            type: ACTION_SET_PLAYER_BACKEND_TYPE,
            playerBackendType: otherBackendType,
        });

        this.createSubjectWithOrigin('https://staging-or-something-www.twitch.tv');
        this.subject.hookUpForm(this.root);
        this.subject.recordClip();

        this.verifyInputAndSubmission(assert, nonPlayerCoreRoundedOffsetValue, otherBackendType);
    });

    QUnit.test('recording a clip with player-core backend uses rounded stream time value ', function(assert) {
        const otherBackendType = BACKEND_PLAYER_CORE;
        const otherBackend = {
            elapsedTime: () => {
                return streamTimeValue;
            },
            getBackend() {
                return otherBackendType;
            },
        };

        this.store.dispatch({
            type: ACTION_SET_BACKEND,
            backend: otherBackend,
        });

        this.store.dispatch({
            type: ACTION_SET_PLAYER_BACKEND_TYPE,
            playerBackendType: otherBackendType,
        });

        this.createSubjectWithOrigin('https://staging-or-something-www.twitch.tv');
        this.subject.hookUpForm(this.root);
        this.subject.recordClip();

        this.verifyInputAndSubmission(assert, roundedOffsetValue, otherBackendType);
    });

    QUnit.test('recording a clip with mediaplayer backend uses rounded stream time value ', function(assert) {
        const otherBackendType = BACKEND_MEDIA_PLAYER;
        const otherBackend = {
            elapsedTime: () => {
                return streamTimeValue;
            },
            getBackend() {
                return otherBackendType;
            },
        };

        this.store.dispatch({
            type: ACTION_SET_BACKEND,
            backend: otherBackend,
        });

        this.store.dispatch({
            type: ACTION_SET_PLAYER_BACKEND_TYPE,
            playerBackendType: otherBackendType,
        });

        this.createSubjectWithOrigin('https://staging-or-something-www.twitch.tv');
        this.subject.hookUpForm(this.root);
        this.subject.recordClip();

        this.verifyInputAndSubmission(assert, roundedOffsetValue, otherBackendType);
    });
});
