import assign from 'lodash/assign';
import { init as initStore } from 'state';
import { CollectionManager } from 'collection-manager';
import * as PlaybackActions from 'actions/playback';
import * as CollectionActions from 'actions/collection';
import * as ScreenActions from 'actions/screen';
import * as StreamActions from 'actions/stream';
import * as UiActions from 'actions/ui';
import { setWindow } from 'actions/window';
import { ACTION_VIDEO_API_PAUSE } from 'actions/video-api';
import { VODTwitchContentStream } from 'stream/twitch-vod';

import sinon from 'sinon';
import { unitTest } from 'tests/utils/module';
import { waitFor } from 'tests/utils/waitFor';
import { buildFakeWindow } from 'tests/fakes/window.fake';
import { TEST_COLLECTION, TEST_NORMALIZED_COLLECTION } from 'tests/fixtures/collection';

// fake selectCollectionVideo action
const FAKE_SELECT_COLLECTION_VIDEO_ACTION = 'fake select collection video';

unitTest('collection-manager', function(hooks) {
    hooks.beforeEach(function() {
        this.api.setLoggedIn(true);

        this.store = initStore();
        sinon.spy(this.store, 'dispatch');

        this.collectionManager = new CollectionManager(this.store);

        this.collection = TEST_COLLECTION;
        this.api.setLoggedIn(false);
        this.api.expectCollectionMetadata(this.collection._id);
        this.api.expectCollectionItems(this.collection._id);

        this.fakeWindow = buildFakeWindow();
        this.store.dispatch(setWindow(this.fakeWindow));

        sinon.stub(PlaybackActions, 'selectCollectionVideo', (contentId, collectionId) => dispatch => {
            dispatch({
                type: FAKE_SELECT_COLLECTION_VIDEO_ACTION,
                contentId,
                collectionId,
            });
        });

        sinon.stub(PlaybackActions, 'setStartTime', startTime => {
            return {
                type: PlaybackActions.ACTION_SET_START_TIME,
                startTime,
            };
        });

        sinon.stub(StreamActions, 'setStream', ({ contentType, contentId }) => dispatch => {
            dispatch({
                type: StreamActions.ACTION_SET_STREAM,
                stream: {
                    type: contentType,
                    contentId,
                },
            });
        });
    });

    hooks.afterEach(function() {
        this.store.dispatch.reset();
        PlaybackActions.selectCollectionVideo.restore();
        PlaybackActions.setStartTime.restore();
        StreamActions.setStream.restore();
    });

    QUnit.module('_fetchCollectionInfo', function() {
        QUnit.test('returns a pre-normalized collection object with items', function(assert) {
            return this.collectionManager._fetchCollectionInfo(this.collection._id).then(collectionData => {
                assert.deepEqual(
                    collectionData,
                    this.collection,
                    'promise resolves with collection data including items'
                );
            });
        });
    });

    QUnit.module('_collectionHasVideo', function() {
        QUnit.test('returns true if video is in collection', function(assert) {
            const collectionItems = this.collection.items;
            const _collectionHasVideo = this.collectionManager._collectionHasVideo;
            assert.equal(
                _collectionHasVideo(collectionItems, `v${collectionItems[0].item_id}`),
                true,
                'true for 1st item'
            );
            assert.equal(
                _collectionHasVideo(collectionItems, `v${collectionItems[1].item_id}`),
                true,
                'true for 2nd item'
            );
            assert.equal(
                _collectionHasVideo(collectionItems, `v${collectionItems[2].item_id}`),
                true,
                'true for 3rd item'
            );
        });

        QUnit.test('returns false if video is not in collection', function(assert) {
            return this.collectionManager._fetchCollectionInfo(this.collection._id).then(collectionData => {
                assert.deepEqual(
                    collectionData,
                    this.collection,
                    'promise resolves with collection data including items'
                );
            });
        });
    });

    QUnit.module('when collection is activated', function() {
        QUnit.module('with an empty collection', function(hooks) {
            hooks.beforeEach(function() {
                this.collection = assign({}, TEST_COLLECTION, {
                    items: [],
                });
                this.api.clearFakeResponses();
                this.api.setLoggedIn(true);
                this.api.expectCollectionMetadata(this.collection._id);
                this.api.expectCollectionItems(this.collection._id, []);
                this.store.dispatch(CollectionActions.requestCollection(this.collection._id));
                this.store.dispatch.reset();
            });

            QUnit.test('empty collection screen is pushed to the screen stack', function(assert) {
                return waitFor(() => this.store.dispatch.callCount === 3).then(() => {
                    const [pushScreenAction] = this.store.dispatch.secondCall.args;
                    const [pauseAction] = this.store.dispatch.thirdCall.args;
                    assert.equal(
                        pushScreenAction.type,
                        ScreenActions.ACTION_PUSH_SCREEN,
                        'push screen action is dispatched'
                    );
                    assert.equal(
                        pushScreenAction.screen,
                        ScreenActions.COLLECTION_EMPTY_SCREEN,
                        'empty collection screen was pushed'
                    );
                    assert.deepEqual(
                        pauseAction,
                        {
                            type: ACTION_VIDEO_API_PAUSE,
                        },
                        'content is paused'
                    );
                });
            });
        });

        QUnit.module('with only a collection', function(hooks) {
            hooks.beforeEach(function() {
                this.store.dispatch(CollectionActions.requestCollection(this.collection._id));
                this.store.dispatch.reset();
            });

            QUnit.test('selects vod as first video and sets collection info', function(assert) {
                return waitFor(() => this.store.dispatch.callCount === 5).then(() => {
                    const [setCollectionThunk] = this.store.dispatch.firstCall.args;
                    const dispatch = sinon.spy();
                    setCollectionThunk(dispatch);
                    const [setCollectionAction] = dispatch.secondCall.args;
                    assert.equal(
                        setCollectionAction.type,
                        CollectionActions.ACTION_SET_COLLECTION,
                        'first dispatch is type set collection'
                    );
                    assert.deepEqual(
                        setCollectionAction.collection,
                        TEST_NORMALIZED_COLLECTION,
                        'set collection called with normalized collection'
                    );

                    const [loadedCollectionItemAction] = this.store.dispatch.secondCall.args;
                    assert.equal(
                        loadedCollectionItemAction.type,
                        CollectionActions.ACTION_LOADED_COLLECTION_ITEM,
                        'loaded collection item action dispatched'
                    );

                    const [setStreamThunk] = this.store.dispatch.thirdCall.args;
                    dispatch.reset();
                    setStreamThunk(dispatch);
                    const [setStreamAction] = dispatch.firstCall.args;
                    assert.equal(
                        setStreamAction.type,
                        StreamActions.ACTION_SET_STREAM,
                        'first dispatch is set stream'
                    );
                    assert.deepEqual(
                        setStreamAction.stream,
                        {
                            type: StreamActions.TYPE_VIDEO,
                            contentId: `v${this.collection.items[0].item_id}`,
                        },
                        'sets the first video in the collection'
                    );
                });
            });
        });

        QUnit.module('with collection and matching video param', function(hooks) {
            hooks.beforeEach(function() {
                this.videoId = `v${this.collection.items[1].item_id}`;
                this.store.dispatch(CollectionActions.requestCollection(this.collection._id, this.videoId));
                this.store.dispatch.reset();
            });

            QUnit.test('selects vod as provided video and sets collection info', function(assert) {
                return waitFor(() => this.store.dispatch.callCount === 5).then(() => {
                    const dispatch = sinon.spy();
                    const [setCollectionThunk] = this.store.dispatch.firstCall.args;
                    setCollectionThunk(dispatch);
                    const [setCollectionAction] = dispatch.secondCall.args;
                    assert.equal(
                        setCollectionAction.type,
                        CollectionActions.ACTION_SET_COLLECTION,
                        'first dispatch is type set collection'
                    );
                    assert.deepEqual(
                        setCollectionAction.collection,
                        TEST_NORMALIZED_COLLECTION,
                        'set collection called with normalized collection'
                    );

                    const [loadedCollectionItemAction] = this.store.dispatch.secondCall.args;
                    assert.equal(
                        loadedCollectionItemAction.type,
                        CollectionActions.ACTION_LOADED_COLLECTION_ITEM,
                        'second loaded collection item action dispatched'
                    );

                    const [setStreamThunk] = this.store.dispatch.thirdCall.args;
                    dispatch.reset();
                    setStreamThunk(dispatch);
                    const [setStreamAction] = dispatch.firstCall.args;
                    assert.equal(
                        setStreamAction.type,
                        StreamActions.ACTION_SET_STREAM,
                        'third dispatch is setStream action'
                    );
                    assert.equal(
                        setStreamAction.stream.type,
                        StreamActions.TYPE_VIDEO,
                        'stream type is video'
                    );
                    assert.equal(
                        setStreamAction.stream.contentId,
                        this.videoId,
                        'stream set matches passed-in video id'
                    );

                    const [startTimeAction] = this.store.dispatch.getCall(4).args;
                    assert.equal(
                        startTimeAction.type,
                        PlaybackActions.ACTION_SET_START_TIME,
                        'last dispatch is set start time action'
                    );
                    assert.equal(
                        startTimeAction.startTime,
                        '',
                        'empty string is passed in for the start time'
                    );
                });
            });
        });

        QUnit.module('with collection and non-matching video param', function() {
            QUnit.test('Ignores collection param, sets videoId if preferVideo is true', function(assert) {
                this.store.dispatch(CollectionActions.requestCollection(
                    this.collection._id,
                    'v123456',
                    '',
                    true
                ));
                this.store.dispatch.reset();

                return waitFor(() => this.store.dispatch.callCount === 2).then(() => {
                    const [setStreamThunk] = this.store.dispatch.firstCall.args;
                    const [startTimeAction] = this.store.dispatch.secondCall.args;
                    const dispatch = sinon.spy();
                    setStreamThunk(dispatch);
                    const [setStreamAction] = dispatch.firstCall.args;

                    assert.equal(
                        setStreamAction.type,
                        StreamActions.ACTION_SET_STREAM,
                        'first dispatch is setStream action'
                    );
                    assert.equal(
                        setStreamAction.stream.type,
                        StreamActions.TYPE_VIDEO,
                        'stream type is video'
                    );
                    assert.equal(
                        setStreamAction.stream.contentId,
                        'v123456',
                        'stream set matches passed-in video id'
                    );

                    assert.equal(
                        startTimeAction.type,
                        PlaybackActions.ACTION_SET_START_TIME,
                        'second dispatch is set start time action'
                    );
                    assert.equal(
                        startTimeAction.startTime,
                        '',
                        'empty string is passed in for the start time'
                    );
                });
            });

            QUnit.test('loads first video in collection if preferVideo is false', function(assert) {
                this.store.dispatch(CollectionActions.requestCollection(
                    this.collection._id,
                    'v123456',
                    '',
                    false
                ));
                this.store.dispatch.reset();

                return waitFor(() => this.store.dispatch.callCount === 5).then(() => {
                    const dispatch = sinon.spy();
                    const [setCollectionThunk] = this.store.dispatch.firstCall.args;
                    setCollectionThunk(dispatch);
                    const [setCollectionAction] = dispatch.secondCall.args;
                    assert.equal(
                        setCollectionAction.type,
                        CollectionActions.ACTION_SET_COLLECTION,
                        'first dispatch is type set collection'
                    );
                    assert.deepEqual(
                        setCollectionAction.collection,
                        TEST_NORMALIZED_COLLECTION,
                        'set collection called with normalized collection'
                    );

                    const [loadedCollectionItemAction] = this.store.dispatch.secondCall.args;
                    assert.equal(
                        loadedCollectionItemAction.type,
                        CollectionActions.ACTION_LOADED_COLLECTION_ITEM,
                        'loaded collection item action dispatched'
                    );

                    dispatch.reset();
                    const [setStreamThunk] = this.store.dispatch.thirdCall.args;
                    setStreamThunk(dispatch);
                    const [setStreamAction] = dispatch.firstCall.args;
                    assert.equal(
                        setStreamAction.type,
                        StreamActions.ACTION_SET_STREAM,
                        'first dispatch is set stream'
                    );
                    assert.deepEqual(
                        setStreamAction.stream,
                        {
                            type: StreamActions.TYPE_VIDEO,
                            contentId: `v${this.collection.items[0].item_id}`,
                        },
                        'sets the first video in the collection'
                    );
                });
            });
        });

        QUnit.module('with collection, matching video param, and timestamp', function(hooks) {
            hooks.beforeEach(function() {
                this.videoId = `v${this.collection.items[1].item_id}`;
                this.timestamp = 123;
                this.store.dispatch(CollectionActions.requestCollection(
                    this.collection._id,
                    this.videoId,
                    this.timestamp
                ));
                this.store.dispatch.reset();
            });

            QUnit.test('sets startTime, selects vod as provided video, sets collection info', function(assert) {
                return waitFor(() => this.store.dispatch.callCount === 5).then(() => {
                    const dispatch = sinon.spy();
                    const [setCollectionThunk] = this.store.dispatch.firstCall.args;
                    setCollectionThunk(dispatch);
                    const [setCollectionAction] = dispatch.secondCall.args;
                    assert.equal(
                        setCollectionAction.type,
                        CollectionActions.ACTION_SET_COLLECTION,
                        'first dispatch is type set collection'
                    );
                    assert.deepEqual(
                        setCollectionAction.collection,
                        TEST_NORMALIZED_COLLECTION,
                        'set collection called with normalized collection'
                    );

                    const [loadedCollectionItemAction] = this.store.dispatch.secondCall.args;
                    assert.equal(
                        loadedCollectionItemAction.type,
                        CollectionActions.ACTION_LOADED_COLLECTION_ITEM,
                        'loaded collection item action dispatched'
                    );

                    dispatch.reset();
                    const [setStreamThunk] = this.store.dispatch.thirdCall.args;
                    setStreamThunk(dispatch);
                    const [setStreamAction] = dispatch.firstCall.args;
                    assert.equal(
                        setStreamAction.type,
                        StreamActions.ACTION_SET_STREAM,
                        'third dispatch is setStream action'
                    );
                    assert.equal(
                        setStreamAction.stream.type,
                        StreamActions.TYPE_VIDEO,
                        'stream type is video'
                    );
                    assert.equal(
                        setStreamAction.stream.contentId,
                        this.videoId,
                        'stream set matches passed-in video id'
                    );

                    const [startTimeAction] = this.store.dispatch.getCall(4).args;
                    assert.equal(
                        startTimeAction.type,
                        PlaybackActions.ACTION_SET_START_TIME,
                        'last dispatch is set start time action'
                    );
                    assert.equal(
                        startTimeAction.startTime,
                        this.timestamp,
                        'timestamp is passed in for the start time'
                    );
                });
            });
        });
    });

    QUnit.module('when onStreamChange is triggered by stream or collection change', function() {
        QUnit.test('without a collection, does nothing', function(assert) {
            this.store.dispatch(CollectionActions.clearCollection());

            this.store.dispatch.reset();
            this.store.dispatch({
                type: StreamActions.ACTION_SET_STREAM,
                stream: new VODTwitchContentStream(`v${this.collection.items[0].item_id}`),
            });

            assert.equal(this.store.dispatch.callCount, 1, 'no dispatches after set stream are made');
        });

        QUnit.test('with an empty collection, does nothing', function(assert) {
            this.collection = assign({}, TEST_COLLECTION, {
                items: [],
            });

            this.store.dispatch.reset();
            this.store.dispatch(CollectionActions.setCollection(this.collection));

            assert.equal(this.store.dispatch.callCount, 1, 'no dispatches after set collection are made');
        });

        QUnit.test('with a collection, when the new stream is last in the collection', function(assert) {
            this.store.dispatch(CollectionActions.setCollection(this.collection));
            this.store.dispatch.reset();

            this.store.dispatch({
                type: StreamActions.ACTION_SET_STREAM,
                stream: new VODTwitchContentStream(`v${this.collection.items[2].item_id}`),
            });

            assert.equal(this.store.dispatch.callCount, 2, 'one dispatch is made after set stream');
            const [loadedLastCollectionItemAction] = this.store.dispatch.secondCall.args;
            assert.equal(
                loadedLastCollectionItemAction.type,
                CollectionActions.ACTION_LOADED_LAST_COLLECTION_ITEM,
                'dispatched action is to indicate loaded last collection item'
            );
        });

        QUnit.test('with a collection, when the new stream is not the last in the collection', function(assert) {
            this.store.dispatch(CollectionActions.setCollection(this.collection));
            this.store.dispatch.reset();

            this.store.dispatch({
                type: StreamActions.ACTION_SET_STREAM,
                stream: new VODTwitchContentStream(`v${this.collection.items[0].item_id}`),
            });

            assert.equal(this.store.dispatch.callCount, 2, 'one dispatch is made after set stream');
            const [loadedCollectionItemAction] = this.store.dispatch.secondCall.args;
            assert.equal(
                loadedCollectionItemAction.type,
                CollectionActions.ACTION_LOADED_COLLECTION_ITEM,
                'dispatched action is to indicate loaded collection item'
            );
        });
    });

    QUnit.module('when a video stream ends', function(hooks) {
        hooks.beforeEach(function() {
            this.store.dispatch({
                type: CollectionActions.ACTION_SET_COLLECTION,
                collection: TEST_NORMALIZED_COLLECTION,
            });

            this.store.dispatch({
                type: StreamActions.ACTION_SET_STREAM,
                stream: new VODTwitchContentStream(`v${this.collection.items[0].item_id}`),
            });
        });

        QUnit.test('plays next collection video in collection', function(assert) {
            this.store.dispatch({
                type: PlaybackActions.ACTION_ENDED,
            });

            assert.equal(this.fakeWindow.setTimeout.callCount, 1);

            const [selectCollectionVideoThunk] = this.store.dispatch.lastCall.args;
            const dispatch = sinon.spy();
            selectCollectionVideoThunk(dispatch);
            const [selectCollectionVideoAction] = dispatch.firstCall.args;

            assert.equal(
                selectCollectionVideoAction.type,
                FAKE_SELECT_COLLECTION_VIDEO_ACTION,
                'dispatch is type set vod action'
            );
            assert.equal(
                selectCollectionVideoAction.contentId,
                `v${this.collection.items[1].item_id}`,
                'selected vod matches the next video id'
            );
            assert.equal(
                selectCollectionVideoAction.collectionId,
                TEST_NORMALIZED_COLLECTION.id,
                'correct collection id is passed in'
            );
        });

        QUnit.test('do not play next collection video if player is mini', function(assert) {
            sinon.spy(this.collectionManager, '_playNextVideo');
            this.store.dispatch(UiActions.setMiniPlayerMode(true));
            this.store.dispatch({
                type: PlaybackActions.ACTION_ENDED,
            });

            assert.equal(
                this.collectionManager._playNextVideo.callCount,
                0,
                'play next video helper is not called'
            );
        });
    });
});
