import sinon from 'sinon';
import { unitTest } from 'tests/utils/module';
import * as Bridge from 'embed/client';

const PLAYER_STATE_UPDATE = 'playerstateupdate';

unitTest('embed | client', function(hooks) {
    hooks.beforeEach(function() {
        this.root = window.document.getElementById('qunit-fixture');
    });

    QUnit.module('api', function() {
        QUnit.test('calling callPlayerMethod posts a message to host', function(assert) {
            const client = new Bridge.EmbedClient(this.root, {});
            client._host.postMessage = sinon.spy();
            client._onHostReady = Promise.resolve();

            const method = QUnit.config.current.testId;
            const args = 11111;
            const expectedPayload = {
                namespace: Bridge.BRIDGE_HOST_NAMESPACE,
                method,
                args: [args],
            };

            return client.callPlayerMethod(method, args).
                then(() => {
                    assert.equal(
                        client._host.postMessage.callCount,
                        1,
                        'called once'
                    );

                    assert.deepEqual(
                        client._host.postMessage.firstCall.args[0],
                        expectedPayload,
                        'passes expected payload'
                    );

                    assert.equal(
                        client._host.postMessage.firstCall.args[1],
                        '*',
                        'passes * as origin'
                    );
                });
        });

        QUnit.test('can add and remove player state listeners', function(assert) {
            // eslint-disable-next-line no-unused-vars
            const client = new Bridge.EmbedClient(this.root, {});
            const listener = a => a;
            assert.equal(
                client._playerStateEmitter.getListeners(PLAYER_STATE_UPDATE).length,
                0,
                'should have 0 listeners to start with'
            );

            client.addPlayerStateListener(listener);
            assert.equal(
                client._playerStateEmitter.getListeners(PLAYER_STATE_UPDATE).length,
                1,
                'should have added one listener'
            );
            assert.deepEqual(
                client._playerStateEmitter.getListeners(PLAYER_STATE_UPDATE)[0].listener,
                listener,
                'should have added the right listener'
            );

            client.removePlayerStateListener(listener);
            assert.equal(
                client._playerStateEmitter.getListeners(PLAYER_STATE_UPDATE).length,
                0,
                'should have removed the listener'
            );
        });

        QUnit.test('only removes player state listeners if correct callback passed in', function(assert) {
            // eslint-disable-next-line no-unused-vars
            const client = new Bridge.EmbedClient(this.root, {});
            const listener = a => a;
            const otherListener = () => {};
            assert.equal(
                client._playerStateEmitter.getListeners(PLAYER_STATE_UPDATE).length,
                0,
                'should have 0 listeners to start with'
            );

            client.addPlayerStateListener(listener);
            assert.equal(
                client._playerStateEmitter.getListeners(PLAYER_STATE_UPDATE).length,
                1,
                'should have added one listener'
            );
            assert.deepEqual(
                client._playerStateEmitter.getListeners(PLAYER_STATE_UPDATE)[0].listener,
                listener,
                'should have added the right listener'
            );

            client.removePlayerStateListener(otherListener);
            assert.deepEqual(
                client._playerStateEmitter.getListeners(PLAYER_STATE_UPDATE)[0].listener,
                listener,
                'should not remove unrelated listener'
            );
        });
    });

    QUnit.module('when applying attributes to iframe', function() {
        QUnit.test('sets default attributes correctly on init', function(assert) {
            // eslint-disable-next-line no-unused-vars
            const client = new Bridge.EmbedClient(this.root, {});
            const $iframe = this.root.querySelector('iframe');
            assert.equal($iframe.getAttribute('frameBorder'), 0);
            assert.equal($iframe.getAttribute('scrolling'), 'no');
            assert.equal($iframe.getAttribute('width'), '640');
            assert.equal($iframe.getAttribute('height'), '390');
        });

        // eslint-disable-next-line max-len
        QUnit.test('adds `allowfullscreen` attribute when options.allowfullscreen is true', function(assert) {
            // eslint-disable-next-line no-unused-vars
            const client = new Bridge.EmbedClient(this.root, { allowfullscreen: true });
            const $iframe = this.root.querySelector('iframe');
            assert.ok($iframe.hasAttribute('allowfullscreen'));
        });

        // eslint-disable-next-line max-len
        QUnit.test('adds `allowfullscreen` attribute when options.allowfullscreen is undefined', function(assert) {
            // eslint-disable-next-line no-unused-vars
            const client = new Bridge.EmbedClient(this.root, { });
            const $iframe = this.root.querySelector('iframe');
            assert.ok($iframe.hasAttribute('allowfullscreen'));
        });

        // eslint-disable-next-line max-len
        QUnit.test('adds `allowfullscreen` attribute when options.allowfullscreen is not a boolean', function(assert) {
            // eslint-disable-next-line no-unused-vars
            const client = new Bridge.EmbedClient(this.root, { allowfullscreen: 'hello' });
            const $iframe = this.root.querySelector('iframe');
            assert.ok($iframe.hasAttribute('allowfullscreen'));
        });

        // eslint-disable-next-line max-len
        QUnit.test('does not add `allowfullscreen` attribute when options.allowfullscreen is false', function(assert) {
            // eslint-disable-next-line no-unused-vars
            const client = new Bridge.EmbedClient(this.root, { allowfullscreen: false });
            const $iframe = this.root.querySelector('iframe');
            assert.ok(!$iframe.hasAttribute('allowfullscreen'));
        });

        QUnit.test('sets `width` attribute based on options.width', function(assert) {
            // eslint-disable-next-line no-unused-vars
            const client = new Bridge.EmbedClient(this.root, { width: 500 });
            const $iframe = this.root.querySelector('iframe');
            assert.equal($iframe.getAttribute('width'), 500);
        });

        QUnit.test('sets `height` attribute based on options.length', function(assert) {
            // eslint-disable-next-line no-unused-vars
            const client = new Bridge.EmbedClient(this.root, { height: 500 });
            const $iframe = this.root.querySelector('iframe');
            assert.equal($iframe.getAttribute('height'), 500);
        });

        QUnit.test('setWidth function sets `width` attribute correctly', function(assert) {
            const client = new Bridge.EmbedClient(this.root, { width: 500 });
            client.setWidth(100);
            const $iframe = this.root.querySelector('iframe');
            assert.equal($iframe.getAttribute('width'), 100);
            client.setWidth(-10);
            assert.equal($iframe.getAttribute('width'), 100);
            client.setWidth('hello');
            assert.equal($iframe.getAttribute('width'), 100);
        });

        QUnit.test('setWidth function sets `height` attribute correctly', function(assert) {
            const client = new Bridge.EmbedClient(this.root, { height: 500 });
            client.setHeight(100);
            const $iframe = this.root.querySelector('iframe');
            assert.equal($iframe.getAttribute('height'), 100);
            client.setHeight(-10);
            assert.equal($iframe.getAttribute('height'), 100);
            client.setHeight('hello');
            assert.equal($iframe.getAttribute('height'), 100);
        });
    });

    QUnit.module('when receiving a message from the wrong embed', function(hooks) {
        hooks.beforeEach(function() {
            sinon.spy(window, 'postMessage');
            this.client = new Bridge.EmbedClient(this.root, {});
            this.iframe = document.createElement('iframe');
            this.root.appendChild(this.iframe);
            this.window = this.iframe.contentWindow;
        });

        hooks.afterEach(function() {
            window.postMessage.restore();
        });

        QUnit.test('ignores the message', function(assert) {
            const eventName = 'playing';
            const eventListener = sinon.spy();
            const stateUpdateMessage = {
                source: this.window,
                data: {
                    namespace: Bridge.BRIDGE_CLIENT_NAMESPACE,
                    method: Bridge.BRIDGE_STATE_UPDATE,
                    args: [{ not: 'ok' }],
                },
            };
            const playerEventMessage = {
                source: this.window,
                data: {
                    namespace: Bridge.BRIDGE_CLIENT_NAMESPACE,
                    method: Bridge.BRIDGE_PLAYER_EVENT,
                    args: [eventName],
                },
            };
            const storeStateUpdate = {
                source: this.window,
                data: {
                    namespace: Bridge.BRIDGE_CLIENT_NAMESPACE,
                    method: Bridge.BRIDGE_STORE_STATE_UPDATE,
                    args: [{ also: 'not ok' }],
                },
            };

            this.client.addEventListener(eventName, eventListener);

            this.client.handleEvent(stateUpdateMessage);
            assert.notEqual(this.client.getPlayerState(), stateUpdateMessage.data.args[0]);

            this.client.handleEvent(playerEventMessage);
            assert.equal(eventListener.callCount, 0);

            this.client.handleEvent(storeStateUpdate);
            assert.notEqual(this.client.getStoreState(), storeStateUpdate.data.args[0]);
        });
    });

    QUnit.module('when receiving a message from the correct embed', function(hooks) {
        hooks.beforeEach(function() {
            sinon.spy(window, 'postMessage');
            this.client = new Bridge.EmbedClient(this.root, {});
            this.window = this.client._iframe.contentWindow;
        });

        hooks.afterEach(function() {
            window.postMessage.restore();
        });

        QUnit.test('updates the state when receiving BRIDGE_STATE_UPDATE events', function(assert) {
            const stateUpdateMessage = {
                source: this.window,
                data: {
                    namespace: Bridge.BRIDGE_CLIENT_NAMESPACE,
                    method: Bridge.BRIDGE_STATE_UPDATE,
                    args: [{ not: 'ok' }],
                },
            };

            this.client.handleEvent(stateUpdateMessage);

            assert.equal(this.client.getPlayerState(), stateUpdateMessage.data.args[0]);
        });

        QUnit.test('calls the player state listener when receiving BRIDGE_STATE_UPDATE events', function(assert) {
            const stateUpdateMessage = {
                source: this.window,
                data: {
                    namespace: Bridge.BRIDGE_CLIENT_NAMESPACE,
                    method: Bridge.BRIDGE_STATE_UPDATE,
                    args: [{ not: 'ok' }],
                },
            };
            const spy = sinon.spy();

            this.client.addPlayerStateListener(spy);

            this.client.handleEvent(stateUpdateMessage);

            assert.ok(spy.called);
        });

        QUnit.test('forwards the player event when receiving BRIDGE_PLAYER_EVENT', function(assert) {
            const eventName = 'playing';
            const eventListener = sinon.spy();
            const playerEventMessage = {
                source: this.window,
                data: {
                    namespace: Bridge.BRIDGE_CLIENT_NAMESPACE,
                    method: Bridge.BRIDGE_PLAYER_EVENT,
                    args: [{
                        event: eventName,
                    }],
                },
            };

            this.client.addEventListener(eventName, eventListener);

            this.client.handleEvent(playerEventMessage);
            assert.equal(eventListener.callCount, 1);
        });

        QUnit.test('forwards the player event when receiving BRIDGE_PLAYER_EVENT_WITH_PAYLOAD', function(assert) {
            const eventName = 'transitionToCollectionVod';
            const eventListener = sinon.spy();
            const payload = {
                vodID: 'v932190413',
                collectionID: '3Fp1Z0mn103',
            };
            const playerEventMessage = {
                source: this.window,
                data: {
                    namespace: Bridge.BRIDGE_CLIENT_NAMESPACE,
                    method: Bridge.BRIDGE_PLAYER_EVENT_WITH_PAYLOAD,
                    args: [{
                        event: eventName,
                        data: payload,
                    }],
                },
            };

            this.client.addEventListener(eventName, eventListener);

            this.client.handleEvent(playerEventMessage);
            assert.equal(eventListener.callCount, 1);
            assert.ok(eventListener.calledWith(payload));
        });

        QUnit.test('updates the store state when receiving BRIDGE_STORE_STATE_UPDATE', function(assert) {
            const storeStateUpdate = {
                source: this.window,
                data: {
                    namespace: Bridge.BRIDGE_CLIENT_NAMESPACE,
                    method: Bridge.BRIDGE_STORE_STATE_UPDATE,
                    args: [{ also: 'not ok' }],
                },
            };

            this.client.handleEvent(storeStateUpdate);

            assert.equal(this.client.getStoreState(), storeStateUpdate.data.args[0]);
        });

        QUnit.test('destroys iframe when receiving BRIDGE_DESTROY', function(assert) {
            const destroyMessage = {
                source: this.window,
                data: {
                    namespace: Bridge.BRIDGE_CLIENT_NAMESPACE,
                    method: Bridge.BRIDGE_DESTROY,
                },
            };

            this.client.handleEvent(destroyMessage);

            assert.equal(this.client._iframe, undefined);
            assert.equal(this.client._host, undefined);
        });
    });
});
