import { PlayerEmbed } from 'embed/embedv1';
import ERROR_METADATA from 'error-metadata';
import * as Bridge from 'embed/client';
import sinon from 'sinon';

QUnit.module('embed | embed', function(hooks) {
    hooks.beforeEach(function() {
        this.options = {
            width: 854,
            height: 480,
            channel: 'monstercat',
        };
        const $root = window.document.createElement('div');
        $root.id = 'root';
        window.document.getElementById('qunit-fixture').appendChild($root);
    });

    QUnit.test('window constants are correct', function(assert) {
        assert.deepEqual(window.Twitch.Analytics, {
            MINUTE_WATCHED: Bridge.EVENT_EMBED_MINUTE_WATCHED,
            VIDEO_PLAY: Bridge.EVENT_EMBED_VIDEO_PLAY,
            BUFFER_EMPTY: Bridge.EVENT_EMBED_BUFFER_EMPTY,
            VIDEO_ERROR: Bridge.EVENT_EMBED_VIDEO_PLAYBACK_ERROR,
        });
        assert.deepEqual(window.Twitch.Errors, {
            ABORTED: ERROR_METADATA.ABORTED.code,
            NETWORK: ERROR_METADATA.NETWORK.code,
            DECODE: ERROR_METADATA.DECODE.code,
            FORMAT_NOT_SUPPORTED: ERROR_METADATA.FORMAT_NOT_SUPPORTED.code,
            CONTENT_NOT_AVAILABLE: ERROR_METADATA.CONTENT_NOT_AVAILABLE.code,
            RENDERER_NOT_AVAILABLE: ERROR_METADATA.RENDERER_NOT_AVAILABLE.code,
        });
        assert.deepEqual(window.Twitch.EmbedEvents, {
            TRANSITION_TO_RECOMMENDED_VOD: Bridge.EVENT_EMBED_TRANSITION_TO_REC_VOD,
        });
    });

    QUnit.test('event constants are the correct strings', function(assert) {
        assert.ok(PlayerEmbed.READY === Bridge.EVENT_EMBED_READY);
        assert.ok(PlayerEmbed.PLAY === Bridge.EVENT_EMBED_PLAY);
        assert.ok(PlayerEmbed.PAUSE === Bridge.EVENT_EMBED_PAUSE);
        assert.ok(PlayerEmbed.ENDED === Bridge.EVENT_EMBED_ENDED);
        assert.ok(PlayerEmbed.OFFLINE === Bridge.EVENT_EMBED_OFFLINE);
        assert.ok(PlayerEmbed.ONLINE === Bridge.EVENT_EMBED_ONLINE);
        assert.ok(PlayerEmbed.ERROR === Bridge.EVENT_EMBED_ERROR);
    });

    QUnit.test('setWidth calls bridge correctly', function(assert) {
        const playerEmbed = new PlayerEmbed('root', this.options);
        const setWidthSpy = sinon.spy(playerEmbed._bridge, 'setWidth');
        playerEmbed.setWidth(500);
        assert.ok(setWidthSpy.calledWith(500));
    });

    QUnit.test('setHeight calls bridge correctly', function(assert) {
        const playerEmbed = new PlayerEmbed('root', this.options);
        const setHeightSpy = sinon.spy(playerEmbed._bridge, 'setHeight');
        playerEmbed.setHeight(500);
        assert.ok(setHeightSpy.calledWith(500));
    });

    QUnit.test('setCollection calls bridge correctly', function(assert) {
        const collectionId = 'acollectionod';
        const videoId = 'avideoid';
        const timestamp = 12345;
        const playerEmbed = new PlayerEmbed('root', this.options);
        const callPlayerMethodSpy = sinon.spy(playerEmbed._bridge, 'callPlayerMethod');
        playerEmbed.setCollection(collectionId, videoId, timestamp);
        assert.equal(callPlayerMethodSpy.callCount, 1);
        // eslint-disable-next-line max-len
        assert.equal(callPlayerMethodSpy.firstCall.args[0], Bridge.METHOD_SET_COLLECTION, 'calls correct bridge method');
        assert.equal(callPlayerMethodSpy.firstCall.args[1], collectionId, 'passes in collection id');
        assert.equal(callPlayerMethodSpy.firstCall.args[2], videoId, 'passes in video id');
        assert.equal(callPlayerMethodSpy.firstCall.args[3], timestamp, 'passes in timestamp');
    });

    QUnit.test('setCollection calls bridge correctly with correct defaults', function(assert) {
        const collectionId = 'acollectionod';
        const playerEmbed = new PlayerEmbed('root', this.options);
        const callPlayerMethodSpy = sinon.spy(playerEmbed._bridge, 'callPlayerMethod');
        playerEmbed.setCollection(collectionId);
        assert.equal(callPlayerMethodSpy.callCount, 1);
        // eslint-disable-next-line max-len
        assert.equal(callPlayerMethodSpy.firstCall.args[0], Bridge.METHOD_SET_COLLECTION, 'calls correct bridge method');
        assert.equal(callPlayerMethodSpy.firstCall.args[1], collectionId, 'passes in collection id');
        assert.equal(callPlayerMethodSpy.firstCall.args[2], '', 'passes in empty string for video id');
        assert.equal(callPlayerMethodSpy.firstCall.args[3], '', 'passes in empty string for timestamp');
    });

    QUnit.test('setContent calls bridge correctly', function(assert) {
        const contentId = 'acontentid';
        const customerId = 'acustomerid';
        const playerEmbed = new PlayerEmbed('root', this.options);
        const callPlayerMethodSpy = sinon.spy(playerEmbed._bridge, 'callPlayerMethod');
        playerEmbed.setContent({
            customerId,
            contentId,
        });
        assert.equal(callPlayerMethodSpy.callCount, 1);
        // eslint-disable-next-line max-len
        assert.equal(callPlayerMethodSpy.firstCall.args[0], Bridge.METHOD_SET_CONTENT, 'calls correct bridge method');
        assert.equal(callPlayerMethodSpy.firstCall.args[1], contentId, 'passes in content id');
        assert.equal(callPlayerMethodSpy.firstCall.args[2], customerId, 'passes in customer id');
    });

    QUnit.test('getContentId returns stream.contentId on store state', function(assert) {
        const playerEmbed = new PlayerEmbed('root', this.options);
        const storeState = {
            stream: {
                contentId: 'acontentid',
            },
        };
        sinon.stub(playerEmbed._bridge, 'getStoreState', () => storeState);

        // eslint-disable-next-line max-len
        assert.equal(playerEmbed.getContentId(), storeState.stream.contentId, 'returns stream.contentId on store state');
    });

    QUnit.test('getCustomerId returns stream.customerId on store state', function(assert) {
        const playerEmbed = new PlayerEmbed('root', this.options);
        const storeState = {
            stream: {
                customerId: 'acustomerid',
            },
        };
        sinon.stub(playerEmbed._bridge, 'getStoreState', () => storeState);

        // eslint-disable-next-line max-len
        assert.equal(playerEmbed.getCustomerId(), storeState.stream.customerId, 'returns stream.customerId on store state');
    });

    QUnit.test('setClip calls bridge correctly', function(assert) {
        const slug = 'someSlug';
        const playerEmbed = new PlayerEmbed('root', this.options);
        const callPlayerMethodSpy = sinon.spy(playerEmbed._bridge, 'callPlayerMethod');
        playerEmbed.setClip(slug);
        assert.equal(callPlayerMethodSpy.callCount, 1);
        // eslint-disable-next-line max-len
        assert.equal(callPlayerMethodSpy.firstCall.args[0], Bridge.METHOD_SET_CLIP, 'calls correct bridge method');
        assert.equal(callPlayerMethodSpy.firstCall.args[1], slug, 'passes in slug');
    });

    QUnit.test('setChannelId calls bridge correctly', function(assert) {
        const channelId = '123456';
        const playerEmbed = new PlayerEmbed('root', this.options);
        const callPlayerMethodSpy = sinon.spy(playerEmbed._bridge, 'callPlayerMethod');
        playerEmbed.setChannelId(channelId);
        assert.equal(callPlayerMethodSpy.callCount, 1, 'setChannelId calls bridge');
        // eslint-disable-next-line max-len
        assert.equal(callPlayerMethodSpy.firstCall.args[0], Bridge.METHOD_SET_CHANNEL_ID, 'calls correct bridge method');
        assert.equal(callPlayerMethodSpy.firstCall.args[1], channelId, 'passes in chanenlId');
    });

    QUnit.test('getChannelId returns player state channelId', function(assert) {
        const playerEmbed = new PlayerEmbed('root', this.optios);
        const playerState = {
            channelId: '27446517',
        };
        const getPlayerStateSpy = sinon.stub(playerEmbed._bridge, 'getPlayerState', () => playerState);
        assert.equal(playerEmbed.getChannelId(), playerState.channelId, 'returns channelId');
        assert.equal(getPlayerStateSpy.callCount, 1, 'getChannelId calls bridge');
    });

    QUnit.test('setVideoSource calls bridge correctly', function(assert) {
        const url = 'https://twitch.tv/some.mp4';
        const playerEmbed = new PlayerEmbed('root', this.options);
        const callPlayerMethodSpy = sinon.spy(playerEmbed._bridge, 'callPlayerMethod');
        playerEmbed.setVideoSource(url);
        assert.equal(callPlayerMethodSpy.callCount, 1);
        // eslint-disable-next-line max-len
        assert.equal(callPlayerMethodSpy.firstCall.args[0], Bridge.METHOD_SET_VIDEO_SOURCE, 'calls correct bridge method');
        assert.equal(callPlayerMethodSpy.firstCall.args[1], url, 'passes in url');
    });

    QUnit.test('_addPlayerStateListener adds a listener and the callback gets called', function(assert) {
        const playerEmbed = new PlayerEmbed('root', this.options);
        const listenerSpy = sinon.spy();
        const addSpy = sinon.spy(playerEmbed._bridge, 'addPlayerStateListener');

        playerEmbed._addPlayerStateListener(listenerSpy);
        assert.ok(addSpy.calledWith(listenerSpy));
        assert.ok(listenerSpy.calledWith(playerEmbed._bridge.getPlayerState()));
    });

    QUnit.test('_removePlayerStateListener removes the player state listener', function(assert) {
        const playerEmbed = new PlayerEmbed('root', this.options);
        const listener = () => {};
        const removeSpy = sinon.spy(playerEmbed._bridge, 'removePlayerStateListener');

        playerEmbed._removePlayerStateListener(listener);
        assert.ok(removeSpy.calledWith(listener));
    });
});
