import { setCollection, openCollectionSidebar, closeCollectionSidebar,
         ACTION_CLOSE_COLLECTION_SIDEBAR } from 'actions/collection';
import { setPlayerType } from 'actions/env';
import { ACTION_VIDEO_API_CHANGE_VOLUME, ACTION_VIDEO_API_PAUSE,
         ACTION_VIDEO_API_PLAY, ACTION_VIDEO_API_CHANGE_PLAYBACK_RATE,
         play, changePlaybackRate  } from 'actions/video-api';
import { init as initStore } from 'state';
import { PlayerHotkeys } from 'hotkeys';
import { unitTest } from 'tests/utils/module';
import { TEST_COLLECTION } from 'tests/fixtures/collection';
import sinon from 'sinon';
import { volumeStepAmount } from 'settings';
import { setTheatreMode, ACTION_SET_THEATRE_MODE, ACTION_TOGGLE_FULLSCREEN } from 'actions/screen-mode';
import * as PlayerType from 'util/player-type';

const KEY_ESC = 'Escape';
const KEY_DOWN_COMPAT = 'Down';
const KEY_DOWN = 'ArrowDown';
const KEY_UP_COMPAT = 'Up';
const KEY_UP = 'ArrowUp';
const KEY_F = 'f';
const KEY_F_CAP = 'F';
const KEY_CTRL = 'Control';
const KEY_SPACE = ' ';
const KEY_SPACEBAR = 'Spacebar';
const KEY_COMMA = ',';
const KEY_PERIOD = '.';
const KEY_LEFT = 'ArrowLeft';
const KEY_RIGHT = 'ArrowRight';

unitTest('hotkeys', function(hooks) {
    hooks.beforeEach(function() {
        this.root = window.document.createElement('div');
        this.root.id = 'root';
        window.document.getElementById('qunit-fixture').appendChild(this.root);
        this.store = initStore();
        sinon.spy(this.store, 'dispatch');

        this.fakeVideo = {};
        const options = {};
        new PlayerHotkeys(this.fakeVideo, this.root, this.store, options);

        this.simulateKeyPress = key => {
            const event = new KeyboardEvent('keydown', {
                key,
            });
            this.root.dispatchEvent(event);
        };
    });

    QUnit.module('hitting escape key', function() {
        QUnit.test('does nothing when collections is not enabled', function(assert) {
            this.simulateKeyPress(KEY_ESC);

            assert.equal(this.store.dispatch.callCount, 0, 'does not call dispatch');
        });

        QUnit.module('when in theater mode', function() {
            QUnit.test('while embedded', function(assert) {
                sinon.stub(PlayerType, 'isEmbed', () => {
                    return true;
                });

                assert.ok(PlayerType.isEmbed(), 'is embedded');

                this.store.dispatch(setTheatreMode(true));
                this.store.dispatch.reset();
                this.simulateKeyPress(KEY_ESC);

                assert.equal(this.store.dispatch.callCount, 1, 'calls dispatch once');
                const [action] = this.store.dispatch.firstCall.args;
                assert.equal(action.type, ACTION_SET_THEATRE_MODE, 'dispatches set theatre mode action');

                PlayerType.isEmbed.restore();
            });
            QUnit.test('while on twitch.tv', function(assert) {
                sinon.stub(PlayerType, 'isEmbed', () => {
                    return false;
                });

                assert.ok(!PlayerType.isEmbed(), 'is on twitch.tv');

                this.store.dispatch(setTheatreMode(true));
                this.store.dispatch.reset();
                this.simulateKeyPress(KEY_ESC);

                assert.equal(this.store.dispatch.callCount, 0, 'never calls dispatch');

                PlayerType.isEmbed.restore();
            });
        });

        QUnit.module('when collections is enabled', function(hooks) {
            hooks.beforeEach(function() {
                setCollection(TEST_COLLECTION);
            });

            QUnit.test('closes the sidebar when the collections sidebar is open', function(assert) {
                this.store.dispatch(openCollectionSidebar());
                this.store.dispatch.reset();
                this.simulateKeyPress(KEY_ESC);

                assert.equal(this.store.dispatch.callCount, 1, 'calls dispatch once');
                const [action] = this.store.dispatch.firstCall.args;
                assert.equal(action.type, ACTION_CLOSE_COLLECTION_SIDEBAR, 'dispatches close sidebar action');
            });

            QUnit.test('does nothing when the collections sidebar is closed', function(assert) {
                this.store.dispatch(closeCollectionSidebar());
                this.simulateKeyPress(KEY_ESC);
                this.store.dispatch.reset();

                assert.equal(this.store.dispatch.callCount, 0, 'does not call dispatch');
            });
        });
    });

    QUnit.module('hitting down arrow key', function() {
        function testDownArrowKeysLowerVolume(key, message) {
            QUnit.test(message, function(assert) {
                const currentVolume = this.store.getState().playback.volume;
                const newVolume = currentVolume - volumeStepAmount;

                this.simulateKeyPress(key);

                assert.equal(this.store.dispatch.callCount, 1, 'calls dispatch once');
                assert.ok(this.store.dispatch.calledWith({
                    type: ACTION_VIDEO_API_CHANGE_VOLUME,
                    volume: newVolume,
                }));
            });
        }

        testDownArrowKeysLowerVolume(KEY_DOWN, 'down arrow key lowers volume');
        testDownArrowKeysLowerVolume(KEY_DOWN_COMPAT, 'down arrow compat key lowers volume');
    });

    QUnit.module('hitting up arrow key', function() {
        function testUpArrowKeysRaisesVolume(key, message) {
            QUnit.test(message, function(assert) {
                const currentVolume = this.store.getState().playback.volume;
                const newVolume = currentVolume + volumeStepAmount;

                this.simulateKeyPress(key);

                assert.equal(this.store.dispatch.callCount, 1, 'calls dispatch once');
                assert.ok(this.store.dispatch.calledWith({
                    type: ACTION_VIDEO_API_CHANGE_VOLUME,
                    volume: newVolume,
                }));
            });
        }

        testUpArrowKeysRaisesVolume(KEY_UP, 'up arrow key raises volume');
        testUpArrowKeysRaisesVolume(KEY_UP_COMPAT, 'up arrow key compat raises volume');
    });

    QUnit.module('fullscreen is togglable', function() {
        function testControlFFullscreensVideo(ctrl, key, message) {
            QUnit.test(message, function(assert) {
                this.simulateKeyPress(ctrl);
                this.simulateKeyPress(key);

                assert.equal(this.store.dispatch.callCount, 1, 'calls dispatch once');
                assert.ok(this.store.dispatch.calledWith({
                    type: ACTION_TOGGLE_FULLSCREEN,
                }));
            });
        }

        testControlFFullscreensVideo(KEY_CTRL, KEY_F_CAP, 'with ctrl+F');
        testControlFFullscreensVideo(KEY_CTRL, KEY_F, 'with ctrl+F');
    });

    QUnit.module('fullscreen is disabled on mini player', function() {
        function testHotkeysOnMiniPlayer(ctrl, key, message) {
            QUnit.test(message, function(assert) {
                const getStateStub = sinon.stub(this.store, 'getState');
                getStateStub.returns({
                    ui: {
                        isMini: true,
                    },
                    env: {
                        playerType: 'popout',
                    },
                });

                this.simulateKeyPress(ctrl);
                this.simulateKeyPress(key);
                assert.ok(this.store.dispatch.notCalled, 'dispatch is not called');
                getStateStub.restore();
            });
        }

        testHotkeysOnMiniPlayer(KEY_CTRL, KEY_F_CAP, 'with ctrl+F');
        testHotkeysOnMiniPlayer(KEY_CTRL, KEY_F, 'with ctrl+f');
    });

    QUnit.module('hitting spacebar key plays video', function() {
        function testSpacebarKeyPlaysPlayback(key, message) {
            QUnit.test(message, function(assert) {
                const getStateStub = sinon.stub(this.store, 'getState');
                getStateStub.returns({
                    playback: {
                        paused: true,
                    },
                    ui: {
                        isMini: false,
                    },
                    env: {
                        playerType: 'popout',
                    },
                });

                this.simulateKeyPress(key);

                assert.equal(this.store.dispatch.callCount, 1, 'calls dispatch once');
                assert.ok(this.store.dispatch.calledWith({
                    type: ACTION_VIDEO_API_PLAY,
                }));
                getStateStub.restore();
            });
        }

        testSpacebarKeyPlaysPlayback(KEY_SPACE, 'for KEY_SPACE');
        testSpacebarKeyPlaysPlayback(KEY_SPACEBAR, 'for KEY_SPACEBAR');
    });

    QUnit.module('hitting spacebar key pauses video', function() {
        function testSpacebarKeyPausesPlayback(key, message) {
            QUnit.test(message, function(assert) {
                this.simulateKeyPress(key);

                assert.equal(this.store.dispatch.callCount, 1, 'calls dispatch once');
                assert.ok(this.store.dispatch.calledWith({
                    type: ACTION_VIDEO_API_PAUSE,
                }));
            });
        }

        testSpacebarKeyPausesPlayback(KEY_SPACE, 'for KEY_SPACE');
        testSpacebarKeyPausesPlayback(KEY_SPACEBAR, 'for KEY_SPACEBAR');
    });

    QUnit.module('when player is clips-viewing type', function(hooks) {
        hooks.beforeEach(function() {
            this.store.dispatch(setPlayerType(PlayerType.PLAYER_CLIPS_VIEWING));
        });

        QUnit.test('spacebar toggles playback', function(assert) {
            this.store.dispatch(play());
            this.store.dispatch.reset();

            this.simulateKeyPress(KEY_SPACE);

            assert.equal(this.store.dispatch.callCount, 1, 'calls dispatch once');
            assert.ok(this.store.dispatch.calledWith({ type: ACTION_VIDEO_API_PAUSE }), 'dispatched pause action');
        });

        QUnit.test('comma slows playback rate', function(assert) {
            this.store.dispatch(changePlaybackRate(1));
            this.store.dispatch.reset();

            this.simulateKeyPress(KEY_COMMA);

            assert.equal(this.store.dispatch.callCount, 1, 'calls dispatch once');
            assert.ok(this.store.dispatch.calledWith({
                type: ACTION_VIDEO_API_CHANGE_PLAYBACK_RATE,
                playbackRate: 0.75,
            }), 'dispatched change playback rate action with 0.75');
        });

        QUnit.test('period speeds up playback rate', function(assert) {
            this.store.dispatch(changePlaybackRate(1));
            this.store.dispatch.reset();

            this.simulateKeyPress(KEY_PERIOD);

            assert.equal(this.store.dispatch.callCount, 1, 'calls dispatch once');
            assert.ok(this.store.dispatch.calledWith({
                type: ACTION_VIDEO_API_CHANGE_PLAYBACK_RATE,
                playbackRate: 1.25,
            }), 'dispatched change playback rate action with 1.25');
        });

        QUnit.module('when seeking current time using keyboard shortcuts', function(hooks) {
            hooks.beforeEach(function() {
                // Pass non-seekable checks, can't stub or spy on undefined functions
                this.fakeVideo.getPaused = () => false;
                this.fakeVideo.getChannel = () => null;
                this.fakeVideo.getEnded = () => false;
                this.fakeVideo.getCurrentTime = () => 10;
                this.fakeVideo.setCurrentTime = () => {};
                sinon.spy(this.fakeVideo, 'setCurrentTime');

                this.getStateStub = sinon.stub(this.store, 'getState');
                this.getStateStub.returns({
                    playback: {
                        paused: true,
                        duration: 20,
                    },
                    ui: {
                        isMini: false,
                    },
                    env: {
                        playerType: 'clips-viewing',
                    },
                });
            });

            hooks.afterEach(function() {
                this.getStateStub.restore();
                this.fakeVideo.setCurrentTime.restore();
            });

            QUnit.test('left key seek back by clips seek amount', function(assert) {
                this.simulateKeyPress(KEY_LEFT);

                assert.equal(this.fakeVideo.setCurrentTime.callCount, 1, 'player setCurrentTime called once');
                assert.ok(this.fakeVideo.setCurrentTime.calledWith(8), 'current time set to 8s (10 - 2)');
            });

            QUnit.test('right key seeks forward by clips seek amount', function(assert) {
                this.simulateKeyPress(KEY_RIGHT);

                assert.equal(this.fakeVideo.setCurrentTime.callCount, 1, 'player setCurrentTime called once');
                assert.ok(this.fakeVideo.setCurrentTime.calledWith(12), 'current time set to 12s (10 + 2)');
            });
        });
    });
});
