import sinon from 'sinon';
import { createAnalyticsTrackerMiddleware } from 'middleware/analytics-tracker-middleware';
import * as AnalyticsTrackerActions from 'actions/analytics-tracker';
import { NullAnalyticsTracker } from 'defaults/null-analytics-tracker';

class Tracker {
    constructor() {
        this.trackEvent = sinon.spy();
    }
}

QUnit.module('middleware | analytics-tracker', function(hooks) {
    hooks.beforeEach(function() {
        this.nullTracker = new NullAnalyticsTracker();
        sinon.spy(this.nullTracker, 'trackEvent');

        const middleware = createAnalyticsTrackerMiddleware(this.nullTracker);

        const dispatch = () => {};
        const getState = () => {};

        this.next = middleware({
            dispatch,
            getState,
        });

        this.dispatchAction = this.next(() => {});
    });

    QUnit.test('it must return a function to handle the action', function(assert) {
        const actionHandler = this.next();
        assert.equal(typeof actionHandler, 'function');
    });

    QUnit.test('it must pass the action into the next callback', function(assert) {
        const nextSpy = sinon.spy();
        const mockAction = { type: 'mock action' };
        const actionHandler = this.next(nextSpy);
        actionHandler(mockAction);

        const [dispatchedAction] = nextSpy.firstCall.args;
        assert.deepEqual(dispatchedAction, mockAction);
    });

    QUnit.module('setAnalyticsTracker', function() {
        QUnit.test('should use nullTracker if nothing is loaded', function(assert) {
            this.dispatchAction(AnalyticsTrackerActions.trackEvent('fake event'));
            assert.equal(this.nullTracker.trackEvent.callCount, 1);
        });

        QUnit.test('should use loaded tracker instance if one exists', function(assert) {
            const tracker = new Tracker();
            this.dispatchAction(AnalyticsTrackerActions.setAnalyticsTracker(tracker));
            this.dispatchAction(AnalyticsTrackerActions.trackEvent('fake event'));
            assert.equal(tracker.trackEvent.callCount, 1, '`trackEvent` invoked on loaded tracker');
            assert.equal(this.nullTracker.trackEvent.callCount, 0, '`trackEvent` not invoked on previous tracker');
        });
    });

    QUnit.module('analytics tracker api actions', function(hooks) {
        hooks.beforeEach(function() {
            this.mockTracker = new Tracker();
            this.dispatchAction(AnalyticsTrackerActions.setAnalyticsTracker(this.mockTracker));
        });

        QUnit.test('dispatching trackEvent should call .trackEvent() on tracker instance', function(assert) {
            const eventName = 'my cool event';
            const eventProperties = {
                prop: 'llama is very awesome',
            };
            this.dispatchAction(AnalyticsTrackerActions.trackEvent(eventName, eventProperties));
            assert.equal(this.mockTracker.trackEvent.callCount, 1, 'one event emitted');
            assert.equal(this.mockTracker.trackEvent.firstCall.args[0], eventName, 'correct event tracked');
            assert.equal(this.mockTracker.trackEvent.firstCall.args[1], eventProperties, 'correct properties tracked');
        });
    });
});
