import sinon from 'sinon';
import { init as initStore } from 'state';
import { createEmbedHostMiddleware } from 'middleware/embed-host-middleware';
import * as EmbedEventEmitterActions from 'actions/embed-event-emitter';
import { ACTION_EMIT_PLAYER_READY, emitTransitionToCollection,
         emitTransitionToRecommendedVOD } from 'actions/event-emitter';
import { NullEmbedHost } from 'defaults/null-embed-host';
import { IMA_PROVIDER } from 'ads/ima-manager';
import { COMPANION_RENDERED } from 'ads/advertisement-event';
import { EVENT_EMBED_FLASH_AD_COMPANION_RENDERED } from 'embed/client';
import { updateStats } from 'actions/stats';
import { ACTION_PLAYER_SEEKED } from 'actions/playback';

class EmbedHost {
    constructor() {
        this._sendAdCompanionEvent = sinon.spy();
        this._sendStatsUpdateEvent = sinon.spy();
        this._sendReadyEvent = sinon.spy();
        this._sendTransitionToCollectionEvent = sinon.spy();
        this._sendTransitionToRecommendedVodEvent = sinon.spy();
        this._sendSeekedEvent = sinon.spy();
        this._emitWheelEvent = sinon.spy();
        this._sendFirstPartyAnalyticEvent = sinon.spy();
    }
}

QUnit.module('middleware | embed-event-emitter', function(hooks) {
    hooks.beforeEach(function() {
        this.nullEmbedHost = new NullEmbedHost();
        sinon.spy(this.nullEmbedHost, '_sendAdCompanionEvent');

        const middleware = createEmbedHostMiddleware(this.nullEmbedHost);

        const dispatch = () => {};
        const getState = () => initStore().getState();

        this.next = middleware({
            dispatch,
            getState,
        });

        this.dispatchAction = this.next(() => {});
    });

    QUnit.test('it must return a function to handle the action', function(assert) {
        const actionHandler = this.next();
        assert.equal(typeof actionHandler, 'function');
    });

    QUnit.test('it must pass the action into the next callback', function(assert) {
        const nextSpy = sinon.spy();
        const mockAction = { type: 'mock action' };
        const actionHandler = this.next(nextSpy);
        actionHandler(mockAction);

        const [dispatchedAction] = nextSpy.firstCall.args;
        assert.deepEqual(dispatchedAction, mockAction);
    });

    QUnit.module('embedHostLoaded', function() {
        QUnit.test('should use NullEmbedHost if nothing is loaded', function(assert) {
            this.dispatchAction(EmbedEventEmitterActions.adCompanionRendered(COMPANION_RENDERED, {
                provider: IMA_PROVIDER,
            }));
            assert.equal(this.nullEmbedHost._sendAdCompanionEvent.callCount, 1);
        });

        QUnit.test('should use loaded embed instance if one exists', function(assert) {
            const embedHost = new EmbedHost();
            this.dispatchAction(EmbedEventEmitterActions.embedHostLoaded(embedHost));
            this.dispatchAction(EmbedEventEmitterActions.adCompanionRendered(COMPANION_RENDERED, {
                provider: IMA_PROVIDER,
            }));

            assert.equal(embedHost._sendAdCompanionEvent.callCount, 1);
            assert.equal(this.nullEmbedHost._sendAdCompanionEvent.callCount, 0);
        });
    });

    QUnit.module('embed api actions', function() {
        QUnit.test('dispatch ima ad companion calls _sendAdCompanionEvent on Embed instance', function(assert) {
            const embedHost = new EmbedHost();
            this.dispatchAction(EmbedEventEmitterActions.embedHostLoaded(embedHost));
            this.dispatchAction(EmbedEventEmitterActions.adCompanionRendered(COMPANION_RENDERED, {
                provider: IMA_PROVIDER,
            }));

            assert.equal(embedHost._sendAdCompanionEvent.callCount, 1);
        });

        QUnit.test('dispatch flash ad companion calls _sendAdCompanionEvent on Embed instance', function(assert) {
            const data = 'flash';
            const embedHost = new EmbedHost();
            this.dispatchAction(EmbedEventEmitterActions.embedHostLoaded(embedHost));
            this.dispatchAction(EmbedEventEmitterActions.adCompanionRendered(
                EVENT_EMBED_FLASH_AD_COMPANION_RENDERED,
                data
            ));

            assert.equal(embedHost._sendAdCompanionEvent.callCount, 1);
        });

        QUnit.test('dispatch emit wheel event calls _emitWheelEvent on Embed instance', function(assert) {
            const data = {
                deltaMode: 0,
                deltaX: 0,
                deltaY: 12,
                shiftKey: false,
            };
            const embedHost = new EmbedHost();
            this.dispatchAction(EmbedEventEmitterActions.embedHostLoaded(embedHost));
            this.dispatchAction(EmbedEventEmitterActions.emitWheelEvent(data));

            assert.equal(embedHost._emitWheelEvent.callCount, 1);
        });

        QUnit.test('dispatch transition calls _sendTransitionToCollectionEvent on Embed instance', function(assert) {
            const embedHost = new EmbedHost();
            this.dispatchAction(EmbedEventEmitterActions.embedHostLoaded(embedHost));
            this.dispatchAction(emitTransitionToCollection('129580scjkseLDSKJ', '203984520'));

            assert.equal(embedHost._sendTransitionToCollectionEvent.callCount, 1);
        });

        // eslint-disable-next-line max-len
        QUnit.test('dispatch transition calls _sendTransitionToRecommendedVodEvent on Embed instance', function(assert) {
            const embedHost = new EmbedHost();
            this.dispatchAction(EmbedEventEmitterActions.embedHostLoaded(embedHost));
            this.dispatchAction(emitTransitionToRecommendedVOD('203984520'));

            assert.equal(embedHost._sendTransitionToRecommendedVodEvent.callCount, 1);
        });

        QUnit.test('dispatch stats update calls _sendStatsUpdate on Embed instance', function(assert) {
            const embedHost = new EmbedHost();
            this.dispatchAction(EmbedEventEmitterActions.embedHostLoaded(embedHost));
            this.dispatchAction(updateStats({}));
            assert.equal(embedHost._sendStatsUpdateEvent.callCount, 1);
        });

        QUnit.test('dispatched player ready calls _sendReadyEvent on Embed instance', function(assert) {
            const embedHost = new EmbedHost();
            this.dispatchAction(EmbedEventEmitterActions.embedHostLoaded(embedHost));
            this.dispatchAction({
                type: ACTION_EMIT_PLAYER_READY,
            });

            assert.equal(embedHost._sendReadyEvent.callCount, 1);
        });

        QUnit.test('dispatched player seeked calls _sendSeekedEvent on Embed instance', function(assert) {
            const embedHost = new EmbedHost();
            this.dispatchAction(EmbedEventEmitterActions.embedHostLoaded(embedHost));
            this.dispatchAction({
                type: ACTION_PLAYER_SEEKED,
                currentTime: 0,
            });

            assert.equal(embedHost._sendSeekedEvent.callCount, 1, 'embed instance seeked function called');
        });

        // eslint-disable-next-line max-len
        QUnit.test('dispatched first party analytics event calls _sendFirstPartyAnalyticEvent on Embed instance', function(assert) {
            const event = 'event name';
            const data = {
                someData: 'someData',
            };
            const embedHost = new EmbedHost();
            this.dispatchAction(EmbedEventEmitterActions.embedHostLoaded(embedHost));
            this.dispatchAction({
                type: EmbedEventEmitterActions.ACTION_FIRST_PARTY_ANALYTIC_EVENT,
                event,
                data,
            });

            // eslint-disable-next-line max-len
            assert.equal(embedHost._sendFirstPartyAnalyticEvent.callCount, 1, 'embed instance first party event function called');
            assert.equal(embedHost._sendFirstPartyAnalyticEvent.firstCall.args[0], event);
            assert.equal(embedHost._sendFirstPartyAnalyticEvent.firstCall.args[1], data);
        });
    });
});
