import sinon from 'sinon';
import { init as initStore } from 'state';
import { createEventEmitterMiddleware } from 'middleware/event-emitter-middleware';
import * as EventEmitterActions from 'actions/event-emitter';
import { setTheatreMode, toggleTheatreMode } from 'actions/screen-mode';
import { NullEventEmitter } from 'defaults/null-event-emitter';
import { ACTION_PROMPT_LOGIN_MODAL } from 'actions/follow';
import { FULLSCREEN_CHANGE } from 'util/fullscreen';
import { ACTION_SET_PERSISTENCE } from 'actions/settings';
import * as TwitchEvents from 'backend/events/twitch-event';

class EventEmitter {
    constructor() {
        this.emit = sinon.spy();
    }
}

QUnit.module('middleware | event-emitter', function(hooks) {
    hooks.beforeEach(function() {
        this.nullEmitter = new NullEventEmitter();
        sinon.spy(this.nullEmitter, 'emit');

        const middleware = createEventEmitterMiddleware(this.nullEmitter);

        const dispatch = () => {};
        const getState = () => initStore().getState();

        this.next = middleware({
            dispatch,
            getState,
        });

        this.dispatchAction = this.next(() => {});
    });

    QUnit.test('it must return a function to handle the action', function(assert) {
        const actionHandler = this.next();
        assert.equal(typeof actionHandler, 'function');
    });

    QUnit.test('it must pass the action into the next callback', function(assert) {
        const nextSpy = sinon.spy();
        const mockAction = { type: 'mock action' };
        const actionHandler = this.next(nextSpy);
        actionHandler(mockAction);

        const [dispatchedAction] = nextSpy.firstCall.args;
        assert.deepEqual(dispatchedAction, mockAction);
    });

    QUnit.module('useEventEmitter', function() {
        QUnit.test('should use nullEmitter if nothing is loaded', function(assert) {
            this.dispatchAction(EventEmitterActions.emitOpenStream());
            assert.equal(this.nullEmitter.emit.callCount, 1);
        });

        QUnit.test('should use loaded emitter instance if one exists', function(assert) {
            const emitter = new EventEmitter();
            this.dispatchAction(EventEmitterActions.eventEmitterLoaded(emitter));
            this.dispatchAction(EventEmitterActions.emitOpenStream());
            assert.equal(emitter.emit.callCount, 1);
            assert.equal(this.nullEmitter.emit.callCount, 0);
        });
    });

    QUnit.module('event emitter api actions', function(hooks) {
        hooks.beforeEach(function() {
            this.mockEmitter = new EventEmitter();
            this.dispatchAction(EventEmitterActions.eventEmitterLoaded(this.mockEmitter));

            this.createDispatcherWithState = state => {
                const middleware = createEventEmitterMiddleware(this.nullEmitter);
                const next = middleware({
                    dispatch() {},
                    getState: () => state,
                });
                return next(() => {});
            };
        });

        QUnit.test('dispatching emitOpenStream should call .emit() on emitter instance', function(assert) {
            this.dispatchAction(EventEmitterActions.emitOpenStream());
            assert.equal(this.mockEmitter.emit.callCount, 1, 'one event emitted');
            // eslint-disable-next-line max-len
            assert.equal(this.mockEmitter.emit.firstCall.args[0], TwitchEvents.OPEN_STREAM, 'correct event name emitted');
        });

        QUnit.test('dispatching promptLoginModal should call .emit event with channelName', function(assert) {
            const channelName = QUnit.config.current.testId;
            this.dispatchAction({
                type: ACTION_PROMPT_LOGIN_MODAL,
                channelName,
            });
            assert.equal(this.mockEmitter.emit.callCount, 1, 'one event emitted');
            assert.equal(
                this.mockEmitter.emit.firstCall.args[0],
                TwitchEvents.PROMPT_LOGIN_MODAL,
                'correct event name emitted'
            );
            assert.deepEqual(
                this.mockEmitter.emit.firstCall.args[1],
                { channelDisplayName: channelName },
                'correct event name emitted'
            );
        });

        QUnit.test('dispatching ACTION_SET_PERSISTENCE should call .emit() on emitter instance', function(assert) {
            this.dispatchAction({
                type: ACTION_SET_PERSISTENCE,
                value: true,
            });
            assert.equal(this.mockEmitter.emit.callCount, 1, 'one event emitted');
            assert.equal(
                this.mockEmitter.emit.firstCall.args[0],
                TwitchEvents.PERSISTENT_PLAYER_TOGGLE,
                'correct event name emitted'
            );
        });

        QUnit.test('dispatching emitFullscreenChange should call .emit() on emitter instance', function(assert) {
            this.dispatchAction(EventEmitterActions.emitFullscreenChange());
            assert.equal(this.mockEmitter.emit.callCount, 1, 'one event emitted');
            assert.equal(this.mockEmitter.emit.firstCall.args[0], FULLSCREEN_CHANGE, 'correct event name emitted');
        });

        QUnit.test('when fullscreen is false, emits expected theatre mode value', function(assert) {
            const dispatchAction = this.createDispatcherWithState({
                screenMode: {
                    isFullScreen: false,
                    isTheatreMode: false,
                },
            });
            dispatchAction(EventEmitterActions.eventEmitterLoaded(this.mockEmitter));

            dispatchAction(setTheatreMode(true));
            assert.equal(this.mockEmitter.emit.callCount, 1, 'one event emitted');
            assert.equal(
                this.mockEmitter.emit.firstCall.args[0],
                TwitchEvents.THEATRE_CHANGE,
                'correct event name emitted'
            );
            assert.equal(this.mockEmitter.emit.firstCall.args[1], true);
        });

        QUnit.test('when fullscreen is true, always emit `false` value', function(assert) {
            const dispatchAction = this.createDispatcherWithState({
                screenMode: {
                    isFullScreen: true,
                    isTheatreMode: false,
                },
            });
            dispatchAction(EventEmitterActions.eventEmitterLoaded(this.mockEmitter));
            dispatchAction(setTheatreMode(true));

            assert.equal(this.mockEmitter.emit.callCount, 1, 'one event emitted');
            assert.equal(
                this.mockEmitter.emit.firstCall.args[0],
                TwitchEvents.THEATRE_CHANGE,
                'correct event name emitted'
            );
            assert.deepEqual(this.mockEmitter.emit.firstCall.args[1], false);
        });

        QUnit.test('action_toggle_theatremode emits true value when isTheatreMode is false', function(assert) {
            const dispatchAction = this.createDispatcherWithState({
                screenMode: {
                    isFullScreen: false,
                    isTheatreMode: false,
                },
            });
            dispatchAction(EventEmitterActions.eventEmitterLoaded(this.mockEmitter));

            dispatchAction(toggleTheatreMode());

            assert.equal(this.mockEmitter.emit.callCount, 1, 'one event emitted');
            assert.equal(
                this.mockEmitter.emit.firstCall.args[0],
                TwitchEvents.THEATRE_CHANGE,
                'correct event name emitted'
            );
            assert.equal(this.mockEmitter.emit.firstCall.args[1], true);
        });

        QUnit.test('action_toggle_theatremode emits false value when isTheatreMode is true', function(assert) {
            const dispatchAction = this.createDispatcherWithState({
                screenMode: {
                    isFullScreen: false,
                    isTheatreMode: true,
                },
            });
            dispatchAction(EventEmitterActions.eventEmitterLoaded(this.mockEmitter));

            dispatchAction(toggleTheatreMode());

            assert.equal(this.mockEmitter.emit.callCount, 1, 'one event emitted');
            assert.equal(
                this.mockEmitter.emit.firstCall.args[0],
                TwitchEvents.THEATRE_CHANGE,
                'correct event name emitted'
            );
            assert.equal(this.mockEmitter.emit.firstCall.args[1], false);
        });

        QUnit.test('dispatching setTheatreMode should not call .emit when no change has occured', function(assert) {
            const dispatchAction = this.createDispatcherWithState({
                screenMode: {
                    isFullScreen: false,
                    isTheatreMode: false,
                },
            });
            dispatchAction(EventEmitterActions.eventEmitterLoaded(this.mockEmitter));

            dispatchAction(setTheatreMode(false));
            assert.equal(this.mockEmitter.emit.callCount, 0, 'no event emitted');
        });

        // eslint-disable-next-line max-len
        QUnit.test('dispatching emitTransitionToRecommendedVOD should call .emit() on emitter instance', function(assert) {
            const vodId = QUnit.config.current.testId;
            this.dispatchAction(EventEmitterActions.emitTransitionToRecommendedVOD(vodId));
            assert.equal(this.mockEmitter.emit.callCount, 1, 'one event emitted');
            assert.equal(
                this.mockEmitter.emit.firstCall.args[0],
                TwitchEvents.TRANSITION_TO_RECOMMENDED_VOD,
                'correct event name emitted'
            );
            assert.deepEqual(this.mockEmitter.emit.firstCall.args[1], { vodId });
        });

        QUnit.test('dispatching emitTransitionToCollection should call .emit() on emitter instance', function(assert) {
            const vodId = QUnit.config.current.testId;
            const collectionId = 'random thing';
            this.dispatchAction(EventEmitterActions.emitTransitionToCollection(collectionId, vodId));
            assert.equal(this.mockEmitter.emit.callCount, 1, 'one event emitted');
            assert.equal(
                this.mockEmitter.emit.firstCall.args[0],
                TwitchEvents.TRANSITION_TO_COLLECTION_VOD,
                'correct event name emitted'
            );
            assert.deepEqual(this.mockEmitter.emit.firstCall.args[1], {
                vodId,
                collectionId,
            });
        });

        QUnit.test('dispatching emitAdPlayEvent should call .emit() on emitter instance', function(assert) {
            this.dispatchAction(EventEmitterActions.emitAdPlayEvent());
            assert.equal(this.mockEmitter.emit.callCount, 1, 'one event emitted');
            assert.equal(this.mockEmitter.emit.firstCall.args[0], 'play');
        });

        QUnit.test('dispatching emitAdPlayingEvent should call .emit() on emitter instance', function(assert) {
            this.dispatchAction(EventEmitterActions.emitAdPlayingEvent());
            assert.equal(this.mockEmitter.emit.callCount, 1, 'one event emitted');
            assert.equal(this.mockEmitter.emit.firstCall.args[0], 'playing');
        });

        QUnit.test('dispatching emitAdPauseEvent should call .emit() on emitter instance', function(assert) {
            this.dispatchAction(EventEmitterActions.emitAdPauseEvent());
            assert.equal(this.mockEmitter.emit.callCount, 1, 'one event emitted');
            assert.equal(this.mockEmitter.emit.firstCall.args[0], 'pause');
        });

        QUnit.test('action ACTION_EMIT_PLAYER_READY should call emit() on emitter instance', function(assert) {
            this.dispatchAction({
                type: EventEmitterActions.ACTION_EMIT_PLAYER_READY,
            });
            assert.equal(this.mockEmitter.emit.callCount, 1, 'one event emitted');
            assert.equal(
                this.mockEmitter.emit.firstCall.args[0],
                TwitchEvents.PLAYER_READY,
                `emits ${TwitchEvents.PLAYER_READY}`
            );
        });
    });
});
