import sinon from 'sinon';
import { createVideoApiMiddleware } from 'middleware/video-api-middleware';
import { videoAPILoaded, play, pause,
         changeVolume, mutePlayer, changeBackend,
         submitVideoIssueReport, changePlaybackRate, seek,
         automatedPause } from 'actions/video-api';
import { startCast, stopCast } from 'actions/chromecast';
import { waitFor } from 'tests/utils/waitFor';

class Video {
    constructor() {
        this.play = sinon.spy();
        this.pause = sinon.spy();
        this.automatedPause = sinon.spy();
        this.setVolume = sinon.spy();
        this.setMuted = sinon.spy();
        this.setBackend = sinon.spy();
        this.submitVideoIssueReport = sinon.spy();
        this.setPlaybackRate = sinon.spy();
        this.startCast = sinon.spy();
        this.stopCast = sinon.spy();
        this.setCurrentTime = sinon.spy();
    }
}

QUnit.module('middleware | video-api', function(hooks) {
    hooks.beforeEach(function() {
        const videoApiMiddleware = createVideoApiMiddleware();

        const dispatch = () => {};
        const getState = () => {};

        this.next = videoApiMiddleware({
            dispatch,
            getState,
        });

        this.dispatchAction = this.next(() => {});
    });

    QUnit.test('it must return an function to handle the action', function(assert) {
        const actionHandler = this.next();
        assert.equal(typeof actionHandler, 'function');
    });

    QUnit.test('it must pass the action into the next callback', function(assert) {
        const nextSpy = sinon.spy();
        const mockAction = { type: 'mock action' };
        const actionHandler = this.next(nextSpy);
        actionHandler(mockAction);

        const [dispatchedAction] = nextSpy.firstCall.args;
        assert.deepEqual(dispatchedAction, mockAction);
    });

    QUnit.module('videoAPILoaded', function() {
        QUnit.test('should use loaded video instance if one exists', function(assert) {
            assert.expect(0);
            const video = new Video();
            this.dispatchAction(videoAPILoaded(video));
            this.dispatchAction(play());
            return waitFor(() => video.play.callCount === 1);
        });

        QUnit.test('should chain actions to run when api is loaded', function(assert) {
            const video = new Video();
            this.dispatchAction(play());
            this.dispatchAction(pause());
            this.dispatchAction(videoAPILoaded(video));
            return waitFor(() => video.pause.callCount === 1).then(() => {
                assert.ok(video.play.calledBefore(video.pause), 'play called before pause');
            });
        });
    });

    QUnit.module('video api actions', function(hooks) {
        hooks.beforeEach(function() {
            this.mockVideo = new Video();
            this.dispatchAction(videoAPILoaded(this.mockVideo));
        });

        QUnit.test('dispatching play should call .play() on Video instance', function(assert) {
            assert.expect(0);
            this.dispatchAction(play());
            return waitFor(() => this.mockVideo.play.callCount === 1);
        });

        QUnit.test('dispatching pause should call .pause() on Video instance', function(assert) {
            assert.expect(0);
            this.dispatchAction(pause());
            return waitFor(() => this.mockVideo.pause.callCount === 1);
        });

        QUnit.test('dispatching automated pause should call .pause() on backend', function(assert) {
            assert.expect(0);
            this.dispatchAction(automatedPause());
            return waitFor(() => this.mockVideo.automatedPause.callCount === 1);
        });

        QUnit.test('dispatching changeVolume should call .setVolume() on Video instance', function(assert) {
            const newVolume = 5;
            this.dispatchAction(changeVolume(newVolume));
            return waitFor(() => this.mockVideo.setVolume.called).then(() => {
                assert.equal(this.mockVideo.setVolume.callCount, 1, 'setVolume is called');
                assert.equal(this.mockVideo.setVolume.firstCall.args[0], newVolume, 'new volume value is passed in');
            });
        });

        QUnit.test('dispatching mutePlayer should call .setMuted() on Video instance', function(assert) {
            const muted = false;
            const automated = false;
            this.dispatchAction(mutePlayer(muted, automated));
            return waitFor(() => this.mockVideo.setMuted.called).then(() => {
                assert.equal(this.mockVideo.setMuted.callCount, 1, 'setMuted is called');
                assert.equal(this.mockVideo.setMuted.firstCall.args[0], muted, 'muted value is passed in');
                assert.equal(this.mockVideo.setMuted.firstCall.args[1], automated, 'automated value is passed in');
            });
        });

        QUnit.test('dispatching changeBackend should call .setBackend() on Video instance', function(assert) {
            const backendType = 'backendtype';
            this.dispatchAction(changeBackend(backendType));
            return waitFor(() => this.mockVideo.setBackend.called).then(() => {
                assert.equal(this.mockVideo.setBackend.callCount, 1, 'setBackend is called');
                assert.equal(this.mockVideo.setBackend.firstCall.args[0], backendType, 'backend type is passed in');
            });
        });

        QUnit.test('reportVideoIsssue should call .submitVideoIssueReport() on Video instance', function(assert) {
            const issueType = 'too many ads';
            this.dispatchAction(submitVideoIssueReport(issueType));
            return waitFor(() => this.mockVideo.submitVideoIssueReport.called).then(() => {
                assert.equal(this.mockVideo.submitVideoIssueReport.callCount, 1, 'submitVideoIssueReport is called');
                assert.equal(
                    this.mockVideo.submitVideoIssueReport.firstCall.args[0],
                    issueType,
                    'issue type is passed in'
                );
            });
        });

        QUnit.test('dispatching changePlaybackRate should call .setPlaybackRate() on Video instance', function(assert) {
            const playbackRate = 2.0;
            this.dispatchAction(changePlaybackRate(playbackRate));
            return waitFor(() => this.mockVideo.setPlaybackRate.called).then(() => {
                assert.equal(this.mockVideo.setPlaybackRate.callCount, 1, 'setBackend is called');
                assert.equal(this.mockVideo.setPlaybackRate.firstCall.args[0], playbackRate, 'rate is passed in');
            });
        });

        QUnit.test('dispatching startCast should call .startCast on video instance', function(assert) {
            assert.expect(0);
            this.dispatchAction(startCast());
            return waitFor(() => this.mockVideo.startCast.callCount === 1);
        });

        QUnit.test('dispatching stopCast should call .stopCast on video instance', function(assert) {
            assert.expect(0);
            this.dispatchAction(stopCast());
            return waitFor(() => this.mockVideo.stopCast.callCount === 1);
        });

        QUnit.test('dispatching seek should call .setCurrentTime on video instance', function(assert) {
            const seekTime = 100;
            this.dispatchAction(seek(seekTime));
            return waitFor(() => this.mockVideo.setCurrentTime.called).then(() => {
                assert.equal(this.mockVideo.setCurrentTime.callCount, 1, 'setCurrentTime is called');
                assert.equal(this.mockVideo.setCurrentTime.firstCall.args[0], seekTime, 'seekTime is passed in');
            });
        });
    });
});
