import { init as initStore } from 'state';
import { OfflineRecommendationsManager } from 'offline-recommendations-manager';
import { unitTest } from '../../utils/module';
import { ACTION_SET_STREAM } from 'actions/stream';
import { ACTION_UPDATE_PLAYER_DIMENSIONS } from 'actions/player-dimensions';
import { LiveTwitchContentStream } from 'stream/twitch-live';
import { VOD_RECOMMENDATION_SCREEN, ACTION_PUSH_SCREEN, ADVERTISEMENT_SCREEN } from 'actions/screen';
import { isWatched, CHANNEL_VODS, ACTION_SET_RECOMMENDED_VODS,
         OFFLINE_RECOMMENDATIONS_TYPE } from 'actions/recommendations';
import { V5_VIDEOS_RESPONSE, NORMALIZED_V5_RECOMMENDED_VIDEOS } from 'tests/fixtures/recommendations';
import { TEST_NORMALIZED_FEATURED_COLLECTION_METADATA } from 'tests/fixtures/collection';
import { ACTION_FETCHED_FEATURED_COLLECTION } from 'actions/collection';
import { ACTION_SET_STREAMMETADATA } from 'actions/stream-metadata';
import assign from 'lodash/assign';
import find from 'lodash/find';
import { userLoggedIn } from 'actions/user';
import { ACTION_SET_ONLINE } from 'actions/online';
import { setPlayerType } from 'actions/env';
import { apiHost as API_HOST } from 'settings';
import { PLAYER_SITE } from 'util/player-type';
import { CONTENT_MODE_VOD } from 'stream/twitch-vod';
import { buildFakeWindow } from 'tests/fakes/window.fake';
import { setWindow } from 'actions/window';
import { ACTION_PLAYING, ACTION_ENDED } from 'actions/playback';
import { ACTION_SET_AD_METADATA, AdContentTypes, AdRollTypes } from 'actions/ads';
import * as api from 'api';
import sinon from 'sinon';

const MAX_JITTER_TIMEOUT = 5000;

unitTest('offline-recommendations-manager', function(hooks) {
    hooks.beforeEach(function() {
        this.store = initStore();
        this.userID = 123456;
        this.channelID = V5_VIDEOS_RESPONSE.videos[0].channel._id;
        this.channelName = V5_VIDEOS_RESPONSE.videos[0].channel.name;

        this.store.dispatch({
            type: ACTION_SET_STREAM,
            stream: new LiveTwitchContentStream(this.channelName, Promise.resolve({}), {}, {}, {}),
        });

        this.store.dispatch({
            type: ACTION_SET_STREAMMETADATA,
            streamMetadata: {
                channelName: this.channelName,
                channel: {
                    id: this.channelID,
                },
            },
        });

        this.store.dispatch(setPlayerType(PLAYER_SITE));
        this.store.dispatch({
            type: ACTION_SET_ONLINE,
            online: false,
        });

        this.store.dispatch({
            type: ACTION_UPDATE_PLAYER_DIMENSIONS,
            playerDimensions: {
                height: 300,
                width: 560,
            },
        });

        this.setTimeoutStub = sinon.stub().callsArg(0);
        this.store.dispatch(setWindow(buildFakeWindow({
            setTimeout: this.setTimeoutStub,
        })));

        this.api.expectV5VODs(this.channelID, V5_VIDEOS_RESPONSE);
        this.api.expectResumeTimes(this.userID);
        this.api.expectFeaturedCollection(this.channelID, true);
        this.api.setLoggedIn(true);

        this.getRecommendationsState = () => {
            return this.store.getState().recommendations;
        };

        this.setRecommendationType = (recommendations, type) => {
            return recommendations.map(r => assign({}, r, { recommendationType: type }));
        };
    });

    // eslint-disable-next-line max-len
    QUnit.module('when live stream is offline, not in collections, the player type is valid', function(hooks) {
        hooks.beforeEach(function() {
            this.store.dispatch(userLoggedIn({ id: this.userID }));
            this.store.dispatch(setPlayerType(PLAYER_SITE));
            sinon.spy(api, 'getFeaturedCollection');
            sinon.spy(this.store, 'dispatch');
        });

        hooks.afterEach(function() {
            api.getFeaturedCollection.restore();
            this.store.dispatch.restore();
        });

        // eslint-disable-next-line max-len
        QUnit.test('and content has not played, tryFetchOfflineRecs() fetches channel videos from correct endpoint and featured collections with 0 jitter', function(assert) {
            const expectedURL = `${API_HOST}/kraken/channels/${this.channelID}/videos?limit=30`;

            assert.equal(this.api.calls().matched.length, 0);
            assert.equal(this.setTimeoutStub.callCount, 0);
            // eslint-disable-next-line no-unused-vars
            const recommendationManager = new OfflineRecommendationsManager(this.store);

            return recommendationManager.tryFetchOfflineRecs().
                then(() => {
                    assert.equal(this.setTimeoutStub.callCount, 1);
                    assert.equal(this.setTimeoutStub.firstCall.args[1], 0, 'jitter period is 0');
                    const request = find(this.api.calls().matched, req => req[0] === expectedURL);
                    assert.ok(request, 'request made for channel videos');
                    assert.equal(api.getFeaturedCollection.callCount, 1, 'featured collection fetched');
                });
        });

        // eslint-disable-next-line max-len
        QUnit.test('and content has played, tryFetchOfflineRecs() fetches channel videos and collections from correct endpoint and featured collections with a non zero jitter', function(assert) {
            this.store.dispatch({ type: ACTION_PLAYING });
            this.store.dispatch({ type: ACTION_ENDED });

            const expectedURL = `${API_HOST}/kraken/channels/${this.channelID}/videos?limit=30`;

            assert.equal(this.api.calls().matched.length, 0);
            assert.equal(this.setTimeoutStub.callCount, 0);
            // eslint-disable-next-line no-unused-vars
            const recommendationManager = new OfflineRecommendationsManager(this.store);

            return recommendationManager.tryFetchOfflineRecs().
                then(() => {
                    assert.equal(this.setTimeoutStub.callCount, 1);
                    const jitterPeriod = this.setTimeoutStub.firstCall.args[1];
                    assert.ok(jitterPeriod >= 1, 'jitter period is 1 or greater');
                    assert.ok(jitterPeriod < (1 + MAX_JITTER_TIMEOUT), 'jitter period is less than 1 + max jitter');
                    const request = find(this.api.calls().matched, req => req[0] === expectedURL);
                    assert.ok(request, 'request made for channel videos');
                    assert.equal(api.getFeaturedCollection.callCount, 1, 'featured collection fetched');
                });
        });

        QUnit.test('tryFetchOfflineRecs() stores featured collection', function(assert) {
            // eslint-disable-next-line no-unused-vars
            const recommendationManager = new OfflineRecommendationsManager(this.store);

            return recommendationManager.tryFetchOfflineRecs().
                then(() => {
                    const collectionActionDispatchArgs = find(this.store.dispatch.args, arg => {
                        return arg[0].type === ACTION_FETCHED_FEATURED_COLLECTION;
                    });

                    assert.deepEqual(collectionActionDispatchArgs[0], {
                        type: ACTION_FETCHED_FEATURED_COLLECTION,
                        featuredCollection: TEST_NORMALIZED_FEATURED_COLLECTION_METADATA,
                    });
                });
        });

        // eslint-disable-next-line max-len
        QUnit.test('tryFetchOfflineRecs() stores normalized video recommendation with recommendationType set', function(assert) {
            const expectedRecommendations = this.setRecommendationType(
                NORMALIZED_V5_RECOMMENDED_VIDEOS,
                CHANNEL_VODS
            );

            // eslint-disable-next-line no-unused-vars
            const recommendationManager = new OfflineRecommendationsManager(this.store);

            return recommendationManager.tryFetchOfflineRecs().
                then(() => {
                    assert.deepEqual(this.getRecommendationsState().videos, expectedRecommendations);
                });
        });

        QUnit.test('tryFetchOfflineRecs() filters out watched videos', function(assert) {
            const resumeTimesResponse = {
                videos: [
                    /* eslint-disable camelcase */
                    {
                        video_id: V5_VIDEOS_RESPONSE.videos[0]._id,
                        position: V5_VIDEOS_RESPONSE.videos[0].length - 30,
                        type: CONTENT_MODE_VOD,
                    }, // watched
                    {
                        video_id: V5_VIDEOS_RESPONSE.videos[1]._id,
                        position: V5_VIDEOS_RESPONSE.videos[1].length * 0.05,
                        type: CONTENT_MODE_VOD,
                    }, // unwatched
                    {
                        video_id: V5_VIDEOS_RESPONSE.videos[2]._id,
                        position: 0,
                        type: CONTENT_MODE_VOD,
                    }, // watched
                    {
                        video_id: V5_VIDEOS_RESPONSE.videos[3]._id,
                        position: V5_VIDEOS_RESPONSE.videos[3].length * 0.5,
                        type: CONTENT_MODE_VOD,
                    }, // watched
                    // all other video resume times are undefined so unwatched
                    /* eslint-enable camelcase */
                ],
            };

            const expectedFilteredRecommendations = this.setRecommendationType(
                NORMALIZED_V5_RECOMMENDED_VIDEOS,
                CHANNEL_VODS
            ).filter(rec => !isWatched(resumeTimesResponse.videos, rec));

            this.api.expectResumeTimes(this.userID, resumeTimesResponse);

            // eslint-disable-next-line no-unused-vars
            const recommendationManager = new OfflineRecommendationsManager(this.store);

            return recommendationManager.tryFetchOfflineRecs().
                then(() => {
                    assert.deepEqual(this.getRecommendationsState().videos, expectedFilteredRecommendations);
                });
        });
    });

    QUnit.module('after storing recommendations, stream is offline, and player is valid size', function(hooks) {
        hooks.beforeEach(function() {
            this.store.dispatch({
                type: ACTION_SET_RECOMMENDED_VODS,
                recommendationsType: OFFLINE_RECOMMENDATIONS_TYPE,
                videos: NORMALIZED_V5_RECOMMENDED_VIDEOS,
            });

            this.store.dispatch({
                type: ACTION_UPDATE_PLAYER_DIMENSIONS,
                playerDimensions: {
                    height: 300,
                    width: 560,
                },
            });
        });

        QUnit.test('tryShowOfflineRecs() pushes vod recommendation screen if stream has not played', function(assert) {
            assert.notOk(this.store.getState().playback.hasPlayed, 'stream has not played');
            assert.notEqual(this.store.getState().screen[0], VOD_RECOMMENDATION_SCREEN);
            // eslint-disable-next-line no-unused-vars
            const recommendationManager = new OfflineRecommendationsManager(this.store);
            recommendationManager.tryShowOfflineRecs();

            assert.equal(this.store.getState().screen[0], VOD_RECOMMENDATION_SCREEN);
        });

        QUnit.test('tryShowOfflineRecs() pushes vod recommendation screen if stream has ended', function(assert) {
            assert.notEqual(this.store.getState().screen[0], VOD_RECOMMENDATION_SCREEN);
            this.store.dispatch({ type: ACTION_ENDED });

            // eslint-disable-next-line no-unused-vars
            const recommendationManager = new OfflineRecommendationsManager(this.store);
            recommendationManager.tryShowOfflineRecs();

            assert.equal(this.store.getState().screen[0], VOD_RECOMMENDATION_SCREEN);
        });

        // eslint-disable-next-line max-len
        QUnit.test('tryShowOfflineRecs() does not push vod recommendation screen if ads are showing', function(assert) {
            assert.notEqual(this.store.getState().screen[0], VOD_RECOMMENDATION_SCREEN);
            this.store.dispatch({
                type: ACTION_SET_AD_METADATA,
                currentMetadata: {
                    rollType: AdRollTypes.POSTROLL,
                    contentType: AdContentTypes.IMA,
                },
            });
            // eslint-disable-next-line no-unused-vars
            const recommendationManager = new OfflineRecommendationsManager(this.store);
            recommendationManager.tryShowOfflineRecs();

            assert.notEqual(this.store.getState().screen[0], VOD_RECOMMENDATION_SCREEN);
        });

        // eslint-disable-next-line max-len
        QUnit.test('tryShowOfflineRecs() does not push vod recommendation screen if vod screen is already on the stack', function(assert) {
            assert.notEqual(this.store.getState().screen[0], VOD_RECOMMENDATION_SCREEN);
            this.store.dispatch({
                type: ACTION_PUSH_SCREEN,
                screen: VOD_RECOMMENDATION_SCREEN,
            });
            this.store.dispatch({
                type: ACTION_PUSH_SCREEN,
                screen: ADVERTISEMENT_SCREEN,
            });

            // eslint-disable-next-line no-unused-vars
            const recommendationManager = new OfflineRecommendationsManager(this.store);
            recommendationManager.tryShowOfflineRecs();

            assert.notEqual(this.store.getState().screen[0], VOD_RECOMMENDATION_SCREEN);
        });

        // eslint-disable-next-line max-len
        QUnit.test('onDimensionsChange() pops vod screen if vod screen is on top and dimensions changes to an invalid height', function(assert) {
            this.store.dispatch({
                type: ACTION_PUSH_SCREEN,
                screen: VOD_RECOMMENDATION_SCREEN,
            });

            this.store.dispatch({
                type: ACTION_UPDATE_PLAYER_DIMENSIONS,
                playerDimensions: {
                    height: 299,
                    width: 560,
                },
            });

            assert.equal(this.store.getState().screen[0], VOD_RECOMMENDATION_SCREEN);

            // eslint-disable-next-line no-unused-vars
            const recommendationManager = new OfflineRecommendationsManager(this.store);
            recommendationManager.onDimensionsChange();

            assert.notEqual(this.store.getState().screen[0], VOD_RECOMMENDATION_SCREEN);
        });

        // eslint-disable-next-line max-len
        QUnit.test('onDimensionsChange() pops vod screen if vod screen is on top and dimensions changes to an invalid height', function(assert) {
            this.store.dispatch({
                type: ACTION_PUSH_SCREEN,
                screen: VOD_RECOMMENDATION_SCREEN,
            });

            this.store.dispatch({
                type: ACTION_UPDATE_PLAYER_DIMENSIONS,
                playerDimensions: {
                    height: 300,
                    width: 559,
                },
            });

            assert.equal(this.store.getState().screen[0], VOD_RECOMMENDATION_SCREEN);

            // eslint-disable-next-line no-unused-vars
            const recommendationManager = new OfflineRecommendationsManager(this.store);
            recommendationManager.onDimensionsChange();

            assert.notEqual(this.store.getState().screen[0], VOD_RECOMMENDATION_SCREEN);
        });
    });
});
