import { unitTest } from 'tests/utils/module';
import { waitFor } from 'tests/utils/waitFor';
import { localStore } from 'tests/fakes/local-store.fake';
import assign from 'lodash/assign';
import * as sentinel from 'analytics/sentinel';
import sinon from 'sinon';
import Player from 'player'; // eslint-disable-line no-unused-vars
import { BACKEND_PLAYER_CORE } from 'backend/player-core';
import { BACKEND_MEDIA_PLAYER } from 'backend/mediaplayer';

function createPlayer() {
    const options = {
        width: 854,
        height: 480,
        channel: 'monstercat',
    };
    return new Twitch.Player('root', options);
}

unitTest('player', function(hooks) {
    hooks.beforeEach(function() {
        const $root = window.document.createElement('div');
        $root.id = 'root';
        window.document.getElementById('qunit-fixture').appendChild($root);

        sinon.stub(sentinel, 'getSentinel').returns(Promise.resolve(true));
    });

    hooks.afterEach(function() {
        sentinel.getSentinel.restore();
        delete window.Twitch.sentinel;
    });

    QUnit.test('can instantiate player within root div', function(assert) {
        const player = createPlayer();
        player.setVolume(0.5);
        assert.ok(player, 'Player was instantiated!');
    });

    QUnit.test('can destroy the player instance', function(assert) {
        const player = createPlayer();
        assert.notDeepEqual(player.getEventEmitter()._getEvents(), {}, 'Player has events!');
        player.destroy();
        assert.deepEqual(player.getEventEmitter()._getEvents(), {}, 'All player events destroyed!');
    });

    QUnit.test('recordClip() is defined', function(assert) {
        const player = createPlayer();
        assert.equal(typeof player.recordClip, 'function');
    });

    QUnit.test('isPlaying() is defined', function(assert) {
        const player = createPlayer();
        assert.equal(typeof player.isPlaying, 'function');
    });

    // TODO actually test isPlaying behavior
    // (waiting on reimplementation after https://git-aws.internal.justin.tv/video/player-ui/pull/1366)
    // UPDATE: The aforementioned PR has been killed :(
    //         The things it is trying to fix are still in the works though, so doing this is
    //         still deferred, as the current implementation is pretty much untestable
    QUnit.skip('isPlaying() returns true if the player is playing', function() {

    });

    QUnit.skip('isPlaying returns false if the player isn\'t playing', function() {

    });

    QUnit.test('calling setChannel should check adblock status before setting channel', function(assert) {
        const player = createPlayer();
        player.setChannel('streamerhouse');
        assert.equal(sentinel.getSentinel.callCount, 2);
        player.destroy();
    });

    QUnit.module('setVideo', function(hooks) {
        hooks.beforeEach(function() {
            this.player = createPlayer();
        });

        hooks.afterEach(function() {
            this.player.destroy();
        });

        QUnit.test('should check adblock status before setting video', function(assert) {
            this.player.setVideo('v123456123');
            assert.equal(sentinel.getSentinel.callCount, 2);
        });

        QUnit.test('accepts digits only vod id strings', function(assert) {
            assert.expect(0);
            const videoId = '123456123';
            this.player.setVideo(videoId);

            return waitFor(() => this.player.getVideo() === `v${videoId}`);
        });

        QUnit.test('accepts v prefixed digits vod id strings', function(assert) {
            assert.expect(0);
            const videoId = 'v123456123';
            this.player.setVideo(videoId);

            return waitFor(() => this.player.getVideo() === videoId);
        });
    });

    QUnit.module('setLiveToVod', function(hooks) {
        hooks.beforeEach(function() {
            this.streamId = 123123;
            this.vodId = 123123;
            this.channelId = '99999';
            this.api.setLoggedIn(false);

            const options = {
                width: 854,
                height: 480,
            };
            this.player = new Twitch.Player('root', options);
        });

        hooks.afterEach(function() {
            this.api.clearFakeResponses();
            this.player.destroy();
        });

        QUnit.test('when endpoint responds with live stream, we call setChannel', function(assert) {
            const setChannelSpy = sinon.spy(this.player, 'setChannel');

            const ltvResponse = {
                stream: {
                    _id: this.streamId,
                },
                channel: {
                    name: 'someName',
                },
                videos: [],
            };

            this.api.expectLTVStream(ltvResponse);
            this.player._setLiveToVod(this.streamId.toString(), this.channelId);

            return waitFor(() => setChannelSpy.called).then(() => {
                assert.ok(
                    setChannelSpy.calledWith(ltvResponse.channel.name),
                    'setChannel should be called with returned channel name'
                );
            });
        });

        QUnit.test('when endpoint responds only with archive vod, we call setVideo', function(assert) {
            const setVideoSpy = sinon.spy(this.player, 'setVideo');

            const ltvResponse = {
                stream: null,
                videos: [
                    {
                        _id: this.vodId,
                        // eslint-disable-next-line camelcase
                        broadcast_type: 'archive',
                    },
                ],
                channel: {
                    name: 'someName',
                },
            };

            this.api.expectLTVStream(ltvResponse);
            this.player._setLiveToVod(this.streamId.toString(), this.channelId);

            return waitFor(() => setVideoSpy.called).then(() => {
                assert.ok(setVideoSpy.calledWith(this.vodId), 'setVideo should be called with returned vod id');
            });
        });

        // eslint-disable-next-line max-len
        QUnit.test('when endpoint responds only with highlight vod, we call setVideo', function(assert) {
            const setVideoSpy = sinon.spy(this.player, 'setVideo');

            const ltvResponse = {
                stream: null,
                videos: [
                    {
                        _id: this.vodId,
                        // eslint-disable-next-line camelcase
                        broadcast_type: 'highlight',
                    },
                ],
                channel: {
                    name: 'someName',
                },
            };

            this.api.expectLTVStream(ltvResponse);
            this.player._setLiveToVod(this.streamId.toString(), this.channelId);

            return waitFor(() => setVideoSpy.called).then(() => {
                assert.ok(setVideoSpy.calledWith(this.vodId), 'setVideo should be called with returned vod id');
            });
        });

        // eslint-disable-next-line max-len
        QUnit.test('when endpoint responds wth both vod types, we call setVideo on archive vod', function(assert) {
            const setVideoSpy = sinon.spy(this.player, 'setVideo');

            const ltvResponse = {
                stream: null,
                videos: [
                    {
                        _id: this.vodId + 1,
                        // eslint-disable-next-line camelcase
                        broadcast_type: 'highlight',
                    },
                    {
                        _id: this.vodId,
                        // eslint-disable-next-line camelcase
                        broadcast_type: 'archive',
                    },
                ],
                channel: {
                    name: 'someName',
                },
            };

            this.api.expectLTVStream(ltvResponse);
            this.player._setLiveToVod(this.streamId.toString(), this.channelId);

            return waitFor(() => setVideoSpy.called).then(() => {
                assert.ok(setVideoSpy.calledWith(this.vodId), 'setVideo should be called with archive vod id');
            });
        });
    });

    QUnit.test('when in standalone player, requests result from Sentinel for adblock detection', function(assert) {
        const player = createPlayer();

        assert.equal(sentinel.getSentinel.callCount, 1);
        player.destroy();
    });

    QUnit.module('when the player is inline in the site', function(hooks) {
        hooks.beforeEach(function() {
            assign(window.Twitch, {
                sentinel: {
                    detect: {
                        _result: true,
                    },
                },
            });
        });

        hooks.afterEach(function() {
            delete window.Twitch.sentinel;
        });

        QUnit.test('when in site player, uses page result from Sentinel for adblock detection', function(assert) {
            const player = createPlayer();
            assert.equal(sentinel.getSentinel.callCount, 0);
            player.destroy();
        });
    });

    QUnit.test('transitions local store `backend` value from `player-core` to `mediaplayer`', function(assert) {
        localStore.set('backend', BACKEND_PLAYER_CORE);
        const player = createPlayer();
        assert.equal(localStore.get('backend'), BACKEND_MEDIA_PLAYER);
        player.destroy();
    });
});
