import assign from 'lodash/assign';
import PubSubClient from 'pubsub-js-client';
import { ACTION_SET_STREAM } from 'actions/stream';
import { ACTION_SET_STREAMMETADATA } from 'actions/stream-metadata';
import { updateViewerCount } from 'actions/viewercount';
import { setOnline } from 'actions/online';
import { setWindow } from 'actions/window';
import { LiveTwitchContentStream } from 'stream/twitch-live';
import { VODTwitchContentStream } from 'stream/twitch-vod';
import { PubSub } from 'pubsub';
import { init as initStore } from 'state';
import sinon from 'sinon';
import { unitTest } from 'tests/utils/module';
import { waitFor } from 'tests/utils/waitFor';
import { DEFAULT_STREAM_METADATA } from 'tests/fixtures/stream-metadata';

function setupLiveStream(dispatch, prefix = '') {
    const channelIdent = `${prefix}_${Math.floor(Math.random() * 1e9).toString(36).toLowerCase()}`;
    const channelName = `${prefix}_channel_${channelIdent}`;
    const streamMetadata = assign({}, DEFAULT_STREAM_METADATA, {
        channelName: channelName,
        channel: {
            name: channelName,
            id: channelIdent,
        },
    });

    const stream = new LiveTwitchContentStream(channelName, Promise.resolve(null), {}, {});

    dispatch({
        type: ACTION_SET_STREAM,
        stream: stream,
    });

    dispatch({
        type: ACTION_SET_STREAMMETADATA,
        streamMetadata,
    });
}

function setupVODStream(dispatch) {
    const vodID = `v${parseInt(QUnit.config.current.testId.toLowerCase(), 36)}`;

    const streamMetadata = assign({}, DEFAULT_STREAM_METADATA, {
        videoId: vodID,
        channel: {
            name: 'channelName',
            id: 'channelId',
        },
    });

    const stream = new VODTwitchContentStream(vodID, Promise.resolve(null), {}, {});

    dispatch({
        type: ACTION_SET_STREAM,
        stream: stream,
    });

    dispatch({
        type: ACTION_SET_STREAMMETADATA,
        streamMetadata,
    });

    return stream;
}

unitTest('pubsub', function(hooks) {
    hooks.beforeEach(function() {
        this.store = initStore();

        this.fakeClient = PubSubClient.getInstance();
        this.fakeClient._reset();
    });

    hooks.afterEach(function() {
        this.fakeClient._reset();
    });

    QUnit.test('listens for stream topic when state stream changed to live stream', function(assert) {
        new PubSub(this.store, {});

        setupLiveStream(this.store.dispatch);
        const streamMetadata = this.store.getState().streamMetadata;

        return waitFor(() => (this.fakeClient.Listen.callCount === 1)).then(() => {
            assert.ok(this.fakeClient.Listen.calledWith(sinon.match({
                topic: `video-playback-by-id.${streamMetadata.channel.id}`,
                message: sinon.match.func,
            })));
        });
    });

    QUnit.test('unlistens from stream topic when state stream changed to different live stream', function(assert) {
        new PubSub(this.store, {});

        setupLiveStream(this.store.dispatch, 'old');
        const oldStreamMetadata = this.store.getState().streamMetadata;
        setupLiveStream(this.store.dispatch, 'new');

        return waitFor(() => (this.fakeClient.Unlisten.callCount === 1)).then(() => {
            assert.ok(this.fakeClient.Unlisten.calledWith(sinon.match({
                topic: `video-playback-by-id.${oldStreamMetadata.channel.id}`,
                message: this.fakeClient.Listen.firstCall.args[0].message,
            })), 'should call unlisten on last set channel');
        });
    });

    QUnit.test('does not listen to stream topic for VOD stream', function(assert) {
        new PubSub(this.store, {});
        setupVODStream(this.store.dispatch);
        assert.equal(this.fakeClient.Listen.callCount, 0);
    });

    QUnit.test('does not unlisten to stream topic when switched from VOD stream', function(assert) {
        new PubSub(this.store, {});
        setupVODStream(this.store.dispatch, 'old');
        setupLiveStream(this.store.dispatch, 'new');

        assert.equal(this.fakeClient.Unlisten.callCount, 0, 'Unlisten should not be called');
    });

    QUnit.module('when listening to a specific topic', function(hooks) {
        hooks.beforeEach(function() {
            this.pubsub = new PubSub(this.store, {});
            setupLiveStream(this.store.dispatch);

            sinon.spy(this.store, 'dispatch');
            sinon.spy(console, 'error');
        });

        hooks.afterEach(function() {
            // eslint-disable-next-line no-console
            console.error.restore();
        });

        QUnit.test('on `stream-up` message, attempts to load live stream', function(assert) {
            const liveStream = this.store.getState().stream;
            const pubsubMessage = {
                type: 'stream-up',
            };

            this.fakeClient.sendMessage(pubsubMessage);

            return waitFor(() => this.store.dispatch.withArgs(sinon.match.func).firstCall).
                then(actionCall => {
                    const dispatch = sinon.spy();
                    const action = actionCall.args[0];

                    action(dispatch, this.store.getState);
                    const setStreamSpy = dispatch.withArgs(sinon.match({ type: ACTION_SET_STREAM }));

                    assert.equal(setStreamSpy.callCount, 1);
                    assert.equal(setStreamSpy.firstCall.args[0].stream.channel, liveStream.channel);
                    assert.notEqual(setStreamSpy.firstCall.args[0].stream, liveStream);
                });
        });

        QUnit.test('on `stream-down` message, changes online state to "offline"', function(assert) {
            const pubsubMessage = {
                type: 'stream-down',
            };

            this.fakeClient.sendMessage(pubsubMessage);

            assert.ok(this.store.dispatch.calledWith(setOnline(false)));
        });

        QUnit.test('on `viewcount` message, updates the viewer count', function(assert) {
            const pubsubMessage = {
                type: 'viewcount',
                viewers: Math.floor(Math.random() * 1000),
            };

            this.fakeClient.sendMessage(pubsubMessage);

            assert.ok(this.store.dispatch.calledWith(updateViewerCount(pubsubMessage.viewers)));
        });

        QUnit.test('on `tos-strike` message, reloads the page', function(assert) {
            const pubsubMessage = {
                type: 'tos-strike',
            };
            const win = {
                document: {
                    location: {
                        reload: sinon.spy(),
                    },
                },
            };
            this.store.dispatch(setWindow(win));

            this.fakeClient.sendMessage(pubsubMessage);

            assert.equal(win.document.location.reload.callCount, 1);
        });

        QUnit.test('abort when the message is not a valid JSON string', function(assert) {
            const pubsubMessage = 'lol not actually json';
            this.fakeClient.sendRawMessage(pubsubMessage);

            assert.equal(this.store.dispatch.callCount, 0);
            // eslint-disable-next-line no-console
            assert.equal(console.error.callCount, 1);
        });
    });

    QUnit.module('destroy', function() {
        QUnit.test('unsubscribes from store', function(assert) {
            const pubsub = new PubSub(this.store, {});

            pubsub.destroy();

            setupLiveStream(this.store.dispatch);

            assert.equal(this.fakeClient.Listen.callCount, 0);
            assert.equal(this.fakeClient.Unlisten.callCount, 0);
        });

        QUnit.test('unlistens to live channel', function(assert) {
            const pubsub = new PubSub(this.store, {});
            setupLiveStream(this.store.dispatch);
            const streamMetadata = this.store.getState().streamMetadata;

            assert.equal(this.fakeClient.Unlisten.callCount, 0, 'Unlisten should not be called');

            pubsub.destroy();

            assert.ok(this.fakeClient.Unlisten.calledWith(sinon.match({
                topic: `video-playback-by-id.${streamMetadata.channel.id}`,
                message: this.fakeClient.Listen.firstCall.args[0].message,
            })));
        });

        QUnit.test('does not unlisten to vod stream', function(assert) {
            const pubsub = new PubSub(this.store, {});
            setupVODStream(this.store.dispatch);

            assert.equal(this.fakeClient.Unlisten.callCount, 0);

            pubsub.destroy();

            assert.equal(this.fakeClient.Unlisten.callCount, 0);
        });
    });
});
