import { PlayerUIResume } from '../../../src/js/resume-vod';
import * as MediaEvent from 'backend/events/media-event';
import { init as initStore } from 'state';
import * as ResumeWatchActions from 'actions/resume-watch';
import * as PlaybackActions from 'actions/playback';
import * as CollectionActions from 'actions/collection';
import { setWindow } from 'actions/window';
import * as Settings from 'settings';
import { setAnalyticsTracker } from 'actions/analytics-tracker';
import { setOnline } from 'actions/online';
import { ACTION_SET_STREAM, TYPE_VIDEO, TYPE_CHANNEL,
         setStream } from 'actions/stream';
import { CONTENT_MODE_VOD, VODTwitchContentStream } from 'stream/twitch-vod';
import { CONTENT_MODE_LIVE } from 'stream/twitch-live';
import { oauthToken } from 'api';
import { TEST_COLLECTION } from 'tests/fixtures/collection';
import { waitFor } from 'tests/utils/waitFor';
import { unitTest } from 'tests/utils/module';
import sinon from 'sinon';

unitTest('resume-vod', function(hooks) {
    hooks.beforeEach(function() {
        this.state = {
            addEventListener() {},
        };
        this.store = initStore();

        this.tracker = {
            trackEvent: sinon.spy(),
        };
        this.store.dispatch(setAnalyticsTracker(this.tracker));

        this.player = {
            setCurrentTime: sinon.spy(),
        };
        this.options = {
            time: null,
        };
        this.window = {
            document: {},
            Date: {
                now() {
                    return 1e8;
                },
            },
        };

        this.store.dispatch(setWindow(this.window));
        sinon.spy(this.store, 'dispatch');

        this.broadcastID = `${parseInt(QUnit.config.current.testId, 36)}`;
        this.channelID = `${Math.floor(Math.random() * 1e8)}`;
        this.videoID = `v${Math.floor(Math.random() * 1e8)}`;

        /* eslint-disable camelcase */
        this.videoData = {
            broadcast_id: this.broadcastID,
            channel: {
                name: 'channeldName',
                partner: false,
            },
        };

        /* eslint-enable camelcase */

        oauthToken.cache.clear();

        this.api.setLoggedIn(true, {});
        const videoDataResponse = this.api.expectVideoInfo(this.videoID, this.videoData);
        const channelDataResponse = this.api.expectChannelInfo(this.videoData.channel.name, this.videoData.channel);
        this.videoInfoResponse = videoDataResponse;
        this.videoInfoResponse.channel = channelDataResponse;
    });

    QUnit.module('when a VOD is loaded and has no resume time', function(hooks) {
        hooks.beforeEach(function() {
            this.store.dispatch(setStream({
                contentType: TYPE_VIDEO,
                contentId: this.videoID,
            }));
            this.duration = 30 * Settings.cancelResumeAmount;
            this.store.dispatch(PlaybackActions.updateDuration(this.duration));
        });

        QUnit.module('if the user is not logged in', function() {
            QUnit.test('a duration update should call _seekToResumeTimeLocal', function(assert) {
                const subject = new PlayerUIResume(this.player, this.state, this.store, this.options);
                sinon.spy(subject, '_seekToResumeTimeLocal');

                subject.handleEvent(MediaEvent.LOADED_METADATA);

                return waitFor(() => subject._seekToResumeTimeLocal.called).
                then(() => {
                    assert.ok(subject._seekToResumeTimeLocal.calledOnce);
                    assert.deepEqual(subject._seekToResumeTimeLocal.firstCall.args[0], this.videoInfoResponse);
                    assert.ok(subject._seekToResumeTimeLocal.calledWith(this.videoInfoResponse));
                });
            });

            QUnit.test('_seekToResumeTimeLocal should not seek in the video', function(assert) {
                this.store.dispatch(PlaybackActions.updateDuration(this.duration));
                const subject = new PlayerUIResume(this.player, this.state, this.store, this.options);
                subject._seekToResumeTimeLocal({
                    // eslint-disable-next-line camelcase
                    broadcast_id: this.broadcastID,
                });
                assert.notOk(this.player.setCurrentTime.called);
            });
        });

        QUnit.module('if the user is logged in', function(hooks) {
            hooks.beforeEach(function() {
                this.userId = '12345';
                this.store.dispatch(ResumeWatchActions.setUser(this.userId));
            });
            QUnit.test('a duration update should call _seekToResumeTime', function(assert) {
                const subject = new PlayerUIResume(this.player, this.state, this.store, this.options);
                sinon.spy(subject, '_seekToResumeTime');

                this.api.expectResumeTimes(this.userId);

                subject.handleEvent(MediaEvent.LOADED_METADATA);

                return waitFor(() => subject._seekToResumeTime.called).
                then(() => {
                    assert.ok(subject._seekToResumeTime.calledOnce);
                    assert.ok(subject._seekToResumeTime.calledWith(this.videoInfoResponse, []));
                });
            });
            QUnit.test('_seekToResumeTime should not seek in the video', function(assert) {
                this.store.dispatch(PlaybackActions.updateDuration(this.duration));
                const subject = new PlayerUIResume(this.player, this.state, this.store, this.options);
                subject._seekToResumeTime({
                    // eslint-disable-next-line camelcase
                    broadcast_id: this.broadcastID,
                }, []);
                assert.notOk(this.player.setCurrentTime.called);
            });
        });
    });

    QUnit.module('when a VOD is loaded and has only a VOD resume time', function(hooks) {
        hooks.beforeEach(function() {
            this.store.dispatch(PlaybackActions.setLoading(false));
            this.store.dispatch(setStream({
                contentType: TYPE_VIDEO,
                contentId: this.videoID,
            }));
            this.resumeTime = 30 * Settings.cancelResumeAmount;
            this.duration = 60 * Settings.cancelResumeAmount;
            this.store.dispatch(PlaybackActions.updateDuration(this.duration));
        });

        QUnit.module('if the user is not logged in', function(hooks) {
            hooks.beforeEach(function() {
                this.store.dispatch(
                    ResumeWatchActions.setVodResumeTime(this.videoID, this.channelID, this.resumeTime)
                );
            });

            QUnit.test('a duration update should call _seekToResumeTimeLocal', function(assert) {
                const subject = new PlayerUIResume(this.player, this.state, this.store, this.options);
                sinon.spy(subject, '_seekToResumeTimeLocal');

                subject.handleEvent(MediaEvent.LOADED_METADATA);

                return waitFor(() => subject._seekToResumeTimeLocal.called).
                then(() => {
                    assert.ok(subject._seekToResumeTimeLocal.calledOnce);
                    assert.ok(subject._seekToResumeTimeLocal.calledWith(this.videoInfoResponse));
                });
            });

            QUnit.test('_seekToResumeTimeLocal should seek there', function(assert) {
                this.store.dispatch(PlaybackActions.updateDuration(this.duration));
                const subject = new PlayerUIResume(this.player, this.state, this.store, this.options);

                subject._seekToResumeTimeLocal({
                    // eslint-disable-next-line camelcase
                    broadcast_id: this.broadcastID,
                });

                assert.deepEqual(
                    this.player.setCurrentTime.firstCall.args,
                    [this.resumeTime]
                );
            });
        });
        QUnit.module('if the user is logged in', function(hooks) {
            hooks.beforeEach(function() {
                this.userId = '12345';
                this.store.dispatch(ResumeWatchActions.setUser(this.userId));
                this.updatedAt = Date.now();
                this.videoInfo = {
                    video_id: this.videoID, // eslint-disable-line camelcase
                    position: this.resumeTime,
                    type: CONTENT_MODE_VOD,
                    updated_at: this.updatedAt, // eslint-disable-line camelcase
                };

                this.api.expectResumeTimes(this.userId, [this.videoInfo]);
                this.store.dispatch(PlaybackActions.updateDuration(this.duration));
            });
            QUnit.test('a duration update should call _seekToResumeTime', function(assert) {
                const subject = new PlayerUIResume(this.player, this.state, this.store, this.options);
                sinon.spy(subject, '_seekToResumeTime');

                subject.handleEvent(MediaEvent.LOADED_METADATA);

                return waitFor(() => subject._seekToResumeTime.called).
                then(() => {
                    assert.ok(subject._seekToResumeTime.calledOnce);
                    assert.ok(subject._seekToResumeTime.calledWith(this.videoInfoResponse, [this.videoInfo]));
                });
            });

            QUnit.test('_seekToResumeTime should seek there', function(assert) {
                this.store.dispatch(PlaybackActions.updateDuration(this.duration));
                const subject = new PlayerUIResume(this.player, this.state, this.store, this.options);

                subject._seekToResumeTime({
                    // eslint-disable-next-line camelcase
                    broadcast_id: this.broadcastID,
                }, [this.videoInfo]);

                assert.deepEqual(
                    this.player.setCurrentTime.firstCall.args,
                    [this.resumeTime]
                );
            });
        });
    });

    QUnit.module('when a VOD is loaded and has only a livestream resume time', function(hooks) {
        hooks.beforeEach(function() {
            this.store.dispatch(PlaybackActions.setLoading(false));
            this.store.dispatch(setStream({
                contentType: TYPE_VIDEO,
                contentId: this.videoID,
            }));
            this.resumeTime = 30 * Settings.cancelResumeAmount;
            this.duration = 60 * Settings.cancelResumeAmount;
            this.store.dispatch(
                ResumeWatchActions.setLivestreamResumeTime(this.broadcastID, this.channelID, this.resumeTime)
            );
            this.store.dispatch(PlaybackActions.updateDuration(this.duration));
        });

        QUnit.module('if the user is not logged in', function() {
            QUnit.test('a duration update should call _seekToResumeTimeLocal', function(assert) {
                const subject = new PlayerUIResume(this.player, this.state, this.store, this.options);
                sinon.spy(subject, '_seekToResumeTimeLocal');

                subject.handleEvent(MediaEvent.LOADED_METADATA);

                return waitFor(() => subject._seekToResumeTimeLocal.called).
                then(() => {
                    assert.ok(subject._seekToResumeTimeLocal.calledOnce);
                    assert.ok(subject._seekToResumeTimeLocal.calledWith(this.videoInfoResponse));
                });
            });

            QUnit.test('_seekToResumeTimeLocal should seek there', function(assert) {
                const subject = new PlayerUIResume(this.player, this.state, this.store, this.options);

                subject._seekToResumeTimeLocal({
                    // eslint-disable-next-line camelcase
                    broadcast_id: this.broadcastID,
                });

                assert.deepEqual(
                    this.player.setCurrentTime.firstCall.args,
                    [this.resumeTime - Settings.livestreamResumePushback]
                );
            });

            // eslint-disable-next-line max-len
            QUnit.test('_seekToResumeTimeLocal should remove livestream resume time from local storage', function(assert) {
                const dispatch = sinon.spy();

                const subject = new PlayerUIResume(this.player, this.state, this.store, this.options);

                subject._seekToResumeTimeLocal({
                    // eslint-disable-next-line camelcase
                    broadcast_id: this.broadcastID,
                });
                const action = this.store.dispatch.getCall(this.store.dispatch.callCount - 1).args[0];
                action(dispatch, this.store.getState);

                assert.ok(dispatch.calledWith({
                    type: ResumeWatchActions.ACTION_LIVESTREAM_CANCEL_RESUME,
                    broadcastID: this.broadcastID,
                }));
            });

            QUnit.test('_seekToResumeTimeLocal should set isSeeked to true', function(assert) {
                const subject = new PlayerUIResume(this.player, this.state, this.store, this.options);

                assert.ok(!this.store.getState().resumeWatch.isSeeked);

                subject._seekToResumeTimeLocal({
                    // eslint-disable-next-line camelcase
                    broadcast_id: this.broadcastID,
                });

                assert.ok(this.store.getState().resumeWatch.isSeeked);
            });
        });

        QUnit.module('if the user is logged in', function(hooks) {
            hooks.beforeEach(function() {
                this.userId = '12345';
                this.store.dispatch(ResumeWatchActions.setUser(this.userId));
                this.updatedAt = Date.now();
                this.videoInfo = {
                    video_id: this.broadcastID, // eslint-disable-line camelcase
                    position: this.resumeTime,
                    type: CONTENT_MODE_LIVE,
                    updated_at: this.updatedAt, // eslint-disable-line camelcase
                };
                this.api.expectResumeTimes(this.userId, [this.videoInfo]);
                this.store.dispatch(PlaybackActions.updateDuration(this.duration));
            });

            QUnit.test('a duration update should call _seekToResumeTime', function(assert) {
                const subject = new PlayerUIResume(this.player, this.state, this.store, this.options);

                sinon.spy(subject, '_seekToResumeTime');

                subject.handleEvent(MediaEvent.LOADED_METADATA);

                return waitFor(() => subject._seekToResumeTime.called).
                then(() => {
                    assert.ok(subject._seekToResumeTime.calledOnce);
                    assert.ok(subject._seekToResumeTime.calledWith(this.videoInfoResponse, [this.videoInfo]));
                });
            });

            QUnit.test('_seekToResumeTime should seek there', function(assert) {
                const subject = new PlayerUIResume(this.player, this.state, this.store, this.options);
                subject._seekToResumeTime({
                    // eslint-disable-next-line camelcase
                    broadcast_id: this.broadcastID,
                }, [this.videoInfo]);

                assert.deepEqual(
                    this.player.setCurrentTime.firstCall.args,
                    [this.resumeTime - Settings.livestreamResumePushback]
                );
            });

            QUnit.test('_seekToResumeTime should remove livestream resume time from backend', function(assert) {
                const subject = new PlayerUIResume(this.player, this.state, this.store, this.options);
                subject._seekToResumeTime({
                    // eslint-disable-next-line camelcase
                    broadcast_id: this.broadcastID,
                }, [this.videoInfo]);
                const action = this.store.dispatch.getCall(this.store.dispatch.callCount - 1).args[0];
                assert.deepEqual(
                    action.toString(),
                    ResumeWatchActions.cancelLivestreamResumeTime(this.broadcastID, this.channelID).toString()
                );
                // this does not test the input to the function(this.broadcastID, this.channelID) is correct
            });

            QUnit.test('_seekToResumeTime should set isSeeked to true', function(assert) {
                const subject = new PlayerUIResume(this.player, this.state, this.store, this.options);

                assert.ok(!this.store.getState().resumeWatch.isSeeked);

                subject._seekToResumeTime({
                    // eslint-disable-next-line camelcase
                    broadcast_id: this.broadcastID,
                }, [this.videoInfo]);

                assert.ok(this.store.getState().resumeWatch.isSeeked);
            });
        });
    });

    QUnit.module('when a VOD is loaded and has only a livestream resume time near the end', function(hooks) {
        hooks.beforeEach(function() {
            this.store.dispatch(PlaybackActions.setLoading(false));
            this.store.dispatch(setStream({
                contentType: TYPE_VIDEO,
                contentId: this.videoID,
            }));
            this.resumeTime = 30 * Settings.cancelResumeAmount;
            this.duration = this.resumeTime + Settings.cancelResumeAmount - 1;
            this.store.dispatch(
                ResumeWatchActions.setLivestreamResumeTime(this.broadcastID, this.channelID, this.resumeTime)
            );
            this.store.dispatch(PlaybackActions.updateDuration(this.duration));
        });

        QUnit.module('if the user is not logged in', function() {
            QUnit.test('a duration update should call _seekToResumeTimeLocal', function(assert) {
                const subject = new PlayerUIResume(this.player, this.state, this.store, this.options);
                sinon.spy(subject, '_seekToResumeTimeLocal');

                subject.handleEvent(MediaEvent.LOADED_METADATA);

                return waitFor(() => subject._seekToResumeTimeLocal.called).
                then(() => {
                    assert.ok(subject._seekToResumeTimeLocal.calledOnce);
                    assert.ok(subject._seekToResumeTimeLocal.calledWith(this.videoInfoResponse));
                });
            });

            QUnit.test('the video should not be seeked', function(assert) {
                const subject = new PlayerUIResume(this.player, this.state, this.store, this.options);

                this.store.dispatch(
                    ResumeWatchActions.setLivestreamResumeTime(this.broadcastID, this.channelID, this.resumeTime)
                );

                subject._seekToResumeTimeLocal({
                    // eslint-disable-next-line camelcase
                    broadcast_id: this.broadcastID,
                });

                assert.notOk(this.player.setCurrentTime.called);
            });
        });

        QUnit.module('if the user is logged in', function(hooks) {
            hooks.beforeEach(function() {
                this.userId = '12345';
                this.store.dispatch(ResumeWatchActions.setUser(this.userId));
                this.updatedAt = Date.now();
                this.videoInfo = {
                    video_id: this.broadcastID, // eslint-disable-line camelcase
                    position: this.resumeTime,
                    type: CONTENT_MODE_LIVE,
                    updated_at: this.updatedAt, // eslint-disable-line camelcase
                };
                this.api.expectResumeTimes(this.userId, [this.videoInfo]);
            });
            QUnit.test('a duration update should call _seekToResumeTime', function(assert) {
                const subject = new PlayerUIResume(this.player, this.state, this.store, this.options);
                sinon.spy(subject, '_seekToResumeTime');

                subject.handleEvent(MediaEvent.LOADED_METADATA);

                return waitFor(() => subject._seekToResumeTime.called).
                then(() => {
                    assert.ok(subject._seekToResumeTime.calledOnce);
                    assert.ok(subject._seekToResumeTime.calledWith(this.videoInfoResponse, [this.videoInfo]));
                });
            });

            QUnit.test('the video should not be seeked', function(assert) {
                const subject = new PlayerUIResume(this.player, this.state, this.store, this.options);

                this.store.dispatch(
                    ResumeWatchActions.setLivestreamResumeTime(this.broadcastID, this.channelID, this.resumeTime)
                );

                subject._seekToResumeTime({
                    // eslint-disable-next-line camelcase
                    broadcast_id: this.broadcastID,
                }, [this.videoInfo]);

                assert.notOk(this.player.setCurrentTime.called);
            });
        });
    });

    QUnit.module('when a VOD is loaded and has only an early livestream resume time', function(hooks) {
        hooks.beforeEach(function() {
            this.store.dispatch(PlaybackActions.setLoading(false));
            this.store.dispatch(setStream({
                contentType: TYPE_VIDEO,
                contentId: this.videoID,
            }));
            this.resumeTime = Settings.livestreamResumePushback - 1;
            this.duration = this.resumeTime + 30 * Settings.cancelResumeAmount;
            this.store.dispatch(
                ResumeWatchActions.setLivestreamResumeTime(this.broadcastID, this.channelID, this.resumeTime)
            );
            this.store.dispatch(PlaybackActions.updateDuration(this.duration));
        });

        QUnit.module('if the user is not logged in', function() {
            QUnit.test('a duration update should call _seekToResumeTimeLocal', function(assert) {
                const subject = new PlayerUIResume(this.player, this.state, this.store, this.options);
                sinon.spy(subject, '_seekToResumeTimeLocal');

                subject.handleEvent(MediaEvent.LOADED_METADATA);

                return waitFor(() => subject._seekToResumeTimeLocal.called).
                then(() => {
                    assert.ok(subject._seekToResumeTimeLocal.calledOnce);
                    assert.ok(subject._seekToResumeTimeLocal.calledWith(this.videoInfoResponse));
                });
            });

            QUnit.test('_seekToResumeTimeLocal should seek to the beginning', function(assert) {
                const subject = new PlayerUIResume(this.player, this.state, this.store, this.options);

                subject._seekToResumeTimeLocal({
                    // eslint-disable-next-line camelcase
                    broadcast_id: this.broadcastID,
                });

                assert.deepEqual(this.player.setCurrentTime.firstCall.args, [0]);
            });
        });
        QUnit.module('if the user is logged in', function(hooks) {
            hooks.beforeEach(function() {
                this.userId = '12345';
                this.store.dispatch(ResumeWatchActions.setUser(this.userId));
                this.updatedAt = Date.now();
                this.videoInfo = {
                    video_id: this.broadcastID, // eslint-disable-line camelcase
                    position: this.resumeTime,
                    type: CONTENT_MODE_LIVE,
                    updated_at: this.updatedAt, // eslint-disable-line camelcase
                };
                this.api.expectResumeTimes(this.userId, [this.videoInfo]);
            });

            QUnit.test('a duration update should call _seekToResumeTime', function(assert) {
                const subject = new PlayerUIResume(this.player, this.state, this.store, this.options);
                sinon.spy(subject, '_seekToResumeTime');

                subject.handleEvent(MediaEvent.LOADED_METADATA);

                return waitFor(() => subject._seekToResumeTime.called).
                then(() => {
                    assert.ok(subject._seekToResumeTime.calledOnce);
                    assert.ok(subject._seekToResumeTime.calledWith(this.videoInfoResponse, [this.videoInfo]));
                });
            });

            QUnit.test('_seekToResumeTime should seek to the beginning', function(assert) {
                const subject = new PlayerUIResume(this.player, this.state, this.store, this.options);

                subject._seekToResumeTime({
                    // eslint-disable-next-line camelcase
                    broadcast_id: this.broadcastID,
                }, [this.videoInfo]);

                assert.deepEqual(this.player.setCurrentTime.firstCall.args, [0]);
            });
        });
    });

    QUnit.module('when a VOD is loaded and has both a VOD resume time and a livestream resume time', function(hooks) {
        hooks.beforeEach(function() {
            this.store.dispatch(PlaybackActions.setLoading(false));
            this.store.dispatch(setStream({
                contentType: TYPE_VIDEO,
                contentId: this.videoID,
            }));
            this.duration = 60 * Settings.cancelResumeAmount;
            this.store.dispatch(PlaybackActions.updateDuration(this.duration));
        });

        QUnit.module('if the user is not logged in', function() {
            QUnit.test('a duration update should call _seekToResumeTimeLocal', function(assert) {
                const subject = new PlayerUIResume(this.player, this.state, this.store, this.options);
                sinon.spy(subject, '_seekToResumeTimeLocal');
                const vodResumeTime = 30 * Settings.cancelResumeAmount;
                const livestreamResumeTime = 20 * Settings.cancelResumeAmount;

                this.store.dispatch(
                    ResumeWatchActions.setVodResumeTime(this.videoID, this.channelID, vodResumeTime)
                );
                this.store.dispatch(
                    ResumeWatchActions.setLivestreamResumeTime(
                        this.broadcastID,
                        this.channelID,
                        livestreamResumeTime
                    )
                );

                subject.handleEvent(MediaEvent.LOADED_METADATA);

                return waitFor(() => subject._seekToResumeTimeLocal.called).
                then(() => {
                    assert.ok(subject._seekToResumeTimeLocal.calledOnce);
                    assert.ok(subject._seekToResumeTimeLocal.calledWith(this.videoInfoResponse));
                });
            });

            QUnit.test('_seekToResumeTimeLocal should seek to VOD time when VOD time is later',function(assert) {
                const vodResumeTime = 30 * Settings.cancelResumeAmount;
                const livestreamResumeTime = 20 * Settings.cancelResumeAmount;

                this.store.dispatch(
                    ResumeWatchActions.setVodResumeTime(this.videoID, this.channelID, vodResumeTime)
                );
                this.store.dispatch(
                    ResumeWatchActions.setLivestreamResumeTime(
                        this.broadcastID,
                        this.channelID,
                        livestreamResumeTime
                    )
                );
                const subject = new PlayerUIResume(this.player, this.state, this.store, this.options);

                subject._seekToResumeTimeLocal(this.videoInfoResponse);
                return waitFor(() => this.player.setCurrentTime.called).
                then(() => {
                    assert.deepEqual(
                        this.player.setCurrentTime.firstCall.args,
                        [vodResumeTime]
                    );
                });
            });

            QUnit.test('_seekToResumeTimeLocal should seek to VOD time when VOD time is earlier', function(assert) {
                const vodResumeTime = 20 * Settings.cancelResumeAmount;
                const livestreamResumeTime = 30 * Settings.cancelResumeAmount;

                this.store.dispatch(
                    ResumeWatchActions.setVodResumeTime(this.videoID, this.channelID, vodResumeTime)
                );
                this.store.dispatch(
                    ResumeWatchActions.setLivestreamResumeTime(
                        this.broadcastID,
                        this.channelID,
                        livestreamResumeTime
                    )
                );
                const subject = new PlayerUIResume(this.player, this.state, this.store, this.options);

                subject._seekToResumeTimeLocal({
                    // eslint-disable-next-line camelcase
                    broadcast_id: this.broadcastID,
                });
                return waitFor(() => this.player.setCurrentTime.called).
                then(() => {
                    assert.deepEqual(
                        this.player.setCurrentTime.firstCall.args,
                        [vodResumeTime]
                    );
                });
            });

            // eslint-disable-next-line max-len
            QUnit.test('_seekToResumeTimeLocal should remove livestream time when VOD time is later from local storage', function(assert) {
                const dispatch = sinon.spy();
                const vodResumeTime = 30 * Settings.cancelResumeAmount;
                const livestreamResumeTime = 20 * Settings.cancelResumeAmount;

                this.store.dispatch(
                    ResumeWatchActions.setVodResumeTime(this.videoID, this.channelID, vodResumeTime)
                );
                this.store.dispatch(
                    ResumeWatchActions.setLivestreamResumeTime(
                        this.broadcastID,
                        this.channelID,
                        livestreamResumeTime
                    )
                );
                const subject = new PlayerUIResume(this.player, this.state, this.store, this.options);

                subject._seekToResumeTimeLocal({
                    // eslint-disable-next-line camelcase
                    broadcast_id: this.broadcastID,
                });
                const action = this.store.dispatch.getCall(this.store.dispatch.callCount - 1).args[0];
                action(dispatch, this.store.getState);

                assert.ok(dispatch.calledWith({
                    type: ResumeWatchActions.ACTION_LIVESTREAM_CANCEL_RESUME,
                    broadcastID: this.broadcastID,
                }));
            });

            // eslint-disable-next-line max-len
            QUnit.test('_seekToResumeTimeLocal should remove livestream time when VOD time is earlier from local storage', function(assert) {
                const dispatch = sinon.spy();
                const vodResumeTime = 20 * Settings.cancelResumeAmount;
                const livestreamResumeTime = 30 * Settings.cancelResumeAmount;

                this.store.dispatch(
                    ResumeWatchActions.setVodResumeTime(this.videoID, this.channelID, vodResumeTime)
                );
                this.store.dispatch(
                    ResumeWatchActions.setLivestreamResumeTime(
                        this.broadcastID,
                        this.channelID,
                        livestreamResumeTime
                    )
                );
                const subject = new PlayerUIResume(this.player, this.state, this.store, this.options);

                subject._seekToResumeTimeLocal({
                    // eslint-disable-next-line camelcase
                    broadcast_id: this.broadcastID,
                });
                const action = this.store.dispatch.getCall(this.store.dispatch.callCount - 1).args[0];
                action(dispatch, this.store.getState);

                assert.ok(dispatch.calledWith({
                    type: ResumeWatchActions.ACTION_LIVESTREAM_CANCEL_RESUME,
                    broadcastID: this.broadcastID,
                }));
            });

            QUnit.test('_seekToResumeTimeLocal should set isSeeked to true', function(assert) {
                const vodResumeTime = 30 * Settings.cancelResumeAmount;
                const livestreamResumeTime = 20 * Settings.cancelResumeAmount;

                this.store.dispatch(
                    ResumeWatchActions.setVodResumeTime(this.videoID, this.channelID, vodResumeTime)
                );
                this.store.dispatch(
                    ResumeWatchActions.setLivestreamResumeTime(
                        this.broadcastID,
                        this.channelID,
                        livestreamResumeTime
                    )
                );
                const subject = new PlayerUIResume(this.player, this.state, this.store, this.options);

                assert.ok(!this.store.getState().resumeWatch.isSeeked);

                subject._seekToResumeTimeLocal({
                    // eslint-disable-next-line camelcase
                    broadcast_id: this.broadcastID,
                });

                assert.ok(this.store.getState().resumeWatch.isSeeked);
            });
        });

        QUnit.module('if the user is logged in', function(hooks) {
            hooks.beforeEach(function() {
                this.userId = '12345';
                this.store.dispatch(ResumeWatchActions.setUser(this.userId));
            });

            QUnit.test('a duration update should call _seekToResumeTime', function(assert) {
                const subject = new PlayerUIResume(this.player, this.state, this.store, this.options);
                sinon.spy(subject, '_seekToResumeTime');
                const vodResumeTime = 30 * Settings.cancelResumeAmount;
                const livestreamResumeTime = 20 * Settings.cancelResumeAmount;
                const updatedAt = Date.now();
                const streamInfo = {
                    video_id: this.broadcastID, // eslint-disable-line camelcase
                    position: livestreamResumeTime,
                    type: CONTENT_MODE_LIVE,
                    updated_at: updatedAt, // eslint-disable-line camelcase
                };
                const videoInfo = {
                    video_id: this.videoID, // eslint-disable-line camelcase
                    position: vodResumeTime,
                    type: CONTENT_MODE_VOD,
                    updated_at: updatedAt, // eslint-disable-line camelcase
                };
                this.api.expectResumeTimes(this.userId, [videoInfo, streamInfo]);

                subject.handleEvent(MediaEvent.LOADED_METADATA);

                return waitFor(() => subject._seekToResumeTime.called).
                then(() => {
                    assert.ok(subject._seekToResumeTime.calledOnce);
                    assert.ok(subject._seekToResumeTime.calledWith(this.videoInfoResponse));
                });
            });

            QUnit.test('_seekToResumeTime should seek to VOD time when VOD time is most recent',function(assert) {
                const vodResumeTime = 30 * Settings.cancelResumeAmount;
                const livestreamResumeTime = 20 * Settings.cancelResumeAmount;
                const streamUpdatedAt = new Date(2016, 6, 30);
                const videoUpdatedAt = new Date(2016, 7, 1);
                const streamInfo = {
                    video_id: this.broadcastID, // eslint-disable-line camelcase
                    position: livestreamResumeTime,
                    type: CONTENT_MODE_LIVE,
                    updated_at: streamUpdatedAt, // eslint-disable-line camelcase
                };
                const videoInfo = {
                    video_id: this.videoID, // eslint-disable-line camelcase
                    position: vodResumeTime,
                    type: CONTENT_MODE_VOD,
                    updated_at: videoUpdatedAt, // eslint-disable-line camelcase
                };

                const subject = new PlayerUIResume(this.player, this.state, this.store, this.options);

                subject._seekToResumeTime({
                    // eslint-disable-next-line camelcase
                    broadcast_id: this.broadcastID,
                }, [videoInfo, streamInfo]);
                return waitFor(() => this.player.setCurrentTime.called).
                then(() => {
                    assert.deepEqual(
                        this.player.setCurrentTime.firstCall.args,
                        [vodResumeTime]
                    );
                });
            });

            // eslint-disable-next-line max-len
            QUnit.test('_seekToResumeTime should seek to livestream time when livestream time is most recent', function(assert) {
                const vodResumeTime = 20 * Settings.cancelResumeAmount;
                const livestreamResumeTime = 30 * Settings.cancelResumeAmount;
                const streamUpdatedAt = new Date(2016, 7, 1);
                const videoUpdatedAt = new Date(2016, 6, 30);
                const streamInfo = {
                    video_id: this.broadcastID, // eslint-disable-line camelcase
                    position: livestreamResumeTime,
                    type: CONTENT_MODE_LIVE,
                    updated_at: streamUpdatedAt, // eslint-disable-line camelcase
                };
                const videoInfo = {
                    video_id: this.videoID, // eslint-disable-line camelcase
                    position: vodResumeTime,
                    type: CONTENT_MODE_VOD,
                    updated_at: videoUpdatedAt, // eslint-disable-line camelcase
                };
                const subject = new PlayerUIResume(this.player, this.state, this.store, this.options);

                subject._seekToResumeTime({
                    // eslint-disable-next-line camelcase
                    broadcast_id: this.broadcastID,
                }, [streamInfo, videoInfo]);
                return waitFor(() => this.player.setCurrentTime.called).
                then(() => {
                    assert.deepEqual(
                        this.player.setCurrentTime.firstCall.args,
                        [livestreamResumeTime - Settings.livestreamResumePushback]
                    );
                });
            });

            QUnit.test('_seekToResumeTime should remove livestream time from backend', function(assert) {
                const vodResumeTime = 30 * Settings.cancelResumeAmount;
                const livestreamResumeTime = 20 * Settings.cancelResumeAmount;
                const streamUpdatedAt = new Date(2016, 7, 1);
                const videoUpdatedAt = new Date(2016, 6, 30);
                const streamInfo = {
                    video_id: this.broadcastID, // eslint-disable-line camelcase
                    position: livestreamResumeTime,
                    type: CONTENT_MODE_LIVE,
                    updated_at: streamUpdatedAt, // eslint-disable-line camelcase
                };
                const videoInfo = {
                    video_id: this.videoID, // eslint-disable-line camelcase
                    position: vodResumeTime,
                    type: CONTENT_MODE_VOD,
                    updated_at: videoUpdatedAt, // eslint-disable-line camelcase
                };
                const subject = new PlayerUIResume(this.player, this.state, this.store, this.options);

                subject._seekToResumeTime({
                    // eslint-disable-next-line camelcase
                    broadcast_id: this.broadcastID,
                }, [streamInfo, videoInfo]);

                const action = this.store.dispatch.getCall(this.store.dispatch.callCount - 1).args[0];
                assert.deepEqual(
                    action.toString(),
                    ResumeWatchActions.cancelLivestreamResumeTime(this.broadcastID, this.channelID).toString()
                );
                // this does not test the input to the function(this.broadcastID) is correct
            });

            QUnit.test('_seekToResumeTime should set isSeeked to true', function(assert) {
                const vodResumeTime = 30 * Settings.cancelResumeAmount;
                const livestreamResumeTime = 20 * Settings.cancelResumeAmount;
                const streamUpdatedAt = new Date(2016, 7, 1);
                const videoUpdatedAt = new Date(2016, 6, 30);
                const streamInfo = {
                    video_id: this.broadcastID, // eslint-disable-line camelcase
                    position: livestreamResumeTime,
                    type: CONTENT_MODE_LIVE,
                    updated_at: streamUpdatedAt, // eslint-disable-line camelcase
                };
                const videoInfo = {
                    video_id: this.videoID, // eslint-disable-line camelcase
                    position: vodResumeTime,
                    type: CONTENT_MODE_VOD,
                    updated_at: videoUpdatedAt, // eslint-disable-line camelcase
                };
                const subject = new PlayerUIResume(this.player, this.state, this.store, this.options);

                assert.ok(!this.store.getState().resumeWatch.isSeeked);

                subject._seekToResumeTime({
                    // eslint-disable-next-line camelcase
                    broadcast_id: this.broadcastID,
                }, [streamInfo, videoInfo]);

                assert.ok(this.store.getState().resumeWatch.isSeeked);
            });
        });
    });

    QUnit.module('while a VOD has not played yet', function(hooks) {
        hooks.beforeEach(function() {
            this.store.dispatch(setStream({
                contentType: TYPE_VIDEO,
                contentId: this.videoID,
            }));
            this.store.dispatch(PlaybackActions.setLoading(true));
            this.store.dispatch(PlaybackActions.updateDuration(60 * Settings.cancelResumeAmount));
            this.store.dispatch.reset();
            this.subject = new PlayerUIResume(this.player, this.state, this.store, this.options);
        });

        QUnit.test('a currentTime state change should do nothing', function(assert) {
            const { duration } = this.store.getState().playback;
            this.store.dispatch(PlaybackActions.updateCurrentTime(duration - 2 * Settings.cancelResumeAmount));
            assert.equal(this.store.dispatch.callCount, 1, 'only 1 dispatch made');
            const [action] = this.store.dispatch.firstCall.args;
            assert.equal(action.type, PlaybackActions.ACTION_UPDATE_CURRENT_TIME, 'to update current time');
        });
    });

    QUnit.module('while a VOD is playing', function(hooks) {
        hooks.beforeEach(function() {
            this.store.dispatch(PlaybackActions.setLoading(false));
            this.store.dispatch(setStream({
                contentType: TYPE_VIDEO,
                contentId: this.videoID,
            }));
            this.store.dispatch(PlaybackActions.updateDuration(60 * Settings.cancelResumeAmount));
            this.subject = new PlayerUIResume(this.player, this.state, this.store, this.options);
        });

        QUnit.module('when the current time is not near the end of the VOD', function(hooks) {
            hooks.beforeEach(function() {
                const { duration } = this.store.getState().playback;
                this.currentTime = duration - 2 * Settings.cancelResumeAmount;
                this.store.dispatch(PlaybackActions.updateCurrentTime(this.currentTime));
            });

            QUnit.test('a currentTime state change should record the current playhead', function(assert) {
                const dispatch = sinon.spy();
                const action = this.store.dispatch.lastCall.args[0];
                action(dispatch, this.store.getState);

                assert.ok(dispatch.calledWith({
                    type: ResumeWatchActions.ACTION_VOD_SET_RESUME_TIME,
                    videoID: this.videoID,
                    time: this.currentTime,
                }));
            });
        });

        QUnit.module('when the current time is near the end of the VOD', function(hooks) {
            hooks.beforeEach(function() {
                const { duration } = this.store.getState().playback;
                this.currentTime = duration - 0.5 * Settings.cancelResumeAmount;
            });

            QUnit.module('if the user is not logged in', function() {
                // eslint-disable-next-line max-len
                QUnit.test('a currentTime state change should remove the resume time from local storage', function(assert) {
                    const { duration } = this.store.getState().playback;
                    const dispatch = sinon.spy();
                    this.store.dispatch(
                        ResumeWatchActions.setVodResumeTime(this.videoID, this.channelID, duration / 2)
                    );

                    this.store.dispatch(PlaybackActions.updateCurrentTime(this.currentTime));
                    const action = this.store.dispatch.lastCall.args[0];
                    action(dispatch, this.store.getState);

                    assert.ok(dispatch.calledWith({
                        type: ResumeWatchActions.ACTION_VOD_CANCEL_RESUME,
                        videoID: this.videoID,
                    }));
                });
            });

            QUnit.module('if the user is logged in', function(hooks) {
                hooks.beforeEach(function() {
                    this.userId = '12345';
                    this.store.dispatch(ResumeWatchActions.setUser(this.userId));
                });

                // eslint-disable-next-line max-len
                QUnit.test('a currentTime state change should remove the resume time from backend', function(assert) {
                    const { duration } = this.store.getState().playback;
                    this.store.dispatch(
                        ResumeWatchActions.setVodResumeTime(this.videoID, this.channelID, duration / 2)
                    );

                    this.store.dispatch(PlaybackActions.updateCurrentTime(this.currentTime));
                    const action = this.store.dispatch.lastCall.args[0];
                    assert.deepEqual(
                        action.toString(),
                        ResumeWatchActions.cancelVodResumeTime(this.videoID, this.channelID).toString()
                    );
                });
            });
        });
    });

    QUnit.module('while a live stream is playing', function(hooks) {
        hooks.beforeEach(function() {
            this.store.dispatch(setStream({
                contentType: TYPE_CHANNEL,
                contentId: this.channelID,
            }));
            this.player.getVideoInfo = function() {
                return {
                    // eslint-disable-next-line camelcase
                    stream_time_offset: 45 * Settings.cancelResumeAmount,
                    // eslint-disable-next-line camelcase
                    broadcast_id: parseInt(QUnit.config.current.testId, 36),
                };
            };
            this.subject = new PlayerUIResume(this.player, this.state, this.store, this.options);
        });

        QUnit.module('while the live stream is online', function(hooks) {
            hooks.beforeEach(function() {
                this.currentTime = 60 * Settings.cancelResumeAmount;
                this.store.dispatch(setOnline(true));
            });

            QUnit.test('a currentTime state change should record the current playhead', function(assert) {
                const broadcastId = this.store.getState().streamMetadata.broadcastID;
                const streamTimeOffset = this.player.getVideoInfo().stream_time_offset;

                // Faking a reload of a stream
                this.subject.handleEvent(MediaEvent.LOADED_METADATA);

                this.store.dispatch(PlaybackActions.updateCurrentTime(this.currentTime));
                assert.ok(!this.player.setCurrentTime.called);

                const dispatch = sinon.spy();
                const action = this.store.dispatch.lastCall.args[0];
                action(dispatch, this.store.getState);

                assert.ok(dispatch.calledWith({
                    type: ResumeWatchActions.ACTION_LIVESTREAM_SET_RESUME_TIME,
                    // eslint-disable-next-line camelcase
                    broadcastID: broadcastId,
                    // eslint-disable-next-line camelcase
                    time: streamTimeOffset + this.currentTime,
                }));
            });
        });

        QUnit.module('while the live stream is offline', function(hooks) {
            hooks.beforeEach(function() {
                this.store.dispatch(setOnline(false));
            });

            QUnit.test('a currentTime state change should remove the resume time', function(assert) {
                const broadcastId = this.store.getState().streamMetadata.broadcastID;
                const dispatch = sinon.spy();

                this.store.dispatch(
                    ResumeWatchActions.setLivestreamResumeTime(
                        broadcastId,
                        this.channelID,
                        60 * Settings.cancelResumeAmount
                    )
                );

                this.store.dispatch(PlaybackActions.updateCurrentTime(this.currentTime));
                const action = this.store.dispatch.lastCall.args[0];
                action(dispatch, this.store.getState);

                assert.ok(dispatch.calledWith({
                    type: ResumeWatchActions.ACTION_LIVESTREAM_CANCEL_RESUME,
                    // eslint-disable-next-line camelcase
                    broadcastID: broadcastId,
                }));
            });
        });
    });

    QUnit.module('when a VOD is loaded from a collection and has a VOD resume time', function(hooks) {
        hooks.beforeEach(function() {
            this.collection = TEST_COLLECTION;
            this.videoID = `v${this.collection.items[0].item_id}`;
            this.resumeTime = 30 * Settings.cancelResumeAmount;
            this.duration = 60 * Settings.cancelResumeAmount;

            this.playerUIResume = new PlayerUIResume(this.player, this.state, this.store, this.options);
            sinon.spy(this.playerUIResume, '_seekToResumeTimeLocal');
            sinon.spy(this.playerUIResume, '_seekToResumeTime');

            this.store.dispatch({
                type: ACTION_SET_STREAM,
                stream: new VODTwitchContentStream(this.videoID),
            });
            this.store.dispatch(CollectionActions.setCollection(this.collection));
            this.store.dispatch(PlaybackActions.updateDuration(this.duration));
        });

        hooks.afterEach(function() {
            this.playerUIResume._seekToResumeTimeLocal.reset();
            this.playerUIResume._seekToResumeTime.reset();
        });

        QUnit.test('if the user is not logged in, a currentTime state change should do nothing', function(assert) {
            this.store.dispatch(
                ResumeWatchActions.setVodResumeTime(this.videoID, this.channelID, this.resumeTime)
            );

            this.playerUIResume.handleEvent(MediaEvent.LOADED_METADATA);

            assert.equal(
                this.api.called(),
                false,
                'server not called for video info'
            );
            assert.equal(
                this.playerUIResume._seekToResumeTimeLocal.called,
                false,
                'not seeked to local storage resume time'
            );
            assert.equal(
                this.playerUIResume._seekToResumeTimeLocal.called,
                false,
                'not seeked to resume time'
            );
        });

        QUnit.test('if the user is logged in, a currentTime state change should do nothing', function(assert) {
            this.userId = '12345';
            this.store.dispatch(ResumeWatchActions.setUser(this.userId));
            this.updatedAt = Date.now();
            this.videoInfo = {
                video_id: this.videoID, // eslint-disable-line camelcase
                position: this.resumeTime,
                type: CONTENT_MODE_VOD,
                updated_at: this.updatedAt, // eslint-disable-line camelcase
            };
            this.api.expectResumeTimes(this.userId, [this.videoInfo]);

            this.store.dispatch(PlaybackActions.updateDuration(this.duration));

            this.playerUIResume.handleEvent(MediaEvent.LOADED_METADATA);

            assert.equal(
                this.api.called(),
                false,
                'server not called for video info'
            );
            assert.equal(
                this.playerUIResume._seekToResumeTimeLocal.called,
                false,
                'not seeked to local storage resume time'
            );
            assert.equal(
                this.playerUIResume._seekToResumeTimeLocal.called,
                false,
                'not seeked to resume time'
            );
        });
    });

    QUnit.module('when a VOD is loaded has a VOD resume time and a start time is set', function(hooks) {
        hooks.beforeEach(function() {
            this.videoID = 'a video id';
            this.resumeTime = 30 * Settings.cancelResumeAmount;
            this.duration = 60 * Settings.cancelResumeAmount;

            this.playerUIResume = new PlayerUIResume(this.player, this.state, this.store, this.options);
            sinon.spy(this.playerUIResume, '_seekToResumeTimeLocal');
            sinon.spy(this.playerUIResume, '_seekToResumeTime');

            this.store.dispatch({
                type: ACTION_SET_STREAM,
                stream: new VODTwitchContentStream(this.videoID),
            });
            this.store.dispatch(PlaybackActions.updateDuration(this.duration));
            this.store.dispatch(PlaybackActions.setStartTime(1));
        });

        hooks.afterEach(function() {
            this.playerUIResume._seekToResumeTimeLocal.reset();
            this.playerUIResume._seekToResumeTime.reset();
        });

        QUnit.test('if the user is not logged in, on LOADED_METADATA firing, it should do nothing', function(assert) {
            this.store.dispatch(
                ResumeWatchActions.setVodResumeTime(this.videoID, this.channelID, this.resumeTime)
            );

            this.playerUIResume.handleEvent(MediaEvent.LOADED_METADATA);

            assert.equal(
                this.api.called(),
                false,
                'server not called for video info'
            );
            assert.equal(
                this.playerUIResume._seekToResumeTimeLocal.called,
                false,
                'not seeked to local storage resume time'
            );
            assert.equal(
                this.playerUIResume._seekToResumeTimeLocal.called,
                false,
                'not seeked to resume time'
            );
        });

        QUnit.test('if the user is logged in, on LOADED_METADATA firing, it should do nothing', function(assert) {
            this.userId = '12345';
            this.store.dispatch(ResumeWatchActions.setUser(this.userId));
            this.updatedAt = Date.now();
            this.videoInfo = {
                video_id: this.videoID, // eslint-disable-line camelcase
                position: this.resumeTime,
                type: CONTENT_MODE_VOD,
                updated_at: this.updatedAt, // eslint-disable-line camelcase
            };
            this.api.expectResumeTimes(this.userId, [this.videoInfo]);

            this.store.dispatch(PlaybackActions.updateDuration(this.duration));

            this.playerUIResume.handleEvent(MediaEvent.LOADED_METADATA);

            assert.equal(
                this.api.called(),
                false,
                'server not called for video info'
            );
            assert.equal(
                this.playerUIResume._seekToResumeTimeLocal.called,
                false,
                'not seeked to local storage resume time'
            );
            assert.equal(
                this.playerUIResume._seekToResumeTimeLocal.called,
                false,
                'not seeked to resume time'
            );
        });
    });
});
