import { unitTest } from 'tests/utils/module';
import * as AdsActions from 'actions/ads';
import { ads } from 'state/ads';
import { VOD_MIDROLL_SETTING_INSERTED } from 'vod-midroll-manager';

const { AdContentTypes, AdRollTypes } = AdsActions;

unitTest('state | ads', function() {
    QUnit.test('defaults to sensible values', function(assert) {
        const beforeState = undefined;
        const action = {
            type: '@@init',
        };

        assert.deepEqual(ads(beforeState, action), {
            adblock: false,
            clickThrough: '',
            currentMetadata: {
                contentType: AdContentTypes.NONE,
                rollType: AdRollTypes.NONE,
            },
            imaScriptLoaded: false,
            vodMidrollArchiveSetting: VOD_MIDROLL_SETTING_INSERTED,
        });
    });

    QUnit.test('responds to the Set Clickthrough action', function(assert) {
        const beforeState = {
            adblock: false,
            clickThrough: '',
            currentMetadata: {
                contentType: AdContentTypes.NONE,
                rollType: AdRollTypes.NONE,
            },
        };
        const action = {
            type: AdsActions.ACTION_SET_CLICKTHROUGH_URL,
            URL: 'testURL',
        };
        const afterState = {
            adblock: false,
            clickThrough: action.URL,
            currentMetadata: {
                contentType: AdContentTypes.NONE,
                rollType: AdRollTypes.NONE,
            },
        };

        assert.deepEqual(ads(beforeState, action), afterState);
    });

    QUnit.test('responds to the Set Adblock Detected action', function(assert) {
        const beforeState = {
            adblock: false,
            clickThrough: '',
            currentMetadata: {
                contentType: AdContentTypes.NONE,
                rollType: AdRollTypes.NONE,
            },
        };
        const action = {
            type: AdsActions.ACTION_SET_ADBLOCK_DETECTED,
            detected: true,
        };
        const afterState = {
            adblock: true,
            clickThrough: '',
            currentMetadata: {
                contentType: AdContentTypes.NONE,
                rollType: AdRollTypes.NONE,
            },
        };

        assert.deepEqual(ads(beforeState, action), afterState);
    });

    QUnit.test('responds to the Set Ad action', function(assert) {
        const beforeState = {
            adblock: false,
            clickThrough: '',
            currentMetadata: {
                contentType: AdContentTypes.NONE,
                rollType: AdRollTypes.NONE,
            },
        };
        const action = {
            type: AdsActions.ACTION_SET_AD_METADATA,
            currentMetadata: {
                contentType: AdContentTypes.IMA,
                rollType: AdRollTypes.PREROLL,
            },
        };
        const afterState = {
            adblock: false,
            clickThrough: '',
            currentMetadata: action.currentMetadata,
        };

        assert.deepEqual(ads(beforeState, action), afterState);
    });

    QUnit.test('responds to the Set Vod Midroll Archive Setting action', function(assert) {
        const beforeState = {
            adblock: false,
            clickThrough: '',
            currentMetadata: {
                contentType: AdContentTypes.NONE,
                rollType: AdRollTypes.NONE,
            },
            imaScriptLoaded: false,
            vodMidrollArchiveSetting: '',
        };
        const action = {
            type: AdsActions.ACTION_SET_VOD_MIDROLL_ARCHIVE_SETTING,
            setting: 'test',
        };
        const afterState = {
            adblock: false,
            clickThrough: '',
            currentMetadata: {
                contentType: AdContentTypes.NONE,
                rollType: AdRollTypes.NONE,
            },
            imaScriptLoaded: false,
            vodMidrollArchiveSetting: 'test',
        };

        assert.deepEqual(ads(beforeState, action), afterState);
    });

    QUnit.test('responds to ima script loading', function(assert) {
        const beforeState = {
            adblock: false,
            clickThrough: '',
            currentMetadata: {
                contentType: AdContentTypes.NONE,
                rollType: AdRollTypes.NONE,
            },
            imaScriptLoaded: false,
        };
        const action = {
            type: AdsActions.ACTION_IMA_SCRIPT_LOADED,
            imaScriptLoaded: true,
        };
        const afterState = {
            adblock: false,
            clickThrough: '',
            currentMetadata: {
                contentType: AdContentTypes.NONE,
                rollType: AdRollTypes.NONE,
            },
            imaScriptLoaded: true,
        };

        assert.deepEqual(ads(beforeState, action), afterState);
    });

    QUnit.test('responds to ima script failing to load', function(assert) {
        const beforeState = {
            adblock: false,
            clickThrough: '',
            currentMetadata: {
                contentType: AdContentTypes.NONE,
                rollType: AdRollTypes.NONE,
            },
            imaScriptLoaded: false,
        };
        const action = {
            type: AdsActions.ACTION_IMA_SCRIPT_LOADED,
            imaScriptLoaded: false,
        };
        const afterState = {
            adblock: false,
            clickThrough: '',
            currentMetadata: {
                contentType: AdContentTypes.NONE,
                rollType: AdRollTypes.NONE,
            },
            imaScriptLoaded: false,
        };

        assert.deepEqual(ads(beforeState, action), afterState);
    });

    QUnit.test('resets clickthrough on Set Ad actions with non stitched ads', function(assert) {
        const beforeState = {
            adblock: false,
            clickThrough: 'testURL',
            currentMetadata: {
                contentType: AdContentTypes.NONE,
                rollType: AdRollTypes.NONE,
            },
        };

        const IMA_PREROLL_AD = {
            contentType: AdContentTypes.IMA,
            rollType: AdRollTypes.PREROLL,
        };
        const IMA_MIDROLL_AD = {
            contentType: AdContentTypes.IMA,
            rollType: AdRollTypes.PREROLL,
        };
        const IMA_POSTROLL_AD = {
            contentType: AdContentTypes.IMA,
            rollType: AdRollTypes.PREROLL,
        };
        const NONE_AD = {
            contentType: AdContentTypes.IMA,
            rollType: AdRollTypes.PREROLL,
        };

        const NON_STITCHED_ADS = [
            IMA_PREROLL_AD,
            IMA_MIDROLL_AD,
            IMA_POSTROLL_AD,
            NONE_AD,
        ];

        NON_STITCHED_ADS.forEach(ad => {
            const action = {
                type: AdsActions.ACTION_SET_AD_METADATA,
                currentMetadata: ad,
            };
            const afterState = {
                adblock: false,
                clickThrough: '',
                currentMetadata: action.currentMetadata,
            };
            assert.deepEqual(ads(beforeState, action), afterState);
        });
    });

    QUnit.test('does not reset clickthrough on Set Ad actions with stitched ads', function(assert) {
        const beforeState = {
            adblock: false,
            clickThrough: 'testURL',
            currentMetadata: {
                contentType: AdContentTypes.NONE,
                rollType: AdRollTypes.NONE,
            },
        };

        const STITCHED_PREROLL_AD = {
            contentType: AdContentTypes.STITCHED,
            rollType: AdRollTypes.PREROLL,
        };
        const STITCHED_MIDROLL_AD = {
            contentType: AdContentTypes.STITCHED,
            rollType: AdRollTypes.PREROLL,
        };
        const STITCHED_POSTROLL_AD = {
            contentType: AdContentTypes.STITCHED,
            rollType: AdRollTypes.PREROLL,
        };

        const STITCHED_ADS = [
            STITCHED_PREROLL_AD,
            STITCHED_MIDROLL_AD,
            STITCHED_POSTROLL_AD,
        ];

        STITCHED_ADS.forEach(ad => {
            const action = {
                type: AdsActions.ACTION_SET_AD_METADATA,
                currentMetadata: ad,
            };
            const afterState = {
                adblock: false,
                clickThrough: beforeState.clickThrough,
                currentMetadata: action.currentMetadata,
            };
            assert.deepEqual(ads(beforeState, action), afterState);
        });
    });

    QUnit.test('returns the currentMetadata state of ads otherwise', function(assert) {
        const beforeState = {
            adblock: false,
            clickThrough: '',
            currentMetadata: {
                contentType: AdContentTypes.NONE,
                rollType: AdRollTypes.NONE,
            },
        };
        const action = {
            type: 'not a real action',
            detected: true,
        };

        assert.equal(ads(beforeState, action), beforeState);
    });
});
