import { unitTest } from 'tests/utils/module';
import * as AnalyticsActions from 'actions/analytics';
import { analytics } from 'state/analytics';
import assign from 'lodash/assign';

export const DEFAULT_ANALYTICS = {
    qualityChangeCount: 0,
    playSessionId: '',
    playSessionStartTime: 0,
    trackingClients: [],
};

unitTest('state | analytics', function() {
    QUnit.test('defaults to sensible values', function(assert) {
        const beforeState = undefined;
        const action = {
            type: '@@init',
        };

        assert.deepEqual(analytics(beforeState, action), {
            qualityChangeCount: 0,
            playSessionId: '',
            playSessionStartTime: 0,
            trackingClients: [],
        });
    });

    QUnit.test('responds to the increment quality change count action', function(assert) {
        const beforeState = {
            qualityChangeCount: 2,
            playSessionId: '',
            playSessionStartTime: 0,
            trackingClients: [],
        };

        const action = {
            type: AnalyticsActions.ACTION_INCREMENT_QUALITY_CHANGE_COUNT,
        };

        const afterState = assign({}, beforeState, {
            qualityChangeCount: beforeState.qualityChangeCount + 1,
        });

        assert.deepEqual(analytics(beforeState, action), afterState);
    });

    QUnit.test('responds to the reset quality change count action', function(assert) {
        const beforeState = {
            qualityChangeCount: 10,
            playSessionId: 'an id',
            playSessionStartTime: 1,
            trackingClients: [],
        };

        const action = {
            type: AnalyticsActions.ACTION_RESET_QUALITY_CHANGE_COUNT,
        };

        const afterState = assign({}, beforeState, {
            qualityChangeCount: 0,
        });

        assert.deepEqual(analytics(beforeState, action), afterState);
    });

    QUnit.test('responds to the reset play session action', function(assert) {
        const beforeState = {
            qualityChangeCount: 10,
            playSessionId: 'an id',
            playSessionStartTime: 10,
            trackingClients: [],
        };

        const action = {
            type: AnalyticsActions.ACTION_RESET_PLAY_SESSION,
            playSessionId: 'another id',
            playSessionStartTime: 10000,
        };

        const afterState = assign({}, beforeState, {
            playSessionId: action.playSessionId,
            playSessionStartTime: action.playSessionStartTime,
        });

        assert.deepEqual(analytics(beforeState, action), afterState);
    });

    QUnit.test('responds to the set tracking clients action', function(assert) {
        const beforeState = {
            qualityChangeCount: 10,
            playSessionId: 'an id',
            playSessionStartTime: 10,
            trackingClients: [],
        };

        const action = {
            type: AnalyticsActions.ACTION_SET_TRACKING_CLIENTS,
            clients: ['a client', 'another client'],
        };

        const afterState = assign({}, beforeState, {
            trackingClients: action.clients,
        });

        assert.deepEqual(analytics(beforeState, action), afterState);
    });
});
