import { ACTION_INITIALIZE_CAPTIONS_SETTINGS, ACTION_SET_CAPTIONS_DATA, ACTION_SET_CAPTIONS_PRESET,
         ACTION_TOGGLE_CAPTIONS, ACTION_SHOW_CC_MODAL } from 'actions/captions';
import { ACTION_SET_STREAM } from 'actions/stream';
import { ACTION_UPDATE_PLAYER_DIMENSIONS } from 'actions/player-dimensions';
import {
    CAPTIONS_AUTO_RESIZE_THRESHOLD_HEIGHT,
    CAPTIONS_AUTO_RESIZE_THRESHOLD_WIDTH,
    CAPTIONS_AUTO_RESIZE_SCALE,
    captions,
} from 'state/captions';
import assign from 'lodash/assign';

QUnit.module('state | captions', function(hooks) {
    hooks.beforeEach(function() {
        this.defaultState = {
            enabled: false,
            available: false,
            preset: 'white-on-black',
            data: null,
            style: {
                backgroundColorName: 'black',
                backgroundOpacity: 'solid',
                fontColorName: 'white',
                fontSize: 20,
                fontOpacity: 'solid',
                font: 'prop-sans-serif',
                alignment: 'left',
                edge: 'none',
                verticalPosition: 'bottom',
                windowColorName: 'transparent',
                windowOpacity: 'solid',
                fontBold: 'none',
                fontItalic: 'none',
                fontUnderline: 'none',
            },
            showCCModal: false,
        };
    });

    QUnit.test('sets default state', function(assert) {
        const action = {
            type: '@@init',
        };

        assert.deepEqual(captions(undefined, action), this.defaultState);
    });

    QUnit.test('reponds to the showCCModal action', function(assert) {
        const action = {
            type: ACTION_SHOW_CC_MODAL,
            showCCModal: true,
        };

        const beforeState = {
            showCCModal: false,
        };

        assert.deepEqual(captions(beforeState, action), {
            showCCModal: true,
        });
    });

    QUnit.test('responds to the Initialize Captions Settings action', function(assert) {
        const action = {
            type: ACTION_INITIALIZE_CAPTIONS_SETTINGS,
            captions: {
                before: false,
            },
        };
        const beforeState = {
            before: true,
        };

        assert.deepEqual(captions(beforeState, action), action.captions);
    });

    // eslint-disable-next-line max-len
    QUnit.test('Set Captions Data action sets captions.available to true if captions data is non-zero', function(assert) {
        const action = {
            type: ACTION_SET_CAPTIONS_DATA,
            captions: {
                data: {
                    data: ['something', 'else'],
                },
            },
        };

        const resultState = this.defaultState;
        resultState.available = true;
        resultState.data = action.captions.data;

        assert.deepEqual(captions(this.defaultState, action), resultState);
    });

    // eslint-disable-next-line max-len
    QUnit.test('Set Captions Data action leaves captions.available as false if captions data is empty', function(assert) {
        const action = {
            type: ACTION_SET_CAPTIONS_DATA,
            captions: {
                data: {
                    data: [],
                },
            },
        };

        const resultState = this.defaultState;
        resultState.data = action.captions.data;

        assert.deepEqual(captions(this.defaultState, action), resultState);
    });

    // eslint-disable-next-line max-len
    QUnit.test('Set Captions Data action leaves captions.available as true if captions data is empty', function(assert) {
        const action = {
            type: ACTION_SET_CAPTIONS_DATA,
            captions: {
                data: {
                    data: [],
                },
            },
        };

        const beforeState = this.defaultState;
        beforeState.available = true;
        const resultState = this.defaultState;
        resultState.data = action.captions.data;
        resultState.available = true;

        assert.deepEqual(captions(beforeState, action), resultState);
    });

    QUnit.test('responds to the Set Captions Preset action', function(assert) {
        const action = {
            type: ACTION_SET_CAPTIONS_PRESET,
            captions: {
                before: false,
            },
        };
        const beforeState = {
            before: true,
        };

        assert.deepEqual(captions(beforeState, action), action.captions);
    });

    QUnit.test('responds to the Toggle Captions action', function(assert) {
        const action = {
            type: ACTION_TOGGLE_CAPTIONS,
            captions: {
                before: false,
            },
        };
        const beforeState = {
            before: true,
        };

        assert.deepEqual(captions(beforeState, action), action.captions);
    });

    QUnit.test('responds to the Set Stream action', function(assert) {
        const action = {
            type: ACTION_SET_STREAM,
        };

        const beforeState = {
            enabled: true,
            available: true,
            preset: 'another preset',
            data: [],
            style: {
                backgroundColorName: 'red',
                backgroundOpacity: 'clear',
                fontColorName: 'white',
                fontSize: 10,
                fontOpacity: 'solid',
                font: 'prop-sans-serif',
                alignment: 'left',
                edge: 'none',
                verticalPosition: 'bottom',
                windowColorName: 'transparent',
                windowOpacity: 'solid',
                fontBold: 'none',
                fontItalic: 'none',
                fontUnderline: 'none',
            },
        };

        const resultState = assign({}, beforeState, {
            available: this.defaultState.available,
            data: this.defaultState.data,
        });

        assert.deepEqual(captions(beforeState, action), resultState);
    });

    QUnit.test('returns the current Captions state otherwise', function(assert) {
        const action = {
            type: 'not a real action',
        };
        const beforeState = {
            before: true,
        };

        assert.deepEqual(captions(beforeState, action), beforeState);
    });

    QUnit.module('updatePlayerDimensions', function() {
        QUnit.test(`scales fontSize to ${CAPTIONS_AUTO_RESIZE_SCALE} of height if at threshold`, function(assert) {
            const action = {
                type: ACTION_UPDATE_PLAYER_DIMENSIONS,
                playerDimensions: {
                    height: CAPTIONS_AUTO_RESIZE_THRESHOLD_HEIGHT,
                    width: 1000,
                },
            };

            const beforeState = {
                style: {
                    fontSize: QUnit.config.testId,
                },
            };

            const expectedFontSize = Math.round(action.playerDimensions.height * CAPTIONS_AUTO_RESIZE_SCALE);

            assert.equal(captions(beforeState, action).style.fontSize, expectedFontSize);
        });

        QUnit.test(`scales fontSize to ${CAPTIONS_AUTO_RESIZE_SCALE} of width if at threshold`, function(assert) {
            const action = {
                type: ACTION_UPDATE_PLAYER_DIMENSIONS,
                playerDimensions: {
                    height: 1000,
                    width: CAPTIONS_AUTO_RESIZE_THRESHOLD_WIDTH,
                },
            };

            const beforeState = {
                style: {
                    fontSize: QUnit.config.testId,
                },
            };

            const expectedFontSize = Math.round(action.playerDimensions.width * CAPTIONS_AUTO_RESIZE_SCALE);

            assert.equal(captions(beforeState, action).style.fontSize, expectedFontSize);
        });

        QUnit.test('scales fontSize to smaller of the two dimensions if both at threshold', function(assert) {
            const action = {
                type: ACTION_UPDATE_PLAYER_DIMENSIONS,
                playerDimensions: {
                    height: CAPTIONS_AUTO_RESIZE_THRESHOLD_HEIGHT,
                    width: CAPTIONS_AUTO_RESIZE_THRESHOLD_WIDTH,
                },
            };

            const beforeState = {
                style: {
                    fontSize: QUnit.config.testId,
                },
            };

            const smallestDimension = Math.min(action.playerDimensions.height, action.playerDimensions.width);

            const expectedFontSize = Math.round(smallestDimension * CAPTIONS_AUTO_RESIZE_SCALE);

            assert.equal(captions(beforeState, action).style.fontSize, expectedFontSize);
        });

        QUnit.test('does not update fontSize if height and width are above threshold', function(assert) {
            const action = {
                type: ACTION_UPDATE_PLAYER_DIMENSIONS,
                playerDimensions: {
                    height: CAPTIONS_AUTO_RESIZE_THRESHOLD_HEIGHT + 1,
                    width: CAPTIONS_AUTO_RESIZE_THRESHOLD_WIDTH + 1,
                },
            };

            const beforeState = {
                style: {
                    fontSize: QUnit.config.testId,
                },
            };

            assert.deepEqual(captions(beforeState, action), beforeState);
        });
    });
});
