import * as ClipsActions from 'actions/clips';
import {
    clips,
    DEFAULT_CLIPS_STATE,
    isValidChannelState,
    isValidPlaybackState,
    isValidStreamState,
    userCanClipSelector,
    NO_CLIPS_CHANNELS,
} from 'state/clips';
import { CONTENT_MODE_LIVE } from 'stream/twitch-live';
import { CONTENT_MODE_VOD } from 'stream/twitch-vod';
import { ONLINE_STATUS } from 'state/online-status';
import { PLAYER_HIGHLIGHTER, PLAYER_SITE } from 'util/player-type';
import { BACKEND_HLS } from 'backend/hls';
import { BACKEND_PLAYER_CORE } from 'backend/player-core';
import { LOGGED_IN } from 'state/user';
import assign from 'lodash/assign';

const getValidChannelState = () => ({
    streamMetadata: {
        broadcastID: 'an id, not null',
        channel: {
            name: 'llama',
        },
    },
});

const getValidLivePlaybackState = () => ({
    playback: {
        hasPlayed: true,
        paused: false,
    },
    stream: {
        contentType: CONTENT_MODE_LIVE,
    },
});

const getValidVODPlaybackState = () => ({
    playback: {
        hasPlayed: true,
        paused: true,
    },
    stream: {
        contentType: CONTENT_MODE_VOD,
    },
});

const getValidStreamState = () => ({
    stream: {
        contentType: CONTENT_MODE_LIVE,
    },
    onlineStatus: ONLINE_STATUS,
    env: {
        playerType: PLAYER_SITE,
    },
    backendInfo: {
        playerBackendType: BACKEND_PLAYER_CORE,
    },
});

export const getValidClippingState = () => (
    assign(
        {
            user: {
                loggedInStatus: LOGGED_IN,
            },
        },
        getValidChannelState(),
        getValidLivePlaybackState(),
        getValidStreamState()
    )
);

QUnit.module('state | clips', function() {
    QUnit.test('defaults to the proper state', function(assert) {
        const action = {
            type: '@@init',
        };

        const expectedState = DEFAULT_CLIPS_STATE;
        assert.propEqual(clips(undefined, action), expectedState);
    });

    QUnit.test('responds to the clip generator loaded action', function(assert) {
        const fakeGenerator = {
            recordClip() {},
        };
        const action = {
            type: ClipsActions.ACTION_CLIP_GENERATOR_LOADED,
            generator: fakeGenerator,
        };

        const beforeState = {
            generator: {},
        };

        const afterState = clips(beforeState, action);

        assert.deepEqual(afterState.generator, fakeGenerator);
    });

    QUnit.test('isValidChannelState is true when channel is valid', function(assert) {
        assert.equal(isValidChannelState(getValidChannelState()), true);
    });

    QUnit.test('isValidChannelState is false when there is no broadcast id', function(assert) {
        const invalidChannelState = getValidChannelState();
        invalidChannelState.streamMetadata.broadcastID = null;

        assert.equal(isValidChannelState(invalidChannelState), false);
    });

    QUnit.test('isValidChannelState is false when there is an unsupported channel', function(assert) {
        const invalidChannelState = getValidChannelState();
        invalidChannelState.streamMetadata.channel.name = NO_CLIPS_CHANNELS[0];

        assert.equal(isValidChannelState(invalidChannelState), false);
    });

    QUnit.test('isValidPlaybackState is true when the playback state is valid', function(assert) {
        assert.equal(isValidPlaybackState(getValidLivePlaybackState()), true);
        assert.equal(isValidPlaybackState(getValidVODPlaybackState()), true);
    });

    QUnit.test('isValidPlaybackState is false when live is paused', function(assert) {
        const invalidPlaybackState = getValidLivePlaybackState();
        invalidPlaybackState.playback.paused = true;

        assert.equal(isValidPlaybackState(invalidPlaybackState), false);
    });

    QUnit.test('isValidPlaybackState is false when live or vod has not played', function(assert) {
        const invalidLivePlaybackState = getValidLivePlaybackState();
        const invalidVODPlaybackState = getValidVODPlaybackState();
        invalidLivePlaybackState.playback.hasPlayed = false;
        invalidVODPlaybackState.playback.hasPlayed = false;

        assert.equal(isValidPlaybackState(invalidLivePlaybackState), false);
        assert.equal(isValidPlaybackState(invalidVODPlaybackState), false);
    });

    QUnit.test('isValidStreamState is true when the stream state is valid', function(assert) {
        assert.equal(isValidStreamState(getValidStreamState()), true);
    });

    QUnit.test('isValidStreamState is false when live is offline', function(assert) {
        const invalidStreamState = getValidStreamState();
        invalidStreamState.onlineStatus = false;

        assert.equal(isValidStreamState(invalidStreamState), false);
    });

    QUnit.test('isValidStreamState is false when there is an invalid stream type', function(assert) {
        const invalidStreamState = getValidStreamState();
        invalidStreamState.stream.contentType = 'invalid type';

        assert.equal(isValidStreamState(invalidStreamState), false);
    });

    QUnit.test('isValidStreamState is false when there is an invalid player type', function(assert) {
        const invalidStreamState = getValidStreamState();
        invalidStreamState.env.playerType = PLAYER_HIGHLIGHTER;

        assert.equal(isValidStreamState(invalidStreamState), false);
    });

    QUnit.test('isValidStreamState is false when there is an unsupported backend', function(assert) {
        const invalidStreamState = getValidStreamState();
        invalidStreamState.backendInfo.playerBackendType = BACKEND_HLS;

        assert.equal(isValidStreamState(invalidStreamState), false);
    });

    QUnit.test('userCanClipSelector is true when the state is valid', function(assert) {
        assert.equal(userCanClipSelector(getValidClippingState()), true);
    });

    QUnit.test('userCanClipSelector is false when the user is not logged in', function(assert) {
        const invalidClippingState = getValidClippingState();
        invalidClippingState.user.loggedInStatus = false;

        assert.equal(userCanClipSelector(invalidClippingState), false);
    });

    QUnit.test('userCanClipSelector is false when isValidChannelState is false', function(assert) {
        const invalidClippingState = getValidClippingState();
        invalidClippingState.streamMetadata.broadcastID = null;

        assert.equal(userCanClipSelector(invalidClippingState), false);
    });

    QUnit.test('userCanClipSelector is false when isValidClippingState is false', function(assert) {
        const invalidClippingState = getValidClippingState();
        invalidClippingState.playback.paused = true;

        assert.equal(userCanClipSelector(invalidClippingState), false);
    });

    QUnit.test('userCanClipSelector is false when isValidStreamState is false', function(assert) {
        const invalidClippingState = getValidClippingState();
        invalidClippingState.onlineStatus = false;

        assert.equal(userCanClipSelector(invalidClippingState), false);
    });
});
