import { ACTION_CLEAR_COLLECTION, ACTION_SET_COLLECTION, openCollectionSidebar,
         closeCollectionSidebar, featuredCollectionFetched, ACTION_REQUEST_COLLECTION } from 'actions/collection';
import { COLLAPSED_VIEW, SIDEBAR_VIEW, NO_COLLECTION_VIEW,
         collection } from 'state/collection';
import { TEST_COLLECTION, TEST_COLLECTION_METADATA, TEST_NORMALIZED_COLLECTION,
         TEST_NORMALIZED_COLLECTION_METADATA, TEST_NORMALIZED_COLLECTION_ITEMS } from 'tests/fixtures/collection';
import assign from 'lodash/assign';
import omit from 'lodash/omit';

const DEFAULT_FEATURED_COLLECTION = Object.freeze({
    id: '',
    itemsCount: 0,
    owner: {
        displayName: '',
        id: '',
        name: '',
    },
    thumbnails: {
        large: 'https://static-cdn.jtvnw.net/jtv_user_pictures/xarth/404_user_70x70.png',
        medium: 'https://static-cdn.jtvnw.net/jtv_user_pictures/xarth/404_user_70x70.png',
        small: 'https://static-cdn.jtvnw.net/jtv_user_pictures/xarth/404_user_70x70.png',
        template: 'https://static-cdn.jtvnw.net/jtv_user_pictures/xarth/404_user_70x70.png',
    },
    title: '',
});

const makeDefaultCollection = () => ({
    id: '',
    owner: {
        displayName: '',
        id: '',
        name: '',
    },
    thumbnails: {
        large: 'https://static-cdn.jtvnw.net/jtv_user_pictures/xarth/404_user_70x70.png',
        medium: 'https://static-cdn.jtvnw.net/jtv_user_pictures/xarth/404_user_70x70.png',
        small: 'https://static-cdn.jtvnw.net/jtv_user_pictures/xarth/404_user_70x70.png',
        template: 'https://static-cdn.jtvnw.net/jtv_user_pictures/xarth/404_user_70x70.png',
    },
    title: '',
    totalDuration: 0,
    views: 0,
    items: [],
    currentView: NO_COLLECTION_VIEW,
    pendingRequest: {
        collectionId: '',
        videoId: '',
        timestamp: '',
        preferVideo: false,
    },
    featuredCollection: DEFAULT_FEATURED_COLLECTION,
});

QUnit.module('state | collection', function(hooks) {
    hooks.beforeEach(function() {
        this.normalizedCollection = TEST_NORMALIZED_COLLECTION;
    });

    QUnit.test('responds to the `fetched featured collection` action', function(assert) {
        const action = featuredCollectionFetched(TEST_COLLECTION_METADATA);
        const beforeState = makeDefaultCollection();
        const resultState = assign({}, beforeState, {
            featuredCollection: {
                id: TEST_COLLECTION_METADATA._id,
                owner: {
                    id: TEST_COLLECTION_METADATA.owner._id,
                    displayName: TEST_COLLECTION_METADATA.owner.display_name,
                    name: TEST_COLLECTION_METADATA.owner.name,
                },
                thumbnails: TEST_COLLECTION_METADATA.thumbnails,
                title: TEST_COLLECTION_METADATA.title,
                itemsCount: TEST_COLLECTION_METADATA.items_count,
            },
        });

        assert.deepEqual(collection(beforeState, action), resultState);
    });

    QUnit.test('If no thumbnails are availible for `fetched featured collection`, use default', function(assert) {
        const noThumbsFeaturedCollection = omit(TEST_COLLECTION_METADATA, 'thumbnails');
        const action = featuredCollectionFetched(noThumbsFeaturedCollection);
        const beforeState = makeDefaultCollection();
        const resultState = collection(beforeState, action);

        assert.deepEqual(resultState.featuredCollection.thumbnails, DEFAULT_FEATURED_COLLECTION.thumbnails);
    });

    QUnit.test('responds to the `request collection` action', function(assert) {
        const request = {
            collectionId: 'a collection id',
            videoId: 'a video id',
            timestamp: 12356,
            preferVideo: true,
        };
        const action = {
            type: ACTION_REQUEST_COLLECTION,
            request,
        };
        const beforeState = makeDefaultCollection();
        const resultState = collection(beforeState, action);
        const expectedState = makeDefaultCollection();
        expectedState.pendingRequest = request;

        assert.deepEqual(resultState, expectedState, '`pendingRequest` object should be set correctly');
    });

    QUnit.test('responds to the `set collection` action', function(assert) {
        const action = {
            type: ACTION_SET_COLLECTION,
            collection: this.normalizedCollection,
        };
        const beforeState = makeDefaultCollection();
        const resultState = collection(beforeState, action);

        assert.equal(resultState.id, this.normalizedCollection.id, '`id` is set to correct id');
    });

    QUnit.test('responds to the `clear collection` action', function(assert) {
        const action = { type: ACTION_CLEAR_COLLECTION };
        const beforeState = { collection: this.normalizedCollection };
        const resultState = collection(beforeState, action);

        assert.deepEqual(resultState, makeDefaultCollection(), '`id` is set to default');
    });

    QUnit.test('responds to the `open collection sidebar` action', function(assert) {
        const action = openCollectionSidebar();
        const beforeState = {
            currentView: NO_COLLECTION_VIEW,
        };
        const resultState = collection(beforeState, action);
        const expectedState = {
            currentView: SIDEBAR_VIEW,
        };
        assert.deepEqual(resultState, expectedState, '`currentView` is set to sidebar');
    });

    QUnit.test('responds to the `close collection sidebar` action', function(assert) {
        const action = closeCollectionSidebar();
        const beforeState = {
            currentView: NO_COLLECTION_VIEW,
        };
        const resultState = collection(beforeState, action);
        const expectedState = {
            currentView: COLLAPSED_VIEW,
        };

        assert.deepEqual(resultState, expectedState, '`currentView` is set to collapsed');
    });

    QUnit.test('setting a new collection replaces previous collection', function(assert) {
        const action = {
            type: ACTION_SET_COLLECTION,
            collection: this.normalizedCollection,
        };
        const firstState = collection(TEST_COLLECTION, action);
        assert.equal(firstState.id, this.normalizedCollection.id, 'first `id` is set to correct id');

        // Make another test collection with one less item
        const newCollection = assign({}, TEST_NORMALIZED_COLLECTION_METADATA, {
            id: '456456456',
            items: TEST_NORMALIZED_COLLECTION_ITEMS.slice(-1),
        });

        const secondState = collection(firstState, {
            type: ACTION_SET_COLLECTION,
            collection: newCollection,
        });
        assert.equal(secondState.id, newCollection.id, 'second `id` is set to correct id');
        assert.equal(secondState.items.length, newCollection.items.length, 'second `length` is correct');
    });
});
