import assign from 'lodash/assign';
import { unitTest } from 'tests/utils/module';
import * as PlayerType from 'util/player-type';
import * as EnvActions from 'actions/env';
import { ACTION_SET_WINDOW } from 'actions/window';
import { PLATFORM_MOBILE_WEB, PLATFORM_WEB, env } from 'state/env';

unitTest('state | env', function() {
    QUnit.test('defaults to sensible values', function(assert) {
        const beforeState = undefined;
        const action = {
            type: '@@init',
        };

        assert.deepEqual(env(beforeState, action), {
            deviceId: '',
            platform: PLATFORM_WEB,
            playerType: PlayerType.PLAYER_POPOUT,
            isMobileLocation: false,
        });
    });

    QUnit.test('responds to the Set Environment action', function(assert) {
        const deviceId = `device_${QUnit.config.current.testId}`;
        const beforeState = env(undefined, { type: '@@init' });
        const action = {
            type: EnvActions.ACTION_SET_ENVIRONMENT,
            deviceId,
        };
        const afterState = assign({}, beforeState, {
            deviceId,
        });

        assert.deepEqual(env(beforeState, action), afterState);
    });

    QUnit.test('responds to the Set Player Type action', function(assert) {
        const beforeState = {
            playerType: PlayerType.PLAYER_POPOUT,
        };
        const action = {
            type: EnvActions.ACTION_SET_PLAYER_TYPE,
            playerType: PlayerType.PLAYER_EMBED,
        };
        const afterState = {
            playerType: PlayerType.PLAYER_EMBED,
        };

        assert.deepEqual(env(beforeState, action), afterState);
    });

    QUnit.test('does not set player type if an invalid player type is used', function(assert) {
        const beforeState = {
            playerType: PlayerType.PLAYER_POPOUT,
        };
        const action = {
            type: EnvActions.ACTION_SET_PLAYER_TYPE,
            playerType: 'invalidplayertype',
        };
        const afterState = {
            playerType: PlayerType.PLAYER_POPOUT,
        };

        assert.deepEqual(env(beforeState, action), afterState);
    });

    QUnit.test('Set Player Type action preserves platform', function(assert) {
        const beforeState = {
            playerType: PlayerType.PLAYER_POPOUT,
            platform: PLATFORM_MOBILE_WEB,
        };
        const action = {
            type: EnvActions.ACTION_SET_PLAYER_TYPE,
            playerType: PlayerType.PLAYER_EMBED,
        };
        const afterState = {
            playerType: PlayerType.PLAYER_EMBED,
            platform: PLATFORM_MOBILE_WEB,
        };

        assert.deepEqual(env(beforeState, action), afterState);
    });

    QUnit.test('returns the current environment otherwise', function(assert) {
        const beforeState = {};
        const action = {
            type: 'not a real action',
            playerType: PlayerType.PLAYER_EMBED,
        };

        assert.equal(env(beforeState, action), beforeState);
    });

    QUnit.test('properly sets mobile web on set window action', function(assert) {
        const beforeState = env(undefined, { type: '@@init' });
        const action = {
            type: ACTION_SET_WINDOW,
            window: {
                document: {},
                navigator: {
                    userAgent: 'Mozilla/5.0 (iPhone; U; CPU iPhone OS 3_2 like Mac OS X; en-us) AppleWebKit/531.21.20' +
                    ' (KHTML,like Gecko) Mobile/7B298g',
                },
            },
        };
        const afterState = assign({}, beforeState, {
            platform: PLATFORM_MOBILE_WEB,
            isMobileLocation: false,
        });

        assert.deepEqual(env(beforeState, action), afterState);
    });

    QUnit.test('properly sets mobile web on set window action when device is a tablet', function(assert) {
        const beforeState = env(undefined, { type: '@@init' });
        const action = {
            type: ACTION_SET_WINDOW,
            window: {
                document: {},
                navigator: {
                    userAgent: 'Mozilla/5.0(iPad; U; CPU iPhone OS 3_2 like Mac OS X; en-us) AppleWebKit/531.21.10 ' +
                    ' (KHTML,like Gecko) Mobile/7B298g',
                },
            },
        };
        const afterState = assign({}, beforeState, {
            platform: PLATFORM_MOBILE_WEB,
            isMobileLocation: false,
        });

        assert.deepEqual(env(beforeState, action), afterState);
    });

    QUnit.test('setting platform preserves playerType', function(assert) {
        const beforeState = {
            playerType: PlayerType.PLAYER_EMBED,
        };
        const action = {
            type: ACTION_SET_WINDOW,
            window: {
                document: {},
                navigator: {
                    userAgent: 'Mozilla/5.0 (iPhone; U; CPU iPhone OS 3_2 like Mac OS X; en-us) AppleWebKit/531.21.20' +
                    ' (KHTML,like Gecko) Mobile/7B298g',
                },
            },
        };
        assert.deepEqual(env(beforeState, action), {
            platform: PLATFORM_MOBILE_WEB,
            playerType: PlayerType.PLAYER_EMBED,
            isMobileLocation: false,
        });
    });

    QUnit.test('properly sets isMobileLocation to true for mobile web staging', function(assert) {
        const beforeState = env(undefined, { type: '@@init' });
        const action = {
            type: ACTION_SET_WINDOW,
            window: {
                top: {},
                document: {
                    referrer: 'https://im.a.staging.environment.m.twitch.tv',
                },
            },
        };

        assert.equal(env(beforeState, action).isMobileLocation, true);
    });

    QUnit.test('properly sets isMobileLocation to true in response to set window', function(assert) {
        const beforeState = env(undefined, { type: '@@init' });
        const action = {
            type: ACTION_SET_WINDOW,
            window: {
                top: {},
                document: {
                    referrer: 'https://m.twitch.tv',
                },
            },
        };

        assert.equal(env(beforeState, action).isMobileLocation, true);
    });

    QUnit.test('sets isMobileLocation to false w/ bad protocol in response to set window', function(assert) {
        const beforeState = env(undefined, { type: '@@init' });
        const action = {
            type: ACTION_SET_WINDOW,
            window: {
                top: {},
                document: {
                    referrer: 'http://m.twitch.tv',
                },
            },
        };

        assert.equal(env(beforeState, action).isMobileLocation, false);
    });

    QUnit.test('properly sets isMobileLocation to false w/ bad hostname in response to set window', function(assert) {
        const beforeState = env(undefined, { type: '@@init' });
        const action = {
            type: ACTION_SET_WINDOW,
            window: {
                top: {},
                document: {
                    referrer: 'http://m.reddit.com',
                },
            },
        };

        assert.equal(env(beforeState, action).isMobileLocation, false);
    });

    QUnit.test('set isMobileLocation to true when embed is inferred through a security error', function(assert) {
        const beforeState = {
            isMobileLocation: true,
        };

        const action = {
            type: ACTION_SET_WINDOW,
            window: {
                get parent() {
                    throw new Error('test error');
                },
                document: {
                    referrer: 'https://m.twitch.tv',
                },
            },
        };

        assert.equal(env(beforeState, action).isMobileLocation, true);
    });

    QUnit.test('set isMobileLocation when embed is inferred but not on m.twitch.tv', function(assert) {
        const beforeState = {
            isMobileLocation: true,
        };

        const action = {
            type: ACTION_SET_WINDOW,
            window: {
                get parent() {
                    throw new Error('test error');
                },
                document: {
                    referrer: 'https://m.reddit.com',
                },
            },
        };

        assert.equal(env(beforeState, action).isMobileLocation, false);
    });

    QUnit.test('responds to the Set Twitch Everywhere Params action', function(assert) {
        const beforeState = {
            twitchEverywhere: { targetOrigin: null },
        };
        const action = {
            type: EnvActions.ACTION_SET_TWITCH_EVERYWHERE_PARAMS,
            params: { targetOrigin: 'foo' },
        };
        const afterState = {
            twitchEverywhere: { targetOrigin: 'foo' },
        };

        assert.deepEqual(env(beforeState, action), afterState);
    });
});
