import { unitTest } from 'tests/utils/module';
import * as ExtensionsActions from 'actions/extensions';
import { extensions } from 'state/extensions';
import { createExtensionParsed } from '../../../fixtures/extensions';
import { EXTENSION_PERMISSION_STATE_GRANTED } from 'util/extensions';
import merge from 'lodash/merge';

const TEST_CHANNEL_NAME = 'twitch';

const DEFAULT_EXTENSION_STATE = {
    channel: undefined,
    activeOverlayExtensions: [],
    loadingState: ExtensionsActions.EXTENSIONS_NOT_LOADED,
    modalRequest: undefined,
    purchaseIntent: null,
};

unitTest('state | extensions', function() {
    QUnit.test('defaults to sensible values', function(assert) {
        const beforeState = undefined;
        const action = {
            type: '@@init',
        };

        assert.deepEqual(extensions(beforeState, action), DEFAULT_EXTENSION_STATE);
    });

    QUnit.test('responds to Set Extensions action', function(assert) {
        const beforeState = {
            activeOverlayExtensions: [],
            loadingState: ExtensionsActions.EXTENSIONS_NOT_LOADED,
        };
        const extensionItems = createExtensionParsed();
        const action = {
            type: ExtensionsActions.ACTION_SET_EXTENSIONS,
            channel: TEST_CHANNEL_NAME,
            extensions: extensionItems,
        };
        const afterState = {
            channel: TEST_CHANNEL_NAME,
            activeOverlayExtensions: extensionItems,
            loadingState: ExtensionsActions.EXTENSIONS_LOADED,
        };

        assert.deepEqual(extensions(beforeState, action), afterState);
    });

    QUnit.test('responds to Clear Extensions action', function(assert) {
        const extensionItems = createExtensionParsed();
        const beforeState = {
            activeOverlayExtensions: extensionItems,
            loadingState: ExtensionsActions.EXTENSIONS_LOADED,
            purchaseIntent: null,
        };
        const action = { type: ExtensionsActions.ACTION_CLEAR_EXTENSIONS };
        assert.deepEqual(extensions(beforeState, action), DEFAULT_EXTENSION_STATE);
    });

    QUnit.test('responds to Fetching Extensions action', function(assert) {
        const extensionItems = createExtensionParsed();
        const beforeState = {
            activeOverlayExtensions: extensionItems,
            loadingState: ExtensionsActions.EXTENSIONS_NOT_LOADED,
        };
        const action = {
            type: ExtensionsActions.ACTION_FETCHING_EXTENSIONS,
            channel: TEST_CHANNEL_NAME,
        };
        const afterState = {
            channel: TEST_CHANNEL_NAME,
            activeOverlayExtensions: [],
            loadingState: ExtensionsActions.EXTENSIONS_LOADING,
        };
        assert.deepEqual(extensions(beforeState, action), afterState);
    });

    QUnit.test('responds to Set Identity action', function(assert) {
        const extension = createExtensionParsed()[0];
        const beforeState = {
            channel: TEST_CHANNEL_NAME,
            activeOverlayExtensions: [
                extension,
            ],
            loadingState: ExtensionsActions.EXTENSIONS_LOADED,
        };
        const action = {
            type: ExtensionsActions.ACTION_SET_EXTENSION_IDENTITY,
            extensionId: extension.id,
            isLinked: true,
        };
        const afterState = {
            channel: TEST_CHANNEL_NAME,
            activeOverlayExtensions: [
                merge({}, extension, {
                    lastUserIdentityLinkState: action.isLinked,
                    token: {
                        permissionsState: EXTENSION_PERMISSION_STATE_GRANTED,
                    },
                }),
            ],
            loadingState: ExtensionsActions.EXTENSIONS_LOADED,
        };
        assert.deepEqual(extensions(beforeState, action), afterState);
    });

    QUnit.test('responsed to Toggle Identity action and enables identity link', function(assert) {
        const extension = createExtensionParsed()[0];
        const beforeState = {
            channel: TEST_CHANNEL_NAME,
            activeOverlayExtensions: [
                extension,
            ],
        };
        const action = {
            type: ExtensionsActions.ACTION_REQUEST_TOGGLE_EXTENSION_IDENTITY,
            extensionId: extension.id,
        };
        const afterState = {
            channel: TEST_CHANNEL_NAME,
            activeOverlayExtensions: [
                merge({}, extension, {
                    lastUserIdentityLinkState: true,
                }),
            ],
        };
        assert.deepEqual(extensions(beforeState, action), afterState);
    });

    QUnit.test('responsed to Toggle Identity action and disabled identity link', function(assert) {
        const extension = merge({}, createExtensionParsed()[0], {
            lastUserIdentityLinkState: true,
            token: {
                permissionsState: EXTENSION_PERMISSION_STATE_GRANTED,
            },
        });
        extension.lastUserIdentityLinkState = true;

        const beforeState = {
            channel: TEST_CHANNEL_NAME,
            activeOverlayExtensions: [
                extension,
            ],
        };
        const action = {
            type: ExtensionsActions.ACTION_REQUEST_TOGGLE_EXTENSION_IDENTITY,
            extensionId: extension.id,
        };
        const afterState = {
            channel: TEST_CHANNEL_NAME,
            activeOverlayExtensions: [
                merge({}, extension, {
                    lastUserIdentityLinkState: false,
                }),
            ],
        };
        assert.deepEqual(extensions(beforeState, action), afterState);
    });

    QUnit.test('responds to Extension Activated action', function(assert) {
        const beforeState = DEFAULT_EXTENSION_STATE;
        const action = {
            type: ExtensionsActions.ACTION_EXTENSION_ACTIVATED,
            channel: TEST_CHANNEL_NAME,
        };

        const afterState = {
            channel: TEST_CHANNEL_NAME,
            activeOverlayExtensions: DEFAULT_EXTENSION_STATE.activeOverlayExtensions,
            loadingState: ExtensionsActions.EXTENSIONS_RELOADING,
            modalRequest: DEFAULT_EXTENSION_STATE.modalRequest,
            purchaseIntent: null,
        };

        assert.deepEqual(extensions(beforeState, action), afterState);
    });

    QUnit.test('responds to Set Modal action', function(assert) {
        const beforeState = DEFAULT_EXTENSION_STATE;
        const action = {
            type: ExtensionsActions.ACTION_SET_EXTENSION_MODAL ,
            modalRequest: {
                action: 'twitch-ext-random-modal-request',
                options: {},
                resultCallback() {},
                defaultResult: {},
            },
        };

        const afterState = {
            channel: DEFAULT_EXTENSION_STATE.channel,
            activeOverlayExtensions: DEFAULT_EXTENSION_STATE.activeOverlayExtensions,
            loadingState: DEFAULT_EXTENSION_STATE.loadingState,
            modalRequest: action.modalRequest,
            purchaseIntent: null,
        };

        assert.deepEqual(extensions(beforeState, action), afterState);
    });
});
