import assign from 'lodash/assign';
import * as CollectionActions from 'actions/collection';
import * as PlaybackActions from 'actions/playback';
import * as StreamActions from 'actions/stream';
import { playback, TRANSITION_TYPE_COLLECTION, TRANSITION_TYPE_RECOMMENDATIONS,
         isPlayingSound } from 'state/playback';
import { nullContentStream } from 'stream/null';

QUnit.module('state | playback', function() {
    QUnit.test('responds to the Set Stream action', function(assert) {
        const action = {
            type: StreamActions.ACTION_SET_STREAM,
        };
        const beforeState = {
            playing: true,
            hasPlayed: true,
            contentShowing: true,
            bufferEmpties: 3,
            startTime: 10,
            startTimeSet: true,
            isSeeking: true,
            restrictedQualityError: true,
        };
        const afterState = {
            playing: false,
            hasPlayed: false,
            contentShowing: false,
            bufferEmpties: 0,
            startTime: 0,
            startTimeSet: false,
            isSeeking: false,
            restrictedQualityError: false,
        };

        assert.deepEqual(playback(beforeState, action), afterState);
    });

    QUnit.test('responds to the Set Stream with nullContentStream action', function(assert) {
        const action = {
            type: StreamActions.ACTION_SET_STREAM,
            stream: nullContentStream,
        };
        const beforeState = {
            playing: true,
            hasPlayed: true,
            contentShowing: true,
            bufferEmpties: 3,
            paused: true,
            startTime: 10,
            startTimeSet: true,
            isSeeking: true,
            restrictedQualityError: true,
        };
        const afterState = {
            playing: false,
            hasPlayed: false,
            contentShowing: false,
            bufferEmpties: 0,
            paused: false,
            startTime: 0,
            startTimeSet: false,
            isSeeking: false,
            restrictedQualityError: false,
        };

        assert.deepEqual(playback(beforeState, action), afterState);
    });

    QUnit.test('responds to the ACTION_PAUSE action', function(assert) {
        const action = {
            type: PlaybackActions.ACTION_PAUSE,
        };
        const beforeState = {
            playing: true,
            paused: false,
            ended: false,
        };
        const afterState = {
            playing: false,
            paused: true,
            ended: false,
        };

        assert.deepEqual(playback(beforeState, action), afterState);
    });

    QUnit.test('responds to the ACTION_PLAYING action', function(assert) {
        const action = {
            type: PlaybackActions.ACTION_PLAYING,
        };
        const beforeState = {
            playing: false,
            hasPlayed: false,
            paused: true,
            ended: true,
            contentShowing: false,
            isSeeking: true,
        };
        const afterState = {
            playing: true,
            hasPlayed: true,
            paused: false,
            ended: false,
            contentShowing: true,
            isSeeking: false,
        };

        assert.deepEqual(playback(beforeState, action), afterState);
    });

    QUnit.test('responds to the ACTION_WAITING action', function(assert) {
        const action = {
            type: PlaybackActions.ACTION_WAITING,
        };
        const beforeState = {
            ended: true,
        };
        const afterState = {
            ended: false,
        };

        assert.deepEqual(playback(beforeState, action), afterState);
    });

    QUnit.test('responds to the ACTION_ENDED action', function(assert) {
        const action = {
            type: PlaybackActions.ACTION_ENDED,
        };
        const beforeState = {
            playing: true,
            paused: false,
            ended: false,
        };
        const afterState = {
            playing: false,
            paused: true,
            ended: true,
        };

        assert.deepEqual(playback(beforeState, action), afterState);
    });

    QUnit.test('responds to the ACTION_PLAYER_SEEKING action', function(assert) {
        const action = {
            type: PlaybackActions.ACTION_PLAYER_SEEKING,
        };
        const beforeState = {
            isSeeking: false,
        };
        const afterState = {
            isSeeking: true,
        };

        assert.deepEqual(playback(beforeState, action), afterState);
    });

    QUnit.test('responds to the ACTION_PLAYER_SEEKED action', function(assert) {
        const action = {
            type: PlaybackActions.ACTION_PLAYER_SEEKED,
        };
        const beforeState = {
            isSeeking: true,
        };
        const afterState = {
            isSeeking: false,
        };

        assert.deepEqual(playback(beforeState, action), afterState);
    });

    QUnit.test('responds to the Update Playback State action when duration changes', function(assert) {
        const newDuration = parseInt(QUnit.config.current.testId, 36);
        const oldDuration = Math.floor(0.5 * newDuration);
        const action = {
            type: PlaybackActions.ACTION_UPDATE_PLAYBACK_DURATION,
            playback: {
                duration: newDuration,
            },
        };

        const beforeState = { duration: oldDuration };
        const afterState = action.playback;

        assert.deepEqual(playback(beforeState, action), afterState);
    });

    QUnit.test('ignores Update Playback State actions with unknown keys', function(assert) {
        const action = {
            type: PlaybackActions.ACTION_UPDATE_PLAYBACK_DURATION,
            playback: {
                notarealkey: QUnit.config.current.testId,
                duration: parseInt(QUnit.config.current.testId, 36),
            },
        };

        const beforeState = playback(undefined, { type: '@@init' });
        const afterState = assign({}, beforeState, { duration: action.playback.duration });

        assert.deepEqual(playback(beforeState, action), afterState);
    });

    QUnit.test('responds to the Quality Restricted Error action', function(assert) {
        const action = {
            type: PlaybackActions.ACTION_QUALITY_RESTRICTED_ERROR,
        };
        const beforeState = {
            restrictedQualityError: false,
        };
        const afterState = {
            restrictedQualityError: true,
        };

        assert.deepEqual(playback(beforeState, action), afterState);
    });

    QUnit.test('responds to the Clear Quality Restricted Error action', function(assert) {
        const action = {
            type: PlaybackActions.ACTION_CLEAR_QUALITY_RESTRICTED_ERROR,
        };
        const beforeState = {
            restrictedQualityError: true,
        };
        const afterState = {
            restrictedQualityError: false,
        };

        assert.deepEqual(playback(beforeState, action), afterState);
    });

    QUnit.test('responds to the Volume Changed action', function(assert) {
        const action = {
            type: PlaybackActions.ACTION_VOLUME_CHANGED,
            volume: 0.8,
        };
        const beforeState = {
            volume: 0.3,
        };
        const afterState = {
            volume: 0.8,
        };

        assert.deepEqual(playback(beforeState, action), afterState);
    });

    QUnit.test('responds to the Player Muted action', function(assert) {
        const action = {
            type: PlaybackActions.ACTION_PLAYER_MUTED,
            muted: true,
        };
        const beforeState = {
            muted: false,
        };
        const afterState = {
            muted: true,
        };

        assert.deepEqual(playback(beforeState, action), afterState);
    });

    QUnit.test('returns the current playback state otherwise', function(assert) {
        const action = {
            type: 'not a real action',
            playback: {
                muted: true,
            },
        };
        const beforeState = { muted: false };
        assert.equal(playback(beforeState, action), beforeState);
    });

    QUnit.test('responds to the Set Autoplay State action', function(assert) {
        const action = {
            type: PlaybackActions.ACTION_SET_AUTOPLAY_STATE,
            playback: {
                autoplay: false,
                paused: true,
            },
        };

        const beforeState = {
            autoplay: true,
            paused: false,
        };
        const afterState = action.playback;

        assert.deepEqual(
            playback(beforeState, action),
            afterState,
            'if autoplay is false, set paused to true'
        );
    });

    QUnit.test('responds to the set loading state action', function(assert) {
        const action = PlaybackActions.setLoading(true);
        const beforeState = {
            isLoading: false,
        };

        const afterState = {
            isLoading: true,
        };

        assert.deepEqual(playback(beforeState, action), afterState);
    });

    QUnit.test('responds to the update current time action', function(assert) {
        const action = PlaybackActions.updateCurrentTime(5);
        const beforeState = {
            currentTime: 0,
        };

        const afterState = {
            currentTime: 5,
        };

        assert.deepEqual(playback(beforeState, action), afterState);
    });

    QUnit.test('responds to the update buffer values action', function(assert) {
        const start = 5;
        const end = 10;
        const action = PlaybackActions.updateBufferValues(start, end);
        const beforeState = {
            buffer: {
                start: 0,
                end: 0,
            },
        };

        const afterState = {
            buffer: {
                start,
                end,
            },
        };

        assert.deepEqual(playback(beforeState, action), afterState);
    });

    QUnit.test('responds to the increment bufferEmpties action', function(assert) {
        const action = {
            type: PlaybackActions.ACTION_INCREMENT_BUFFER_EMPTIES,
        };

        const beforeState = {
            bufferEmpties: 0,
        };

        const afterState = {
            bufferEmpties: 1,
        };

        assert.deepEqual(playback(beforeState, action), afterState);
    });

    QUnit.test('responds to the ACTION_CONTENT_IS_SHOWING action', function(assert) {
        const action = {
            type: PlaybackActions.ACTION_CONTENT_IS_SHOWING,
        };
        const beforeState = {
            contentShowing: false,
        };

        const afterState = {
            contentShowing: true,
        };

        assert.deepEqual(playback(beforeState, action), afterState, 'should set itself to true');
    });

    QUnit.test('responds to the ACTION_LOADED_LAST_COLLECTION_ITEM action', function(assert) {
        const action = {
            type: CollectionActions.ACTION_LOADED_LAST_COLLECTION_ITEM,
        };
        const beforeState = {
            transitionScheme: TRANSITION_TYPE_COLLECTION,
        };

        const afterState = {
            transitionScheme: TRANSITION_TYPE_RECOMMENDATIONS,
        };

        assert.deepEqual(playback(beforeState, action), afterState, 'should set transition scheme to recommendations');
    });

    QUnit.test('responds to the ACTION_LOADED_COLLECTION_ITEM action', function(assert) {
        const action = {
            type: CollectionActions.ACTION_LOADED_COLLECTION_ITEM,
        };
        const beforeState = {
            transitionScheme: TRANSITION_TYPE_RECOMMENDATIONS,
        };

        const afterState = {
            transitionScheme: TRANSITION_TYPE_COLLECTION,
        };

        assert.deepEqual(playback(beforeState, action), afterState, 'should set transition scheme to collection');
    });

    QUnit.test('responds to the ACTION_PLAYBACK_RATE_CHANGED action', function(assert) {
        const playbackRate = 2.0;
        const action = {
            type: PlaybackActions.ACTION_PLAYBACK_RATE_CHANGED,
            playbackRate,
        };
        const beforeState = {
            playbackRate: 1.0,
        };

        const afterState = {
            playbackRate: 2.0,
        };

        assert.deepEqual(playback(beforeState, action), afterState, 'should update playbackRate value');
    });

    QUnit.test('responds to the ACTION_SET_START_TIME action', function(assert) {
        const startTime = 100;
        const action = {
            type: PlaybackActions.ACTION_SET_START_TIME,
            startTime,
        };
        const beforeState = {
            startTimeSet: false,
            startTime: 0,
        };

        const resultState = playback(beforeState, action);

        assert.equal(resultState.startTime, startTime, 'should update startTime value');
        assert.ok(resultState.startTimeSet, 'should update startTimeSet to true');
    });

    function testIsPlayingSound(name, { volume, muted, expected }) {
        QUnit.test(name, function(assert) {
            const state = {
                playback: {
                    volume,
                    muted,
                },
            };
            // eslint-disable-next-line max-len
            assert.equal(isPlayingSound(state), expected, `should be ${expected} when volume=${volume} and muted=${muted})`);
        });
    }

    testIsPlayingSound('isPlayingSound returns false when muted and nonzero volume', {
        volume: 0.1,
        muted: true,
        expected: false,
    });

    testIsPlayingSound('isPlayingSound returns false when muted and zero volume', {
        volume: 0,
        muted: true,
        expected: false,
    });

    testIsPlayingSound('isPlayingSound returns false when unmuted and zero volume', {
        volume: 0,
        muted: false,
        expected: false,
    });

    testIsPlayingSound('isPlayingSound returns true when unmuted and nonzero volume', {
        volume: 0.1,
        muted: false,
        expected: true,
    });
});
