import { unitTest } from 'tests/utils/module';
import * as ResumeWatchActions from 'actions/resume-watch';
import { ACTION_USER_LOGGED_IN } from 'actions/user';
import { LiveTwitchContentStream } from 'stream/twitch-live';
import { VODTwitchContentStream } from 'stream/twitch-vod';
import { ACTION_SET_STREAM } from 'actions/stream';
import * as ResumeWatchState from 'state/resume-watch';

unitTest('state | resume-watch', function() {
    QUnit.test('defaults to an empty struct', function(assert) {
        const beforeState = undefined;
        const action = {
            type: '@@init',
        };

        assert.deepEqual(ResumeWatchState.resumeWatch(beforeState, action), {
            times: {},
            watch: {},
            streamTimes: {},
            userId: null,
            lastTimeStamp: 0,
            initUpdateOffset: 0,
            updateInterval: 0,
            isSeeked: false,
        });
    });

    QUnit.test('responds to the Init Resume Times', function(assert) {
        const videoID = `v${parseInt(QUnit.config.current.testId, 36)}`;
        const time = 9283;
        const broadcastID = `${Math.floor(Math.random() * 1e8)}`;
        const streamTime = 7465;
        const beforeState = {};
        const action = {
            type: ResumeWatchActions.ACTION_VOD_INIT_RESUME,
            times: {
                [videoID]: time,
            },
            watch: {
                [videoID]: true,
            },
            streamTimes: {
                [broadcastID]: streamTime,
            },
            lastTimeStamp: 0,
            userId: null,
            initUpdateOffset: ResumeWatchState.INIT_UPDATE_OFFSET_VOD,
            updateInterval: ResumeWatchState.UPDATE_INTERVAL_VOD,
            isSeeked: false,
        };

        assert.deepEqual(ResumeWatchState.resumeWatch(beforeState, action), {
            times: {
                [videoID]: time,
            },
            watch: {
                [videoID]: true,
            },
            streamTimes: {
                [broadcastID]: streamTime,
            },
            userId: null,
            lastTimeStamp: 0,
            initUpdateOffset: ResumeWatchState.INIT_UPDATE_OFFSET_VOD,
            updateInterval: ResumeWatchState.UPDATE_INTERVAL_VOD,
            isSeeked: false,
        });
    });

    QUnit.test('responds to the Set Resume Times action', function(assert) {
        const beforeState = {
            times: {},
            watch: {},
            streamTimes: {},
            lastTimeStamp: 25,
            userId: '12345',
            initUpdateOffset: ResumeWatchState.INIT_UPDATE_OFFSET_VOD,
            updateInterval: ResumeWatchState.UPDATE_INTERVAL_VOD,
            isSeeked: false,
        };

        /* eslint-disable camelcase */
        const vodResumeTime = {
            position: 278,
            type: 'vod',
            updated_at: '2016-09-20T20:32:28.781542Z',
            video_id: 'v90236935',
        };

        const liveResumeTime = {
            position: 11,
            type: 'live',
            updated_at: '2016-09-20T20:32:28.781542Z',
            video_id: '12353253252',
        };
        /* eslint-enable camelcase */

        const action = {
            type: ResumeWatchActions.ACTION_SET_RESUME_TIMES,
            resumeTimes: [vodResumeTime, liveResumeTime],
        };

        assert.deepEqual(ResumeWatchState.resumeWatch(beforeState, action), {
            times: {
                [vodResumeTime.video_id]: vodResumeTime.position,
            },
            watch: {
                [vodResumeTime.video_id]: true,
            },
            streamTimes: {
                [liveResumeTime.video_id]: liveResumeTime.position,
            },
            userId: '12345',
            lastTimeStamp: 25,
            initUpdateOffset: ResumeWatchState.INIT_UPDATE_OFFSET_VOD,
            updateInterval: ResumeWatchState.UPDATE_INTERVAL_VOD,
            isSeeked: false,
        });
    });

    QUnit.test('responds to the Set VOD Resume Time action', function(assert) {
        const beforeState = {
            times: {
                v12345: 123,
            },
            watch: {
                v12345: true,
            },
            streamTimes: {
                54321: 543,
            },
            lastTimeStamp: 25,
            userId: '12345',
            initUpdateOffset: ResumeWatchState.INIT_UPDATE_OFFSET_VOD,
            updateInterval: ResumeWatchState.UPDATE_INTERVAL_VOD,
            isSeeked: false,
        };
        const action = {
            type: ResumeWatchActions.ACTION_VOD_SET_RESUME_TIME,
            videoID: 'v23456',
            time: 352,
        };

        assert.deepEqual(ResumeWatchState.resumeWatch(beforeState, action), {
            times: {
                v12345: beforeState.times.v12345,
                [action.videoID]: action.time,
            },
            watch: {
                v12345: true,
                [action.videoID]: true,
            },
            streamTimes: {
                54321: 543,
            },
            userId: '12345',
            lastTimeStamp: 25,
            initUpdateOffset: ResumeWatchState.INIT_UPDATE_OFFSET_VOD,
            updateInterval: ResumeWatchState.UPDATE_INTERVAL_VOD,
            isSeeked: false,
        });
    });

    QUnit.test('responds to the Set Livestream Resume Time action', function(assert) {
        const beforeState = {
            times: {
                v12345: 123,
            },
            watch: {
                v12345: true,
            },
            streamTimes: {
                54321: 543,
            },
            userId: '12345',
            lastTimeStamp: 25,
            initUpdateOffset: Math.ceil(Math.random() * ResumeWatchState.MAX_INIT_UPDATE_OFFSET_LIVESTREAM),
            updateInterval: ResumeWatchState.UPDATE_INTERVAL_LIVESTREAM,
            isSeeked: false,
        };
        const action = {
            type: ResumeWatchActions.ACTION_LIVESTREAM_SET_RESUME_TIME,
            broadcastID: '65432',
            time: 352,
        };

        assert.deepEqual(ResumeWatchState.resumeWatch(beforeState, action), {
            times: {
                v12345: beforeState.times.v12345,
            },
            watch: {
                v12345: true,
            },
            streamTimes: {
                54321: 543,
                [action.broadcastID]: action.time,
            },
            userId: '12345',
            lastTimeStamp: 25,
            initUpdateOffset: beforeState.initUpdateOffset,
            updateInterval: ResumeWatchState.UPDATE_INTERVAL_LIVESTREAM,
            isSeeked: false,
        });
    });

    QUnit.test('responds to the Cancel VOD Resume action', function(assert) {
        const vodID = `v${parseInt(QUnit.config.current.testId, 36)}`;
        const cancelID = `v${Math.floor(Math.random() * 1e8)}`;
        const broadcastID = `${Math.floor(Math.random() * 1e8)}`;
        const beforeState = {
            times: {
                [vodID]: Math.floor(Math.random() * 1e3),
                [cancelID]: Math.floor(Math.random() * 1e3),
            },
            watch: {
                [vodID]: true,
                [cancelID]: true,
            },
            streamTimes: {
                [broadcastID]: Math.floor(Math.random() * 1e3),
            },
            lastTimeStamp: 25,
            userId: '12345',
            initUpdateOffset: ResumeWatchState.INIT_UPDATE_OFFSET_VOD,
            updateInterval: ResumeWatchState.UPDATE_INTERVAL_VOD,
            isSeeked: false,
        };
        const action = {
            type: ResumeWatchActions.ACTION_VOD_CANCEL_RESUME,
            videoID: cancelID,
        };

        assert.deepEqual(ResumeWatchState.resumeWatch(beforeState, action), {
            times: {
                [vodID]: beforeState.times[vodID],
            },
            watch: {
                [vodID]: true,
            },
            streamTimes: {
                [broadcastID]: beforeState.streamTimes[broadcastID],
            },
            lastTimeStamp: 25,
            userId: '12345',
            initUpdateOffset: ResumeWatchState.INIT_UPDATE_OFFSET_VOD,
            updateInterval: ResumeWatchState.UPDATE_INTERVAL_VOD,
            isSeeked: false,
        });
    });

    QUnit.test('responds to the Cancel Livestream Resume action', function(assert) {
        const vodID = `v${parseInt(QUnit.config.current.testId, 36)}`;
        const broadcastID = `${Math.floor(Math.random() * 1e8)}`;
        const cancelID = `${Math.floor(Math.random() * 1e8)}`;
        const beforeState = {
            times: {
                [vodID]: Math.floor(Math.random() * 1e3),
            },
            watch: {
                [vodID]: true,
            },
            streamTimes: {
                [broadcastID]: Math.floor(Math.random() * 1e3),
                [cancelID]: Math.floor(Math.random() * 1e3),

            },
            lastTimeStamp: 25,
            userId: '12345',
            initUpdateOffset: Math.ceil(Math.random() * ResumeWatchState.MAX_INIT_UPDATE_OFFSET_LIVESTREAM),
            updateInterval: ResumeWatchState.UPDATE_INTERVAL_LIVESTREAM,
            isSeeked: false,
        };
        const action = {
            type: ResumeWatchActions.ACTION_LIVESTREAM_CANCEL_RESUME,
            broadcastID: cancelID,
        };

        assert.deepEqual(ResumeWatchState.resumeWatch(beforeState, action), {
            times: {
                [vodID]: beforeState.times[vodID],
            },
            watch: {
                [vodID]: true,
            },
            streamTimes: {
                [broadcastID]: beforeState.streamTimes[broadcastID],
            },
            lastTimeStamp: 25,
            userId: '12345',
            initUpdateOffset: beforeState.initUpdateOffset,
            updateInterval: ResumeWatchState.UPDATE_INTERVAL_LIVESTREAM,
            isSeeked: false,
        });
    });

    QUnit.test('responds to the set user action', function(assert) {
        const vodId = `v${parseInt(QUnit.config.current.testId, 36)}`;
        const broadcastID = `${Math.floor(Math.random() * 1e8)}`;
        const beforeState = {
            times: {
                [vodId]: Math.floor(Math.random() * 1e3),
            },
            watch: {
                [vodId]: true,
            },
            streamTimes: {
                [broadcastID]: Math.floor(Math.random() * 1e3),
            },
            lastTimeStamp: 25,
            userId: null,
            initUpdateOffset: ResumeWatchState.INIT_UPDATE_OFFSET_VOD,
            updateInterval: ResumeWatchState.UPDATE_INTERVAL_VOD,
            isSeeked: false,
        };
        const action = {
            type: ResumeWatchActions.ACTION_VOD_SET_USER,
            userId: '12345',
        };

        assert.deepEqual(ResumeWatchState.resumeWatch(beforeState, action), {
            times: {
                [vodId]: beforeState.times[vodId],
            },
            watch: {
                [vodId]: true,
            },
            streamTimes: {
                [broadcastID]: beforeState.streamTimes[broadcastID],
            },
            lastTimeStamp: 25,
            userId: '12345',
            initUpdateOffset: ResumeWatchState.INIT_UPDATE_OFFSET_VOD,
            updateInterval: ResumeWatchState.UPDATE_INTERVAL_VOD,
            isSeeked: false,
        });
    });

    QUnit.test('responds to the user logged in action', function(assert) {
        const vodId = `v${parseInt(QUnit.config.current.testId, 36)}`;
        const broadcastID = `${Math.floor(Math.random() * 1e8)}`;
        const userID = 12345;
        const beforeState = {
            times: {
                [vodId]: Math.floor(Math.random() * 1e3),
            },
            watch: {
                [vodId]: true,
            },
            streamTimes: {
                [broadcastID]: Math.floor(Math.random() * 1e3),
            },
            lastTimeStamp: 25,
            userId: null,
            initUpdateOffset: ResumeWatchState.INIT_UPDATE_OFFSET_VOD,
            updateInterval: ResumeWatchState.UPDATE_INTERVAL_VOD,
            isSeeked: false,
        };
        const action = {
            type: ACTION_USER_LOGGED_IN,
            userInfo: {
                id: userID,
            },
        };

        assert.deepEqual(ResumeWatchState.resumeWatch(beforeState, action), {
            times: {
                [vodId]: beforeState.times[vodId],
            },
            watch: {
                [vodId]: true,
            },
            streamTimes: {
                [broadcastID]: beforeState.streamTimes[broadcastID],
            },
            lastTimeStamp: 25,
            userId: userID,
            initUpdateOffset: ResumeWatchState.INIT_UPDATE_OFFSET_VOD,
            updateInterval: ResumeWatchState.UPDATE_INTERVAL_VOD,
            isSeeked: false,
        });
    });

    QUnit.test('responds to the set isSeeked action', function(assert) {
        const vodId = `v${parseInt(QUnit.config.current.testId, 36)}`;
        const broadcastID = `${Math.floor(Math.random() * 1e8)}`;
        const beforeState = {
            times: {
                [vodId]: Math.floor(Math.random() * 1e3),
            },
            watch: {
                [vodId]: true,
            },
            streamTimes: {
                [broadcastID]: Math.floor(Math.random() * 1e3),
            },
            lastTimeStamp: 25,
            userId: '12345',
            initUpdateOffset: ResumeWatchState.INIT_UPDATE_OFFSET_VOD,
            updateInterval: ResumeWatchState.UPDATE_INTERVAL_VOD,
            isSeeked: false,
        };
        const action = {
            type: ResumeWatchActions.ACTION_VOD_SET_IS_SEEKED,
            isSeeked: true,
        };

        assert.deepEqual(ResumeWatchState.resumeWatch(beforeState, action), {
            times: {
                [vodId]: beforeState.times[vodId],
            },
            watch: {
                [vodId]: true,
            },
            streamTimes: {
                [broadcastID]: beforeState.streamTimes[broadcastID],
            },
            lastTimeStamp: 25,
            userId: '12345',
            initUpdateOffset: ResumeWatchState.INIT_UPDATE_OFFSET_VOD,
            updateInterval: ResumeWatchState.UPDATE_INTERVAL_VOD,
            isSeeked: true,
        });
    });

    QUnit.test('responds to the set stream action with a VOD', function(assert) {
        const vodId = `v${parseInt(QUnit.config.current.testId, 36)}`;
        const broadcastID = `${Math.floor(Math.random() * 1e8)}`;
        const beforeState = {
            times: {
                [vodId]: Math.floor(Math.random() * 1e3),
            },
            watch: {
                [vodId]: true,
            },
            streamTimes: {
                [broadcastID]: Math.floor(Math.random() * 1e3),
            },
            lastTimeStamp: 25,
            userId: '12345',
            initUpdateOffset: 0,
            updateInterval: 0,
            isSeeked: true,
        };
        const action = {
            type: ACTION_SET_STREAM,
            stream: new VODTwitchContentStream(vodId, 'oauth', 'usher', {}),
        };

        assert.deepEqual(ResumeWatchState.resumeWatch(beforeState, action), {
            times: {
                [vodId]: beforeState.times[vodId],
            },
            watch: {
                [vodId]: true,
            },
            streamTimes: {
                [broadcastID]: beforeState.streamTimes[broadcastID],
            },
            lastTimeStamp: 0,
            userId: '12345',
            initUpdateOffset: ResumeWatchState.INIT_UPDATE_OFFSET_VOD,
            updateInterval: ResumeWatchState.UPDATE_INTERVAL_VOD,
            isSeeked: false,
        });
    });

    QUnit.test('responds to the set stream action with a live stream', function(assert) {
        const vodId = `v${parseInt(QUnit.config.current.testId, 36)}`;
        const broadcastID = `${Math.floor(Math.random() * 1e8)}`;
        const beforeState = {
            times: {
                [vodId]: Math.floor(Math.random() * 1e3),
            },
            watch: {
                [vodId]: true,
            },
            streamTimes: {
                [broadcastID]: Math.floor(Math.random() * 1e3),
            },
            lastTimeStamp: 25,
            userId: '12345',
            initUpdateOffset: 0,
            updateInterval: 0,
            isSeeked: true,
        };
        const action = {
            type: ACTION_SET_STREAM,
            stream: new LiveTwitchContentStream('extalix', 'oauth', 'usher', {}, false),
        };

        const afterState = ResumeWatchState.resumeWatch(beforeState, action);

        assert.deepEqual(afterState, {
            times: {
                [vodId]: beforeState.times[vodId],
            },
            watch: {
                [vodId]: true,
            },
            streamTimes: {
                [broadcastID]: beforeState.streamTimes[broadcastID],
            },
            lastTimeStamp: 0,
            userId: '12345',
            initUpdateOffset: afterState.initUpdateOffset,
            updateInterval: ResumeWatchState.UPDATE_INTERVAL_LIVESTREAM,
            isSeeked: false,
        });

        assert.ok(afterState.initUpdateOffset > 0);
    });

    QUnit.test('responds to the vod post backend time action', function(assert) {
        const vodId = `v${parseInt(QUnit.config.current.testId, 36)}`;
        const broadcastID = `${Math.floor(Math.random() * 1e8)}`;
        const beforeState = {
            times: {
                [vodId]: Math.floor(Math.random() * 1e3),
            },
            watch: {
                [vodId]: true,
            },
            streamTimes: {
                [broadcastID]: Math.floor(Math.random() * 1e3),
            },
            lastTimeStamp: 25,
            userId: null,
            initUpdateOffset: ResumeWatchState.INIT_UPDATE_OFFSET_VOD,
            updateInterval: ResumeWatchState.UPDATE_INTERVAL_VOD,
            isSeeked: false,
        };
        const action = {
            type: ResumeWatchActions.ACTION_VOD_POST_BACKEND_TIME,
            lastTimeStamp: 45,
        };

        assert.deepEqual(ResumeWatchState.resumeWatch(beforeState, action), {
            times: {
                [vodId]: beforeState.times[vodId],
            },
            watch: {
                [vodId]: true,
            },
            streamTimes: {
                [broadcastID]: beforeState.streamTimes[broadcastID],
            },
            lastTimeStamp: 45,
            userId: null,
            initUpdateOffset: ResumeWatchState.INIT_UPDATE_OFFSET_VOD,
            updateInterval: ResumeWatchState.UPDATE_INTERVAL_VOD,
            isSeeked: false,
        });
    });

    QUnit.test('returns the current resume watch configuration otherwise', function(assert) {
        const vodID = `v${parseInt(QUnit.config.current.testId, 36)}`;
        const broadcastID = `${Math.floor(Math.random() * 1e8)}`;
        const beforeState = {
            times: {
                [vodID]: Math.floor(Math.random() * 1e3),
            },
            watch: {
                [vodID]: true,
            },
            streamTimes: {
                [broadcastID]: Math.floor(Math.random() * 1e3),
            },
            lastTimeStamp: 45,
            userId: '12345',
            initUpdateOffset: ResumeWatchState.INIT_UPDATE_OFFSET_VOD,
            updateInterval: ResumeWatchState.UPDATE_INTERVAL_VOD,
            isSeeked: false,
        };
        const action = {
            type: 'not a real action',
        };

        assert.equal(ResumeWatchState.resumeWatch(beforeState, action), beforeState);
    });
});
