import { ACTION_SET_FULLSCREEN, ACTION_SET_CAN_FULLSCREEN, ACTION_TOGGLE_FULLSCREEN,
         ACTION_SET_THEATRE_MODE, ACTION_TOGGLE_THEATRE_MODE } from 'actions/screen-mode';
import { screenMode, DEFAULT_SCREEN_MODE } from 'state/screen-mode';
import { ACTION_SET_PROVIDED_OPTIONS } from 'actions/player-options';

QUnit.module('state | screen mode', function() {
    QUnit.test('defaults to the proper state', function(assert) {
        const action = {
            type: '@@init',
        };

        const expectedState = DEFAULT_SCREEN_MODE;

        assert.propEqual(screenMode(undefined, action), expectedState);
    });

    QUnit.test('responds to the SET_FULLSCREEN action', function(assert) {
        const enableAction = {
            type: ACTION_SET_FULLSCREEN,
            isFullScreen: true,
        };

        const disableAction = {
            type: ACTION_SET_FULLSCREEN,
            isFullScreen: false,
        };

        const beforeState = DEFAULT_SCREEN_MODE;

        let state = screenMode(beforeState, enableAction);

        assert.equal(state.isFullScreen, true, 'sets isFullScreen true fullscreen is enabled');

        state = screenMode(state, disableAction);

        assert.equal(state.isFullScreen, false, 'sets isFullScreen false when fullscreen is disabled');
    });

    QUnit.test('responds to the SET_CAN_FULLSCREEN action', function(assert) {
        const enableAction = {
            type: ACTION_SET_CAN_FULLSCREEN,
            canFullScreen: true,
        };

        const disableAction = {
            type: ACTION_SET_CAN_FULLSCREEN,
            canFullScreen: false,
        };

        const beforeState = DEFAULT_SCREEN_MODE;

        let state = screenMode(beforeState, enableAction);

        assert.equal(state.canFullScreen, true, 'sets canFullScreen true when fullscreen is available');

        state = screenMode(state, disableAction);

        assert.equal(state.canFullScreen, false, 'sets canFullScreen false when fullscreen is unavailable');
    });

    QUnit.test('responds to the TOGGLE_FULLSCREEN action if canFullScreen', function(assert) {
        const beforeState = {
            isFullScreen: false,
            canFullScreen: true,
            canTheatreMode: false,
            isTheatreMode: false,
        };
        const toggleAction = {
            type: ACTION_TOGGLE_FULLSCREEN,
        };

        const state = screenMode(beforeState, toggleAction);
        assert.equal(state.isFullScreen, true, 'toggles isFullScreen on');
    });

    QUnit.test('does not respond to the TOGGLE_FULLSCREEN action if canFullScreen is disabled', function(assert) {
        const beforeState = {
            isFullScreen: false,
            canFullScreen: false,
            canTheatreMode: false,
            isTheatreMode: false,
        };
        const toggleAction = {
            type: ACTION_TOGGLE_FULLSCREEN,
        };

        const state = screenMode(beforeState, toggleAction);
        assert.equal(state.isFullScreen, false, 'NO-OP');
    });

    QUnit.test('responds to ACTION_SET_PROVIDED_OPTIONS', function(assert) {
        const state = screenMode(DEFAULT_SCREEN_MODE, {
            type: ACTION_SET_PROVIDED_OPTIONS,
            playerOptions: {
                showtheatre: true,
            },
        });
        assert.equal(state.canTheatreMode, true);
    });

    QUnit.test('responds to the TOGGLE_THEATRE_MODE action', function(assert) {
        const beforeState = {
            isFullScreen: false,
            canFullScreen: false,
            canTheatreMode: true,
            isTheatreMode: false,
        };
        const toggleAction = {
            type: ACTION_TOGGLE_THEATRE_MODE,
        };

        const enableState = screenMode(beforeState, toggleAction);
        assert.equal(enableState.isTheatreMode, true, 'toggles isTheatreMode on');

        const disableState = screenMode(enableState, toggleAction);
        assert.equal(disableState.isTheatreMode, false, 'toggles isTheatreMode off');
    });

    QUnit.test('sets theatre mode to `false` when full screen is toggled', function(assert) {
        const beforeState = Object.freeze({
            isFullScreen: false,
            canFullScreen: true,
            canTheatreMode: true,
            isTheatreMode: false,
        });
        const toggleTheatreAction = {
            type: ACTION_TOGGLE_THEATRE_MODE,
        };
        const toggleFullScreenAction = {
            type: ACTION_TOGGLE_FULLSCREEN,
        };

        const enableTheatreState = screenMode(beforeState, toggleTheatreAction);
        assert.deepEqual(enableTheatreState, {
            isFullScreen: false,
            canFullScreen: true,
            canTheatreMode: true,
            isTheatreMode: true,
        }, 'should toggle theatre mode (theatre: true)');

        const enableFullScreenState = screenMode(enableTheatreState, toggleFullScreenAction);
        assert.deepEqual(enableFullScreenState, {
            isFullScreen: true,
            canFullScreen: true,
            canTheatreMode: true,
            isTheatreMode: true,
        }, 'should preserve theatre mode when in full screen mode (theatre: true, fullscreen: true)');

        const disableFullScreenState = screenMode(enableFullScreenState, toggleFullScreenAction);
        assert.deepEqual(disableFullScreenState, {
            isFullScreen: false,
            canFullScreen: true,
            canTheatreMode: true,
            isTheatreMode: true,
        }, 'should restore theatre mode when full screen mode is unset (theatre: true, fullscreen: false)');
    });

    // eslint-disable-next-line max-len
    QUnit.test('does not respond to the TOGGLE_THEATRE_MODE action if canTheatreMode is false', function(assert) {
        const beforeState = {
            isFullScreen: false,
            canFullScreen: false,
            canTheatreMode: false,
            isTheatreMode: false,
        };

        const toggleAction = {
            type: ACTION_TOGGLE_THEATRE_MODE,
        };

        const state = screenMode(beforeState, toggleAction);
        assert.equal(state.isTheatreMode, false, 'NO-OP');
    });

    QUnit.test('responds to the ACTION_SET_THEATRE_MODE action', function(assert) {
        const enableAction = {
            type: ACTION_SET_THEATRE_MODE,
            isTheatreMode: true,
        };

        const disableAction = {
            type: ACTION_SET_THEATRE_MODE,
            isTheatreMode: false,
        };

        const beforeState = DEFAULT_SCREEN_MODE;

        let state = screenMode(beforeState, enableAction);
        assert.equal(state.isTheatreMode, true, 'sets theatre mode to true');

        state = screenMode(state, disableAction);
        assert.equal(state.isTheatreMode, false, 'sets theatre mode to false');
    });
});
