import { screen } from 'state/screen';
import { ACTION_PUSH_SCREEN, ACTION_POP_SCREEN, CONTENT_SCREEN,
         ADVERTISEMENT_SCREEN, STITCHED_ADVERTISEMENT_SCREEN,
         VOD_RESTRICTED_SCREEN, VOD_RECOMMENDATION_SCREEN,
         ERROR_SCREEN } from 'actions/screen';
import { ACTION_SET_STREAM } from 'actions/stream';
import { ACTION_ERROR, ACTION_CLEAR_ERROR } from 'actions/error';
import { ACTION_SET_AD_METADATA, AdContentTypes, AdRollTypes } from 'actions/ads';

QUnit.module('state | screen', function() {
    QUnit.test('should return default screen mode if none is set', function(assert) {
        const action = {
            type: '@@init',
        };
        const result = screen(undefined, action);

        assert.deepEqual(result, [CONTENT_SCREEN]);
    });
    QUnit.test('should return new screen mode if one is set', function(assert) {
        const action = {
            type: ACTION_PUSH_SCREEN,
            screen: CONTENT_SCREEN,
        };
        const result = screen(undefined, action);

        assert.deepEqual(result[0], CONTENT_SCREEN);
    });
    QUnit.test('should correctly push screen mode', function(assert) {
        const action = {
            type: ACTION_PUSH_SCREEN,
            screen: ADVERTISEMENT_SCREEN,
        };
        const result = screen([CONTENT_SCREEN], action);

        assert.deepEqual(result, [ADVERTISEMENT_SCREEN, CONTENT_SCREEN]);
    });
    QUnit.test('should correctly pop screen mode', function(assert) {
        const action = {
            type: ACTION_POP_SCREEN,
        };
        const result = screen([ADVERTISEMENT_SCREEN, CONTENT_SCREEN], action);

        assert.deepEqual(result, [CONTENT_SCREEN]);
    });
    QUnit.test('should return default screen mode if pop would empty stack', function(assert) {
        const action = {
            type: ACTION_POP_SCREEN,
        };
        const result = screen([ADVERTISEMENT_SCREEN], action);

        assert.deepEqual(result, [CONTENT_SCREEN]);
    });
    QUnit.test('should pop VOD_RECOMMENDATION_SCREEN on ACTION_SET_STREAM', function(assert) {
        const action = {
            type: ACTION_SET_STREAM,
        };

        const result = screen([VOD_RECOMMENDATION_SCREEN, CONTENT_SCREEN], action);
        assert.deepEqual(result, [CONTENT_SCREEN]);
    });
    QUnit.test('should pop VOD_RESTRICTED_SCREEN on ACTION_SET_STREAM', function(assert) {
        const action = {
            type: ACTION_SET_STREAM,
        };

        const result = screen([VOD_RESTRICTED_SCREEN, CONTENT_SCREEN], action);
        assert.deepEqual(result, [CONTENT_SCREEN]);
    });
    QUnit.test('should push ERROR_SCREEN on ACTION_ERROR', function(assert) {
        const action = {
            type: ACTION_ERROR,
        };

        const result = screen([CONTENT_SCREEN], action);
        assert.deepEqual(result, [ERROR_SCREEN, CONTENT_SCREEN]);
    });
    QUnit.test('should pop ERROR_SCREEN on ACTION_CLEAR_ERROR', function(assert) {
        const action = {
            type: ACTION_CLEAR_ERROR,
        };

        const result = screen([ERROR_SCREEN, CONTENT_SCREEN], action);
        assert.deepEqual(result, [CONTENT_SCREEN]);
    });
    // eslint-disable-next-line max-len
    QUnit.test('should push ADVERTISEMENT_SCREEN when ACTION_SET_AD_METADATA is fired with IMA value', function(assert) {
        const action = {
            type: ACTION_SET_AD_METADATA,
            currentMetadata: {
                contentType: AdContentTypes.IMA,
                rollType: AdRollTypes.PREROLL,
            },
        };

        const result = screen([CONTENT_SCREEN], action);
        assert.deepEqual(result, [ADVERTISEMENT_SCREEN, CONTENT_SCREEN]);
    });
    // eslint-disable-next-line max-len
    QUnit.test('should push ADVERTISEMENT_SCREEN once when multiple ACTION_SET_AD_METADATA actions are fired with IMA value', function(assert) {
        const action = {
            type: ACTION_SET_AD_METADATA,
            currentMetadata: {
                contentType: AdContentTypes.IMA,
                rollType: AdRollTypes.PREROLL,
            },
        };

        let result = screen([CONTENT_SCREEN], action);
        result = screen(result, action);
        assert.deepEqual(result, [ADVERTISEMENT_SCREEN, CONTENT_SCREEN]);
    });
    // eslint-disable-next-line max-len
    QUnit.test('should push STITCHED_ADVERTISEMENT_SCREEN when ACTION_SET_AD_METADATA is fired with STITCHED value', function(assert) {
        const action = {
            type: ACTION_SET_AD_METADATA,
            currentMetadata: {
                contentType: AdContentTypes.STITCHED,
                rollType: AdRollTypes.MIDROLL,
            },
        };
        const result = screen([CONTENT_SCREEN], action);
        assert.deepEqual(result, [STITCHED_ADVERTISEMENT_SCREEN, CONTENT_SCREEN]);
    });
    // eslint-disable-next-line max-len
    QUnit.test('should push STITCHED_ADVERTISEMENT_SCREEN once when multiple ACTION_SET_AD_METADATA actions are fired with STITCHED value', function(assert) {
        const action = {
            type: ACTION_SET_AD_METADATA,
            currentMetadata: {
                contentType: AdContentTypes.STITCHED,
                rollType: AdRollTypes.MIDROLL,
            },
        };

        let result = screen([CONTENT_SCREEN], action);
        result = screen(result, action);
        assert.deepEqual(result, [STITCHED_ADVERTISEMENT_SCREEN, CONTENT_SCREEN]);
    });
    // eslint-disable-next-line max-len
    QUnit.test('should pop screen when ADVERTISEMENT_SCREEN is on stack and ACTION_SET_AD_METADATA is fired with NONE value', function(assert) {
        const action = {
            type: ACTION_SET_AD_METADATA,
            currentMetadata: {
                contentType: AdContentTypes.NONE,
                rollType: AdRollTypes.NONE,
            },
        };

        const result = screen([ADVERTISEMENT_SCREEN, CONTENT_SCREEN], action);
        assert.deepEqual(result, [CONTENT_SCREEN]);
    });
    // eslint-disable-next-line max-len
    QUnit.test('should pop screen when STITCHED_ADVERTISEMENT_SCREEN is on stack and ACTION_SET_AD_METADATA is fired with NONE value', function(assert) {
        const action = {
            type: ACTION_SET_AD_METADATA,
            currentMetadata: {
                contentType: AdContentTypes.NONE,
                rollType: AdRollTypes.NONE,
            },
        };

        const result = screen([STITCHED_ADVERTISEMENT_SCREEN, CONTENT_SCREEN], action);
        assert.deepEqual(result, [CONTENT_SCREEN]);
    });
    // eslint-disable-next-line max-len
    QUnit.test('should not pop screen when an ad screen is not on top of stack, and ACTION_SET_AD_METADATA is fired with NONE value', function(assert) {
        const action = {
            type: ACTION_SET_AD_METADATA,
            currentMetadata: {
                contentType: AdContentTypes.NONE,
                rollType: AdRollTypes.NONE,
            },
        };

        const result = screen([VOD_RECOMMENDATION_SCREEN, CONTENT_SCREEN], action);
        assert.deepEqual(result, [VOD_RECOMMENDATION_SCREEN, CONTENT_SCREEN]);
    });
    QUnit.test('should not add ERROR_SCREEN if VOD_RESTRICTED_SCREEN is on top of stack', function(assert) {
        const action = {
            type: ACTION_ERROR,
        };
        const screenStack = [VOD_RESTRICTED_SCREEN, CONTENT_SCREEN];
        const result = screen(screenStack, action);
        assert.deepEqual(result, screenStack);
    });
    QUnit.test('should not add ERROR_SCREEN if ERROR_SCREEN is on top of stack', function(assert) {
        const action = {
            type: ACTION_ERROR,
        };
        const screenStack = [ERROR_SCREEN, CONTENT_SCREEN];
        const result = screen(screenStack, action);
        assert.deepEqual(result, screenStack);
    });
});
