import { LiveTwitchContentStream } from 'stream/twitch-live';
import { VODTwitchContentStream } from 'stream/twitch-vod';
import { ProvidedContentStream } from 'stream/provided';
import { streamMetadata } from 'state/stream-metadata';
import { ACTION_SET_STREAM } from 'actions/stream';
import { ACTION_SET_STREAMMETADATA } from 'actions/stream-metadata';
import { DEFAULT_STREAM_METADATA } from 'tests/fixtures/stream-metadata';

QUnit.module('state | stream-metadata', function() {
    QUnit.test('should return a default streamMetadata if none is set', function(assert) {
        const action = {
            type: '@@init',
        };
        const result = streamMetadata(undefined, action);

        assert.deepEqual(result, DEFAULT_STREAM_METADATA);
    });

    QUnit.test('should update streamMetadata properly on set streammetadata action', function(assert) {
        const action = {
            type: ACTION_SET_STREAMMETADATA,
            /* eslint-disable camelcase */
            streamMetadata: {
                broadcastID: 0,
                type: '',
                channelName: 'monstercat',
                videoId: null,
                contentId: '',
                customerId: '',
                channel: {
                    displayName: 'hi',
                    id: 'id',
                    name: 'hello',
                    partner: true,
                    status: 'online',
                    videoBanner: 'banner',
                    logo: 'logo',
                    mature: false,
                },
                createdAt: 'yup',
                game: 'LoL',
                preview: {
                    template: 'https://yup.com',
                },
                streamType: 'live',
                url: 'https://www.twitch.tv/hello',
                viewers: parseInt(QUnit.config.current.testId, 36),
            },
            /* eslint-enable camelcase */
        };
        const result = streamMetadata(undefined, action);

        assert.deepEqual(result, action.streamMetadata);
    });

    QUnit.test('should update channelName when setStream is called with a live stream', function(assert) {
        const channel = `channel${QUnit.config.current.testId}`;
        const action = {
            type: ACTION_SET_STREAM,
            stream: new LiveTwitchContentStream(channel),
        };
        const result = streamMetadata(undefined, action);
        assert.equal(result.channelName, channel, 'correct channelName is set');
        assert.equal(result.videoId, null, 'no video id is set');
    });

    QUnit.test('should update videoId when setStream is called with a vod stream', function(assert) {
        const videoId = `v${parseInt(QUnit.config.current.testId, 36)}`;
        const action = {
            type: ACTION_SET_STREAM,
            stream: new VODTwitchContentStream(videoId),
        };
        const result = streamMetadata(undefined, action);
        assert.equal(result.channelName, null, 'no channelName is set');
        assert.equal(result.videoId, videoId, 'correct video id is set');
    });

    // eslint-disable-next-line max-len
    QUnit.test('should update contentId and customerId when setStream is called with a provided content stream', function(assert) {
        const contentId = 'acontentid';
        const customerId = 'acustomerid';
        const action = {
            type: ACTION_SET_STREAM,
            stream: new ProvidedContentStream({
                customerId,
                contentId,
            }),
        };
        const result = streamMetadata(undefined, action);
        assert.equal(result.channelName, null, 'no channelName is set');
        assert.equal(result.videoId, null, 'no video id is set');
        assert.equal(result.contentId, contentId, 'correct content id is set');
        assert.equal(result.customerId, customerId, 'correct customer id is set');
    });

    // eslint-disable-next-line max-len
    QUnit.test('should set contentId and customerId to blank string when setStream is called with a non provided content stream', function(assert) {
        const action = {
            type: ACTION_SET_STREAM,
            stream: new VODTwitchContentStream('a vod id'),
        };
        const result = streamMetadata(undefined, action);
        assert.equal(result.contentId, '', 'empty string content id is set');
        assert.equal(result.customerId, '', 'empty string customer id is set');
    });
});
