import { ACTION_SET_MUTED_SEGMENTS, ACTION_SET_PREVIEWS, ACTION_SET_MARKERS } from 'actions/timeline-metadata';
import { timelineMetadata } from 'state/timeline-metadata';
import { ACTION_SET_STREAM } from 'actions/stream';

const DEFAULT_TIMELINE_METADATA = {
    mutedSegments: [],
    previews: {
        count: 0,
        lq: {
            width: 0,
            height: 0,
            rows: 0,
            cols: 0,
            URLs: [''],
        },
        hq: {
            width: 0,
            height: 0,
            rows: 0,
            cols: 0,
            URLs: [''],
        },
    },
    markers: [],
};

const TEST_PREVIEWS_1 = {
    count: 10,
    lq: {
        width: 2,
        height: 3,
        rows: 4,
        cols: 5,
        URLs: ['testURL1'],
    },
    hq: {
        width: 1,
        height: 2,
        rows: 3,
        cols: 4,
        URLs: ['testURL2'],
    },
};

const TEST_PREVIEWS_2 = {
    count: 2,
    lq: {
        width: 1,
        height: 2,
        rows: 3,
        cols: 5,
        URLs: ['atestURL'],
    },
    hq: {
        width: 23,
        height: 4,
        rows: 356,
        cols: 2,
        URLs: ['anotherTestURL'],
    },
};

const TEST_MUTED_SEGMENTS_1 = [
    {
        offset: 15,
        duration: 5,
    },
    {
        offset: 52,
        duration: 5,
    },
];

const TEST_MUTED_SEGMENTS_2 = [
    {
        offset: 10,
        duration: 2,
    },
];

const TEST_MARKERS_1 = [
    {
        info: '<strong>Druid</strong> vs Druid',
        startTime: 924.998,
        title: 'Match 1',
        thumbnail: {
            imageURL: 'testimageurl1',
            x: 0,
            y: 0,
            width: 0,
            height: 0,
            cols: 0,
        },
    },
];

const TEST_MARKERS_2 = [
    {
        info: '<strong>Druid</strong> vs Druid',
        startTime: 924.998,
        title: 'Match 1',
        thumbnail: {
            imageURL: 'testimageurl2',
            x: 1,
            y: 1,
            width: 1,
            height: 1,
            cols: 1,
        },
    },
    {
        info: '<strong>Druid</strong> vs Rogue',
        startTime: 1397,
        title: 'Match 2',
        thumbnail: {
            imageURL: 'testimageurl3',
            x: 2,
            y: 2,
            width: 2,
            height: 2,
            cols: 2,
        },
    },
];

QUnit.module('state | timelineMetadata', function() {
    QUnit.test('should default to default timelineMetadata if none is set', function(assert) {
        const action = {
            type: '@@init',
        };
        const result = timelineMetadata(undefined, action);

        assert.deepEqual(result, DEFAULT_TIMELINE_METADATA);
    });

    QUnit.test('should reset to default timelineMetadata on Set Stream action', function(assert) {
        const action = {
            type: ACTION_SET_STREAM,
        };

        const beforeState = {
            mutedSegments: TEST_MUTED_SEGMENTS_1,
            previews: TEST_PREVIEWS_1,
            markers: TEST_MARKERS_1,
        };

        assert.deepEqual(timelineMetadata(beforeState, action), DEFAULT_TIMELINE_METADATA);
    });

    QUnit.module('mutedSegments', function() {
        QUnit.test('responds to the Set Muted Segments action', function(assert) {
            const action = {
                type: ACTION_SET_MUTED_SEGMENTS,
                mutedSegments: TEST_MUTED_SEGMENTS_2,
            };

            const beforeState = {
                mutedSegments: TEST_MUTED_SEGMENTS_1,
                previews: TEST_PREVIEWS_1,
                markers: TEST_MARKERS_1,
            };

            assert.deepEqual(timelineMetadata(beforeState, action), {
                mutedSegments: TEST_MUTED_SEGMENTS_2,
                previews: beforeState.previews,
                markers: beforeState.markers,
            });
        });

        QUnit.test('returns the current muted segments otherwise', function(assert) {
            const action = {
                type: 'not a real action',
            };

            const beforeState = {
                mutedSegments: TEST_MUTED_SEGMENTS_1,
                previews: TEST_PREVIEWS_1,
                markers: TEST_MARKERS_1,
            };

            assert.deepEqual(timelineMetadata(beforeState, action), beforeState);
        });
    });

    QUnit.module('previews', function() {
        QUnit.test('responds to the Set Previews action', function(assert) {
            const action = {
                type: ACTION_SET_PREVIEWS,
                previews: TEST_PREVIEWS_2,
            };

            const beforeState = {
                mutedSegments: TEST_MUTED_SEGMENTS_1,
                previews: TEST_PREVIEWS_1,
                markers: TEST_MARKERS_1,
            };

            assert.deepEqual(timelineMetadata(beforeState, action), {
                mutedSegments: beforeState.mutedSegments,
                previews: TEST_PREVIEWS_2,
                markers: beforeState.markers,
            });
        });

        QUnit.test('returns the current previews otherwise', function(assert) {
            const action = {
                type: 'not a real action',
            };

            const beforeState = {
                mutedSegments: TEST_MUTED_SEGMENTS_1,
                previews: TEST_PREVIEWS_1,
            };

            assert.deepEqual(timelineMetadata(beforeState, action), beforeState);
        });
    });

    QUnit.module('markers', function() {
        QUnit.test('responds to the Set Markers action', function(assert) {
            const action = {
                type: ACTION_SET_MARKERS,
                markers: TEST_MARKERS_2,
            };

            const beforeState = {
                mutedSegments: TEST_MUTED_SEGMENTS_1,
                previews: TEST_PREVIEWS_1,
                markers: TEST_MARKERS_1,
            };

            assert.deepEqual(timelineMetadata(beforeState, action), {
                mutedSegments: beforeState.mutedSegments,
                previews: beforeState.previews,
                markers: TEST_MARKERS_2,
            });
        });

        QUnit.test('returns the current markers otherwise', function(assert) {
            const action = {
                type: 'not a real action',
            };

            const beforeState = {
                mutedSegments: TEST_MUTED_SEGMENTS_1,
                previews: TEST_PREVIEWS_1,
                markers: TEST_MARKERS_1,
            };

            assert.deepEqual(timelineMetadata(beforeState, action), beforeState);
        });
    });
});
