import { stringify } from 'query-string';
import { unitTest } from 'tests/utils/module';
import { CONTENT_MODE_PROVIDED, ProvidedContentStream } from 'stream/provided';
import { usherHost as USHER_HOST } from 'settings';
import * as DeviceId from 'util/device-id';
import sinon from 'sinon';

unitTest('stream | provided', function(hooks) {
    hooks.beforeEach(function() {
        this.expectedDeviceId = 'TheExpectedDeviceId12345';
        sinon.stub(DeviceId, 'get').withArgs(false).returns(this.expectedDeviceId);
        this.contentUrl = 'afakeurl';
        this.contentId = 'acontentid';
        this.customerId = 'acustomerid';
        this.stream = new ProvidedContentStream({
            customerId: this.customerId,
            contentId: this.contentId,
        });
    });

    hooks.afterEach(function() {
        DeviceId.get.restore();
    });

    QUnit.test('contentType should be provided', function(assert) {
        assert.equal(this.stream.contentType, CONTENT_MODE_PROVIDED);
    });

    QUnit.test('channel should return undefined', function(assert) {
        assert.equal(this.stream.channel, undefined);
    });

    QUnit.test('videoId should return undefined', function(assert) {
        assert.equal(this.stream.videoId, undefined);
    });

    QUnit.test('accessToken is promise that resolves with nothing', function(assert) {
        return this.stream.accessToken.then(function(value) {
            assert.equal(value, undefined);
        });
    });

    QUnit.test('contentId is the same as the provided contentId', function(assert) {
        assert.equal(this.stream.contentId, this.contentId);
    });

    QUnit.test('customerId is the same as the provided customerId', function(assert) {
        assert.equal(this.stream.customerId, this.customerId);
    });

    QUnit.module('streamUrl', function(hooks) {
        hooks.beforeEach(function() {
            this.usherParams = {
                /* eslint-disable camelcase */
                allow_source: true,
                device_id: this.expectedDeviceId,
                /* eslint-enable camelcase */
            };
        });

        QUnit.test('streamUrl returns properly formed base url with the usher params', function(assert) {
            const expectedBaseUrl = `${USHER_HOST}/api/lvs/hls/lvs.${this.customerId}.${this.contentId}.m3u8`;

            return this.stream.streamUrl.then(streamUrl => {
                assert.equal(streamUrl, `${expectedBaseUrl}?${stringify(this.usherParams)}`);
            });
        });

        QUnit.test('streamUrl uses usherHost override if provided during initialization', function(assert) {
            const override = 'newhost';
            const stream = new ProvidedContentStream({
                customerId: this.customerId,
                contentId: this.contentId,
                usherHostOverride: override,
            });

            const expectedBaseUrl = `${override}/api/lvs/hls/lvs.${this.customerId}.${this.contentId}.m3u8`;
            return stream.streamUrl.then(streamUrl => {
                assert.equal(streamUrl, `${expectedBaseUrl}?${stringify(this.usherParams)}`);
            });
        });
    });

    // eslint-disable-next-line max-len
    QUnit.test('castStreamUrl returns properly formed base url with usher params and allow_source removed', function(assert) {
        const params = this.stream._buildUsherParams();
        // eslint-disable-next-line camelcase
        params.allow_source = undefined;
        const expectedBaseUrl = `${USHER_HOST}/api/lvs/hls/lvs.${this.customerId}.${this.contentId}.m3u8`;

        return this.stream.castStreamUrl.then(castStreamUrl => {
            assert.equal(castStreamUrl, `${expectedBaseUrl}?${stringify(params)}`);
        });
    });
});
