import { AbstractStream } from 'stream/stream';
import { unitTest } from 'tests/utils/module';
import sinon from 'sinon';

unitTest('stream | stream', function(hooks) {
    hooks.beforeEach(function() {
        this.contentId = `some_id_${parseInt(QUnit.config.current.testId, 16)}`;
        this.oauthToken = `oauth_${QUnit.config.current.testId}`;
        this.usherParams = {
            cake: false,
        };
        this.accessTokenParams = {
            // eslint-disable-next-line camelcase
            need_https: true,
        };
        this.experimentSettings = {
            someExperiment: Promise.resolve('true'),
        };
        this.stream = new AbstractStream(
            this.contentId,
            Promise.resolve(this.oauthToken),
            this.usherParams,
            this.accessTokenParams,
            this.experimentSettings
        );
    });

    QUnit.module('placeholders', function(hooks) {
        const stringPlaceholders = [
            '_baseStreamUrl',
            'channel',
            'contentType',
            'contentId',
            'customerId',
            'streamUrl',
            'videoId',
        ];

        hooks.beforeEach(function() {
            sinon.stub(this.stream, '_handleInvalidOp');
        });

        hooks.afterEach(function() {
            this.stream._handleInvalidOp.restore();
        });

        stringPlaceholders.forEach(property => {
            QUnit.test(`${property} should call invalid handler and return empty string`, function(assert) {
                assert.equal(this.stream[property], '');
                assert.ok(this.stream._handleInvalidOp.calledOnce);
            });
        });

        QUnit.test('_fetchAccessToken calls invalid handler and return a promise with an empty obj', function(assert) {
            return this.stream._fetchAccessToken().then(accessToken => {
                assert.deepEqual(accessToken, {});
                assert.ok(this.stream._handleInvalidOp.calledOnce);
            });
        });

        QUnit.test('_buildUsherParams calls invalid handler and return an empty obj', function(assert) {
            assert.deepEqual(this.stream._buildUsherParams(), {});
            assert.ok(this.stream._handleInvalidOp.calledOnce);
        });

        QUnit.test('accessToken calls invalid handler and return a promise with an empty obj', function(assert) {
            const nAuthPromise = this.stream.accessToken;
            return nAuthPromise.then(nAuthToken => {
                assert.deepEqual(nAuthToken, {});
                assert.ok(this.stream._handleInvalidOp.calledOnce);
            });
        });

        QUnit.test('castStreamUrl calls invalid handler and return an empty string', function(assert) {
            const castStreamUrlPromise = this.stream.castStreamUrl;
            return castStreamUrlPromise.then(castUrl => {
                assert.equal(castUrl, '');
                // Once for nAuth token, once for castStreamUrl
                assert.ok(this.stream._handleInvalidOp.calledTwice);
            });
        });
    });
});
